/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.cassandra;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.JdkSSLOptions;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.SSLOptions;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.TypeCodec;
import com.datastax.driver.core.exceptions.AuthenticationException;
import com.datastax.driver.core.exceptions.NoHostAvailableException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import javax.net.ssl.SSLContext;
import org.apache.avro.Schema;
import org.apache.avro.SchemaBuilder;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.cassandra.CassandraSessionProviderService;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.security.util.SslContextFactory;
import org.apache.nifi.ssl.SSLContextService;

public abstract class AbstractCassandraProcessor
extends AbstractProcessor {
    public static final int DEFAULT_CASSANDRA_PORT = 9042;
    static final PropertyDescriptor CONNECTION_PROVIDER_SERVICE = new PropertyDescriptor.Builder().name("cassandra-connection-provider").displayName("Cassandra Connection Provider").description("Specifies the Cassandra connection providing controller service to be used to connect to Cassandra cluster.").required(false).identifiesControllerService(CassandraSessionProviderService.class).build();
    static final PropertyDescriptor CONTACT_POINTS = new PropertyDescriptor.Builder().name("Cassandra Contact Points").description("Contact points are addresses of Cassandra nodes. The list of contact points should be comma-separated and in hostname:port format. Example node1:port,node2:port,.... The default client port for Cassandra is 9042, but the port(s) must be explicitly specified.").required(false).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).addValidator(StandardValidators.HOSTNAME_PORT_LIST_VALIDATOR).build();
    static final PropertyDescriptor KEYSPACE = new PropertyDescriptor.Builder().name("Keyspace").description("The Cassandra Keyspace to connect to. If no keyspace is specified, the query will need to include the keyspace name before any table reference, in case of 'query' native processors or if the processor exposes the 'Table' property, the keyspace name has to be provided with the table name in the form of <KEYSPACE>.<TABLE>").required(false).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    static final PropertyDescriptor PROP_SSL_CONTEXT_SERVICE = new PropertyDescriptor.Builder().name("SSL Context Service").description("The SSL Context Service used to provide client certificate information for TLS/SSL connections.").required(false).identifiesControllerService(SSLContextService.class).build();
    static final PropertyDescriptor CLIENT_AUTH = new PropertyDescriptor.Builder().name("Client Auth").description("Client authentication policy when connecting to secure (TLS/SSL) cluster. Possible values are REQUIRED, WANT, NONE. This property is only used when an SSL Context has been defined and enabled.").required(false).allowableValues((Enum[])SSLContextService.ClientAuth.values()).defaultValue("REQUIRED").build();
    static final PropertyDescriptor USERNAME = new PropertyDescriptor.Builder().name("Username").description("Username to access the Cassandra cluster").required(false).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    static final PropertyDescriptor PASSWORD = new PropertyDescriptor.Builder().name("Password").description("Password to access the Cassandra cluster").required(false).sensitive(true).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    static final PropertyDescriptor CONSISTENCY_LEVEL = new PropertyDescriptor.Builder().name("Consistency Level").description("The strategy for how many replicas must respond before results are returned.").required(false).allowableValues((Enum[])ConsistencyLevel.values()).defaultValue("ONE").build();
    static final PropertyDescriptor CHARSET = new PropertyDescriptor.Builder().name("Character Set").description("Specifies the character set of the record data.").required(true).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).defaultValue("UTF-8").addValidator(StandardValidators.CHARACTER_SET_VALIDATOR).build();
    static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("A FlowFile is transferred to this relationship if the operation completed successfully.").build();
    static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("A FlowFile is transferred to this relationship if the operation failed.").build();
    static final Relationship REL_RETRY = new Relationship.Builder().name("retry").description("A FlowFile is transferred to this relationship if the operation cannot be completed but attempting it again may succeed.").build();
    protected static List<PropertyDescriptor> descriptors = new ArrayList<PropertyDescriptor>();
    protected final AtomicReference<Cluster> cluster = new AtomicReference<Object>(null);
    protected final AtomicReference<Session> cassandraSession = new AtomicReference<Object>(null);
    protected static final CodecRegistry codecRegistry;

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        HashSet<ValidationResult> results = new HashSet<ValidationResult>();
        String userName = validationContext.getProperty(USERNAME).evaluateAttributeExpressions().getValue();
        String password = validationContext.getProperty(PASSWORD).evaluateAttributeExpressions().getValue();
        if (StringUtils.isEmpty((CharSequence)userName) != StringUtils.isEmpty((CharSequence)password)) {
            results.add(new ValidationResult.Builder().subject("Username / Password configuration").valid(false).explanation("If username or password is specified, then the other must be specified as well").build());
        }
        boolean connectionProviderIsSet = validationContext.getProperty(CONNECTION_PROVIDER_SERVICE).isSet();
        boolean contactPointsIsSet = validationContext.getProperty(CONTACT_POINTS).isSet();
        if (connectionProviderIsSet && contactPointsIsSet) {
            results.add(new ValidationResult.Builder().subject("Cassandra configuration").valid(false).explanation("both " + CONNECTION_PROVIDER_SERVICE.getDisplayName() + " and processor level Cassandra configuration cannot be provided at the same time.").build());
        }
        if (!connectionProviderIsSet && !contactPointsIsSet) {
            results.add(new ValidationResult.Builder().subject("Cassandra configuration").valid(false).explanation("either " + CONNECTION_PROVIDER_SERVICE.getDisplayName() + " or processor level Cassandra configuration has to be provided.").build());
        }
        return results;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        boolean connectionProviderIsSet = context.getProperty(CONNECTION_PROVIDER_SERVICE).isSet();
        if (connectionProviderIsSet) {
            CassandraSessionProviderService sessionProvider = (CassandraSessionProviderService)context.getProperty(CONNECTION_PROVIDER_SERVICE).asControllerService(CassandraSessionProviderService.class);
            this.cluster.set(sessionProvider.getCluster());
            this.cassandraSession.set(sessionProvider.getCassandraSession());
            return;
        }
        try {
            this.connectToCassandra(context);
        }
        catch (NoHostAvailableException nhae) {
            this.getLogger().error("No host in the Cassandra cluster can be contacted successfully to execute this statement", (Throwable)nhae);
            this.getLogger().error(nhae.getCustomMessage(10, true, false));
            throw new ProcessException((Throwable)nhae);
        }
        catch (AuthenticationException ae) {
            this.getLogger().error("Invalid username/password combination", (Throwable)ae);
            throw new ProcessException((Throwable)ae);
        }
    }

    void connectToCassandra(ProcessContext context) {
        if (this.cluster.get() == null) {
            String password;
            String username;
            SSLContext sslContext;
            ComponentLog log = this.getLogger();
            String contactPointList = context.getProperty(CONTACT_POINTS).evaluateAttributeExpressions().getValue();
            String consistencyLevel = context.getProperty(CONSISTENCY_LEVEL).getValue();
            List<InetSocketAddress> contactPoints = this.getContactPoints(contactPointList);
            SSLContextService sslService = (SSLContextService)context.getProperty(PROP_SSL_CONTEXT_SERVICE).asControllerService(SSLContextService.class);
            String rawClientAuth = context.getProperty(CLIENT_AUTH).getValue();
            if (sslService != null) {
                SSLContextService.ClientAuth clientAuth;
                if (StringUtils.isBlank((CharSequence)rawClientAuth)) {
                    clientAuth = SSLContextService.ClientAuth.REQUIRED;
                } else {
                    try {
                        clientAuth = SSLContextService.ClientAuth.valueOf((String)rawClientAuth);
                    }
                    catch (IllegalArgumentException iae) {
                        throw new IllegalStateException(String.format("Unrecognized client auth '%s'. Possible values are [%s]", rawClientAuth, StringUtils.join((Object[])SslContextFactory.ClientAuth.values(), (String)", ")));
                    }
                }
                sslContext = sslService.createSSLContext(clientAuth);
            } else {
                sslContext = null;
            }
            PropertyValue usernameProperty = context.getProperty(USERNAME).evaluateAttributeExpressions();
            PropertyValue passwordProperty = context.getProperty(PASSWORD).evaluateAttributeExpressions();
            if (usernameProperty != null && passwordProperty != null) {
                username = usernameProperty.getValue();
                password = passwordProperty.getValue();
            } else {
                username = null;
                password = null;
            }
            Cluster newCluster = this.createCluster(contactPoints, sslContext, username, password);
            PropertyValue keyspaceProperty = context.getProperty(KEYSPACE).evaluateAttributeExpressions();
            Session newSession = keyspaceProperty != null ? newCluster.connect(keyspaceProperty.getValue()) : newCluster.connect();
            newCluster.getConfiguration().getQueryOptions().setConsistencyLevel(ConsistencyLevel.valueOf((String)consistencyLevel));
            Metadata metadata = newCluster.getMetadata();
            log.info("Connected to Cassandra cluster: {}", new Object[]{metadata.getClusterName()});
            this.cluster.set(newCluster);
            this.cassandraSession.set(newSession);
        }
    }

    protected Cluster createCluster(List<InetSocketAddress> contactPoints, SSLContext sslContext, String username, String password) {
        Cluster.Builder builder = Cluster.builder().addContactPointsWithPorts(contactPoints);
        if (sslContext != null) {
            JdkSSLOptions sslOptions = JdkSSLOptions.builder().withSSLContext(sslContext).build();
            builder = builder.withSSL((SSLOptions)sslOptions);
        }
        if (username != null && password != null) {
            builder = builder.withCredentials(username, password);
        }
        return builder.build();
    }

    public void stop(ProcessContext context) {
        if (!context.getProperty(CONNECTION_PROVIDER_SERVICE).isSet()) {
            if (this.cassandraSession.get() != null) {
                this.cassandraSession.get().close();
                this.cassandraSession.set(null);
            }
            if (this.cluster.get() != null) {
                this.cluster.get().close();
                this.cluster.set(null);
            }
        }
    }

    protected static Object getCassandraObject(Row row, int i, DataType dataType) {
        if (dataType.equals(DataType.blob())) {
            return row.getBytes(i);
        }
        if (dataType.equals(DataType.varint()) || dataType.equals(DataType.decimal())) {
            return row.getObject(i).toString();
        }
        if (dataType.equals(DataType.cboolean())) {
            return row.getBool(i);
        }
        if (dataType.equals(DataType.cint())) {
            return row.getInt(i);
        }
        if (dataType.equals(DataType.bigint()) || dataType.equals(DataType.counter())) {
            return row.getLong(i);
        }
        if (dataType.equals(DataType.ascii()) || dataType.equals(DataType.text()) || dataType.equals(DataType.varchar())) {
            return row.getString(i);
        }
        if (dataType.equals(DataType.cfloat())) {
            return Float.valueOf(row.getFloat(i));
        }
        if (dataType.equals(DataType.cdouble())) {
            return row.getDouble(i);
        }
        if (dataType.equals(DataType.timestamp())) {
            return row.getTimestamp(i);
        }
        if (dataType.equals(DataType.date())) {
            return row.getDate(i);
        }
        if (dataType.equals(DataType.time())) {
            return row.getTime(i);
        }
        if (dataType.isCollection()) {
            List typeArguments = dataType.getTypeArguments();
            if (typeArguments == null || typeArguments.size() == 0) {
                throw new IllegalArgumentException("Column[" + i + "] " + dataType.getName() + " is a collection but no type arguments were specified!");
            }
            DataType firstArg = (DataType)typeArguments.get(0);
            TypeCodec firstCodec = codecRegistry.codecFor(firstArg);
            if (dataType.equals(DataType.set((DataType)firstArg))) {
                return row.getSet(i, firstCodec.getJavaType());
            }
            if (dataType.equals(DataType.list((DataType)firstArg))) {
                return row.getList(i, firstCodec.getJavaType());
            }
            DataType secondArg = (DataType)typeArguments.get(1);
            TypeCodec secondCodec = codecRegistry.codecFor(secondArg);
            if (dataType.equals(DataType.map((DataType)firstArg, (DataType)secondArg))) {
                return row.getMap(i, firstCodec.getJavaType(), secondCodec.getJavaType());
            }
        } else {
            return row.getObject(i).toString();
        }
        return null;
    }

    protected static Schema getUnionFieldType(String dataType) {
        return (Schema)((SchemaBuilder.UnionAccumulator)((SchemaBuilder.UnionAccumulator)SchemaBuilder.builder().unionOf().nullBuilder().endNull()).and().type(AbstractCassandraProcessor.getSchemaForType(dataType))).endUnion();
    }

    protected static Schema getSchemaForType(String dataType) {
        Schema returnSchema;
        SchemaBuilder.TypeBuilder typeBuilder = SchemaBuilder.builder();
        switch (dataType) {
            case "string": {
                returnSchema = (Schema)typeBuilder.stringType();
                break;
            }
            case "boolean": {
                returnSchema = (Schema)typeBuilder.booleanType();
                break;
            }
            case "int": {
                returnSchema = (Schema)typeBuilder.intType();
                break;
            }
            case "long": {
                returnSchema = (Schema)typeBuilder.longType();
                break;
            }
            case "float": {
                returnSchema = (Schema)typeBuilder.floatType();
                break;
            }
            case "double": {
                returnSchema = (Schema)typeBuilder.doubleType();
                break;
            }
            case "bytes": {
                returnSchema = (Schema)typeBuilder.bytesType();
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown Avro primitive type: " + dataType);
            }
        }
        return returnSchema;
    }

    protected static String getPrimitiveAvroTypeFromCassandraType(DataType dataType) {
        if (dataType.equals(DataType.ascii()) || dataType.equals(DataType.text()) || dataType.equals(DataType.varchar()) || dataType.equals(DataType.timestamp()) || dataType.equals(DataType.timeuuid()) || dataType.equals(DataType.uuid()) || dataType.equals(DataType.inet()) || dataType.equals(DataType.varint())) {
            return "string";
        }
        if (dataType.equals(DataType.cboolean())) {
            return "boolean";
        }
        if (dataType.equals(DataType.cint())) {
            return "int";
        }
        if (dataType.equals(DataType.bigint()) || dataType.equals(DataType.counter())) {
            return "long";
        }
        if (dataType.equals(DataType.cfloat())) {
            return "float";
        }
        if (dataType.equals(DataType.cdouble())) {
            return "double";
        }
        if (dataType.equals(DataType.blob())) {
            return "bytes";
        }
        throw new IllegalArgumentException("createSchema: Unknown Cassandra data type " + dataType.getName() + " cannot be converted to Avro type");
    }

    protected static DataType getPrimitiveDataTypeFromString(String dataTypeName) {
        Set primitiveTypes = DataType.allPrimitiveTypes();
        for (DataType primitiveType : primitiveTypes) {
            if (!primitiveType.toString().equals(dataTypeName)) continue;
            return primitiveType;
        }
        return null;
    }

    protected List<InetSocketAddress> getContactPoints(String contactPointList) {
        if (contactPointList == null) {
            return null;
        }
        List<String> contactPointStringList = Arrays.asList(contactPointList.split(","));
        ArrayList<InetSocketAddress> contactPoints = new ArrayList<InetSocketAddress>();
        for (String contactPointEntry : contactPointStringList) {
            String[] addresses = contactPointEntry.split(":");
            String hostName = addresses[0].trim();
            int port = addresses.length > 1 ? Integer.parseInt(addresses[1].trim()) : 9042;
            contactPoints.add(new InetSocketAddress(hostName, port));
        }
        return contactPoints;
    }

    static {
        descriptors.add(CONNECTION_PROVIDER_SERVICE);
        descriptors.add(CONTACT_POINTS);
        descriptors.add(KEYSPACE);
        descriptors.add(PROP_SSL_CONTEXT_SERVICE);
        descriptors.add(CLIENT_AUTH);
        descriptors.add(USERNAME);
        descriptors.add(PASSWORD);
        descriptors.add(CONSISTENCY_LEVEL);
        descriptors.add(CHARSET);
        codecRegistry = new CodecRegistry();
    }
}

