/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.persistence;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FlowConfigurationArchiveManager {
    private static final Logger logger = LoggerFactory.getLogger(FlowConfigurationArchiveManager.class);
    private final Pattern archiveFilenamePattern = Pattern.compile("^([\\d]{8}T[\\d]{6}([\\+\\-][\\d]{4}|Z))_.+$");
    private final Path flowFile;
    private final Path archiveDir;
    private final long maxTimeMillis;
    private final long maxStorageBytes;

    public FlowConfigurationArchiveManager(Path flowFile, NiFiProperties properties) {
        String archiveDirVal = properties.getFlowConfigurationArchiveDir();
        Path archiveDir = archiveDirVal == null || archiveDirVal.equals("") ? flowFile.getParent().resolve("archive") : new File(archiveDirVal).toPath();
        long archiveMaxTime = FormatUtils.getTimeDuration((String)properties.getFlowConfigurationArchiveMaxTime(), (TimeUnit)TimeUnit.MILLISECONDS);
        long archiveMaxStorage = DataUnit.parseDataSize((String)properties.getFlowConfigurationArchiveMaxStorage(), (DataUnit)DataUnit.B).longValue();
        this.flowFile = flowFile;
        this.archiveDir = archiveDir;
        this.maxTimeMillis = archiveMaxTime;
        this.maxStorageBytes = archiveMaxStorage;
    }

    public FlowConfigurationArchiveManager(Path flowFile, Path archiveDir, long maxTimeMillis, long maxStorageBytes) {
        this.flowFile = flowFile;
        this.archiveDir = archiveDir;
        this.maxTimeMillis = maxTimeMillis;
        this.maxStorageBytes = maxStorageBytes;
    }

    private String createArchiveFileName(String originalFlowFileName) {
        TimeZone tz = TimeZone.getDefault();
        Calendar cal = GregorianCalendar.getInstance(tz);
        int offsetInMillis = tz.getOffset(cal.getTimeInMillis());
        int year = cal.get(1);
        int month = cal.get(2) + 1;
        int day = cal.get(5);
        int hour = cal.get(11);
        int min = cal.get(12);
        int sec = cal.get(13);
        String offset = String.format("%s%02d%02d", offsetInMillis >= 0 ? "+" : "-", Math.abs(offsetInMillis / 3600000), Math.abs(offsetInMillis / 60000 % 60));
        return String.format("%d%02d%02dT%02d%02d%02d%s_%s", year, month, day, hour, min, sec, offset, originalFlowFileName);
    }

    public File setupArchiveFile() throws IOException {
        Files.createDirectories(this.archiveDir, new FileAttribute[0]);
        if (!Files.isDirectory(this.archiveDir, new LinkOption[0])) {
            throw new IOException("Archive directory doesn't appear to be a directory " + this.archiveDir);
        }
        String originalFlowFileName = this.flowFile.getFileName().toString();
        Path archiveFile = this.archiveDir.resolve(this.createArchiveFileName(originalFlowFileName));
        return archiveFile.toFile();
    }

    public File archive() throws IOException {
        String originalFlowFileName = this.flowFile.getFileName().toString();
        File archiveFile = this.setupArchiveFile();
        Files.copy(this.flowFile, archiveFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        long now = System.currentTimeMillis();
        Map oldArchives = Files.walk(this.archiveDir, 1, new FileVisitOption[0]).filter(p -> {
            Matcher matcher;
            String filename = p.getFileName().toString();
            return Files.isRegularFile(p, new LinkOption[0]) && filename.endsWith("_" + originalFlowFileName) && (matcher = this.archiveFilenamePattern.matcher(filename)).matches() && filename.equals(matcher.group(1) + "_" + originalFlowFileName);
        }).collect(Collectors.groupingBy(p -> now - p.toFile().lastModified() > this.maxTimeMillis, Collectors.toList()));
        logger.debug("oldArchives={}", oldArchives);
        List expiredArchives = oldArchives.get(true);
        if (expiredArchives != null) {
            expiredArchives.stream().forEach(p -> {
                try {
                    logger.info("Removing expired archive file {}", p);
                    Files.delete(p);
                }
                catch (IOException e) {
                    logger.warn("Failed to delete expired archive {} due to {}", p, (Object)e.toString());
                }
            });
        }
        List remainingArchives = oldArchives.get(false);
        long totalArchiveSize = remainingArchives.stream().mapToLong(p -> {
            try {
                return Files.size(p);
            }
            catch (IOException e) {
                logger.warn("Failed to get file size of {} due to {}", p, (Object)e.toString());
                return 0L;
            }
        }).sum();
        logger.debug("totalArchiveSize={}", (Object)totalArchiveSize);
        remainingArchives.sort((a, b) -> Long.valueOf(a.toFile().lastModified()).compareTo(b.toFile().lastModified()));
        long reducedTotalArchiveSize = totalArchiveSize;
        for (int i = 0; i < remainingArchives.size() && reducedTotalArchiveSize > this.maxStorageBytes; ++i) {
            Path path = (Path)remainingArchives.get(i);
            try {
                logger.info("Removing archive file {} to reduce storage usage. currentSize={}", (Object)path, (Object)reducedTotalArchiveSize);
                long size = Files.size(path);
                Files.delete(path);
                reducedTotalArchiveSize -= size;
                continue;
            }
            catch (IOException e) {
                logger.warn("Failed to delete {} to reduce storage usage, due to {}", (Object)path, (Object)e.toString());
            }
        }
        return archiveFile;
    }
}

