/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller.repository;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.controller.repository.ContentNotFoundException;
import org.apache.nifi.controller.repository.ContentRepository;
import org.apache.nifi.controller.repository.RepositoryPurgeException;
import org.apache.nifi.controller.repository.claim.ContentClaim;
import org.apache.nifi.controller.repository.claim.ResourceClaim;
import org.apache.nifi.controller.repository.claim.ResourceClaimManager;
import org.apache.nifi.controller.repository.claim.StandardContentClaim;
import org.apache.nifi.controller.repository.io.LimitedInputStream;
import org.apache.nifi.engine.FlowEngine;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.stream.io.ByteCountingOutputStream;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.stream.io.SynchronizedByteCountingOutputStream;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.util.StopWatch;
import org.apache.nifi.util.file.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemRepository
implements ContentRepository {
    public static final int SECTIONS_PER_CONTAINER = 1024;
    public static final long MIN_CLEANUP_INTERVAL_MILLIS = 1000L;
    public static final String ARCHIVE_DIR_NAME = "archive";
    public static final String APPENDABLE_CLAIM_LENGTH_CAP = "100 MB";
    public static final Pattern MAX_ARCHIVE_SIZE_PATTERN = Pattern.compile("\\d{1,2}%");
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemRepository.class);
    private final Logger archiveExpirationLog = LoggerFactory.getLogger((String)(FileSystemRepository.class.getName() + ".archive.expiration"));
    private final Map<String, Path> containers;
    private final List<String> containerNames;
    private final AtomicLong index;
    private final ScheduledExecutorService executor = new FlowEngine(4, "FileSystemRepository Workers", true);
    private final ConcurrentMap<String, BlockingQueue<ResourceClaim>> reclaimable = new ConcurrentHashMap<String, BlockingQueue<ResourceClaim>>();
    private final Map<String, ContainerState> containerStateMap = new HashMap<String, ContainerState>();
    private final BlockingQueue<ClaimLengthPair> writableClaimQueue;
    private final ConcurrentMap<ResourceClaim, ByteCountingOutputStream> writableClaimStreams = new ConcurrentHashMap<ResourceClaim, ByteCountingOutputStream>(100);
    private final boolean archiveData;
    private final long maxAppendableClaimLength;
    private final int maxFlowFilesPerClaim;
    private final long maxArchiveMillis;
    private final Map<String, Long> minUsableContainerBytesForArchive = new HashMap<String, Long>();
    private final boolean alwaysSync;
    private final ScheduledExecutorService containerCleanupExecutor;
    private ResourceClaimManager resourceClaimManager;
    private final Map<String, BlockingQueue<ArchiveInfo>> archivedFiles = new HashMap<String, BlockingQueue<ArchiveInfo>>();
    private final AtomicLong oldestArchiveDate = new AtomicLong(0L);
    private final NiFiProperties nifiProperties;

    public FileSystemRepository() {
        this.containers = null;
        this.containerNames = null;
        this.index = null;
        this.archiveData = false;
        this.maxArchiveMillis = 0L;
        this.alwaysSync = false;
        this.containerCleanupExecutor = null;
        this.nifiProperties = null;
        this.maxAppendableClaimLength = 0L;
        this.maxFlowFilesPerClaim = 0;
        this.writableClaimQueue = null;
    }

    public FileSystemRepository(NiFiProperties nifiProperties) throws IOException {
        long appendableClaimLengthCap;
        this.nifiProperties = nifiProperties;
        Map fileRespositoryPaths = nifiProperties.getContentRepositoryPaths();
        for (Path path : fileRespositoryPaths.values()) {
            Files.createDirectories(path, new FileAttribute[0]);
        }
        this.maxFlowFilesPerClaim = nifiProperties.getMaxFlowFilesPerClaim();
        this.writableClaimQueue = new LinkedBlockingQueue<ClaimLengthPair>(this.maxFlowFilesPerClaim);
        long configuredAppendableClaimLength = DataUnit.parseDataSize((String)nifiProperties.getMaxAppendableClaimSize(), (DataUnit)DataUnit.B).longValue();
        if (configuredAppendableClaimLength > (appendableClaimLengthCap = DataUnit.parseDataSize((String)APPENDABLE_CLAIM_LENGTH_CAP, (DataUnit)DataUnit.B).longValue())) {
            LOG.warn("Configured property '{}' with value {} exceeds cap of {}. Setting value to {}", new Object[]{"nifi.content.claim.max.appendable.size", configuredAppendableClaimLength, APPENDABLE_CLAIM_LENGTH_CAP, APPENDABLE_CLAIM_LENGTH_CAP});
            this.maxAppendableClaimLength = appendableClaimLengthCap;
        } else {
            this.maxAppendableClaimLength = configuredAppendableClaimLength;
        }
        this.containers = new HashMap<String, Path>(fileRespositoryPaths);
        this.containerNames = new ArrayList<String>(this.containers.keySet());
        this.index = new AtomicLong(0L);
        for (String containerName : this.containerNames) {
            this.reclaimable.put(containerName, new LinkedBlockingQueue(10000));
            this.archivedFiles.put(containerName, new LinkedBlockingQueue(100000));
        }
        String enableArchiving = nifiProperties.getProperty("nifi.content.repository.archive.enabled");
        String maxArchiveRetentionPeriod = nifiProperties.getProperty("nifi.content.repository.archive.max.retention.period");
        String maxArchiveSize = nifiProperties.getProperty("nifi.content.repository.archive.max.usage.percentage");
        String archiveBackPressureSize = nifiProperties.getProperty("nifi.content.repository.archive.backpressure.percentage");
        if ("true".equalsIgnoreCase(enableArchiving)) {
            this.archiveData = true;
            if (maxArchiveSize == null) {
                throw new RuntimeException("No value specified for property 'nifi.content.repository.archive.max.usage.percentage' but archiving is enabled. You must configure the max disk usage in order to enable archiving.");
            }
            if (!MAX_ARCHIVE_SIZE_PATTERN.matcher(maxArchiveSize.trim()).matches()) {
                throw new RuntimeException("Invalid value specified for the 'nifi.content.repository.archive.max.usage.percentage' property. Value must be in format: <XX>%");
            }
        } else if ("false".equalsIgnoreCase(enableArchiving)) {
            this.archiveData = false;
        } else {
            LOG.warn("No property set for '{}'; will not archive content", (Object)"nifi.content.repository.archive.enabled");
            this.archiveData = false;
        }
        double maxArchiveRatio = 0.0;
        double archiveBackPressureRatio = 0.01;
        if (maxArchiveSize != null && MAX_ARCHIVE_SIZE_PATTERN.matcher(maxArchiveSize.trim()).matches()) {
            maxArchiveRatio = FileSystemRepository.getRatio(maxArchiveSize);
            archiveBackPressureRatio = archiveBackPressureSize != null && MAX_ARCHIVE_SIZE_PATTERN.matcher(archiveBackPressureSize.trim()).matches() ? FileSystemRepository.getRatio(archiveBackPressureSize) : maxArchiveRatio + 0.02;
        }
        if (maxArchiveRatio > 0.0) {
            for (Map.Entry<String, Path> container : this.containers.entrySet()) {
                String containerName = container.getKey();
                long capacity = container.getValue().toFile().getTotalSpace();
                if (capacity == 0L) {
                    throw new RuntimeException("System returned total space of the partition for " + containerName + " is zero byte. Nifi can not create a zero sized FileSystemRepository");
                }
                long maxArchiveBytes = (long)((double)capacity * (1.0 - (maxArchiveRatio - 0.02)));
                this.minUsableContainerBytesForArchive.put(container.getKey(), maxArchiveBytes);
                LOG.info("Maximum Threshold for Container {} set to {} bytes; if volume exceeds this size, archived data will be deleted until it no longer exceeds this size", (Object)containerName, (Object)maxArchiveBytes);
                long backPressureBytes = (long)((double)container.getValue().toFile().getTotalSpace() * archiveBackPressureRatio);
                ContainerState containerState = new ContainerState(containerName, true, backPressureBytes, capacity);
                this.containerStateMap.put(containerName, containerState);
            }
        } else {
            for (String containerName : this.containerNames) {
                this.containerStateMap.put(containerName, new ContainerState(containerName, false, Long.MAX_VALUE, Long.MAX_VALUE));
            }
        }
        this.maxArchiveMillis = maxArchiveRatio <= 0.0 ? 0L : (StringUtils.isEmpty((CharSequence)maxArchiveRetentionPeriod) ? Long.MAX_VALUE : FormatUtils.getTimeDuration((String)maxArchiveRetentionPeriod, (TimeUnit)TimeUnit.MILLISECONDS));
        this.alwaysSync = Boolean.parseBoolean(nifiProperties.getProperty("nifi.content.repository.always.sync"));
        LOG.info("Initializing FileSystemRepository with 'Always Sync' set to {}", (Object)this.alwaysSync);
        this.initializeRepository();
        this.containerCleanupExecutor = new FlowEngine(this.containers.size(), "Cleanup FileSystemRepository Container", true);
    }

    public void initialize(ResourceClaimManager claimManager) {
        this.resourceClaimManager = claimManager;
        Map fileRespositoryPaths = this.nifiProperties.getContentRepositoryPaths();
        this.executor.scheduleWithFixedDelay(new BinDestructableClaims(), 1L, 1L, TimeUnit.SECONDS);
        for (int i = 0; i < fileRespositoryPaths.size(); ++i) {
            this.executor.scheduleWithFixedDelay(new ArchiveOrDestroyDestructableClaims(), 1L, 1L, TimeUnit.SECONDS);
        }
        long cleanupMillis = this.determineCleanupInterval(this.nifiProperties);
        for (Map.Entry<String, Path> containerEntry : this.containers.entrySet()) {
            String containerName = containerEntry.getKey();
            Path containerPath = containerEntry.getValue();
            DestroyExpiredArchiveClaims cleanup = new DestroyExpiredArchiveClaims(containerName, containerPath);
            this.containerCleanupExecutor.scheduleWithFixedDelay(cleanup, cleanupMillis, cleanupMillis, TimeUnit.MILLISECONDS);
        }
    }

    public void shutdown() {
        this.executor.shutdown();
        this.containerCleanupExecutor.shutdown();
        for (OutputStream out : this.writableClaimStreams.values()) {
            try {
                out.close();
            }
            catch (IOException iOException) {}
        }
    }

    private static double getRatio(String value) {
        String trimmed = value.trim();
        String percentage = trimmed.substring(0, trimmed.length() - 1);
        return (double)Integer.parseInt(percentage) / 100.0;
    }

    private synchronized void initializeRepository() throws IOException {
        HashMap<String, Path> realPathMap = new HashMap<String, Path>();
        ExecutorService executor = Executors.newFixedThreadPool(this.containers.size());
        ArrayList<Future<Long>> futures = new ArrayList<Future<Long>>();
        for (Map.Entry<String, Path> entry : this.containers.entrySet()) {
            String containerName = entry.getKey();
            final ContainerState containerState = this.containerStateMap.get(containerName);
            Path containerPath = entry.getValue();
            boolean pathExists = Files.exists(containerPath, new LinkOption[0]);
            final Path realPath = pathExists ? containerPath.toRealPath(new LinkOption[0]) : Files.createDirectories(containerPath, new FileAttribute[0]).toRealPath(new LinkOption[0]);
            for (int i = 0; i < 1024; ++i) {
                Files.createDirectories(realPath.resolve(String.valueOf(i)), new FileAttribute[0]);
            }
            realPathMap.put(containerName, realPath);
            Callable<Long> scanContainer = new Callable<Long>(){

                @Override
                public Long call() throws IOException {
                    final AtomicLong oldestDateHolder = new AtomicLong(0L);
                    Files.walkFileTree(realPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                        @Override
                        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                            LOG.warn("Content repository contains un-readable file or directory '" + file.getFileName() + "'. Skipping. ", (Throwable)exc);
                            return FileVisitResult.SKIP_SUBTREE;
                        }

                        @Override
                        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                            if (attrs.isDirectory()) {
                                return FileVisitResult.CONTINUE;
                            }
                            Path relativePath = realPath.relativize(file);
                            if (relativePath.getNameCount() > 3 && FileSystemRepository.ARCHIVE_DIR_NAME.equals(relativePath.subpath(1, 2).toString())) {
                                long lastModifiedTime = FileSystemRepository.this.getLastModTime(file);
                                if (lastModifiedTime < oldestDateHolder.get()) {
                                    oldestDateHolder.set(lastModifiedTime);
                                }
                                containerState.incrementArchiveCount();
                            }
                            return FileVisitResult.CONTINUE;
                        }
                    });
                    return oldestDateHolder.get();
                }
            };
            if (!pathExists) continue;
            futures.add(executor.submit(scanContainer));
        }
        executor.shutdown();
        for (Future future : futures) {
            try {
                Long oldestDate = (Long)future.get();
                if (oldestDate >= this.oldestArchiveDate.get()) continue;
                this.oldestArchiveDate.set(oldestDate);
            }
            catch (InterruptedException | ExecutionException e) {
                if (e.getCause() instanceof IOException) {
                    throw (IOException)e.getCause();
                }
                throw new RuntimeException(e);
            }
        }
        this.containers.clear();
        this.containers.putAll(realPathMap);
    }

    public Set<String> getContainerNames() {
        return new HashSet<String>(this.containerNames);
    }

    public long getContainerCapacity(String containerName) throws IOException {
        Path path = this.containers.get(containerName);
        if (path == null) {
            throw new IllegalArgumentException("No container exists with name " + containerName);
        }
        long capacity = FileUtils.getContainerCapacity((Path)path);
        if (capacity == 0L) {
            throw new IOException("System returned total space of the partition for " + containerName + " is zero byte. Nifi can not create a zero sized FileSystemRepository.");
        }
        return capacity;
    }

    public long getContainerUsableSpace(String containerName) throws IOException {
        Path path = this.containers.get(containerName);
        if (path == null) {
            throw new IllegalArgumentException("No container exists with name " + containerName);
        }
        return FileUtils.getContainerUsableSpace((Path)path);
    }

    public String getContainerFileStoreName(String containerName) {
        Path path = this.containers.get(containerName);
        try {
            return Files.getFileStore(path).name();
        }
        catch (IOException e) {
            return null;
        }
    }

    public void cleanup() {
        for (Map.Entry<String, Path> entry : this.containers.entrySet()) {
            String containerName = entry.getKey();
            Path containerPath = entry.getValue();
            File[] sectionFiles = containerPath.toFile().listFiles();
            if (sectionFiles == null) continue;
            for (File sectionFile : sectionFiles) {
                this.removeIncompleteContent(containerName, containerPath, sectionFile.toPath());
            }
        }
    }

    private void removeIncompleteContent(String containerName, Path containerPath, Path fileToRemove) {
        if (Files.isDirectory(fileToRemove, new LinkOption[0])) {
            Path lastPathName = fileToRemove.subpath(1, fileToRemove.getNameCount());
            String fileName = lastPathName.toFile().getName();
            if (fileName.equals(ARCHIVE_DIR_NAME)) {
                return;
            }
            File[] children = fileToRemove.toFile().listFiles();
            if (children != null) {
                for (File child : children) {
                    this.removeIncompleteContent(containerName, containerPath, child.toPath());
                }
            }
            return;
        }
        Path relativePath = containerPath.relativize(fileToRemove);
        Path sectionPath = relativePath.subpath(0, 1);
        if (relativePath.getNameCount() < 2) {
            return;
        }
        Path idPath = relativePath.subpath(1, relativePath.getNameCount());
        String id = idPath.toFile().getName();
        String sectionName = sectionPath.toFile().getName();
        ResourceClaim resourceClaim = this.resourceClaimManager.newResourceClaim(containerName, sectionName, id, false, false);
        if (this.resourceClaimManager.getClaimantCount(resourceClaim) == 0) {
            this.removeIncompleteContent(fileToRemove);
        }
    }

    private void removeIncompleteContent(Path fileToRemove) {
        String fileDescription = null;
        try {
            fileDescription = fileToRemove.toFile().getAbsolutePath() + " (" + Files.size(fileToRemove) + " bytes)";
        }
        catch (IOException e) {
            fileDescription = fileToRemove.toFile().getAbsolutePath() + " (unknown file size)";
        }
        LOG.info("Found unknown file {} in File System Repository; {} file", (Object)fileDescription, (Object)(this.archiveData ? "archiving" : "removing"));
        try {
            if (this.archiveData) {
                this.archive(fileToRemove);
            } else {
                Files.delete(fileToRemove);
            }
        }
        catch (IOException e) {
            String action = this.archiveData ? ARCHIVE_DIR_NAME : "remove";
            LOG.warn("Unable to {} unknown file {} from File System Repository due to {}", new Object[]{action, fileDescription, e.toString()});
            LOG.warn("", (Throwable)e);
        }
    }

    public boolean isActiveResourceClaimsSupported() {
        return true;
    }

    public Set<ResourceClaim> getActiveResourceClaims(String containerName) throws IOException {
        Path containerPath = this.containers.get(containerName);
        if (containerPath == null) {
            return Collections.emptySet();
        }
        ScanForActiveResourceClaims scan = new ScanForActiveResourceClaims(containerPath, containerName, this.resourceClaimManager, this.containers.keySet());
        Files.walkFileTree(containerPath, scan);
        Set<ResourceClaim> activeResourceClaims = scan.getActiveResourceClaims();
        LOG.debug("Obtaining active resource claims, will return a list of {} resource claims for container {}", (Object)activeResourceClaims.size(), (Object)containerName);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Listing of resource claims:");
            activeResourceClaims.forEach(claim -> LOG.trace(claim.toString()));
        }
        return activeResourceClaims;
    }

    private Path getPath(ContentClaim claim) {
        ResourceClaim resourceClaim = claim.getResourceClaim();
        return this.getPath(resourceClaim);
    }

    private Path getPath(ResourceClaim resourceClaim) {
        Path containerPath = this.containers.get(resourceClaim.getContainer());
        if (containerPath == null) {
            return null;
        }
        return containerPath.resolve(resourceClaim.getSection()).resolve(resourceClaim.getId());
    }

    public Path getPath(ContentClaim claim, boolean verifyExists) throws ContentNotFoundException {
        ResourceClaim resourceClaim = claim.getResourceClaim();
        Path containerPath = this.containers.get(resourceClaim.getContainer());
        if (containerPath == null) {
            if (verifyExists) {
                throw new ContentNotFoundException(claim);
            }
            return null;
        }
        Path resolvedPath = containerPath.resolve(resourceClaim.getSection()).resolve(resourceClaim.getId());
        if (!Files.exists(resolvedPath, new LinkOption[0])) {
            resolvedPath = this.getArchivePath(claim.getResourceClaim());
            if (verifyExists && !Files.exists(resolvedPath, new LinkOption[0])) {
                throw new ContentNotFoundException(claim);
            }
        }
        return resolvedPath;
    }

    public ContentClaim create(boolean lossTolerant) throws IOException {
        long resourceOffset;
        ResourceClaim resourceClaim;
        ClaimLengthPair pair = (ClaimLengthPair)this.writableClaimQueue.poll();
        if (pair == null) {
            long currentIndex = this.index.incrementAndGet();
            String containerName = null;
            boolean waitRequired = true;
            ContainerState containerState = null;
            for (long containerIndex = currentIndex; containerIndex < currentIndex + (long)this.containers.size(); ++containerIndex) {
                long modulatedContainerIndex = containerIndex % (long)this.containers.size();
                containerName = this.containerNames.get((int)modulatedContainerIndex);
                containerState = this.containerStateMap.get(containerName);
                if (containerState.isWaitRequired()) continue;
                waitRequired = false;
                break;
            }
            if (waitRequired) {
                containerState.waitForArchiveExpiration();
            }
            long modulatedSectionIndex = currentIndex % 1024L;
            String section = String.valueOf(modulatedSectionIndex).intern();
            String claimId = System.currentTimeMillis() + "-" + currentIndex;
            resourceClaim = this.resourceClaimManager.newResourceClaim(containerName, section, claimId, lossTolerant, true);
            resourceOffset = 0L;
            LOG.debug("Creating new Resource Claim {}", (Object)resourceClaim);
            File file = this.getPath(resourceClaim).toFile();
            SynchronizedByteCountingOutputStream claimStream = new SynchronizedByteCountingOutputStream((OutputStream)new FileOutputStream(file, true), file.length());
            this.writableClaimStreams.put(resourceClaim, (ByteCountingOutputStream)claimStream);
            this.incrementClaimantCount(resourceClaim, true);
        } else {
            resourceClaim = pair.getClaim();
            resourceOffset = pair.getLength();
            LOG.debug("Reusing Resource Claim {}", (Object)resourceClaim);
            this.incrementClaimantCount(resourceClaim, false);
        }
        StandardContentClaim scc = new StandardContentClaim(resourceClaim, resourceOffset);
        return scc;
    }

    public int incrementClaimaintCount(ContentClaim claim) {
        return this.incrementClaimantCount(claim == null ? null : claim.getResourceClaim(), false);
    }

    protected int incrementClaimantCount(ResourceClaim resourceClaim, boolean newClaim) {
        if (resourceClaim == null) {
            return 0;
        }
        return this.resourceClaimManager.incrementClaimantCount(resourceClaim, newClaim);
    }

    public int getClaimantCount(ContentClaim claim) {
        if (claim == null) {
            return 0;
        }
        return this.resourceClaimManager.getClaimantCount(claim.getResourceClaim());
    }

    public int decrementClaimantCount(ContentClaim claim) {
        if (claim == null) {
            return 0;
        }
        return this.resourceClaimManager.decrementClaimantCount(claim.getResourceClaim());
    }

    public boolean remove(ContentClaim claim) {
        if (claim == null) {
            return false;
        }
        return this.remove(claim.getResourceClaim());
    }

    private boolean remove(ResourceClaim claim) {
        File file;
        if (claim == null) {
            return false;
        }
        if (claim.isInUse()) {
            return false;
        }
        Path path = null;
        try {
            path = this.getPath(claim);
        }
        catch (ContentNotFoundException contentNotFoundException) {
            // empty catch block
        }
        ByteCountingOutputStream bcos = (ByteCountingOutputStream)this.writableClaimStreams.remove(claim);
        if (bcos != null) {
            try {
                bcos.close();
            }
            catch (IOException e) {
                LOG.warn("Failed to close Output Stream for {} due to {}", (Object)claim, (Object)e);
            }
        }
        if (!(file = path.toFile()).delete() && file.exists()) {
            LOG.warn("Unable to delete {} at path {}", new Object[]{claim, path});
            return false;
        }
        return true;
    }

    public ContentClaim clone(ContentClaim original, boolean lossTolerant) throws IOException {
        if (original == null) {
            return null;
        }
        ContentClaim newClaim = this.create(lossTolerant);
        try (InputStream in = this.read(original);
             OutputStream out = this.write(newClaim);){
            StreamUtils.copy((InputStream)in, (OutputStream)out);
        }
        catch (IOException ioe) {
            this.decrementClaimantCount(newClaim);
            this.remove(newClaim);
            throw ioe;
        }
        return newClaim;
    }

    public long merge(Collection<ContentClaim> claims, ContentClaim destination, byte[] header, byte[] footer, byte[] demarcator) throws IOException {
        if (claims.contains(destination)) {
            throw new IllegalArgumentException("destination cannot be within claims");
        }
        try (ByteCountingOutputStream out = new ByteCountingOutputStream(this.write(destination));){
            if (header != null) {
                out.write(header);
            }
            int i = 0;
            for (ContentClaim claim : claims) {
                try (InputStream in = this.read(claim);){
                    StreamUtils.copy((InputStream)in, (OutputStream)out);
                }
                if (++i >= claims.size() || demarcator == null) continue;
                out.write(demarcator);
            }
            if (footer != null) {
                out.write(footer);
            }
            long l = out.getBytesWritten();
            return l;
        }
    }

    public long importFrom(Path content, ContentClaim claim) throws IOException {
        try (InputStream in = Files.newInputStream(content, StandardOpenOption.READ);){
            long l = this.importFrom(in, claim);
            return l;
        }
    }

    public long importFrom(InputStream content, ContentClaim claim) throws IOException {
        try (OutputStream out = this.write(claim, false);){
            long l = StreamUtils.copy((InputStream)content, (OutputStream)out);
            return l;
        }
    }

    /*
     * Exception decompiling
     */
    public long exportTo(ContentClaim claim, Path destination, boolean append) throws IOException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    /*
     * Exception decompiling
     */
    public long exportTo(ContentClaim claim, Path destination, boolean append, long offset, long length) throws IOException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    public long exportTo(ContentClaim claim, OutputStream destination) throws IOException {
        if (claim == null) {
            return 0L;
        }
        try (InputStream in = this.read(claim);){
            long l = StreamUtils.copy((InputStream)in, (OutputStream)destination);
            return l;
        }
    }

    public long exportTo(ContentClaim claim, OutputStream destination, long offset, long length) throws IOException {
        if (offset < 0L) {
            throw new IllegalArgumentException("offset cannot be negative");
        }
        long claimSize = this.size(claim);
        if (offset > claimSize) {
            throw new IllegalArgumentException("offset of " + offset + " exceeds claim size of " + claimSize);
        }
        if (offset == 0L && length == claimSize) {
            return this.exportTo(claim, destination);
        }
        try (InputStream in = this.read(claim);){
            int len;
            StreamUtils.skip((InputStream)in, (long)offset);
            byte[] buffer = new byte[8192];
            long copied = 0L;
            while ((len = in.read(buffer, 0, (int)Math.min(length - copied, (long)buffer.length))) > 0) {
                destination.write(buffer, 0, len);
                copied += (long)len;
            }
            long l = copied;
            return l;
        }
    }

    public long size(ContentClaim claim) throws IOException {
        if (claim == null) {
            return 0L;
        }
        if (claim.getLength() < 0L) {
            return Files.size(this.getPath(claim, true)) - claim.getOffset();
        }
        return claim.getLength();
    }

    public InputStream read(ContentClaim claim) throws IOException {
        if (claim == null) {
            return new ByteArrayInputStream(new byte[0]);
        }
        Path path = this.getPath(claim, true);
        FileInputStream fis = new FileInputStream(path.toFile());
        if (claim.getOffset() > 0L) {
            try {
                StreamUtils.skip((InputStream)fis, (long)claim.getOffset());
            }
            catch (IOException ioe) {
                IOUtils.closeQuietly((InputStream)fis);
                throw ioe;
            }
        }
        if (claim.getLength() >= 0L) {
            return new LimitedInputStream((InputStream)fis, () -> ((ContentClaim)claim).getLength());
        }
        return fis;
    }

    public OutputStream write(ContentClaim claim) throws IOException {
        return this.write(claim, false);
    }

    private OutputStream write(ContentClaim claim, boolean append) {
        StandardContentClaim scc = FileSystemRepository.validateContentClaimForWriting(claim);
        ByteCountingOutputStream claimStream = (ByteCountingOutputStream)this.writableClaimStreams.get(scc.getResourceClaim());
        int initialLength = append ? (int)Math.max(0L, scc.getLength()) : 0;
        ByteCountingOutputStream bcos = claimStream;
        ContentRepositoryOutputStream out = new ContentRepositoryOutputStream(scc, bcos, initialLength);
        LOG.debug("Writing to {}", (Object)out);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Stack trace: ", (Throwable)new RuntimeException("Stack Trace for writing to " + out));
        }
        return out;
    }

    public static StandardContentClaim validateContentClaimForWriting(ContentClaim claim) {
        if (claim == null) {
            throw new NullPointerException("ContentClaim cannot be null");
        }
        if (!(claim instanceof StandardContentClaim)) {
            throw new IllegalArgumentException("Cannot write to " + claim + " because that Content Claim does belong to this Content Repository");
        }
        if (claim.getLength() > 0L) {
            throw new IllegalArgumentException("Cannot write to " + claim + " because it has already been written to.");
        }
        return (StandardContentClaim)claim;
    }

    public void purge() {
        for (Path path : this.containers.values()) {
            FileUtils.deleteFilesInDir((File)path.toFile(), null, (Logger)LOG, (boolean)true);
        }
        for (Path path : this.containers.values()) {
            if (!Files.exists(path, new LinkOption[0])) {
                throw new RepositoryPurgeException("File " + path.toFile().getAbsolutePath() + " does not exist");
            }
            boolean writable = false;
            for (int i = 0; i < 10; ++i) {
                if (Files.isWritable(path)) {
                    writable = true;
                    break;
                }
                try {
                    Thread.sleep(100L);
                    continue;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (writable) continue;
            throw new RepositoryPurgeException("File " + path.toFile().getAbsolutePath() + " is not writable");
        }
        this.resourceClaimManager.purge();
    }

    public static Path getArchivePath(Path contentClaimPath) {
        Path sectionPath = contentClaimPath.getParent();
        String claimId = contentClaimPath.toFile().getName();
        return sectionPath.resolve(ARCHIVE_DIR_NAME).resolve(claimId);
    }

    private Path getArchivePath(ResourceClaim claim) {
        String claimId = claim.getId();
        Path containerPath = this.containers.get(claim.getContainer());
        Path archivePath = containerPath.resolve(claim.getSection()).resolve(ARCHIVE_DIR_NAME).resolve(claimId);
        return archivePath;
    }

    public boolean isAccessible(ContentClaim contentClaim) throws IOException {
        if (contentClaim == null) {
            return false;
        }
        Path path = this.getPath(contentClaim);
        if (path == null) {
            return false;
        }
        if (Files.exists(path, new LinkOption[0])) {
            return true;
        }
        return Files.exists(this.getArchivePath(contentClaim.getResourceClaim()), new LinkOption[0]);
    }

    boolean archive(ResourceClaim claim) throws IOException {
        Path curPath;
        if (!this.archiveData) {
            return false;
        }
        if (claim.isInUse()) {
            return false;
        }
        OutputStream out = (OutputStream)this.writableClaimStreams.remove(claim);
        if (out != null) {
            try {
                out.close();
            }
            catch (IOException ioe) {
                LOG.warn("Unable to close Output Stream for " + claim, (Throwable)ioe);
            }
        }
        if ((curPath = this.getPath(claim)) == null) {
            return false;
        }
        boolean archived = this.archive(curPath);
        LOG.debug("Successfully moved {} to archive", (Object)claim);
        return archived;
    }

    protected int getOpenStreamCount() {
        return this.writableClaimStreams.size();
    }

    protected ConcurrentMap<ResourceClaim, ByteCountingOutputStream> getWritableClaimStreams() {
        return this.writableClaimStreams;
    }

    protected ByteCountingOutputStream getWritableClaimStreamByResourceClaim(ResourceClaim rc) {
        return (ByteCountingOutputStream)this.writableClaimStreams.get(rc);
    }

    protected ResourceClaimManager getResourceClaimManager() {
        return this.resourceClaimManager;
    }

    protected BlockingQueue<ClaimLengthPair> getWritableClaimQueue() {
        return this.writableClaimQueue;
    }

    protected long getMaxAppendableClaimLength() {
        return this.maxAppendableClaimLength;
    }

    protected boolean isAlwaysSync() {
        return this.alwaysSync;
    }

    protected boolean archive(Path curPath) throws IOException {
        boolean alreadyArchived = ARCHIVE_DIR_NAME.equals(curPath.getParent().toFile().getName());
        if (alreadyArchived) {
            return false;
        }
        Path archivePath = FileSystemRepository.getArchivePath(curPath);
        if (curPath.equals(archivePath)) {
            LOG.warn("Cannot archive {} because it is already archived", (Object)curPath);
            return false;
        }
        try {
            Files.move(curPath, archivePath, new CopyOption[0]);
            return true;
        }
        catch (NoSuchFileException nsfee) {
            if (Files.exists(curPath, new LinkOption[0])) {
                Files.createDirectories(archivePath.getParent(), new FileAttribute[0]);
                Files.move(curPath, archivePath, new CopyOption[0]);
                return true;
            }
            return false;
        }
    }

    private long getLastModTime(File file) {
        String filename = file.getName();
        int dashIndex = filename.indexOf("-");
        if (dashIndex > 0) {
            String creationTimestamp = filename.substring(0, dashIndex);
            try {
                return Long.parseLong(creationTimestamp);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return file.lastModified();
    }

    private long getLastModTime(Path file) throws IOException {
        return this.getLastModTime(file.toFile());
    }

    private boolean deleteBasedOnTimestamp(BlockingQueue<ArchiveInfo> fileQueue, long removalTimeThreshold) throws IOException {
        ArchiveInfo nextFile = (ArchiveInfo)fileQueue.peek();
        if (nextFile == null) {
            return false;
        }
        long oldestArchiveDate = this.getLastModTime(nextFile.toPath());
        return oldestArchiveDate <= removalTimeThreshold;
    }

    private long destroyExpiredArchives(final String containerName, final Path container) throws IOException {
        ArchiveInfo toDelete;
        this.archiveExpirationLog.debug("Destroying Expired Archives for Container {}", (Object)containerName);
        final ArrayList notYetExceedingThreshold = new ArrayList();
        long removalTimeThreshold = System.currentTimeMillis() - this.maxArchiveMillis;
        long oldestArchiveDateFound = System.currentTimeMillis();
        final Long minRequiredSpace = this.minUsableContainerBytesForArchive.get(containerName);
        if (minRequiredSpace == null) {
            this.archiveExpirationLog.debug("Could not determine minimum required space so will not destroy any archived data");
            return -1L;
        }
        final long usableSpace = this.getContainerUsableSpace(containerName);
        final ContainerState containerState = this.containerStateMap.get(containerName);
        long startNanos = System.nanoTime();
        long toFree = minRequiredSpace - usableSpace;
        BlockingQueue<ArchiveInfo> fileQueue = this.archivedFiles.get(containerName);
        if (this.archiveExpirationLog.isDebugEnabled()) {
            if (toFree < 0L) {
                this.archiveExpirationLog.debug("Currently {} bytes free for Container {}; requirement is {} byte free, so no need to free space until an additional {} bytes are used", new Object[]{usableSpace, containerName, minRequiredSpace, Math.abs(toFree)});
            } else {
                this.archiveExpirationLog.debug("Currently {} bytes free for Container {}; requirement is {} byte free, so need to free {} bytes", new Object[]{usableSpace, containerName, minRequiredSpace, toFree});
            }
        }
        int deleteCount = 0;
        long freed = 0L;
        while ((toDelete = (ArchiveInfo)fileQueue.peek()) != null) {
            try {
                long fileSize = toDelete.getSize();
                removalTimeThreshold = System.currentTimeMillis() - this.maxArchiveMillis;
                if (freed < toFree || this.getLastModTime(toDelete.toPath()) < removalTimeThreshold) {
                    toDelete = (ArchiveInfo)fileQueue.poll();
                    Files.deleteIfExists(toDelete.toPath());
                    containerState.decrementArchiveCount();
                    LOG.debug("Deleted archived ContentClaim with ID {} from Container {} because the archival size was exceeding the max configured size", (Object)toDelete.getName(), (Object)containerName);
                    freed += fileSize;
                    ++deleteCount;
                }
                if (freed < toFree) continue;
                if (this.deleteBasedOnTimestamp(fileQueue, removalTimeThreshold)) {
                    this.archiveExpirationLog.debug("Freed enough space ({} bytes freed, needed to free {} bytes) but will continue to expire data based on timestamp", (Object)freed, (Object)toFree);
                    continue;
                }
                this.archiveExpirationLog.debug("Freed enough space ({} bytes freed, needed to free {} bytes). Finished expiring data", (Object)freed, (Object)toFree);
                ArchiveInfo archiveInfo = (ArchiveInfo)fileQueue.peek();
                long oldestArchiveDate = archiveInfo == null ? System.currentTimeMillis() : this.getLastModTime(archiveInfo.toPath());
                long millis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                if (deleteCount > 0) {
                    LOG.info("Deleted {} files from archive for Container {}; oldest Archive Date is now {}; container cleanup took {} millis", new Object[]{deleteCount, containerName, new Date(oldestArchiveDate), millis});
                } else {
                    LOG.debug("Deleted {} files from archive for Container {}; oldest Archive Date is now {}; container cleanup took {} millis", new Object[]{deleteCount, containerName, new Date(oldestArchiveDate), millis});
                }
                return oldestArchiveDate;
            }
            catch (IOException ioe) {
                LOG.warn("Failed to delete {} from archive due to {}", (Object)toDelete, (Object)ioe.toString());
                if (!LOG.isDebugEnabled()) continue;
                LOG.warn("", (Throwable)ioe);
            }
        }
        this.archiveExpirationLog.debug("Searching for more archived data to expire");
        StopWatch stopWatch = new StopWatch(true);
        for (int i = 0; i < 1024; ++i) {
            Path sectionContainer = container.resolve(String.valueOf(i));
            Path archive = sectionContainer.resolve(ARCHIVE_DIR_NAME);
            if (!Files.exists(archive, new LinkOption[0])) continue;
            try {
                final long timestampThreshold = removalTimeThreshold;
                Files.walkFileTree(archive, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                    @Override
                    public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                        if (attrs.isDirectory()) {
                            return FileVisitResult.CONTINUE;
                        }
                        long lastModTime = FileSystemRepository.this.getLastModTime(file);
                        if (lastModTime < timestampThreshold) {
                            try {
                                Files.deleteIfExists(file);
                                containerState.decrementArchiveCount();
                                LOG.debug("Deleted archived ContentClaim with ID {} from Container {} because it was older than the configured max archival duration", (Object)file.toFile().getName(), (Object)containerName);
                            }
                            catch (IOException ioe) {
                                LOG.warn("Failed to remove archived ContentClaim with ID {} from Container {} due to {}", new Object[]{file.toFile().getName(), containerName, ioe.toString()});
                                if (LOG.isDebugEnabled()) {
                                    LOG.warn("", (Throwable)ioe);
                                }
                            }
                        } else if (usableSpace < minRequiredSpace) {
                            notYetExceedingThreshold.add(new ArchiveInfo(container, file, attrs.size(), lastModTime));
                        }
                        return FileVisitResult.CONTINUE;
                    }
                });
                continue;
            }
            catch (IOException ioe) {
                LOG.warn("Failed to cleanup archived files in {} due to {}", (Object)archive, (Object)ioe.toString());
                if (!LOG.isDebugEnabled()) continue;
                LOG.warn("", (Throwable)ioe);
            }
        }
        long deleteExpiredMillis = stopWatch.getElapsed(TimeUnit.MILLISECONDS);
        Collections.sort(notYetExceedingThreshold, new Comparator<ArchiveInfo>(){

            @Override
            public int compare(ArchiveInfo o1, ArchiveInfo o2) {
                return Long.compare(o1.getLastModTime(), o2.getLastModTime());
            }
        });
        long sortRemainingMillis = stopWatch.getElapsed(TimeUnit.MILLISECONDS) - deleteExpiredMillis;
        this.archiveExpirationLog.debug("Deleting data based on timestamp");
        Iterator itr = notYetExceedingThreshold.iterator();
        int counter = 0;
        while (itr.hasNext()) {
            block20: {
                ArchiveInfo archiveInfo = (ArchiveInfo)itr.next();
                try {
                    Path path = archiveInfo.toPath();
                    Files.deleteIfExists(path);
                    containerState.decrementArchiveCount();
                    LOG.debug("Deleted archived ContentClaim with ID {} from Container {} because the archival size was exceeding the max configured size", (Object)archiveInfo.getName(), (Object)containerName);
                    if (++counter % 25 == 0 && this.getContainerUsableSpace(containerName) > minRequiredSpace) {
                        LOG.debug("Finished cleaning up archive for Container {}", (Object)containerName);
                        break;
                    }
                }
                catch (IOException ioe) {
                    LOG.warn("Failed to delete {} from archive due to {}", (Object)archiveInfo, (Object)ioe.toString());
                    if (!LOG.isDebugEnabled()) break block20;
                    LOG.warn("", (Throwable)ioe);
                }
            }
            itr.remove();
        }
        long deleteOldestMillis = stopWatch.getElapsed(TimeUnit.MILLISECONDS) - sortRemainingMillis - deleteExpiredMillis;
        long oldestContainerArchive = notYetExceedingThreshold.isEmpty() ? System.currentTimeMillis() : ((ArchiveInfo)notYetExceedingThreshold.get(0)).getLastModTime();
        if (oldestContainerArchive < oldestArchiveDateFound) {
            oldestArchiveDateFound = oldestContainerArchive;
        }
        for (ArchiveInfo toEnqueue : notYetExceedingThreshold.subList(0, Math.min(100000, notYetExceedingThreshold.size()))) {
            fileQueue.offer(toEnqueue);
        }
        long cleanupMillis = stopWatch.getElapsed(TimeUnit.MILLISECONDS) - deleteOldestMillis - sortRemainingMillis - deleteExpiredMillis;
        LOG.debug("Oldest Archive Date for Container {} is {}; delete expired = {} ms, sort remaining = {} ms, delete oldest = {} ms, cleanup = {} ms", new Object[]{containerName, new Date(oldestContainerArchive), deleteExpiredMillis, sortRemainingMillis, deleteOldestMillis, cleanupMillis});
        return oldestContainerArchive;
    }

    private long determineCleanupInterval(NiFiProperties properties) {
        long cleanupInterval = 1000L;
        String archiveCleanupFrequency = properties.getProperty("nifi.content.repository.archive.cleanup.frequency");
        if (archiveCleanupFrequency != null) {
            try {
                cleanupInterval = FormatUtils.getTimeDuration((String)archiveCleanupFrequency.trim(), (TimeUnit)TimeUnit.MILLISECONDS);
            }
            catch (Exception e) {
                throw new RuntimeException("Invalid value set for property nifi.content.repository.archive.cleanup.frequency");
            }
            if (cleanupInterval < 1000L) {
                LOG.warn("The value of nifi.content.repository.archive.cleanup.frequency property is set to '" + archiveCleanupFrequency + "' which is below the allowed minimum of 1 second (1000 milliseconds). Minimum value of 1 sec will be used as scheduling interval for archive cleanup task.");
                cleanupInterval = 1000L;
            }
        }
        return cleanupInterval;
    }

    protected class ContentRepositoryOutputStream
    extends OutputStream {
        protected final StandardContentClaim scc;
        protected final ByteCountingOutputStream bcos;
        protected final int initialLength;
        protected long bytesWritten;
        protected boolean recycle;
        protected boolean closed;

        public ContentRepositoryOutputStream(StandardContentClaim scc, ByteCountingOutputStream bcos, int initialLength) {
            this.scc = scc;
            this.bcos = bcos;
            this.initialLength = initialLength;
            this.bytesWritten = 0L;
            this.recycle = true;
            this.closed = false;
        }

        public String toString() {
            return "FileSystemRepository Stream [" + this.scc + "]";
        }

        @Override
        public synchronized void write(int b) throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            try {
                this.bcos.write(b);
            }
            catch (IOException ioe) {
                this.recycle = false;
                throw new IOException("Failed to write to " + this, ioe);
            }
            ++this.bytesWritten;
            this.scc.setLength(this.bytesWritten + (long)this.initialLength);
        }

        @Override
        public synchronized void write(byte[] b) throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            try {
                this.bcos.write(b);
            }
            catch (IOException ioe) {
                this.recycle = false;
                throw new IOException("Failed to write to " + this, ioe);
            }
            this.bytesWritten += (long)b.length;
            this.scc.setLength(this.bytesWritten + (long)this.initialLength);
        }

        @Override
        public synchronized void write(byte[] b, int off, int len) throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            try {
                this.bcos.write(b, off, len);
            }
            catch (IOException ioe) {
                this.recycle = false;
                throw new IOException("Failed to write to " + this, ioe);
            }
            this.bytesWritten += (long)len;
            this.scc.setLength(this.bytesWritten + (long)this.initialLength);
        }

        @Override
        public synchronized void flush() throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            this.bcos.flush();
        }

        @Override
        public synchronized void close() throws IOException {
            this.closed = true;
            if (FileSystemRepository.this.alwaysSync) {
                ((FileOutputStream)this.bcos.getWrappedStream()).getFD().sync();
            }
            if (this.scc.getLength() < 0L) {
                this.scc.setLength(0L);
            }
            long resourceClaimLength = this.scc.getOffset() + this.scc.getLength();
            if (this.recycle && resourceClaimLength < FileSystemRepository.this.maxAppendableClaimLength) {
                boolean enqueued;
                ClaimLengthPair pair = new ClaimLengthPair(this.scc.getResourceClaim(), resourceClaimLength);
                boolean bl = enqueued = FileSystemRepository.this.writableClaimStreams.get(this.scc.getResourceClaim()) != null && FileSystemRepository.this.writableClaimQueue.offer(pair);
                if (enqueued) {
                    LOG.debug("Claim length less than max; Adding {} back to Writable Claim Queue", (Object)this);
                } else {
                    FileSystemRepository.this.writableClaimStreams.remove(this.scc.getResourceClaim());
                    FileSystemRepository.this.resourceClaimManager.freeze(this.scc.getResourceClaim());
                    this.bcos.close();
                    LOG.debug("Claim length less than max; Closing {} because could not add back to queue", (Object)this);
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Stack trace: ", (Throwable)new RuntimeException("Stack Trace for closing " + this));
                    }
                }
            } else {
                FileSystemRepository.this.resourceClaimManager.freeze(this.scc.getResourceClaim());
                FileSystemRepository.this.writableClaimQueue.remove(new ClaimLengthPair(this.scc.getResourceClaim(), resourceClaimLength));
                this.bcos.close();
                LOG.debug("Claim lenth >= max; Closing {}", (Object)this);
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Stack trace: ", (Throwable)new RuntimeException("Stack Trace for closing " + this));
                }
            }
        }
    }

    private static class ScanForActiveResourceClaims
    extends SimpleFileVisitor<Path> {
        private static final Pattern SECTION_NAME_PATTERN = Pattern.compile("\\d{0,4}");
        private final String containerName;
        private final ResourceClaimManager resourceClaimManager;
        private final Set<String> containerNames;
        private final Path rootPath;
        private final Set<ResourceClaim> activeResourceClaims = new HashSet<ResourceClaim>();
        private String sectionName = null;

        public ScanForActiveResourceClaims(Path rootPath, String containerName, ResourceClaimManager resourceClaimManager, Set<String> containerNames) {
            this.rootPath = rootPath;
            this.containerName = containerName;
            this.resourceClaimManager = resourceClaimManager;
            this.containerNames = containerNames;
        }

        public Set<ResourceClaim> getActiveResourceClaims() {
            return this.activeResourceClaims;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            LOG.warn("Content repository contains un-readable file or directory '" + file.getFileName() + "'. Skipping. ", (Throwable)exc);
            return FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (dir.equals(this.rootPath)) {
                return FileVisitResult.CONTINUE;
            }
            String dirName = dir.toFile().getName();
            if (this.containerNames.contains(dirName)) {
                LOG.debug("Obtaining active resource claims, will traverse into Container {}", (Object)dirName);
                return FileVisitResult.CONTINUE;
            }
            if (SECTION_NAME_PATTERN.matcher(dirName).matches()) {
                LOG.debug("Obtaining active resource claims, will traverse into Section {}", (Object)dirName);
                this.sectionName = dirName;
                return FileVisitResult.CONTINUE;
            }
            LOG.debug("Obtaining active resource claims, will NOT traverse into sub-directory {}", (Object)dirName);
            return FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) throws IOException {
            if (attrs.isDirectory()) {
                return FileVisitResult.CONTINUE;
            }
            File file = path.toFile();
            if (this.sectionName == null || !this.sectionName.equals(file.getParentFile().getName())) {
                LOG.debug("Obtaining active resource claims, will NOT consider {} because its parent is not the current section", (Object)file);
                return FileVisitResult.CONTINUE;
            }
            String identifier = file.getName();
            ResourceClaim resourceClaim = this.resourceClaimManager.getResourceClaim(this.containerName, this.sectionName, identifier);
            if (resourceClaim == null) {
                resourceClaim = this.resourceClaimManager.newResourceClaim(this.containerName, this.sectionName, identifier, false, false);
            }
            this.activeResourceClaims.add(resourceClaim);
            return FileVisitResult.CONTINUE;
        }
    }

    protected static class ClaimLengthPair {
        private final ResourceClaim claim;
        private final Long length;

        public ClaimLengthPair(ResourceClaim claim, Long length) {
            this.claim = claim;
            this.length = length;
        }

        public ResourceClaim getClaim() {
            return this.claim;
        }

        public Long getLength() {
            return this.length;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.claim == null ? 0 : this.claim.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ClaimLengthPair other = (ClaimLengthPair)obj;
            return this.claim.equals(other.getClaim());
        }
    }

    private class ContainerState {
        private final String containerName;
        private final AtomicLong archivedFileCount = new AtomicLong(0L);
        private final long backPressureBytes;
        private final long capacity;
        private final boolean archiveEnabled;
        private final Lock lock = new ReentrantLock();
        private final Condition condition = this.lock.newCondition();
        private volatile long bytesUsed = 0L;

        public ContainerState(String containerName, boolean archiveEnabled, long backPressureBytes, long capacity) {
            this.containerName = containerName;
            this.archiveEnabled = archiveEnabled;
            this.backPressureBytes = backPressureBytes;
            this.capacity = capacity;
        }

        public boolean isWaitRequired() {
            if (!this.archiveEnabled) {
                return false;
            }
            long used = this.bytesUsed;
            if (used == 0L) {
                try {
                    long free = FileSystemRepository.this.getContainerUsableSpace(this.containerName);
                    this.bytesUsed = used = this.capacity - free;
                }
                catch (IOException e) {
                    return false;
                }
            }
            return used >= this.backPressureBytes && this.archivedFileCount.get() > 0L;
        }

        public void waitForArchiveExpiration() {
            if (!this.archiveEnabled) {
                return;
            }
            this.lock.lock();
            try {
                while (this.isWaitRequired()) {
                    try {
                        LOG.info("Unable to write to container {} due to archive file size constraints; waiting for archive cleanup", (Object)this.containerName);
                        this.condition.await();
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        public void signalCreationReady() {
            if (!this.archiveEnabled) {
                return;
            }
            this.lock.lock();
            try {
                try {
                    long free = FileSystemRepository.this.getContainerUsableSpace(this.containerName);
                    this.bytesUsed = this.capacity - free;
                }
                catch (Exception e) {
                    this.bytesUsed = 0L;
                }
                LOG.debug("Container {} signaled to allow Content Claim Creation", (Object)this.containerName);
                this.condition.signal();
            }
            finally {
                this.lock.unlock();
            }
        }

        public void incrementArchiveCount() {
            this.archivedFileCount.incrementAndGet();
        }

        public void decrementArchiveCount() {
            this.archivedFileCount.decrementAndGet();
        }
    }

    private class DestroyExpiredArchiveClaims
    implements Runnable {
        private final String containerName;
        private final Path containerPath;

        private DestroyExpiredArchiveClaims(String containerName, Path containerPath) {
            this.containerName = containerName;
            this.containerPath = containerPath;
        }

        @Override
        public void run() {
            try {
                long oldestContainerArchive;
                if (FileSystemRepository.this.oldestArchiveDate.get() > System.currentTimeMillis() - FileSystemRepository.this.maxArchiveMillis) {
                    Long minRequiredSpace = (Long)FileSystemRepository.this.minUsableContainerBytesForArchive.get(this.containerName);
                    if (minRequiredSpace == null) {
                        return;
                    }
                    try {
                        long usableSpace = FileSystemRepository.this.getContainerUsableSpace(this.containerName);
                        if (usableSpace > minRequiredSpace) {
                            return;
                        }
                    }
                    catch (Exception e) {
                        LOG.error("Failed to determine space available in container {}; will attempt to cleanup archive", (Object)this.containerName);
                    }
                }
                Thread.currentThread().setName("Cleanup Archive for " + this.containerName);
                try {
                    oldestContainerArchive = FileSystemRepository.this.destroyExpiredArchives(this.containerName, this.containerPath);
                    ContainerState containerState = (ContainerState)FileSystemRepository.this.containerStateMap.get(this.containerName);
                    containerState.signalCreationReady();
                }
                catch (IOException ioe) {
                    LOG.error("Failed to cleanup archive for container {} due to {}", (Object)this.containerName, (Object)ioe.toString());
                    if (LOG.isDebugEnabled()) {
                        LOG.error("", (Throwable)ioe);
                    }
                    return;
                }
                if (oldestContainerArchive < 0L) {
                    boolean updated;
                    do {
                        long oldest;
                        if (oldestContainerArchive < (oldest = FileSystemRepository.this.oldestArchiveDate.get())) {
                            updated = FileSystemRepository.this.oldestArchiveDate.compareAndSet(oldest, oldestContainerArchive);
                            if (!updated || !LOG.isDebugEnabled()) continue;
                            LOG.debug("Oldest Archive Date is now {}", (Object)new Date(oldestContainerArchive));
                            continue;
                        }
                        updated = true;
                    } while (!updated);
                }
            }
            catch (Throwable t) {
                LOG.error("Failed to cleanup archive for container {} due to {}", (Object)this.containerName, (Object)t.toString());
                LOG.error("", t);
            }
        }
    }

    private static class ArchiveInfo {
        private final Path containerPath;
        private final String relativePath;
        private final String name;
        private final long size;
        private final long lastModTime;

        public ArchiveInfo(Path containerPath, Path path, long size, long lastModTime) {
            this.containerPath = containerPath;
            this.relativePath = containerPath.relativize(path).toString();
            this.name = path.toFile().getName();
            this.size = size;
            this.lastModTime = lastModTime;
        }

        public String getName() {
            return this.name;
        }

        public long getSize() {
            return this.size;
        }

        public long getLastModTime() {
            return this.lastModTime;
        }

        public Path toPath() {
            return this.containerPath.resolve(this.relativePath);
        }
    }

    private class ArchiveOrDestroyDestructableClaims
    implements Runnable {
        private ArchiveOrDestroyDestructableClaims() {
        }

        @Override
        public void run() {
            try {
                int successCount;
                do {
                    successCount = 0;
                    ArrayList toRemove = new ArrayList();
                    for (Map.Entry entry : FileSystemRepository.this.reclaimable.entrySet()) {
                        String container = (String)entry.getKey();
                        ContainerState containerState = (ContainerState)FileSystemRepository.this.containerStateMap.get(container);
                        toRemove.clear();
                        ((BlockingQueue)entry.getValue()).drainTo(toRemove);
                        if (toRemove.isEmpty()) continue;
                        long start = System.nanoTime();
                        for (ResourceClaim claim : toRemove) {
                            if (FileSystemRepository.this.archiveData) {
                                try {
                                    if (!FileSystemRepository.this.archive(claim)) continue;
                                    containerState.incrementArchiveCount();
                                    ++successCount;
                                }
                                catch (Exception e) {
                                    LOG.warn("Failed to archive {} due to {}", (Object)claim, (Object)e.toString());
                                    if (!LOG.isDebugEnabled()) continue;
                                    LOG.warn("", (Throwable)e);
                                }
                                continue;
                            }
                            if (!FileSystemRepository.this.remove(claim)) continue;
                            ++successCount;
                        }
                        long nanos = System.nanoTime() - start;
                        long millis = TimeUnit.NANOSECONDS.toMillis(nanos);
                        if (successCount == 0) {
                            LOG.debug("No ContentClaims archived/removed for Container {}", (Object)container);
                            continue;
                        }
                        LOG.info("Successfully {} {} Resource Claims for Container {} in {} millis", new Object[]{FileSystemRepository.this.archiveData ? "archived" : "destroyed", successCount, container, millis});
                    }
                } while (successCount != 0);
                return;
            }
            catch (Throwable t) {
                LOG.error("Failed to handle destructable claims due to {}", (Object)t.toString());
                if (LOG.isDebugEnabled()) {
                    LOG.error("", t);
                }
                return;
            }
        }
    }

    private class BinDestructableClaims
    implements Runnable {
        private BinDestructableClaims() {
        }

        @Override
        public void run() {
            try {
                ArrayList toDestroy = new ArrayList();
                block4: while (true) {
                    toDestroy.clear();
                    FileSystemRepository.this.resourceClaimManager.drainDestructableClaims(toDestroy, 10000);
                    if (toDestroy.isEmpty()) {
                        return;
                    }
                    Iterator iterator = toDestroy.iterator();
                    block5: while (true) {
                        if (!iterator.hasNext()) continue block4;
                        ResourceClaim claim = (ResourceClaim)iterator.next();
                        String container = claim.getContainer();
                        BlockingQueue claimQueue = (BlockingQueue)FileSystemRepository.this.reclaimable.get(container);
                        try {
                            while (true) {
                                if (claimQueue.offer(claim, 10L, TimeUnit.MINUTES)) continue block5;
                                LOG.warn("Failed to clean up {} because old claims aren't being cleaned up fast enough. This Content Claim will remain in the Content Repository until NiFi is restarted, at which point it will be cleaned up", (Object)claim);
                            }
                        }
                        catch (InterruptedException ie) {
                            LOG.warn("Failed to clean up {} because thread was interrupted", (Object)claim);
                            continue;
                        }
                        break;
                    }
                    break;
                }
            }
            catch (Throwable t) {
                LOG.error("Failed to cleanup content claims due to {}", t);
                return;
            }
        }
    }
}

