/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller.repository.crypto;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.security.KeyManagementException;
import javax.crypto.CipherOutputStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.controller.repository.FileSystemRepository;
import org.apache.nifi.controller.repository.claim.ContentClaim;
import org.apache.nifi.controller.repository.claim.StandardContentClaim;
import org.apache.nifi.security.kms.EncryptionException;
import org.apache.nifi.security.kms.KeyProvider;
import org.apache.nifi.security.repository.RepositoryEncryptorUtils;
import org.apache.nifi.security.repository.RepositoryType;
import org.apache.nifi.security.repository.stream.RepositoryObjectStreamEncryptor;
import org.apache.nifi.security.repository.stream.aes.RepositoryObjectAESCTREncryptor;
import org.apache.nifi.stream.io.ByteCountingOutputStream;
import org.apache.nifi.stream.io.NonCloseableOutputStream;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.util.NiFiProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EncryptedFileSystemRepository
extends FileSystemRepository {
    private static final Logger logger = LoggerFactory.getLogger(EncryptedFileSystemRepository.class);
    private String activeKeyId;
    private KeyProvider keyProvider;

    public EncryptedFileSystemRepository() {
        this.keyProvider = null;
    }

    public EncryptedFileSystemRepository(NiFiProperties niFiProperties) throws IOException {
        super(niFiProperties);
        this.keyProvider = RepositoryEncryptorUtils.validateAndBuildRepositoryKeyProvider((NiFiProperties)niFiProperties, (RepositoryType)RepositoryType.CONTENT);
        this.setActiveKeyId(niFiProperties.getContentRepositoryEncryptionKeyId());
    }

    @Override
    public long importFrom(InputStream content, ContentClaim claim) throws IOException {
        try (OutputStream out = this.write(claim);){
            long l = StreamUtils.copy((InputStream)content, (OutputStream)out);
            return l;
        }
    }

    @Override
    public long exportTo(ContentClaim claim, OutputStream destination) throws IOException {
        logger.warn("Exporting content from {} to output stream {}. This content will be decrypted", (Object)claim.getResourceClaim().getId(), (Object)destination);
        return super.exportTo(claim, destination);
    }

    @Override
    public long exportTo(ContentClaim claim, OutputStream destination, long offset, long length) throws IOException {
        logger.warn("Exporting content from {} (offset: {}, length: {}) to output stream {}. This content will be decrypted", new Object[]{claim.getResourceClaim().getId(), offset, length, destination});
        return super.exportTo(claim, destination, offset, length);
    }

    @Override
    public long exportTo(ContentClaim claim, Path destination, boolean append) throws IOException {
        logger.warn("Exporting content from {} to path {}. This content will be decrypted", (Object)claim.getResourceClaim().getId(), (Object)destination);
        return super.exportTo(claim, destination, append);
    }

    @Override
    public long exportTo(ContentClaim claim, Path destination, boolean append, long offset, long length) throws IOException {
        logger.warn("Exporting content from {} (offset: {}, length: {}) to path {}. This content will be decrypted", new Object[]{claim.getResourceClaim().getId(), offset, length, destination});
        return super.exportTo(claim, destination, append, offset, length);
    }

    @Override
    public InputStream read(ContentClaim claim) throws IOException {
        InputStream inputStream = super.read(claim);
        try {
            String recordId = EncryptedFileSystemRepository.getRecordId(claim);
            logger.debug("Creating decrypted input stream to read flowfile content with record ID: " + recordId);
            InputStream decryptingInputStream = this.getDecryptingInputStream(inputStream, recordId);
            logger.debug("Reading from record ID {}", (Object)recordId);
            if (logger.isTraceEnabled()) {
                logger.trace("Stack trace: ", (Throwable)new RuntimeException("Stack Trace for reading from record ID " + recordId));
            }
            return decryptingInputStream;
        }
        catch (KeyManagementException | EncryptionException e) {
            logger.error("Encountered an error instantiating the encrypted content repository input stream: " + e.getMessage());
            throw new IOException("Error creating encrypted content repository input stream", e);
        }
    }

    private InputStream getDecryptingInputStream(InputStream inputStream, String recordId) throws KeyManagementException, EncryptionException {
        RepositoryObjectAESCTREncryptor encryptor = new RepositoryObjectAESCTREncryptor();
        encryptor.initialize(this.keyProvider);
        return encryptor.decrypt(inputStream, recordId);
    }

    @Override
    public OutputStream write(ContentClaim claim) throws IOException {
        StandardContentClaim scc = EncryptedFileSystemRepository.validateContentClaimForWriting(claim);
        ByteCountingOutputStream claimStream = this.getWritableClaimStreamByResourceClaim(scc.getResourceClaim());
        long startingOffset = claimStream.getBytesWritten();
        try {
            String keyId = this.getActiveKeyId();
            String recordId = EncryptedFileSystemRepository.getRecordId(claim);
            logger.debug("Creating encrypted output stream (keyId: " + keyId + ") to write flowfile content with record ID: " + recordId);
            OutputStream out = this.getEncryptedOutputStream(scc, claimStream, startingOffset, keyId, recordId);
            logger.debug("Writing to {}", (Object)out);
            if (logger.isTraceEnabled()) {
                logger.trace("Stack trace: ", (Throwable)new RuntimeException("Stack Trace for writing to " + out));
            }
            return out;
        }
        catch (KeyManagementException | EncryptionException e) {
            logger.error("Encountered an error instantiating the encrypted content repository output stream: " + e.getMessage());
            throw new IOException("Error creating encrypted content repository output stream", e);
        }
    }

    String getActiveKeyId() {
        return this.activeKeyId;
    }

    public void setActiveKeyId(String activeKeyId) {
        if (StringUtils.isNotBlank((CharSequence)activeKeyId) && this.keyProvider.keyExists(activeKeyId)) {
            this.activeKeyId = activeKeyId;
            logger.debug("Set active key ID to '" + activeKeyId + "'");
        } else {
            logger.warn("Attempted to set active key ID to '" + activeKeyId + "' but that is not a valid or available key ID. Keeping active key ID as '" + this.activeKeyId + "'");
        }
    }

    public static String getRecordId(ContentClaim claim) {
        if (claim != null && claim.getResourceClaim() != null && !StringUtils.isBlank((CharSequence)claim.getResourceClaim().getId())) {
            return "nifi-ecr-rc-" + claim.getResourceClaim().getId() + "+" + claim.getOffset();
        }
        String tempId = "nifi-ecr-ts-" + System.nanoTime();
        logger.error("Cannot determine record ID from null content claim or claim with missing/empty resource claim ID; using timestamp-generated ID: " + tempId + "+0");
        return tempId;
    }

    private OutputStream getEncryptedOutputStream(StandardContentClaim scc, ByteCountingOutputStream claimStream, long startingOffset, String keyId, String recordId) throws KeyManagementException, EncryptionException {
        RepositoryObjectAESCTREncryptor encryptor = new RepositoryObjectAESCTREncryptor();
        encryptor.initialize(this.keyProvider);
        return new EncryptedContentRepositoryOutputStream(scc, claimStream, (RepositoryObjectStreamEncryptor)encryptor, recordId, keyId, startingOffset);
    }

    private class EncryptedContentRepositoryOutputStream
    extends FileSystemRepository.ContentRepositoryOutputStream {
        private final CipherOutputStream cipherOutputStream;
        private final long startingOffset;

        EncryptedContentRepositoryOutputStream(StandardContentClaim scc, ByteCountingOutputStream byteCountingOutputStream, RepositoryObjectStreamEncryptor encryptor, String recordId, String keyId, long startingOffset) throws EncryptionException {
            super(scc, byteCountingOutputStream, 0);
            this.startingOffset = startingOffset;
            this.cipherOutputStream = (CipherOutputStream)encryptor.encrypt((OutputStream)new NonCloseableOutputStream((OutputStream)byteCountingOutputStream), recordId, keyId);
        }

        @Override
        public String toString() {
            return "EncryptedFileSystemRepository Stream [" + this.scc + "]";
        }

        @Override
        public synchronized void write(int b) throws IOException {
            ByteBuffer bb = ByteBuffer.allocate(4);
            bb.putInt(b);
            this.writeBytes(bb.array(), 0, 4);
        }

        @Override
        public synchronized void write(byte[] b) throws IOException {
            this.writeBytes(b, 0, b.length);
        }

        @Override
        public synchronized void write(byte[] b, int off, int len) throws IOException {
            this.writeBytes(b, off, len);
        }

        private void writeBytes(byte[] b, int off, int len) throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            try {
                this.cipherOutputStream.write(b, off, len);
                this.scc.setLength(this.bcos.getBytesWritten() - this.startingOffset);
            }
            catch (IOException ioe) {
                this.recycle = false;
                throw new IOException("Failed to write to " + this, ioe);
            }
        }

        @Override
        public synchronized void flush() throws IOException {
            if (this.closed) {
                throw new IOException("Stream is closed");
            }
            this.cipherOutputStream.flush();
        }

        @Override
        public synchronized void close() throws IOException {
            this.closed = true;
            this.cipherOutputStream.flush();
            this.cipherOutputStream.close();
            this.scc.setLength(this.bcos.getBytesWritten() - this.startingOffset);
            super.close();
        }
    }
}

