/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller.queue.clustered.client.async.nio;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.OptionalInt;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PeerChannel
implements Closeable {
    private static final int END_OF_FILE = -1;
    private static final int EMPTY_BUFFER = 0;
    private static final Logger logger = LoggerFactory.getLogger(PeerChannel.class);
    private final SocketChannel socketChannel;
    private final SSLEngine sslEngine;
    private final String peerDescription;
    private final ByteBuffer singleByteBuffer = ByteBuffer.allocate(1);
    private ByteBuffer destinationBuffer = ByteBuffer.allocate(16384);
    private final ByteBuffer streamBuffer = ByteBuffer.allocate(16384);
    private ByteBuffer applicationBuffer = ByteBuffer.allocate(0);

    public PeerChannel(SocketChannel socketChannel, SSLEngine sslEngine, String peerDescription) {
        this.socketChannel = socketChannel;
        this.sslEngine = sslEngine;
        this.peerDescription = peerDescription;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        try {
            if (this.sslEngine == null) {
                logger.debug("Closing Peer Channel [{}] SSLEngine not configured", (Object)this.peerDescription);
            } else {
                logger.debug("Closing Peer Channel [{}] SSLEngine close started", (Object)this.peerDescription);
                this.sslEngine.closeOutbound();
                ByteBuffer inputBuffer = ByteBuffer.allocate(0);
                ByteBuffer outputBuffer = ByteBuffer.allocate(this.sslEngine.getSession().getPacketBufferSize());
                SSLEngineResult wrapResult = this.sslEngine.wrap(inputBuffer, outputBuffer);
                SSLEngineResult.Status status = wrapResult.getStatus();
                outputBuffer.flip();
                if (SSLEngineResult.Status.OK == status) {
                    this.write(outputBuffer);
                    outputBuffer.clear();
                    wrapResult = this.sslEngine.wrap(inputBuffer, outputBuffer);
                    status = wrapResult.getStatus();
                }
                if (SSLEngineResult.Status.CLOSED != status) {
                    throw new SSLException(String.format("Closing Peer Channel [%s] Invalid Wrap Result Status [%s]", new Object[]{this.peerDescription, status}));
                }
                this.write(outputBuffer);
                logger.debug("Closing Peer Channel [{}] SSLEngine close completed", (Object)this.peerDescription);
            }
        }
        finally {
            logger.debug("Closing Peer Channel [{}] Socket Channel close started", (Object)this.peerDescription);
            this.socketChannel.close();
        }
    }

    public boolean isConnected() {
        return this.socketChannel.isConnected();
    }

    public boolean isOpen() {
        return this.socketChannel.isOpen();
    }

    public String getPeerDescription() {
        return this.peerDescription;
    }

    public boolean write(byte b) throws IOException {
        this.singleByteBuffer.clear();
        this.singleByteBuffer.put(b);
        this.singleByteBuffer.rewind();
        ByteBuffer prepared = this.prepareForWrite(this.singleByteBuffer);
        int bytesWritten = this.write(prepared);
        return bytesWritten > 0;
    }

    public OptionalInt read() throws IOException {
        this.singleByteBuffer.clear();
        int bytesRead = this.read(this.singleByteBuffer);
        if (bytesRead < 0) {
            return OptionalInt.of(-1);
        }
        if (bytesRead == 0) {
            return OptionalInt.empty();
        }
        this.singleByteBuffer.flip();
        byte read = this.singleByteBuffer.get();
        return OptionalInt.of(read & 0xFF);
    }

    public ByteBuffer prepareForWrite(ByteBuffer plaintext) throws IOException {
        logger.trace("Channel [{}] Buffer wrap started: Input Bytes [{}]", (Object)this.peerDescription, (Object)plaintext.remaining());
        if (this.sslEngine == null) {
            return plaintext;
        }
        ByteBuffer prepared = ByteBuffer.allocate(Math.min(85, plaintext.capacity() - plaintext.position()));
        while (plaintext.hasRemaining()) {
            this.encrypt(plaintext);
            int bytesRemaining = prepared.capacity() - prepared.position();
            if (bytesRemaining < this.destinationBuffer.remaining()) {
                ByteBuffer temp = ByteBuffer.allocate(prepared.capacity() + this.sslEngine.getSession().getApplicationBufferSize());
                prepared.flip();
                temp.put(prepared);
                prepared = temp;
            }
            prepared.put(this.destinationBuffer);
        }
        prepared.flip();
        logger.trace("Channel [{}] Buffer wrap completed: Prepared Bytes [{}]", (Object)this.peerDescription, (Object)prepared.remaining());
        return prepared;
    }

    public int write(ByteBuffer preparedBuffer) throws IOException {
        return this.socketChannel.write(preparedBuffer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int read(ByteBuffer dst) throws IOException {
        int bytesCopied = this.copy(this.applicationBuffer, dst);
        if (bytesCopied != 0) {
            return bytesCopied;
        }
        int bytesRead = this.socketChannel.read(this.streamBuffer);
        logger.trace("Channel [{}] Socket read completed: bytes [{}]", (Object)this.peerDescription, (Object)bytesRead);
        if (bytesRead == -1) {
            return -1;
        }
        if (this.streamBuffer.remaining() == 0) {
            return 0;
        }
        this.streamBuffer.flip();
        try {
            if (this.sslEngine == null) {
                this.cloneToApplicationBuffer(this.streamBuffer);
                int n = this.copy(this.applicationBuffer, dst);
                return n;
            }
            boolean decrypted = this.decrypt(this.streamBuffer);
            logger.trace("Channel [{}] Decryption completed [{}]", (Object)this.peerDescription, (Object)decrypted);
            if (decrypted) {
                this.cloneToApplicationBuffer(this.destinationBuffer);
                logger.trace("Cloned destination buffer to application buffer");
                int n = this.copy(this.applicationBuffer, dst);
                return n;
            }
            logger.trace("Channel [{}] Socket Channel read required", (Object)this.peerDescription);
            int n = 0;
            return n;
        }
        finally {
            this.streamBuffer.compact();
        }
    }

    private void cloneToApplicationBuffer(ByteBuffer buffer) {
        if (this.applicationBuffer.capacity() < buffer.remaining()) {
            this.applicationBuffer = ByteBuffer.allocate(buffer.remaining());
        } else {
            this.applicationBuffer.clear();
        }
        this.applicationBuffer.put(buffer);
        this.applicationBuffer.flip();
    }

    private int copy(ByteBuffer src, ByteBuffer dst) {
        if (src != null && src.hasRemaining()) {
            int bytesToCopy = Math.min(dst.remaining(), src.remaining());
            if (bytesToCopy < 1) {
                return bytesToCopy;
            }
            byte[] buff = new byte[bytesToCopy];
            src.get(buff);
            dst.put(buff);
            return bytesToCopy;
        }
        return 0;
    }

    private void encrypt(ByteBuffer plaintext) throws IOException {
        if (this.sslEngine == null) {
            throw new SSLException("Unable to encrypt message because no SSLEngine has been configured");
        }
        this.destinationBuffer.clear();
        while (true) {
            SSLEngineResult result = this.sslEngine.wrap(plaintext, this.destinationBuffer);
            this.logOperationResult("WRAP", result);
            switch (result.getStatus()) {
                case OK: {
                    this.destinationBuffer.flip();
                    return;
                }
                case CLOSED: {
                    throw new IOException("Failed to encrypt data to write to Peer " + this.peerDescription + " because Peer unexpectedly closed connection");
                }
                case BUFFER_OVERFLOW: {
                    ByteBuffer tempBuffer = ByteBuffer.allocate(this.destinationBuffer.capacity() + this.sslEngine.getSession().getApplicationBufferSize());
                    this.destinationBuffer.flip();
                    tempBuffer.put(this.destinationBuffer);
                    this.destinationBuffer = tempBuffer;
                    break;
                }
                case BUFFER_UNDERFLOW: {
                    throw new IOException("Received unexpected Buffer Underflow result when encrypting data to write to Peer " + this.peerDescription);
                }
            }
        }
    }

    private boolean decrypt(ByteBuffer encrypted) throws IOException {
        if (this.sslEngine == null) {
            throw new SSLException("Unable to decrypt message because no SSLEngine has been configured");
        }
        this.destinationBuffer.clear();
        block6: while (true) {
            SSLEngineResult result = this.sslEngine.unwrap(encrypted, this.destinationBuffer);
            this.logOperationResult("UNWRAP", result);
            switch (result.getStatus()) {
                case OK: {
                    if (SSLEngineResult.HandshakeStatus.FINISHED == result.getHandshakeStatus()) continue block6;
                    this.destinationBuffer.flip();
                    return true;
                }
                case CLOSED: {
                    throw new IOException("Failed to decrypt data from Peer " + this.peerDescription + " because Peer unexpectedly closed connection");
                }
                case BUFFER_OVERFLOW: {
                    ByteBuffer tempBuffer = ByteBuffer.allocate(encrypted.position() + this.sslEngine.getSession().getApplicationBufferSize());
                    this.destinationBuffer.flip();
                    tempBuffer.put(this.destinationBuffer);
                    this.destinationBuffer = tempBuffer;
                    break;
                }
                case BUFFER_UNDERFLOW: {
                    return false;
                }
            }
        }
    }

    public void performHandshake() throws IOException {
        if (this.sslEngine == null) {
            return;
        }
        this.sslEngine.beginHandshake();
        ByteBuffer emptyMessage = ByteBuffer.allocate(0);
        ByteBuffer unwrapBuffer = ByteBuffer.allocate(0);
        block6: while (true) {
            SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
            this.logHandshakeStatus(handshakeStatus);
            block0 : switch (handshakeStatus) {
                case FINISHED: 
                case NOT_HANDSHAKING: {
                    this.streamBuffer.clear();
                    this.destinationBuffer.clear();
                    this.logHandshakeCompleted();
                    return;
                }
                case NEED_TASK: {
                    Runnable runnable;
                    while ((runnable = this.sslEngine.getDelegatedTask()) != null) {
                        runnable.run();
                    }
                    continue block6;
                }
                case NEED_WRAP: {
                    this.encrypt(emptyMessage);
                    int bytesWritten = this.write(this.destinationBuffer);
                    this.logHandshakeStatusBytes(handshakeStatus, "Socket write completed", bytesWritten);
                    break;
                }
                case NEED_UNWRAP: {
                    while (this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_UNWRAP) {
                        boolean decrypted = this.decrypt(unwrapBuffer);
                        SSLEngineResult.HandshakeStatus unwrapHandshakeStatus = this.sslEngine.getHandshakeStatus();
                        if (decrypted || unwrapHandshakeStatus == SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
                            this.logHandshakeStatus(unwrapHandshakeStatus, "Decryption completed");
                            break block0;
                        }
                        if (unwrapBuffer.capacity() - unwrapBuffer.position() < 1) {
                            this.logHandshakeStatus(unwrapHandshakeStatus, "Increasing unwrap buffer for decryption");
                            ByteBuffer tempBuffer = ByteBuffer.allocate(unwrapBuffer.capacity() + this.sslEngine.getSession().getApplicationBufferSize());
                            tempBuffer.put(unwrapBuffer);
                            unwrapBuffer = tempBuffer;
                            unwrapBuffer.flip();
                            continue;
                        }
                        this.logHandshakeStatus(unwrapHandshakeStatus, "Socket read started");
                        unwrapBuffer.compact();
                        int bytesRead = this.socketChannel.read(unwrapBuffer);
                        unwrapBuffer.flip();
                        this.logHandshakeStatusBytes(unwrapHandshakeStatus, "Socket read completed", bytesRead);
                    }
                    continue block6;
                }
            }
        }
    }

    private void logOperationResult(String operation, SSLEngineResult sslEngineResult) {
        logger.trace("Channel [{}] {} [{}]", new Object[]{this.peerDescription, operation, sslEngineResult});
    }

    private void logHandshakeCompleted() {
        SSLSession sslSession = this.sslEngine.getSession();
        logger.debug("Channel [{}] Handshake Completed Protocol [{}] Cipher Suite [{}]", new Object[]{this.peerDescription, sslSession.getProtocol(), sslSession.getCipherSuite()});
    }

    private void logHandshakeStatus(SSLEngineResult.HandshakeStatus handshakeStatus) {
        logger.debug("Channel [{}] Handshake Status [{}]", (Object)this.peerDescription, (Object)handshakeStatus);
    }

    private void logHandshakeStatus(SSLEngineResult.HandshakeStatus handshakeStatus, String operation) {
        logger.debug("Channel [{}] Handshake Status [{}] {}", new Object[]{this.peerDescription, handshakeStatus, operation});
    }

    private void logHandshakeStatusBytes(SSLEngineResult.HandshakeStatus handshakeStatus, String operation, int bytes) {
        logger.debug("Channel [{}] Handshake Status [{}] {} Bytes [{}]", new Object[]{this.peerDescription, handshakeStatus, operation, bytes});
    }
}

