/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller.leader.election;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.leader.LeaderSelector;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListener;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListenerAdapter;
import org.apache.curator.framework.recipes.leader.Participant;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.retry.RetryNTimes;
import org.apache.nifi.controller.cluster.ZooKeeperClientConfig;
import org.apache.nifi.controller.leader.election.CuratorACLProviderFactory;
import org.apache.nifi.controller.leader.election.LeaderElectionManager;
import org.apache.nifi.controller.leader.election.LeaderElectionStateChangeListener;
import org.apache.nifi.engine.FlowEngine;
import org.apache.nifi.util.NiFiProperties;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.common.PathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CuratorLeaderElectionManager
implements LeaderElectionManager {
    private static final Logger logger = LoggerFactory.getLogger(CuratorLeaderElectionManager.class);
    private final FlowEngine leaderElectionMonitorEngine;
    private final ZooKeeperClientConfig zkConfig;
    private CuratorFramework curatorClient;
    private volatile boolean stopped = true;
    private final Map<String, LeaderRole> leaderRoles = new HashMap<String, LeaderRole>();
    private final Map<String, RegisteredRole> registeredRoles = new HashMap<String, RegisteredRole>();

    public CuratorLeaderElectionManager(int threadPoolSize, NiFiProperties properties) {
        this.leaderElectionMonitorEngine = new FlowEngine(threadPoolSize, "Leader Election Notification", true);
        this.zkConfig = ZooKeeperClientConfig.createConfig(properties);
    }

    @Override
    public synchronized void start() {
        if (!this.stopped) {
            return;
        }
        this.stopped = false;
        this.curatorClient = this.createClient();
        for (Map.Entry<String, RegisteredRole> entry : this.registeredRoles.entrySet()) {
            RegisteredRole role = entry.getValue();
            this.register(entry.getKey(), role.getListener(), role.getParticipantId());
        }
        logger.info("{} started", (Object)this);
    }

    @Override
    public void register(String roleName, LeaderElectionStateChangeListener listener) {
        this.register(roleName, listener, null);
    }

    private String getElectionPath(String roleName) {
        String rootPath = this.zkConfig.getRootPath();
        String leaderPath = rootPath + (rootPath.endsWith("/") ? "" : "/") + "leaders/" + roleName;
        return leaderPath;
    }

    @Override
    public synchronized void register(String roleName, LeaderElectionStateChangeListener listener, String participantId) {
        boolean isParticipant;
        logger.debug("{} Registering new Leader Selector for role {}", (Object)this, (Object)roleName);
        LeaderRole currentRole = this.leaderRoles.get(roleName);
        if (currentRole != null && (currentRole.isParticipant() || participantId == null)) {
            logger.info("{} Attempted to register Leader Election for role '{}' but this role is already registered", (Object)this, (Object)roleName);
            return;
        }
        String leaderPath = this.getElectionPath(roleName);
        try {
            PathUtils.validatePath((String)leaderPath);
        }
        catch (IllegalArgumentException e) {
            throw new IllegalStateException("Cannot register leader election for role '" + roleName + "' because this is not a valid role name");
        }
        this.registeredRoles.put(roleName, new RegisteredRole(participantId, listener));
        boolean bl = isParticipant = participantId != null && !participantId.trim().isEmpty();
        if (!this.isStopped()) {
            ElectionListener electionListener = new ElectionListener(roleName, listener);
            LeaderSelector leaderSelector = new LeaderSelector(this.curatorClient, leaderPath, (ExecutorService)this.leaderElectionMonitorEngine, (LeaderSelectorListener)electionListener);
            if (isParticipant) {
                leaderSelector.autoRequeue();
                leaderSelector.setId(participantId);
                leaderSelector.start();
            }
            LeaderRole leaderRole = new LeaderRole(leaderSelector, electionListener, isParticipant);
            this.leaderRoles.put(roleName, leaderRole);
        }
        if (isParticipant) {
            logger.info("{} Registered new Leader Selector for role {}; this node is an active participant in the election.", (Object)this, (Object)roleName);
        } else {
            logger.info("{} Registered new Leader Selector for role {}; this node is a silent observer in the election.", (Object)this, (Object)roleName);
        }
    }

    @Override
    public synchronized void unregister(String roleName) {
        this.registeredRoles.remove(roleName);
        LeaderRole leaderRole = this.leaderRoles.remove(roleName);
        if (leaderRole == null) {
            logger.info("Cannot unregister Leader Election Role '{}' becuase that role is not registered", (Object)roleName);
            return;
        }
        LeaderSelector leaderSelector = leaderRole.getLeaderSelector();
        if (leaderSelector == null) {
            logger.info("Cannot unregister Leader Election Role '{}' becuase that role is not registered", (Object)roleName);
            return;
        }
        leaderSelector.close();
        logger.info("This node is no longer registered to be elected as the Leader for Role '{}'", (Object)roleName);
    }

    @Override
    public synchronized void stop() {
        this.stopped = true;
        for (Map.Entry<String, LeaderRole> entry : this.leaderRoles.entrySet()) {
            LeaderRole role = entry.getValue();
            LeaderSelector selector = role.getLeaderSelector();
            try {
                selector.close();
            }
            catch (Exception e) {
                logger.warn("Failed to close Leader Selector for {}", (Object)entry.getKey(), (Object)e);
            }
        }
        this.leaderRoles.clear();
        if (this.curatorClient != null) {
            this.curatorClient.close();
            this.curatorClient = null;
        }
        logger.info("{} stopped and closed", (Object)this);
    }

    @Override
    public boolean isStopped() {
        return this.stopped;
    }

    public String toString() {
        return "CuratorLeaderElectionManager[stopped=" + this.isStopped() + "]";
    }

    private synchronized LeaderRole getLeaderRole(String roleName) {
        return this.leaderRoles.get(roleName);
    }

    @Override
    public boolean isLeader(String roleName) {
        LeaderRole role = this.getLeaderRole(roleName);
        if (role == null) {
            return false;
        }
        return role.isLeader();
    }

    @Override
    public String getLeader(String roleName) {
        Participant participant;
        if (this.isStopped()) {
            return this.determineLeaderExternal(roleName);
        }
        LeaderRole role = this.getLeaderRole(roleName);
        if (role == null) {
            return this.determineLeaderExternal(roleName);
        }
        try {
            participant = role.getLeaderSelector().getLeader();
        }
        catch (Exception e) {
            logger.debug("Unable to determine leader for role '{}'; returning null", (Object)roleName);
            return null;
        }
        if (participant == null) {
            return null;
        }
        String participantId = participant.getId();
        if (StringUtils.isEmpty((CharSequence)participantId)) {
            return null;
        }
        return participantId;
    }

    @Override
    public boolean isLeaderElected(String roleName) {
        String leaderAddress = this.determineLeaderExternal(roleName);
        return !StringUtils.isEmpty((CharSequence)leaderAddress);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    private String determineLeaderExternal(String roleName) {
        try (CuratorFramework client = this.createClient();){
            LeaderSelectorListener electionListener = new LeaderSelectorListener(){

                public void stateChanged(CuratorFramework client, ConnectionState newState) {
                }

                public void takeLeadership(CuratorFramework client) throws Exception {
                }
            };
            String electionPath = this.getElectionPath(roleName);
            LeaderSelector selector = new LeaderSelector(client, electionPath, electionListener);
            try {
                Participant leader = selector.getLeader();
                String string = leader == null ? null : leader.getId();
                return string;
            }
            catch (KeeperException.NoNodeException nne) {
                String string = null;
                client.close();
                return string;
            }
            catch (Exception e) {
                logger.warn("Unable to determine the Elected Leader for role '{}' due to {}; assuming no leader has been elected", (Object)roleName, (Object)e.toString());
                if (logger.isDebugEnabled()) {
                    logger.warn("", (Throwable)e);
                }
                String string = null;
                {
                    catch (Throwable throwable) {
                        throw throwable;
                    }
                }
                client.close();
                return string;
            }
        }
    }

    private CuratorFramework createClient() {
        RetryNTimes retryPolicy = new RetryNTimes(1, 100);
        CuratorACLProviderFactory aclProviderFactory = new CuratorACLProviderFactory();
        CuratorFramework client = CuratorFrameworkFactory.builder().connectString(this.zkConfig.getConnectString()).sessionTimeoutMs(this.zkConfig.getSessionTimeoutMillis()).connectionTimeoutMs(this.zkConfig.getConnectionTimeoutMillis()).retryPolicy((RetryPolicy)retryPolicy).aclProvider(aclProviderFactory.create(this.zkConfig)).defaultData(new byte[0]).build();
        client.start();
        return client;
    }

    private class ElectionListener
    extends LeaderSelectorListenerAdapter
    implements LeaderSelectorListener {
        private final String roleName;
        private final LeaderElectionStateChangeListener listener;
        private volatile boolean leader;

        public ElectionListener(String roleName, LeaderElectionStateChangeListener listener) {
            this.roleName = roleName;
            this.listener = listener;
        }

        public boolean isLeader() {
            return this.leader;
        }

        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            logger.info("{} Connection State changed to {}", (Object)this, (Object)newState.name());
            super.stateChanged(client, newState);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public void takeLeadership(CuratorFramework client) throws Exception {
            this.leader = true;
            logger.info("{} This node has been elected Leader for Role '{}'", (Object)this, (Object)this.roleName);
            if (this.listener != null) {
                try {
                    this.listener.onLeaderElection();
                }
                catch (Exception e) {
                    logger.error("This node was elected Leader for Role '{}' but failed to take leadership. Will relinquish leadership role. Failure was due to: {}", (Object)this.roleName, (Object)e);
                    logger.error("", (Throwable)e);
                    this.leader = false;
                    Thread.sleep(1000L);
                    return;
                }
            }
            try {
                while (!CuratorLeaderElectionManager.this.isStopped()) {
                    try {
                        Thread.sleep(100L);
                    }
                    catch (InterruptedException ie) {
                        logger.info("{} has been interrupted; no longer leader for role '{}'", (Object)this, (Object)this.roleName);
                        Thread.currentThread().interrupt();
                        this.leader = false;
                        logger.info("{} This node is no longer leader for role '{}'", (Object)this, (Object)this.roleName);
                        if (this.listener == null) return;
                        try {
                            this.listener.onLeaderRelinquish();
                            return;
                        }
                        catch (Exception e) {
                            logger.error("This node is no longer leader for role '{}' but failed to shutdown leadership responsibilities properly due to: {}", (Object)this.roleName, (Object)e.toString());
                            if (!logger.isDebugEnabled()) return;
                            logger.error("", (Throwable)e);
                        }
                        return;
                    }
                }
            }
            finally {
                block16: {
                    this.leader = false;
                    logger.info("{} This node is no longer leader for role '{}'", (Object)this, (Object)this.roleName);
                    if (this.listener != null) {
                        try {
                            this.listener.onLeaderRelinquish();
                        }
                        catch (Exception e) {
                            logger.error("This node is no longer leader for role '{}' but failed to shutdown leadership responsibilities properly due to: {}", (Object)this.roleName, (Object)e.toString());
                            if (!logger.isDebugEnabled()) break block16;
                            logger.error("", (Throwable)e);
                        }
                    }
                }
            }
        }
    }

    private static class RegisteredRole {
        private final LeaderElectionStateChangeListener listener;
        private final String participantId;

        public RegisteredRole(String participantId, LeaderElectionStateChangeListener listener) {
            this.participantId = participantId;
            this.listener = listener;
        }

        public LeaderElectionStateChangeListener getListener() {
            return this.listener;
        }

        public String getParticipantId() {
            return this.participantId;
        }
    }

    private static class LeaderRole {
        private final LeaderSelector leaderSelector;
        private final ElectionListener electionListener;
        private final boolean participant;

        public LeaderRole(LeaderSelector leaderSelector, ElectionListener electionListener, boolean participant) {
            this.leaderSelector = leaderSelector;
            this.electionListener = electionListener;
            this.participant = participant;
        }

        public LeaderSelector getLeaderSelector() {
            return this.leaderSelector;
        }

        public boolean isLeader() {
            return this.electionListener.isLeader();
        }

        public boolean isParticipant() {
            return this.participant;
        }
    }
}

