/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.encrypt;

import java.nio.charset.StandardCharsets;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.encrypt.EncryptionException;
import org.apache.nifi.security.kms.CryptoUtils;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.KeyDerivationFunction;
import org.apache.nifi.security.util.crypto.CipherProvider;
import org.apache.nifi.security.util.crypto.CipherProviderFactory;
import org.apache.nifi.security.util.crypto.CipherUtility;
import org.apache.nifi.security.util.crypto.KeyedCipherProvider;
import org.apache.nifi.security.util.crypto.NiFiLegacyCipherProvider;
import org.apache.nifi.security.util.crypto.PBECipherProvider;
import org.apache.nifi.util.NiFiProperties;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StringEncryptor {
    private static final Logger logger = LoggerFactory.getLogger(StringEncryptor.class);
    private static final List<String> SUPPORTED_ALGORITHMS = new ArrayList<String>();
    private static final List<String> SUPPORTED_PROVIDERS = new ArrayList<String>();
    private final String algorithm;
    private final String provider;
    private final PBEKeySpec password;
    private final SecretKeySpec key;
    private String encoding = "HEX";
    private CipherProvider cipherProvider;
    public static final String NF_SENSITIVE_PROPS_KEY = "nifi.sensitive.props.key";
    public static final String NF_SENSITIVE_PROPS_ALGORITHM = "nifi.sensitive.props.algorithm";
    public static final String NF_SENSITIVE_PROPS_PROVIDER = "nifi.sensitive.props.provider";
    private static final String DEFAULT_SENSITIVE_PROPS_KEY = "nififtw!";

    public StringEncryptor(String algorithm, String provider, String key) {
        this.algorithm = algorithm;
        this.provider = provider;
        this.key = null;
        this.password = new PBEKeySpec(key == null ? DEFAULT_SENSITIVE_PROPS_KEY.toCharArray() : key.toCharArray());
        this.initialize();
    }

    public StringEncryptor(String algorithm, String provider, byte[] key) {
        this.algorithm = algorithm;
        this.provider = provider;
        this.key = new SecretKeySpec(key, this.extractKeyTypeFromAlgorithm(algorithm));
        this.password = null;
        this.initialize();
    }

    protected StringEncryptor() {
        this.algorithm = null;
        this.provider = null;
        this.key = null;
        this.password = null;
    }

    private String extractKeyTypeFromAlgorithm(String algorithm) throws EncryptionException {
        if (StringUtils.isBlank((CharSequence)algorithm)) {
            throw new EncryptionException("The algorithm cannot be null or empty");
        }
        String parsedCipher = CipherUtility.parseCipherFromAlgorithm((String)algorithm);
        if (parsedCipher.equals(algorithm)) {
            throw new EncryptionException("No supported algorithm detected");
        }
        return parsedCipher;
    }

    @Deprecated
    public static StringEncryptor createEncryptor(NiFiProperties niFiProperties) throws EncryptionException {
        String sensitivePropAlgorithmVal = niFiProperties.getProperty(NF_SENSITIVE_PROPS_ALGORITHM);
        String sensitivePropProviderVal = niFiProperties.getProperty(NF_SENSITIVE_PROPS_PROVIDER);
        String sensitivePropValueNifiPropVar = niFiProperties.getProperty(NF_SENSITIVE_PROPS_KEY);
        if (StringUtils.isBlank((CharSequence)sensitivePropValueNifiPropVar)) {
            StringEncryptor.printBlankKeyWarning();
            sensitivePropValueNifiPropVar = DEFAULT_SENSITIVE_PROPS_KEY;
        }
        return StringEncryptor.createEncryptor(sensitivePropAlgorithmVal, sensitivePropProviderVal, sensitivePropValueNifiPropVar);
    }

    public static StringEncryptor createEncryptor(String algorithm, String provider, String password) {
        if (StringUtils.isBlank((CharSequence)algorithm)) {
            throw new EncryptionException("nifi.sensitive.props.algorithm must be set");
        }
        if (StringUtils.isBlank((CharSequence)provider)) {
            throw new EncryptionException("nifi.sensitive.props.provider must be set");
        }
        if (StringUtils.isBlank((CharSequence)password)) {
            StringEncryptor.printBlankKeyWarning();
            password = DEFAULT_SENSITIVE_PROPS_KEY;
        }
        return new StringEncryptor(algorithm, provider, password);
    }

    private static void printBlankKeyWarning() {
        logger.error(StringUtils.repeat((String)"*", (int)80));
        logger.error(StringEncryptor.centerString("A blank sensitive properties key was provided"));
        logger.error(StringEncryptor.centerString("Specify a unique key in nifi.properties"));
        logger.error(StringEncryptor.centerString("for nifi.sensitive.props.key"));
        logger.error(StringEncryptor.centerString(""));
        logger.error(StringEncryptor.centerString("The Encrypt Config Tool in NiFi Toolkit can be used to"));
        logger.error(StringEncryptor.centerString("migrate the flow to the new key"));
        logger.error(StringUtils.repeat((String)"*", (int)80));
    }

    private static String centerString(String msg) {
        return "*" + StringUtils.center((String)msg, (int)78, (String)" ") + "*";
    }

    protected void initialize() {
        if (this.isInitialized()) {
            logger.debug("Attempted to initialize an already-initialized StringEncryptor");
            return;
        }
        if (this.paramsAreValid()) {
            this.cipherProvider = CipherUtility.isPBECipher((String)this.algorithm) ? CipherProviderFactory.getCipherProvider((KeyDerivationFunction)KeyDerivationFunction.NIFI_LEGACY) : CipherProviderFactory.getCipherProvider((KeyDerivationFunction)KeyDerivationFunction.NONE);
        } else {
            throw new EncryptionException("Cannot initialize the StringEncryptor because some configuration values are invalid");
        }
    }

    private boolean paramsAreValid() {
        boolean algorithmAndProviderValid = StringEncryptor.algorithmIsValid(this.algorithm) && StringEncryptor.providerIsValid(this.provider);
        boolean secretIsValid = false;
        if (CipherUtility.isPBECipher((String)this.algorithm)) {
            secretIsValid = this.passwordIsValid(this.password);
        } else if (CipherUtility.isKeyedCipher((String)this.algorithm)) {
            secretIsValid = this.keyIsValid(this.key, this.algorithm);
        }
        return algorithmAndProviderValid && secretIsValid;
    }

    private boolean keyIsValid(SecretKeySpec key, String algorithm) {
        return key != null && CipherUtility.getValidKeyLengthsForAlgorithm((String)algorithm).contains(key.getEncoded().length * 8);
    }

    private boolean passwordIsValid(PBEKeySpec password) {
        try {
            return password.getPassword().length > 0;
        }
        catch (IllegalStateException | NullPointerException e) {
            return false;
        }
    }

    public void setEncoding(String base) {
        if ("HEX".equalsIgnoreCase(base)) {
            this.encoding = "HEX";
        } else if ("BASE64".equalsIgnoreCase(base)) {
            this.encoding = "BASE64";
        } else {
            throw new IllegalArgumentException("The encoding base must be 'HEX' or 'BASE64'");
        }
    }

    public String encrypt(String clearText) throws EncryptionException {
        try {
            if (this.isInitialized()) {
                byte[] rawBytes = CipherUtility.isPBECipher((String)this.algorithm) ? this.encryptPBE(clearText) : this.encryptKeyed(clearText);
                return this.encode(rawBytes);
            }
            throw new EncryptionException("The encryptor is not initialized");
        }
        catch (Exception e) {
            throw new EncryptionException(e);
        }
    }

    private byte[] encryptPBE(String plaintext) {
        PBECipherProvider pbecp = (PBECipherProvider)this.cipherProvider;
        EncryptionMethod encryptionMethod = EncryptionMethod.forAlgorithm((String)this.algorithm);
        byte[] salt = pbecp instanceof NiFiLegacyCipherProvider ? ((NiFiLegacyCipherProvider)pbecp).generateSalt(encryptionMethod) : pbecp.generateSalt();
        int keyLength = CipherUtility.parseKeyLengthFromAlgorithm((String)this.algorithm);
        try {
            Cipher cipher = pbecp.getCipher(encryptionMethod, new String(this.password.getPassword()), salt, keyLength, true);
            byte[] cipherBytes = cipher.doFinal(plaintext.getBytes(StandardCharsets.UTF_8));
            return CryptoUtils.concatByteArrays((byte[][])new byte[][]{salt, cipherBytes});
        }
        catch (Exception e) {
            throw new EncryptionException("Could not encrypt sensitive value", e);
        }
    }

    private byte[] encryptKeyed(String plaintext) {
        KeyedCipherProvider keyedcp = (KeyedCipherProvider)this.cipherProvider;
        try {
            SecureRandom sr = new SecureRandom();
            byte[] iv = new byte[16];
            sr.nextBytes(iv);
            Cipher cipher = keyedcp.getCipher(EncryptionMethod.forAlgorithm((String)this.algorithm), (SecretKey)this.key, iv, true);
            byte[] cipherBytes = cipher.doFinal(plaintext.getBytes(StandardCharsets.UTF_8));
            return CryptoUtils.concatByteArrays((byte[][])new byte[][]{iv, cipherBytes});
        }
        catch (Exception e) {
            throw new EncryptionException("Could not encrypt sensitive value", e);
        }
    }

    private String encode(byte[] rawBytes) {
        if (this.encoding.equalsIgnoreCase("HEX")) {
            return Hex.encodeHexString((byte[])rawBytes);
        }
        return Base64.toBase64String((byte[])rawBytes);
    }

    public String decrypt(String cipherText) throws EncryptionException {
        try {
            if (this.isInitialized()) {
                byte[] cipherBytes = this.decode(cipherText);
                byte[] plainBytes = CipherUtility.isPBECipher((String)this.algorithm) ? this.decryptPBE(cipherBytes) : this.decryptKeyed(cipherBytes);
                return new String(plainBytes, StandardCharsets.UTF_8);
            }
            throw new EncryptionException("The encryptor is not initialized");
        }
        catch (Exception e) {
            throw new EncryptionException(e);
        }
    }

    private byte[] decryptPBE(byte[] cipherBytes) {
        PBECipherProvider pbecp = (PBECipherProvider)this.cipherProvider;
        EncryptionMethod encryptionMethod = EncryptionMethod.forAlgorithm((String)this.algorithm);
        int saltLength = CipherUtility.getSaltLengthForAlgorithm((String)this.algorithm);
        byte[] salt = new byte[saltLength];
        System.arraycopy(cipherBytes, 0, salt, 0, saltLength);
        byte[] actualCipherBytes = Arrays.copyOfRange(cipherBytes, saltLength, cipherBytes.length);
        int keyLength = CipherUtility.parseKeyLengthFromAlgorithm((String)this.algorithm);
        try {
            Cipher cipher = pbecp.getCipher(encryptionMethod, new String(this.password.getPassword()), salt, keyLength, false);
            return cipher.doFinal(actualCipherBytes);
        }
        catch (Exception e) {
            throw new EncryptionException("Could not decrypt sensitive value", e);
        }
    }

    private byte[] decryptKeyed(byte[] cipherBytes) {
        KeyedCipherProvider keyedcp = (KeyedCipherProvider)this.cipherProvider;
        try {
            int ivLength = 16;
            byte[] iv = new byte[ivLength];
            System.arraycopy(cipherBytes, 0, iv, 0, ivLength);
            byte[] actualCipherBytes = Arrays.copyOfRange(cipherBytes, ivLength, cipherBytes.length);
            Cipher cipher = keyedcp.getCipher(EncryptionMethod.forAlgorithm((String)this.algorithm), (SecretKey)this.key, iv, false);
            return cipher.doFinal(actualCipherBytes);
        }
        catch (Exception e) {
            throw new EncryptionException("Could not decrypt sensitive value", e);
        }
    }

    private byte[] decode(String encoded) throws DecoderException {
        if (this.encoding.equalsIgnoreCase("HEX")) {
            return Hex.decodeHex((char[])encoded.toCharArray());
        }
        return Base64.decode((String)encoded);
    }

    public boolean isInitialized() {
        return this.cipherProvider != null;
    }

    protected static boolean algorithmIsValid(String algorithm) {
        return SUPPORTED_ALGORITHMS.contains(algorithm);
    }

    protected static boolean providerIsValid(String provider) {
        return SUPPORTED_PROVIDERS.contains(provider);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        StringEncryptor that = (StringEncryptor)o;
        return Objects.equals(this.algorithm, that.algorithm) && Objects.equals(this.provider, that.provider) && Objects.equals(this.encoding, that.encoding) && this.secretsAreEqual(that.password, that.key);
    }

    private boolean secretsAreEqual(PBEKeySpec otherPassword, SecretKeySpec otherKey) {
        return StringEncryptor.isPBEKeySpecEqual(this.password, otherPassword) && Objects.equals(this.key, otherKey);
    }

    private static boolean isPBEKeySpecEqual(PBEKeySpec a, PBEKeySpec b) {
        if (a != null) {
            boolean passwordsEqual;
            if (b == null) {
                return false;
            }
            boolean nonNullsEqual = a.getIterationCount() == b.getIterationCount() && a.getKeyLength() == b.getKeyLength() && Arrays.equals(a.getSalt(), b.getSalt());
            try {
                passwordsEqual = CryptoUtils.constantTimeEquals((char[])a.getPassword(), (char[])b.getPassword());
            }
            catch (IllegalStateException e) {
                logger.warn("Encountered an error trying to compare password equality (one or more passwords have been cleared)");
                return false;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("The PBEKeySpec objects have equal non-null elements ({}) and equal passwords ({})", new Object[]{String.valueOf(nonNullsEqual), String.valueOf(passwordsEqual)});
            }
            return nonNullsEqual && passwordsEqual;
        }
        return b == null;
    }

    public int hashCode() {
        return Objects.hash(this.algorithm, this.provider, this.encoding, this.password, this.key);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("StringEncryptor using ").append(this.algorithm).append(" from ").append(this.provider).append(" with ").append(this.encoding).append(" encoding and cipher provider ").append(this.cipherProvider.getClass().getName());
        return sb.toString();
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
        for (EncryptionMethod encryptionMethod : EncryptionMethod.values()) {
            SUPPORTED_ALGORITHMS.add(encryptionMethod.getAlgorithm());
        }
        logger.debug("Supported encryption algorithms: " + StringUtils.join(SUPPORTED_ALGORITHMS, (String)"\n"));
        for (Provider provider : Security.getProviders()) {
            SUPPORTED_PROVIDERS.add(provider.getName());
        }
        logger.debug("Supported providers: " + StringUtils.join(SUPPORTED_PROVIDERS, (String)"\n"));
    }
}

