/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.provenance;

import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.nifi.provenance.RepositoryConfiguration;
import org.apache.nifi.provenance.StandardProvenanceEventRecord;
import org.apache.nifi.provenance.serialization.RecordReader;
import org.apache.nifi.provenance.serialization.RecordReaders;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexConfiguration {
    private final RepositoryConfiguration repoConfig;
    private final Map<File, List<File>> indexDirectoryMap = new HashMap<File, List<File>>();
    private final Pattern indexNamePattern = Pattern.compile("index-(\\d+)");
    private final Lock lock = new ReentrantLock();
    private static final Logger logger = LoggerFactory.getLogger(IndexConfiguration.class);
    private Long maxIndexedId = null;
    private Long minIndexedId = null;

    public IndexConfiguration(RepositoryConfiguration repoConfig) {
        this.repoConfig = repoConfig;
        this.indexDirectoryMap.putAll(this.recoverIndexDirectories());
    }

    private Map<File, List<File>> recoverIndexDirectories() {
        HashMap<File, List<File>> indexDirectoryMap = new HashMap<File, List<File>>();
        for (File storageDirectory : this.repoConfig.getStorageDirectories().values()) {
            ArrayList<File> indexDirectories = new ArrayList<File>();
            File[] matching = storageDirectory.listFiles(new FileFilter(){

                @Override
                public boolean accept(File pathname) {
                    return pathname.isDirectory() && IndexConfiguration.this.indexNamePattern.matcher(pathname.getName()).matches();
                }
            });
            if (matching != null) {
                for (File matchingFile : matching) {
                    indexDirectories.add(matchingFile);
                }
            }
            indexDirectoryMap.put(storageDirectory, indexDirectories);
        }
        return indexDirectoryMap;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Long getFirstEntryTime(File provenanceLogFile) {
        if (provenanceLogFile == null) {
            return null;
        }
        try (RecordReader reader = RecordReaders.newRecordReader(provenanceLogFile, null, Integer.MAX_VALUE);){
            StandardProvenanceEventRecord firstRecord = reader.nextRecord();
            if (firstRecord == null) {
                Long l2 = provenanceLogFile.lastModified();
                return l2;
            }
            Long l = firstRecord.getEventTime();
            return l;
        }
        catch (EOFException | FileNotFoundException fnf) {
            return null;
        }
        catch (IOException ioe) {
            logger.warn("Failed to read first entry in file {} due to {}", (Object)provenanceLogFile, (Object)ioe.toString());
            logger.warn("", (Throwable)ioe);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeIndexDirectory(File indexDirectory) {
        this.lock.lock();
        try {
            HashSet<File> keysToRemove = new HashSet<File>();
            for (Map.Entry<File, List<File>> entry : this.indexDirectoryMap.entrySet()) {
                List<File> list = entry.getValue();
                list.remove(indexDirectory);
                if (!list.isEmpty()) continue;
                keysToRemove.add(entry.getKey());
            }
            for (File keyToRemove : keysToRemove) {
                this.indexDirectoryMap.remove(keyToRemove);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public File getWritableIndexDirectory(File provenanceLogFile, long newIndexTimestamp) {
        return this.getWritableIndexDirectoryForStorageDirectory(provenanceLogFile.getParentFile(), provenanceLogFile, newIndexTimestamp);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File getWritableIndexDirectoryForStorageDirectory(File storageDirectory, File provenanceLogFile, long newIndexTimestamp) {
        this.lock.lock();
        try {
            List<File> indexDirectories = this.indexDirectoryMap.get(storageDirectory);
            if (indexDirectories == null) {
                File newDir = this.addNewIndex(storageDirectory, provenanceLogFile, newIndexTimestamp);
                indexDirectories = new ArrayList<File>();
                indexDirectories.add(newDir);
                this.indexDirectoryMap.put(storageDirectory, indexDirectories);
                File file = newDir;
                return file;
            }
            if (indexDirectories.isEmpty()) {
                File newDir = this.addNewIndex(storageDirectory, provenanceLogFile, newIndexTimestamp);
                indexDirectories.add(newDir);
                File file = newDir;
                return file;
            }
            File lastDir = indexDirectories.get(indexDirectories.size() - 1);
            long size = this.getSize(lastDir);
            if (size > this.repoConfig.getDesiredIndexSize()) {
                File newDir = this.addNewIndex(storageDirectory, provenanceLogFile, newIndexTimestamp);
                indexDirectories.add(newDir);
                File file = newDir;
                return file;
            }
            File file = lastDir;
            return file;
        }
        finally {
            this.lock.unlock();
        }
    }

    private File addNewIndex(File storageDirectory, File provenanceLogFile, long newIndexTimestamp) {
        Long firstEntryTime = this.getFirstEntryTime(provenanceLogFile);
        if (firstEntryTime == null) {
            firstEntryTime = newIndexTimestamp;
        }
        return new File(storageDirectory, "index-" + firstEntryTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<File> getIndexDirectories() {
        this.lock.lock();
        try {
            ArrayList<File> files = new ArrayList<File>();
            for (List<File> list : this.indexDirectoryMap.values()) {
                files.addAll(list);
            }
            ArrayList<File> arrayList = files;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    private long getIndexStartTime(File indexDir) {
        if (indexDir == null) {
            return -1L;
        }
        Matcher matcher = this.indexNamePattern.matcher(indexDir.getName());
        boolean matches = matcher.matches();
        if (matches) {
            return Long.parseLong(matcher.group(1));
        }
        return -1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<File> getIndexDirectories(Long startTime, Long endTime) {
        if (startTime == null && endTime == null) {
            return this.getIndexDirectories();
        }
        ArrayList<File> dirs = new ArrayList<File>();
        this.lock.lock();
        try {
            List<File> sortedIndexDirectories = this.getIndexDirectories();
            Collections.sort(sortedIndexDirectories, new Comparator<File>(){

                @Override
                public int compare(File o1, File o2) {
                    long epochTimestamp1 = IndexConfiguration.this.getIndexStartTime(o1);
                    long epochTimestamp2 = IndexConfiguration.this.getIndexStartTime(o2);
                    return Long.compare(epochTimestamp2, epochTimestamp1);
                }
            });
            for (File indexDir : sortedIndexDirectories) {
                long indexStartTime;
                if (startTime != null && indexDir.lastModified() < startTime || endTime != null && (indexStartTime = this.getIndexStartTime(indexDir)) > endTime) continue;
                dirs.add(indexDir);
            }
            ArrayList<File> arrayList = dirs;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<File> getIndexDirectories(File provenanceLogFile) {
        ArrayList<File> dirs = new ArrayList<File>();
        this.lock.lock();
        try {
            List<File> indices = this.indexDirectoryMap.get(provenanceLogFile.getParentFile());
            if (indices == null) {
                List<File> list = Collections.emptyList();
                return list;
            }
            ArrayList<File> sortedIndexDirectories = new ArrayList<File>(indices);
            Collections.sort(sortedIndexDirectories, new Comparator<File>(){

                @Override
                public int compare(File o1, File o2) {
                    long epochTimestamp1 = IndexConfiguration.this.getIndexStartTime(o1);
                    long epochTimestamp2 = IndexConfiguration.this.getIndexStartTime(o2);
                    return Long.compare(epochTimestamp1, epochTimestamp2);
                }
            });
            Long firstEntryTime = this.getFirstEntryTime(provenanceLogFile);
            if (firstEntryTime == null) {
                logger.debug("Found no records in {} so returning no Indices for it", (Object)provenanceLogFile);
                List<File> list = Collections.emptyList();
                return list;
            }
            boolean foundIndexCreatedLater = false;
            for (File indexDir : sortedIndexDirectories) {
                if (indexDir.lastModified() < firstEntryTime) continue;
                long indexStartTime = this.getIndexStartTime(indexDir);
                if (indexStartTime > provenanceLogFile.lastModified()) {
                    if (foundIndexCreatedLater) continue;
                    foundIndexCreatedLater = true;
                }
                dirs.add(indexDir);
            }
            ArrayList<File> arrayList = dirs;
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    private long getSize(File indexDirectory) {
        if (!indexDirectory.exists()) {
            return 0L;
        }
        if (!indexDirectory.isDirectory()) {
            throw new IllegalArgumentException("Must specify a directory but specified " + indexDirectory);
        }
        File[] files = indexDirectory.listFiles();
        if (files == null) {
            return 0L;
        }
        long sum = 0L;
        for (File file : files) {
            sum += file.length();
        }
        return sum;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getIndexSize() {
        this.lock.lock();
        try {
            long sum = 0L;
            for (File indexDir : this.getIndexDirectories()) {
                sum += this.getSize(indexDir);
            }
            long l = sum;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setMaxIdIndexed(long id) {
        this.lock.lock();
        try {
            if (this.maxIndexedId == null || id > this.maxIndexedId) {
                this.maxIndexedId = id;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public Long getMaxIdIndexed() {
        this.lock.lock();
        try {
            Long l = this.maxIndexedId;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setMinIdIndexed(long id) {
        this.lock.lock();
        try {
            if (this.minIndexedId == null || id > this.minIndexedId) {
                this.minIndexedId = this.maxIndexedId == null || id > this.maxIndexedId ? this.maxIndexedId : Long.valueOf(id);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public Long getMinIdIndexed() {
        this.lock.lock();
        try {
            Long l = this.minIndexedId;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }
}

