/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.provenance.index.lucene;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.nifi.provenance.RepositoryConfiguration;
import org.apache.nifi.provenance.index.lucene.IndexLocation;
import org.apache.nifi.provenance.util.DirectoryUtils;
import org.apache.nifi.util.Tuple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexDirectoryManager {
    private static final Logger logger = LoggerFactory.getLogger(IndexDirectoryManager.class);
    private static final FileFilter INDEX_DIRECTORY_FILTER = f -> f.getName().startsWith("index-");
    private static final Pattern INDEX_FILENAME_PATTERN = Pattern.compile("index-(\\d+)");
    private final RepositoryConfiguration repoConfig;
    private final SortedMap<Long, List<IndexLocation>> indexLocationByTimestamp = new TreeMap<Long, List<IndexLocation>>();
    private final Map<String, IndexLocation> activeIndices = new HashMap<String, IndexLocation>();

    public IndexDirectoryManager(RepositoryConfiguration repoConfig) {
        this.repoConfig = repoConfig;
    }

    public synchronized void initialize() {
        HashMap<File, Tuple> latestIndexByStorageDir = new HashMap<File, Tuple>();
        for (Map.Entry<String, File> entry : this.repoConfig.getStorageDirectories().entrySet()) {
            String partitionName = entry.getKey();
            File storageDir = entry.getValue();
            File[] indexDirs = storageDir.listFiles(INDEX_DIRECTORY_FILTER);
            if (indexDirs == null) {
                logger.warn("Unable to access Provenance Repository storage directory {}", (Object)storageDir);
                continue;
            }
            for (File indexDir : indexDirs) {
                Matcher matcher = INDEX_FILENAME_PATTERN.matcher(indexDir.getName());
                if (!matcher.matches()) continue;
                long startTime = DirectoryUtils.getIndexTimestamp(indexDir);
                List dirsForTimestamp = this.indexLocationByTimestamp.computeIfAbsent(startTime, t -> new ArrayList());
                IndexLocation indexLoc = new IndexLocation(indexDir, startTime, partitionName);
                dirsForTimestamp.add(indexLoc);
                Tuple tuple = (Tuple)latestIndexByStorageDir.get(storageDir);
                if (tuple != null && startTime <= (Long)tuple.getKey()) continue;
                latestIndexByStorageDir.put(storageDir, new Tuple((Object)startTime, (Object)indexLoc));
            }
        }
        for (Tuple tuple : latestIndexByStorageDir.values()) {
            IndexLocation indexLoc = (IndexLocation)tuple.getValue();
            this.activeIndices.put(indexLoc.getPartitionName(), indexLoc);
        }
    }

    public synchronized void deleteDirectory(File directory) {
        Iterator<Map.Entry<Long, List<IndexLocation>>> itr = this.indexLocationByTimestamp.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry<Long, List<IndexLocation>> entry = itr.next();
            List<IndexLocation> locations = entry.getValue();
            IndexLocation locToRemove = new IndexLocation(directory, DirectoryUtils.getIndexTimestamp(directory), directory.getName());
            locations.remove(locToRemove);
            if (!locations.isEmpty()) continue;
            itr.remove();
        }
    }

    public synchronized List<File> getDirectoriesBefore(long timestamp) {
        ArrayList<File> selected = new ArrayList<File>();
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp().stream().collect(Collectors.groupingBy(indexLoc -> indexLoc.getPartitionName()));
        block0: for (List<IndexLocation> locationList : startTimeWithFileByStorageDirectory.values()) {
            for (int i = 0; i < locationList.size(); ++i) {
                String partition;
                IndexLocation activeLocation;
                IndexLocation indexLoc2 = locationList.get(i);
                if (indexLoc2.equals(activeLocation = this.activeIndices.get(partition = indexLoc2.getPartitionName()))) continue;
                Long indexStartTime = indexLoc2.getIndexStartTimestamp();
                if (indexStartTime > timestamp) continue block0;
                long indexEndTime = indexLoc2.getIndexEndTimestamp();
                if (indexEndTime > timestamp) continue;
                logger.debug("Considering Index Location {} older than {} ({}) because its events have an EventTime ranging from {} ({}) to {} ({}) based on the following IndexLocations: {}", new Object[]{indexLoc2, timestamp, new Date(timestamp), indexStartTime, new Date(indexStartTime), indexEndTime, new Date(indexEndTime), locationList});
                selected.add(indexLoc2.getIndexDirectory());
            }
        }
        logger.debug("Returning the following list of index locations because they were finished being written to before {}: {}", (Object)timestamp, selected);
        return selected;
    }

    private List<IndexLocation> flattenDirectoriesByTimestamp() {
        ArrayList<IndexLocation> startTimeWithFile = new ArrayList<IndexLocation>();
        for (Map.Entry<Long, List<IndexLocation>> entry : this.indexLocationByTimestamp.entrySet()) {
            for (IndexLocation indexLoc : entry.getValue()) {
                startTimeWithFile.add(indexLoc);
            }
        }
        return startTimeWithFile;
    }

    public synchronized List<File> getDirectories(Long startTime, Long endTime) {
        ArrayList<File> selected = new ArrayList<File>();
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp().stream().collect(Collectors.groupingBy(indexLoc -> indexLoc.getPartitionName()));
        for (List<IndexLocation> locationList : startTimeWithFileByStorageDirectory.values()) {
            selected.addAll(IndexDirectoryManager.getDirectories(startTime, endTime, locationList));
        }
        return selected;
    }

    public synchronized List<File> getDirectories(Long startTime, Long endTime, String partitionName) {
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp().stream().collect(Collectors.groupingBy(indexLoc -> indexLoc.getPartitionName()));
        List<IndexLocation> indexLocations = startTimeWithFileByStorageDirectory.get(partitionName);
        if (indexLocations == null) {
            return Collections.emptyList();
        }
        return IndexDirectoryManager.getDirectories(startTime, endTime, indexLocations);
    }

    protected static List<File> getDirectories(Long startTime, Long endTime, List<IndexLocation> locations) {
        ArrayList<File> selected = new ArrayList<File>();
        int overlapCount = 0;
        for (int i = 0; i < locations.size(); ++i) {
            IndexLocation nextIndexLoc;
            Long indexEndTimestamp;
            IndexLocation indexLoc = locations.get(i);
            Long indexStartTimestamp = indexLoc.getIndexStartTimestamp();
            if (endTime != null && indexStartTimestamp > endTime) {
                if (overlapCount != 0) continue;
                ++overlapCount;
            }
            if (startTime != null && i < locations.size() - 1 && (indexEndTimestamp = Long.valueOf((nextIndexLoc = locations.get(i + 1)).getIndexStartTimestamp())) < startTime) continue;
            selected.add(indexLoc.getIndexDirectory());
        }
        return selected;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean onIndexCommitted(File indexDir) {
        long indexSize = this.getSize(indexDir);
        IndexDirectoryManager indexDirectoryManager = this;
        synchronized (indexDirectoryManager) {
            String partitionName = null;
            for (Map.Entry<String, IndexLocation> entry : this.activeIndices.entrySet()) {
                if (!indexDir.equals(entry.getValue().getIndexDirectory())) continue;
                partitionName = entry.getKey();
                break;
            }
            if (partitionName == null) {
                logger.debug("Size of Provenance Index at {} is now {}. However, was unable to find the appropriate Active Index to roll over.", (Object)indexDir, (Object)indexSize);
                return true;
            }
            if (indexSize >= this.repoConfig.getDesiredIndexSize()) {
                logger.info("Size of Provenance Index at {} is now {}. Will close this index and roll over to a new one.", (Object)indexDir, (Object)indexSize);
                this.activeIndices.remove(partitionName);
                return true;
            }
            return false;
        }
    }

    public synchronized Optional<File> getActiveIndexDirectory(String partitionName) {
        IndexLocation indexLocation = this.activeIndices.get(partitionName);
        if (indexLocation == null) {
            return Optional.empty();
        }
        return Optional.of(indexLocation.getIndexDirectory());
    }

    private long getSize(File indexDir) {
        if (!indexDir.exists()) {
            return 0L;
        }
        if (!indexDir.isDirectory()) {
            throw new IllegalArgumentException("Must specify a directory but specified " + indexDir);
        }
        File[] files = indexDir.listFiles();
        if (files == null) {
            return 0L;
        }
        long sum = 0L;
        for (File file : files) {
            sum += file.length();
        }
        return sum;
    }

    public synchronized File getWritableIndexingDirectory(long earliestTimestamp, String partitionName) {
        IndexLocation indexLoc = this.activeIndices.get(partitionName);
        if (indexLoc == null) {
            indexLoc = new IndexLocation(this.createIndex(earliestTimestamp, partitionName), earliestTimestamp, partitionName);
            logger.debug("Created new Index Directory {}", (Object)indexLoc);
            this.indexLocationByTimestamp.computeIfAbsent(earliestTimestamp, t -> new ArrayList()).add(indexLoc);
            this.activeIndices.put(partitionName, indexLoc);
        }
        return indexLoc.getIndexDirectory();
    }

    private File createIndex(long earliestTimestamp, String partitionName) {
        File storageDir = this.repoConfig.getStorageDirectories().entrySet().stream().filter(e -> ((String)e.getKey()).equals(partitionName)).map(Map.Entry::getValue).findFirst().orElseThrow(() -> new IllegalArgumentException("Invalid Partition: " + partitionName));
        File indexDir = new File(storageDir, "index-" + earliestTimestamp);
        return indexDir;
    }
}

