/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.record.sink;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnEnabled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.controller.AbstractControllerService;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.oauth2.OAuth2AccessTokenProvider;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.record.sink.RecordSinkService;
import org.apache.nifi.schema.access.SchemaNotFoundException;
import org.apache.nifi.serialization.RecordSetWriter;
import org.apache.nifi.serialization.RecordSetWriterFactory;
import org.apache.nifi.serialization.WriteResult;
import org.apache.nifi.serialization.record.Record;
import org.apache.nifi.serialization.record.RecordSet;
import org.apache.nifi.web.client.api.HttpRequestBodySpec;
import org.apache.nifi.web.client.api.HttpResponseEntity;
import org.apache.nifi.web.client.api.HttpUriBuilder;
import org.apache.nifi.web.client.provider.api.WebClientServiceProvider;

@Tags(value={"http", "post", "record", "sink"})
@CapabilityDescription(value="Format and send Records to a configured uri using HTTP post. The Record Writer formats the records which are sent as the body of the HTTP post request. JsonRecordSetWriter is often used with this processor because many HTTP posts require a JSON body.")
public class HttpRecordSink
extends AbstractControllerService
implements RecordSinkService {
    protected static final String HEADER_AUTHORIZATION = "Authorization";
    protected static final String HEADER_CONTENT_TYPE = "Content-Type";
    public static final PropertyDescriptor API_URL = new PropertyDescriptor.Builder().name("API URL").description("The URL which receives the HTTP requests.").addValidator(StandardValidators.NON_BLANK_VALIDATOR).addValidator(StandardValidators.URL_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).required(true).build();
    static final PropertyDescriptor MAX_BATCH_SIZE = new PropertyDescriptor.Builder().name("Maximum Batch Size").description("Specifies the maximum number of records to send in the body of each HTTP request. Zero means the batch size is not limited, and all records are sent together in a single HTTP request.").defaultValue("0").addValidator(StandardValidators.NON_NEGATIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).required(true).build();
    public static final PropertyDescriptor WEB_SERVICE_CLIENT_PROVIDER = new PropertyDescriptor.Builder().name("Web Service Client Provider").description("Controller service to provide the HTTP client for sending the HTTP requests.").required(true).identifiesControllerService(WebClientServiceProvider.class).build();
    public static final PropertyDescriptor OAUTH2_ACCESS_TOKEN_PROVIDER = new PropertyDescriptor.Builder().name("OAuth2 Access Token Provider").description("OAuth2 service that provides the access tokens for the HTTP requests.").identifiesControllerService(OAuth2AccessTokenProvider.class).required(false).build();
    private String apiUrl;
    private int maxBatchSize;
    private volatile RecordSetWriterFactory writerFactory;
    private WebClientServiceProvider webClientServiceProvider;
    private volatile Optional<OAuth2AccessTokenProvider> oauth2AccessTokenProviderOptional;
    Map<String, String> dynamicHttpHeaders;
    public static final List<PropertyDescriptor> PROPERTIES = Collections.unmodifiableList(Arrays.asList(API_URL, MAX_BATCH_SIZE, RECORD_WRITER_FACTORY, WEB_SERVICE_CLIENT_PROVIDER, OAUTH2_ACCESS_TOKEN_PROVIDER));

    public List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTIES;
    }

    protected PropertyDescriptor getSupportedDynamicPropertyDescriptor(String propertyDescriptorName) {
        if (HttpRecordSink.hasProhibitedName(propertyDescriptorName, HEADER_CONTENT_TYPE)) {
            return HttpRecordSink.getInvalidDynamicPropertyDescriptor(propertyDescriptorName, "is not allowed. Only exact case of Content-Type is allowed.");
        }
        if (HttpRecordSink.hasProhibitedName(propertyDescriptorName, HEADER_AUTHORIZATION)) {
            return HttpRecordSink.getInvalidDynamicPropertyDescriptor(propertyDescriptorName, "is not allowed. Only exact case of Authorization is allowed.");
        }
        return new PropertyDescriptor.Builder().name(propertyDescriptorName).required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).dynamic(true).build();
    }

    private static boolean hasProhibitedName(String userInput, String correctName) {
        return (correctName + ":").equalsIgnoreCase(userInput) || correctName.equalsIgnoreCase(userInput) && !correctName.equals(userInput);
    }

    private static PropertyDescriptor getInvalidDynamicPropertyDescriptor(String propertyDescriptorName, String explanation) {
        return new PropertyDescriptor.Builder().name(propertyDescriptorName).addValidator((subject, input, context) -> new ValidationResult.Builder().explanation(explanation).valid(false).subject(subject).build()).dynamic(true).build();
    }

    @OnEnabled
    public void onEnabled(ConfigurationContext context) {
        this.apiUrl = context.getProperty(API_URL).evaluateAttributeExpressions().getValue();
        this.maxBatchSize = context.getProperty(MAX_BATCH_SIZE).evaluateAttributeExpressions().asInteger();
        this.writerFactory = (RecordSetWriterFactory)context.getProperty(RECORD_WRITER_FACTORY).asControllerService(RecordSetWriterFactory.class);
        this.webClientServiceProvider = (WebClientServiceProvider)context.getProperty(WEB_SERVICE_CLIENT_PROVIDER).asControllerService(WebClientServiceProvider.class);
        if (context.getProperty(OAUTH2_ACCESS_TOKEN_PROVIDER).isSet()) {
            OAuth2AccessTokenProvider oauth2AccessTokenProvider = (OAuth2AccessTokenProvider)context.getProperty(OAUTH2_ACCESS_TOKEN_PROVIDER).asControllerService(OAuth2AccessTokenProvider.class);
            oauth2AccessTokenProvider.getAccessDetails();
            this.oauth2AccessTokenProviderOptional = Optional.of(oauth2AccessTokenProvider);
        } else {
            this.oauth2AccessTokenProviderOptional = Optional.empty();
        }
        this.dynamicHttpHeaders = context.getProperties().keySet().stream().filter(PropertyDescriptor::isDynamic).collect(Collectors.toMap(PropertyDescriptor::getName, p -> context.getProperty(p).evaluateAttributeExpressions().getValue()));
    }

    public WriteResult sendData(RecordSet recordSet, Map<String, String> attributes, boolean sendZeroResults) throws IOException {
        WriteResult writeResult;
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
             RecordSetWriter writer = this.writerFactory.createWriter(this.getLogger(), recordSet.getSchema(), (OutputStream)baos, attributes);){
            writeResult = this.sendRecords(recordSet, writer, baos, this.maxBatchSize);
        }
        catch (SchemaNotFoundException e) {
            String errorMessage = String.format("RecordSetWriter could not be created because the schema was not found. The schema name for the RecordSet to write is %s", recordSet.getSchema().getSchemaName());
            throw new IOException(errorMessage, e);
        }
        return writeResult;
    }

    private WriteResult sendRecords(RecordSet recordSet, RecordSetWriter writer, ByteArrayOutputStream baos, int maxBatchSize) throws IOException {
        WriteResult writeResult = WriteResult.EMPTY;
        Record r = recordSet.next();
        if (r != null) {
            int batchCount = 0;
            do {
                if (maxBatchSize != 1 && batchCount == 0) {
                    writer.beginRecordSet();
                }
                writeResult = writer.write(r);
                r = recordSet.next();
                if (r != null && (maxBatchSize <= 0 || ++batchCount < maxBatchSize)) continue;
                if (maxBatchSize != 1) {
                    writeResult = writer.finishRecordSet();
                }
                writer.flush();
                this.sendHttpRequest(baos.toByteArray(), writer.getMimeType());
                baos.reset();
                batchCount = 0;
            } while (r != null);
        }
        return writeResult;
    }

    public void sendHttpRequest(byte[] body, String mimeType) throws IOException {
        URI apiUri = URI.create(this.apiUrl);
        HttpUriBuilder uriBuilder = this.webClientServiceProvider.getHttpUriBuilder().scheme(apiUri.getScheme()).host(apiUri.getHost()).encodedPath(apiUri.getPath());
        if (apiUri.getPort() != -1) {
            uriBuilder.port(apiUri.getPort());
        }
        URI uri = uriBuilder.build();
        HttpRequestBodySpec requestBodySpec = this.webClientServiceProvider.getWebClientService().post().uri(uri);
        this.dynamicHttpHeaders.forEach((arg_0, arg_1) -> ((HttpRequestBodySpec)requestBodySpec).header(arg_0, arg_1));
        if (StringUtils.isNotBlank((CharSequence)mimeType) && !this.dynamicHttpHeaders.containsKey(HEADER_CONTENT_TYPE)) {
            requestBodySpec.header(HEADER_CONTENT_TYPE, mimeType);
        }
        if (!this.dynamicHttpHeaders.containsKey(HEADER_AUTHORIZATION)) {
            this.oauth2AccessTokenProviderOptional.ifPresent(oauth2AccessTokenProvider -> requestBodySpec.header(HEADER_AUTHORIZATION, "Bearer " + oauth2AccessTokenProvider.getAccessDetails().getAccessToken()));
        }
        ByteArrayInputStream requestBodyInputStream = new ByteArrayInputStream(body);
        try (HttpResponseEntity response = requestBodySpec.body((InputStream)requestBodyInputStream, OptionalLong.of(((InputStream)requestBodyInputStream).available())).retrieve();){
            int statusCode = response.statusCode();
            if (statusCode < 200 || statusCode >= 300) {
                throw new IOException(String.format("HTTP request failed with status code: %s for url: %s and returned response body: %s", statusCode, uri.toString(), response.body() == null ? "none" : IOUtils.toString((InputStream)response.body(), (Charset)StandardCharsets.UTF_8)));
            }
        }
        catch (IOException ioe) {
            throw ioe;
        }
        catch (Exception e) {
            throw new IOException(String.format("HttpRecordSink HTTP request transmission failed for url: %s", uri.toString()), e);
        }
    }
}

