/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.nifi.security.util.KeyStoreUtils;
import org.apache.nifi.security.util.TlsConfiguration;
import org.apache.nifi.security.util.TlsException;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SslContextFactory {
    private static final Logger logger = LoggerFactory.getLogger(SslContextFactory.class);

    public static SSLContext createSslContext(TlsConfiguration tlsConfiguration) throws TlsException {
        return SslContextFactory.createSslContext(tlsConfiguration, ClientAuth.REQUIRED);
    }

    public static SSLContext createSslContext(TlsConfiguration tlsConfiguration, ClientAuth clientAuth) throws TlsException {
        if (TlsConfiguration.isEmpty(tlsConfiguration)) {
            logger.debug("Cannot create SSLContext from empty TLS configuration; returning null");
            return null;
        }
        if (tlsConfiguration.isKeystorePopulated() && !tlsConfiguration.isTruststorePopulated()) {
            logger.error("The TLS config keystore properties were populated but the truststore properties were not");
            if (logger.isDebugEnabled()) {
                logger.debug("Provided TLS config: {}", (Object)tlsConfiguration);
            }
            throw new TlsException("Truststore properties are required if keystore properties are present");
        }
        if (clientAuth == null) {
            clientAuth = ClientAuth.REQUIRED;
            logger.debug("ClientAuth was null so defaulting to {}", (Object)clientAuth);
        }
        KeyManager[] keyManagers = SslContextFactory.getKeyManagers(tlsConfiguration);
        TrustManager[] trustManagers = SslContextFactory.getTrustManagers(tlsConfiguration);
        return SslContextFactory.initializeSSLContext(tlsConfiguration, clientAuth, keyManagers, trustManagers);
    }

    public static X509TrustManager getX509TrustManager(TlsConfiguration tlsConfiguration) throws TlsException {
        TrustManager[] trustManagers = SslContextFactory.getTrustManagers(tlsConfiguration);
        if (trustManagers == null) {
            return null;
        }
        Optional<X509TrustManager> x509TrustManager = Arrays.stream(trustManagers).filter(tm -> tm instanceof X509TrustManager).map(tm -> (X509TrustManager)tm).findFirst();
        return x509TrustManager.orElse(null);
    }

    public static SSLSocketFactory createSSLSocketFactory(TlsConfiguration tlsConfiguration) throws TlsException {
        SSLContext sslContext = SslContextFactory.createSslContext(tlsConfiguration, ClientAuth.REQUIRED);
        if (sslContext == null) {
            if (!TlsConfiguration.isEmpty(tlsConfiguration)) {
                logger.error("The SSLContext could not be formed from the provided TLS configuration. Check the provided keystore and truststore properties");
            }
            return null;
        }
        return sslContext.getSocketFactory();
    }

    protected static KeyManager[] getKeyManagers(TlsConfiguration tlsConfiguration) throws TlsException {
        KeyManager[] keyManagers = null;
        if (tlsConfiguration.isKeystoreValid()) {
            KeyManagerFactory keyManagerFactory = KeyStoreUtils.loadKeyManagerFactory(tlsConfiguration);
            keyManagers = keyManagerFactory.getKeyManagers();
        } else {
            if (tlsConfiguration.isAnyKeystorePopulated()) {
                logger.warn("Some keystore properties are populated ({}, {}, {}, {}) but not valid", (Object[])tlsConfiguration.getKeystorePropertiesForLogging());
                throw new TlsException("The keystore properties are not valid");
            }
            logger.debug("The keystore properties are not populated");
        }
        return keyManagers;
    }

    protected static TrustManager[] getTrustManagers(TlsConfiguration tlsConfiguration) throws TlsException {
        TrustManager[] trustManagers = null;
        if (tlsConfiguration.isTruststoreValid()) {
            TrustManagerFactory trustManagerFactory = KeyStoreUtils.loadTrustManagerFactory(tlsConfiguration);
            trustManagers = trustManagerFactory.getTrustManagers();
        } else {
            if (tlsConfiguration.isAnyTruststorePopulated()) {
                logger.warn("Some truststore properties are populated ({}, {}, {}) but not valid", (Object[])tlsConfiguration.getTruststorePropertiesForLogging());
                throw new TlsException("The truststore properties are not valid");
            }
            logger.debug("The truststore properties are not populated");
        }
        return trustManagers;
    }

    private static SSLContext initializeSSLContext(TlsConfiguration tlsConfiguration, ClientAuth clientAuth, KeyManager[] keyManagers, TrustManager[] trustManagers) throws TlsException {
        try {
            SSLContext sslContext = SSLContext.getInstance(tlsConfiguration.getProtocol());
            sslContext.init(keyManagers, trustManagers, new SecureRandom());
            switch (clientAuth) {
                case REQUIRED: {
                    sslContext.getDefaultSSLParameters().setNeedClientAuth(true);
                    break;
                }
                case WANT: {
                    sslContext.getDefaultSSLParameters().setWantClientAuth(true);
                    break;
                }
                default: {
                    sslContext.getDefaultSSLParameters().setWantClientAuth(false);
                }
            }
            return sslContext;
        }
        catch (KeyManagementException | NoSuchAlgorithmException e) {
            logger.error("Encountered an error creating SSLContext from TLS configuration ({}): {}", (Object)tlsConfiguration.toString(), (Object)e.getLocalizedMessage());
            throw new TlsException("Error creating SSL context", e);
        }
    }

    public static enum ClientAuth {
        WANT("Want", "Requests the client certificate on handshake and validates if present but does not require it"),
        REQUIRED("Required", "Requests the client certificate on handshake and rejects the connection if it is not present and valid"),
        NONE("None", "Does not request the client certificate on handshake");

        private final String type;
        private final String description;

        private ClientAuth(String type, String description) {
            this.type = type;
            this.description = description;
        }

        public String getType() {
            return this.type;
        }

        public String getDescription() {
            return this.description;
        }

        public String toString() {
            ToStringBuilder builder = new ToStringBuilder((Object)this);
            ToStringBuilder.setDefaultStyle((ToStringStyle)ToStringStyle.SHORT_PREFIX_STYLE);
            builder.append("Type", (Object)this.type);
            builder.append("Description", (Object)this.description);
            return builder.toString();
        }

        public static boolean isValidClientAuthType(String type) {
            if (StringUtils.isBlank((String)type)) {
                return false;
            }
            return Arrays.stream(ClientAuth.values()).map(ca -> ca.getType().toLowerCase()).collect(Collectors.toList()).contains(type.toLowerCase());
        }
    }
}

