/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.repository;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import org.apache.nifi.security.kms.CryptoUtils;
import org.apache.nifi.security.kms.EncryptionException;
import org.apache.nifi.security.kms.FileBasedKeyProvider;
import org.apache.nifi.security.kms.KeyProvider;
import org.apache.nifi.security.kms.KeyProviderFactory;
import org.apache.nifi.security.kms.KeyStoreKeyProvider;
import org.apache.nifi.security.kms.StaticKeyProvider;
import org.apache.nifi.security.kms.configuration.FileBasedKeyProviderConfiguration;
import org.apache.nifi.security.kms.configuration.KeyProviderConfiguration;
import org.apache.nifi.security.kms.configuration.KeyStoreKeyProviderConfiguration;
import org.apache.nifi.security.kms.configuration.StaticKeyProviderConfiguration;
import org.apache.nifi.security.repository.RepositoryObjectEncryptionMetadata;
import org.apache.nifi.security.repository.RepositoryType;
import org.apache.nifi.security.repository.config.RepositoryEncryptionConfiguration;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.KeyStoreUtils;
import org.apache.nifi.security.util.TlsException;
import org.apache.nifi.security.util.crypto.AESKeyedCipherProvider;
import org.apache.nifi.stream.io.NonCloseableInputStream;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RepositoryEncryptorUtils {
    private static final Logger logger = LoggerFactory.getLogger(RepositoryEncryptorUtils.class);
    private static final int CONTENT_HEADER_SIZE = 2;
    private static final int IV_LENGTH = 16;
    private static final int MIN_METADATA_LENGTH = 22;
    private static final String EWAPR_CLASS_NAME = "org.apache.nifi.provenance.EncryptedWriteAheadProvenanceRepository";

    public static byte[] serializeEncryptionMetadata(RepositoryObjectEncryptionMetadata metadata) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream outputStream = new ObjectOutputStream(baos);
        outputStream.writeObject(metadata);
        outputStream.close();
        return baos.toByteArray();
    }

    public static Cipher initCipher(AESKeyedCipherProvider aesKeyedCipherProvider, EncryptionMethod method, int mode, SecretKey key, byte[] ivBytes) throws EncryptionException {
        try {
            if (method == null || key == null || ivBytes == null) {
                throw new IllegalArgumentException("Missing critical information");
            }
            return aesKeyedCipherProvider.getCipher(method, key, ivBytes, mode == 1);
        }
        catch (Exception e) {
            logger.error("Encountered an exception initializing the cipher", (Throwable)e);
            throw new EncryptionException(e);
        }
    }

    public static RepositoryObjectEncryptionMetadata extractEncryptionMetadata(byte[] encryptedRecord) throws EncryptionException, IOException, ClassNotFoundException {
        if (encryptedRecord == null || encryptedRecord.length < 22) {
            throw new EncryptionException("The encrypted record is too short to contain the metadata");
        }
        ByteArrayInputStream bais = new ByteArrayInputStream(encryptedRecord);
        try (ObjectInputStream ois = new ObjectInputStream(bais);){
            RepositoryObjectEncryptionMetadata repositoryObjectEncryptionMetadata = (RepositoryObjectEncryptionMetadata)ois.readObject();
            return repositoryObjectEncryptionMetadata;
        }
    }

    public static RepositoryObjectEncryptionMetadata extractEncryptionMetadata(InputStream encryptedRecord) throws EncryptionException, IOException, ClassNotFoundException {
        if (encryptedRecord == null) {
            throw new EncryptionException("The encrypted record is too short to contain the metadata");
        }
        encryptedRecord.read(new byte[2]);
        try (ObjectInputStream ois = new ObjectInputStream((InputStream)new NonCloseableInputStream(encryptedRecord));){
            RepositoryObjectEncryptionMetadata repositoryObjectEncryptionMetadata = (RepositoryObjectEncryptionMetadata)ois.readObject();
            return repositoryObjectEncryptionMetadata;
        }
    }

    public static byte[] extractCipherBytes(byte[] encryptedRecord, RepositoryObjectEncryptionMetadata metadata) {
        int cipherBytesStart = metadata.cipherByteLength > 0 ? encryptedRecord.length - metadata.cipherByteLength : metadata.length() + 2;
        return Arrays.copyOfRange(encryptedRecord, cipherBytesStart, encryptedRecord.length);
    }

    public static boolean isRepositoryEncryptionConfigured(NiFiProperties niFiProperties, RepositoryType repositoryType) {
        switch (repositoryType) {
            case CONTENT: {
                return RepositoryEncryptorUtils.isContentRepositoryEncryptionConfigured(niFiProperties);
            }
            case PROVENANCE: {
                return RepositoryEncryptorUtils.isProvenanceRepositoryEncryptionConfigured(niFiProperties);
            }
            case FLOWFILE: {
                return RepositoryEncryptorUtils.isFlowFileRepositoryEncryptionConfigured(niFiProperties);
            }
        }
        logger.warn("Repository encryption configuration validation attempted for {}, an invalid repository type", (Object)repositoryType);
        return false;
    }

    static boolean isProvenanceRepositoryEncryptionConfigured(NiFiProperties niFiProperties) {
        String implementationClassName = niFiProperties.getProperty("nifi.provenance.repository.implementation");
        boolean encryptedRepo = EWAPR_CLASS_NAME.equals(implementationClassName);
        if (encryptedRepo) {
            return CryptoUtils.isValidKeyProvider(niFiProperties.getProperty("nifi.provenance.repository.encryption.key.provider.implementation"), niFiProperties.getProperty("nifi.provenance.repository.encryption.key.provider.location"), niFiProperties.getProvenanceRepoEncryptionKeyId(), niFiProperties.getProvenanceRepoEncryptionKeys());
        }
        return false;
    }

    static boolean isContentRepositoryEncryptionConfigured(NiFiProperties niFiProperties) {
        String implementationClassName = niFiProperties.getProperty("nifi.content.repository.implementation");
        boolean encryptedRepo = "org.apache.nifi.controller.repository.crypto.EncryptedFileSystemRepository".equals(implementationClassName);
        if (encryptedRepo) {
            return CryptoUtils.isValidKeyProvider(niFiProperties.getProperty("nifi.content.repository.encryption.key.provider.implementation"), niFiProperties.getProperty("nifi.content.repository.encryption.key.provider.location"), niFiProperties.getContentRepositoryEncryptionKeyId(), niFiProperties.getContentRepositoryEncryptionKeys());
        }
        return false;
    }

    static boolean isFlowFileRepositoryEncryptionConfigured(NiFiProperties niFiProperties) {
        String implementationClassName = niFiProperties.getProperty("nifi.flowfile.repository.implementation");
        boolean encryptedRepo = "org.apache.nifi.controller.repository.crypto.EncryptedWriteAheadFlowFileRepository".equals(implementationClassName);
        if (encryptedRepo) {
            return CryptoUtils.isValidKeyProvider(niFiProperties.getProperty("nifi.flowfile.repository.encryption.key.provider.implementation"), niFiProperties.getProperty("nifi.flowfile.repository.encryption.key.provider.location"), niFiProperties.getFlowFileRepoEncryptionKeyId(), niFiProperties.getFlowFileRepoEncryptionKeys());
        }
        return false;
    }

    static String determineKeyProviderImplementationClassName(RepositoryType repositoryType) {
        if (repositoryType == null) {
            logger.warn("Could not determine key provider implementation class name for null repository");
            return "no_such_key_provider_defined";
        }
        switch (repositoryType) {
            case FLOWFILE: {
                return "nifi.flowfile.repository.encryption.key.provider.implementation";
            }
            case CONTENT: {
                return "nifi.content.repository.encryption.key.provider.implementation";
            }
            case PROVENANCE: {
                return "nifi.provenance.repository.encryption.key.provider.implementation";
            }
        }
        logger.warn("Could not determine key provider implementation class name for " + repositoryType.getName());
        return "no_such_key_provider_defined";
    }

    public static KeyProvider validateAndBuildRepositoryKeyProvider(NiFiProperties niFiProperties, RepositoryType repositoryType) throws IOException {
        if (RepositoryEncryptorUtils.isRepositoryEncryptionConfigured(niFiProperties, repositoryType)) {
            try {
                RepositoryEncryptionConfiguration configuration = RepositoryEncryptionConfiguration.fromNiFiProperties(niFiProperties, repositoryType);
                return RepositoryEncryptorUtils.getKeyProvider(configuration);
            }
            catch (KeyManagementException e) {
                String msg = "Encountered an error building the key provider";
                logger.error("Encountered an error building the key provider", (Throwable)e);
                throw new IOException("Encountered an error building the key provider", e);
            }
        }
        throw new IOException("The provided configuration does not support an encrypted " + repositoryType.getName());
    }

    public static KeyProvider validateAndBuildRepositoryKeyProvider(RepositoryEncryptionConfiguration repositoryEncryptionConfiguration) throws IOException {
        try {
            return RepositoryEncryptorUtils.getKeyProvider(repositoryEncryptionConfiguration);
        }
        catch (KeyManagementException e) {
            String msg = "Encountered an error building the key provider";
            logger.error("Encountered an error building the key provider", (Throwable)e);
            throw new IOException("Encountered an error building the key provider", e);
        }
    }

    private static KeyProvider getKeyProvider(RepositoryEncryptionConfiguration repositoryEncryptionConfiguration) throws KeyManagementException {
        String keyId;
        KeyProviderConfiguration<?> keyProviderConfiguration = RepositoryEncryptorUtils.getKeyProviderConfiguration(repositoryEncryptionConfiguration);
        KeyProvider keyProvider = KeyProviderFactory.getKeyProvider(keyProviderConfiguration);
        if (keyProvider.keyExists(keyId = repositoryEncryptionConfiguration.getEncryptionKeyId())) {
            return keyProvider;
        }
        throw new KeyManagementException(String.format("Key Identifier [%s] not found in Key Provider", keyId));
    }

    private static KeyProviderConfiguration<?> getKeyProviderConfiguration(RepositoryEncryptionConfiguration repositoryEncryptionConfiguration) throws KeyManagementException {
        String keyProviderImplementation = repositoryEncryptionConfiguration.getKeyProviderImplementation();
        if (keyProviderImplementation.endsWith(StaticKeyProvider.class.getSimpleName())) {
            return new StaticKeyProviderConfiguration(repositoryEncryptionConfiguration.getEncryptionKeys());
        }
        if (keyProviderImplementation.endsWith(FileBasedKeyProvider.class.getSimpleName())) {
            SecretKey rootKey = CryptoUtils.getRootKey();
            return new FileBasedKeyProviderConfiguration(repositoryEncryptionConfiguration.getKeyProviderLocation(), rootKey);
        }
        if (keyProviderImplementation.endsWith(KeyStoreKeyProvider.class.getSimpleName())) {
            String keyProviderPassword = repositoryEncryptionConfiguration.getKeyProviderPassword();
            if (StringUtils.isBlank((String)keyProviderPassword)) {
                throw new KeyManagementException("Key Provider Password not configured");
            }
            String location = repositoryEncryptionConfiguration.getKeyProviderLocation();
            char[] keyStorePassword = repositoryEncryptionConfiguration.getKeyProviderPassword().toCharArray();
            String keyStoreType = repositoryEncryptionConfiguration.getKeyStoreType();
            try {
                KeyStore keyStore = KeyStoreUtils.loadSecretKeyStore(location, keyStorePassword, keyStoreType);
                return new KeyStoreKeyProviderConfiguration(keyStore, keyStorePassword);
            }
            catch (TlsException e) {
                throw new KeyManagementException("Key Store Provider loading failed", e);
            }
        }
        throw new UnsupportedOperationException(String.format("Key Provider Implementation [%s] not supported", keyProviderImplementation));
    }
}

