/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HexFormat;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import javax.security.auth.x500.X500Principal;
import org.apache.nifi.security.cert.builder.StandardCertificateBuilder;
import org.apache.nifi.security.util.KeystoreType;
import org.apache.nifi.security.util.StandardTlsConfiguration;
import org.apache.nifi.security.util.TlsConfiguration;
import org.apache.nifi.security.util.TlsException;
import org.apache.nifi.security.util.TlsPlatform;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyStoreUtils {
    private static final Logger logger = LoggerFactory.getLogger(KeyStoreUtils.class);
    private static final BouncyCastleProvider BOUNCY_CASTLE_PROVIDER = new BouncyCastleProvider();
    private static final String JKS_EXT = ".jks";
    private static final String PKCS12_EXT = ".p12";
    private static final String BCFKS_EXT = ".bcfks";
    private static final String KEY_ALIAS = "nifi-key";
    private static final String CERT_ALIAS = "nifi-cert";
    private static final X500Principal CERT_DN = new X500Principal("CN=localhost");
    private static final String KEY_ALGORITHM = "RSA";
    private static final int PASSWORD_LENGTH = 16;
    private static final String TEST_KEYSTORE_PREFIX = "test-keystore-";
    private static final String TEST_TRUSTSTORE_PREFIX = "test-truststore-";
    private static final String KEYSTORE_ERROR_MSG = "There was an error creating a Keystore.";
    private static final String TRUSTSTORE_ERROR_MSG = "There was an error creating a Truststore.";
    private static final Map<KeystoreType, String> KEY_STORE_EXTENSIONS = new HashMap<KeystoreType, String>();

    public static KeyStore getKeyStore(String keyStoreType) throws KeyStoreException {
        if (KeystoreType.BCFKS.toString().equals(keyStoreType)) {
            return KeyStore.getInstance(keyStoreType, (Provider)BOUNCY_CASTLE_PROVIDER);
        }
        return KeyStore.getInstance(keyStoreType);
    }

    public static KeyStore getSecretKeyStore(String keystoreTypeName) throws KeyStoreException {
        KeystoreType keystoreType = KeyStoreUtils.getKeystoreType(keystoreTypeName);
        if (KeystoreType.BCFKS == keystoreType) {
            return KeyStore.getInstance(keystoreType.getType(), (Provider)BOUNCY_CASTLE_PROVIDER);
        }
        if (KeystoreType.PKCS12 == keystoreType) {
            return KeyStore.getInstance(keystoreType.getType());
        }
        throw new KeyStoreException(String.format("Keystore Type [%s] does not support Secret Keys", keystoreType.getType()));
    }

    public static KeyStore loadKeyStore(String keystorePath, char[] keystorePassword, String keystoreType) throws TlsException {
        try {
            KeyStore keyStore = KeyStoreUtils.getKeyStore(keystoreType);
            try (FileInputStream keyStoreStream = new FileInputStream(keystorePath);){
                keyStore.load(keyStoreStream, keystorePassword);
            }
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Encountered an error loading keystore: {}", (Object)e.getLocalizedMessage());
            throw new TlsException("Error loading keystore", (Throwable)e);
        }
    }

    public static KeyStore loadSecretKeyStore(String keystorePath, char[] keystorePassword, String keystoreTypeName) throws TlsException {
        try {
            KeyStore keyStore = KeyStoreUtils.getSecretKeyStore(keystoreTypeName);
            try (FileInputStream keyStoreStream = new FileInputStream(keystorePath);){
                keyStore.load(keyStoreStream, keystorePassword);
            }
            return keyStore;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new TlsException(String.format("Loading Secret Keystore [%s] Type [%s] Failed", keystorePath, keystoreTypeName), (Throwable)e);
        }
    }

    public static TlsConfiguration createTlsConfigAndNewKeystoreTruststore(TlsConfiguration tlsConfiguration, int certDurationDays, String[] dnsSubjectAlternativeNames) throws IOException, GeneralSecurityException {
        Path trustStorePath;
        Path keyStorePath;
        String keystorePassword = KeyStoreUtils.isNotBlank(tlsConfiguration.getKeystorePassword()) ? tlsConfiguration.getKeystorePassword() : KeyStoreUtils.generatePassword();
        KeystoreType keystoreType = tlsConfiguration.getKeystoreType() != null ? tlsConfiguration.getKeystoreType() : KeystoreType.PKCS12;
        String keyPassword = KeyStoreUtils.isNotBlank(tlsConfiguration.getKeyPassword()) ? tlsConfiguration.getKeyPassword() : keystorePassword;
        String truststorePassword = KeyStoreUtils.isNotBlank(tlsConfiguration.getTruststorePassword()) ? tlsConfiguration.getTruststorePassword() : KeyStoreUtils.generatePassword();
        KeystoreType truststoreType = tlsConfiguration.getTruststoreType() != null ? tlsConfiguration.getTruststoreType() : KeystoreType.PKCS12;
        try {
            keyStorePath = KeyStoreUtils.generateTempKeystorePath(keystoreType);
        }
        catch (IOException e) {
            logger.error(KEYSTORE_ERROR_MSG, (Throwable)e);
            throw new UncheckedIOException(KEYSTORE_ERROR_MSG, e);
        }
        try {
            trustStorePath = KeyStoreUtils.generateTempTruststorePath(truststoreType);
        }
        catch (IOException e) {
            logger.error(TRUSTSTORE_ERROR_MSG, (Throwable)e);
            throw new UncheckedIOException(TRUSTSTORE_ERROR_MSG, e);
        }
        X509Certificate clientCert = KeyStoreUtils.createKeyStoreAndGetX509Certificate(KEY_ALIAS, keystorePassword, keyPassword, keyStorePath.toString(), keystoreType, certDurationDays, dnsSubjectAlternativeNames);
        KeyStoreUtils.createTrustStore(clientCert, CERT_ALIAS, truststorePassword, trustStorePath.toString(), truststoreType);
        return new StandardTlsConfiguration(keyStorePath.toString(), keystorePassword, keyPassword, keystoreType, trustStorePath.toString(), truststorePassword, truststoreType, TlsPlatform.getLatestProtocol());
    }

    public static KeyManagerFactory getKeyManagerFactoryFromKeyStore(KeyStore keyStore, char[] keystorePassword, char[] keyPassword) throws TlsException {
        try {
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            if (keyPassword == null) {
                keyManagerFactory.init(keyStore, keystorePassword);
            } else {
                keyManagerFactory.init(keyStore, keyPassword);
            }
            return keyManagerFactory;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            logger.error("Encountered an error loading keystore: {}", (Object)e.getLocalizedMessage());
            throw new TlsException("Error loading keystore", (Throwable)e);
        }
    }

    public static KeyManagerFactory loadKeyManagerFactory(TlsConfiguration tlsConfiguration) throws TlsException {
        return KeyStoreUtils.loadKeyManagerFactory(tlsConfiguration.getKeystorePath(), tlsConfiguration.getKeystorePassword(), tlsConfiguration.getFunctionalKeyPassword(), tlsConfiguration.getKeystoreType().getType());
    }

    public static KeyManagerFactory loadKeyManagerFactory(String keystorePath, String keystorePassword, String keyPassword, String keystoreType) throws TlsException {
        if (keystorePassword == null || keystorePassword.isEmpty()) {
            throw new IllegalArgumentException("The keystore password cannot be null or empty");
        }
        char[] keystorePasswordChars = keystorePassword.toCharArray();
        char[] keyPasswordChars = KeyStoreUtils.isNotBlank(keyPassword) ? keyPassword.toCharArray() : keystorePasswordChars;
        KeyStore keyStore = KeyStoreUtils.loadKeyStore(keystorePath, keystorePasswordChars, keystoreType);
        return KeyStoreUtils.getKeyManagerFactoryFromKeyStore(keyStore, keystorePasswordChars, keyPasswordChars);
    }

    public static KeyStore loadTrustStore(String truststorePath, char[] truststorePassword, String truststoreType) throws TlsException {
        try {
            KeyStore trustStore = KeyStoreUtils.getKeyStore(truststoreType);
            try (FileInputStream trustStoreStream = new FileInputStream(truststorePath);){
                trustStore.load(trustStoreStream, truststorePassword);
            }
            return trustStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Encountered an error loading truststore: {}", (Object)e.getLocalizedMessage());
            throw new TlsException("Error loading truststore", (Throwable)e);
        }
    }

    public static TrustManagerFactory getTrustManagerFactoryFromTrustStore(KeyStore trustStore) throws TlsException {
        try {
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(trustStore);
            return trustManagerFactory;
        }
        catch (KeyStoreException | NoSuchAlgorithmException e) {
            logger.error("Encountered an error loading truststore: {}", (Object)e.getLocalizedMessage());
            throw new TlsException("Error loading truststore", (Throwable)e);
        }
    }

    public static TrustManagerFactory loadTrustManagerFactory(TlsConfiguration tlsConfiguration) throws TlsException {
        return KeyStoreUtils.loadTrustManagerFactory(tlsConfiguration.getTruststorePath(), tlsConfiguration.getTruststorePassword(), tlsConfiguration.getTruststoreType().getType());
    }

    public static TrustManagerFactory loadTrustManagerFactory(String truststorePath, String truststorePassword, String truststoreType) throws TlsException {
        if (truststoreType.equalsIgnoreCase(KeystoreType.PKCS12.getType()) && (truststorePassword == null || truststorePassword.isBlank())) {
            throw new IllegalArgumentException("A PKCS12 Truststore Type requires a password");
        }
        char[] truststorePasswordChars = KeyStoreUtils.isNotBlank(truststorePassword) ? truststorePassword.toCharArray() : null;
        KeyStore trustStore = KeyStoreUtils.loadTrustStore(truststorePath, truststorePasswordChars, truststoreType);
        return KeyStoreUtils.getTrustManagerFactoryFromTrustStore(trustStore);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean isStoreValid(URL keystore, KeystoreType keystoreType, char[] password) {
        if (keystore == null) {
            throw new IllegalArgumentException("Keystore may not be null");
        }
        if (keystoreType == null) {
            throw new IllegalArgumentException("Keystore type may not be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password may not be null");
        }
        BufferedInputStream bis = null;
        try {
            bis = new BufferedInputStream(keystore.openStream());
            KeyStore ks = KeyStoreUtils.getKeyStore(keystoreType.name());
            ks.load(bis, password);
            boolean bl = true;
            return bl;
        }
        catch (Exception e) {
            logger.debug("Keystore [{}] Type [{}] load failed", new Object[]{keystore, keystoreType, e});
            boolean bl = false;
            return bl;
        }
        finally {
            if (bis != null) {
                try {
                    bis.close();
                }
                catch (IOException ioe) {
                    logger.warn("Failed to close input stream", (Throwable)ioe);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     */
    public static boolean isKeyPasswordCorrect(URL keystore, KeystoreType keystoreType, char[] password, char[] keyPassword) {
        if (keystore == null) {
            throw new IllegalArgumentException("Keystore may not be null");
        }
        if (keystoreType == null) {
            throw new IllegalArgumentException("Keystore type may not be null");
        }
        if (password == null) {
            throw new IllegalArgumentException("Password may not be null");
        }
        BufferedInputStream bis = null;
        try {
            bis = new BufferedInputStream(keystore.openStream());
            KeyStore ks = KeyStoreUtils.getKeyStore(keystoreType.name());
            ks.load(bis, password);
            String alias = ks.aliases().nextElement();
            try {
                ks.getKey(alias, keyPassword);
                boolean bl = true;
                return bl;
            }
            catch (UnrecoverableKeyException e) {
                boolean ioe;
                block20: {
                    logger.warn("Tried to access a key in keystore " + String.valueOf(keystore) + " with a key password that failed");
                    ioe = false;
                    if (bis == null) break block20;
                    try {
                        bis.close();
                    }
                    catch (IOException ioe2) {
                        logger.warn("Failed to close input stream", (Throwable)ioe2);
                    }
                }
                return ioe;
                catch (Exception e2) {
                    boolean bl = false;
                    return bl;
                }
                catch (Throwable throwable) {
                    throw throwable;
                }
            }
        }
        finally {
            if (bis != null) {
                try {
                    bis.close();
                }
                catch (IOException ioe) {
                    logger.warn("Failed to close input stream", (Throwable)ioe);
                }
            }
        }
    }

    public static KeystoreType getKeystoreTypeFromExtension(String keystorePath) {
        KeystoreType keystoreType = KeystoreType.PKCS12;
        for (Map.Entry<KeystoreType, String> keystoreTypeEntry : KEY_STORE_EXTENSIONS.entrySet()) {
            String extension = keystoreTypeEntry.getValue().toLowerCase();
            if (!keystorePath.endsWith(extension)) continue;
            keystoreType = keystoreTypeEntry.getKey();
            break;
        }
        return keystoreType;
    }

    public static boolean isSecretKeyEntrySupported(KeystoreType keystoreType) {
        return KeystoreType.BCFKS == keystoreType || KeystoreType.PKCS12 == keystoreType;
    }

    private static X509Certificate createKeyStoreAndGetX509Certificate(String alias, String keyStorePassword, String keyPassword, String keyStorePath, KeystoreType keyStoreType, int certDurationDays, String[] dnsSubjectAlternativeNames) throws IOException, KeyStoreException, NoSuchAlgorithmException, CertificateException {
        KeyPair keyPair = KeyPairGenerator.getInstance(KEY_ALGORITHM).generateKeyPair();
        List<Object> dnsNames = dnsSubjectAlternativeNames == null ? Collections.emptyList() : Arrays.asList(dnsSubjectAlternativeNames);
        X509Certificate selfSignedCert = new StandardCertificateBuilder(keyPair, CERT_DN, Duration.ofDays(certDurationDays)).setDnsSubjectAlternativeNames(dnsNames).build();
        try (FileOutputStream outputStream = new FileOutputStream(keyStorePath);){
            KeyStore keyStore = KeyStoreUtils.loadEmptyKeyStore(keyStoreType);
            keyStore.setKeyEntry(alias, keyPair.getPrivate(), keyPassword.toCharArray(), new Certificate[]{selfSignedCert});
            keyStore.store(outputStream, keyStorePassword.toCharArray());
            X509Certificate x509Certificate = selfSignedCert;
            return x509Certificate;
        }
    }

    private static void createTrustStore(X509Certificate cert, String alias, String password, String path, KeystoreType truststoreType) throws KeyStoreException, NoSuchAlgorithmException, CertificateException {
        try (FileOutputStream outputStream = new FileOutputStream(path);){
            KeyStore trustStore = KeyStoreUtils.loadEmptyKeyStore(truststoreType);
            trustStore.setCertificateEntry(alias, cert);
            trustStore.store(outputStream, password.toCharArray());
        }
        catch (IOException e) {
            throw new UncheckedIOException(TRUSTSTORE_ERROR_MSG, e);
        }
    }

    private static Path generateTempKeystorePath(KeystoreType keystoreType) throws IOException {
        return Files.createTempFile(TEST_KEYSTORE_PREFIX, KeyStoreUtils.getKeystoreExtension(keystoreType), new FileAttribute[0]);
    }

    private static Path generateTempTruststorePath(KeystoreType truststoreType) throws IOException {
        return Files.createTempFile(TEST_TRUSTSTORE_PREFIX, KeyStoreUtils.getKeystoreExtension(truststoreType), new FileAttribute[0]);
    }

    private static KeyStore loadEmptyKeyStore(KeystoreType keyStoreType) throws KeyStoreException, CertificateException, NoSuchAlgorithmException {
        try {
            KeyStore keyStore = KeyStoreUtils.getKeyStore(keyStoreType.getType());
            keyStore.load(null, null);
            return keyStore;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Error loading keystore", e);
        }
    }

    private static String getKeystoreExtension(KeystoreType keystoreType) {
        return KEY_STORE_EXTENSIONS.get(keystoreType);
    }

    private static String generatePassword() {
        byte[] password = new byte[16];
        new SecureRandom().nextBytes(password);
        return HexFormat.of().formatHex(password);
    }

    private static KeystoreType getKeystoreType(String keystoreTypeName) {
        String keystoreTypeFilter = keystoreTypeName.toUpperCase();
        Optional<KeystoreType> foundKeystoreType = Arrays.stream(KeystoreType.values()).filter(keystoreType -> keystoreType.getType().equals(keystoreTypeFilter)).findFirst();
        return foundKeystoreType.orElseThrow(() -> new IllegalArgumentException(String.format("Keystore Type [%s] not found", keystoreTypeFilter)));
    }

    private static boolean isNotBlank(String string) {
        return string != null && !string.isBlank();
    }

    static {
        KEY_STORE_EXTENSIONS.put(KeystoreType.JKS, JKS_EXT);
        KEY_STORE_EXTENSIONS.put(KeystoreType.PKCS12, PKCS12_EXT);
        KEY_STORE_EXTENSIONS.put(KeystoreType.BCFKS, BCFKS_EXT);
    }
}

