/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.logging.ProcessorLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.annotation.CapabilityDescription;
import org.apache.nifi.processor.annotation.SideEffectFree;
import org.apache.nifi.processor.annotation.Tags;
import org.apache.nifi.processor.annotation.TriggerSerially;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.util.timebuffer.EntityAccess;
import org.apache.nifi.util.timebuffer.TimedBuffer;

@SideEffectFree
@TriggerSerially
@Tags(value={"rate control", "throttle", "rate", "throughput"})
@CapabilityDescription(value="Controls the rate at which data is transferred to follow-on processors.")
public class ControlRate
extends AbstractProcessor {
    public static final String DATA_RATE = "data rate";
    public static final String FLOWFILE_RATE = "flowfile count";
    public static final String ATTRIBUTE_RATE = "attribute value";
    public static final PropertyDescriptor RATE_CONTROL_CRITERIA = new PropertyDescriptor.Builder().name("Rate Control Criteria").description("Indicates the criteria that is used to control the throughput rate. Changing this value resets the rate counters.").required(true).allowableValues(new String[]{"data rate", "flowfile count", "attribute value"}).defaultValue("data rate").build();
    public static final PropertyDescriptor MAX_RATE = new PropertyDescriptor.Builder().name("Maximum Rate").description("The maximum rate at which data should pass through this processor. The format of this property is expected to be a positive integer, or a Data Size (such as '1 MB') if Rate Control Criteria is set to 'data rate'.").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor RATE_CONTROL_ATTRIBUTE_NAME = new PropertyDescriptor.Builder().name("Rate Controlled Attribute").description("The name of an attribute whose values build toward the rate limit if Rate Control Criteria is set to 'attribute value'. The value of the attribute referenced by this property must be a positive integer, or the FlowFile will be routed to failure. This value is ignored if Rate Control Criteria is not set to 'attribute value'. Changing this value resets the rate counters.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(false).build();
    public static final PropertyDescriptor TIME_PERIOD = new PropertyDescriptor.Builder().name("Time Duration").description("The amount of time to which the Maximum Data Size and Maximum Number of Files pertains. Changing this value resets the rate counters.").required(true).addValidator(StandardValidators.createTimePeriodValidator((long)1L, (TimeUnit)TimeUnit.SECONDS, (long)Integer.MAX_VALUE, (TimeUnit)TimeUnit.SECONDS)).defaultValue("1 min").build();
    public static final PropertyDescriptor GROUPING_ATTRIBUTE_NAME = new PropertyDescriptor.Builder().name("Grouping Attribute").description("By default, a single \"throttle\" is used for all FlowFiles. If this value is specified, a separate throttle is used for each value specified by the attribute with this name. Changing this value resets the rate counters.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(false).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All FlowFiles are transferred to this relationship").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("FlowFiles will be routed to this relationship if they are missing a necessary attribute or the attribute is not in the expected format").build();
    private static final Pattern POSITIVE_LONG_PATTERN = Pattern.compile("0*[1-9][0-9]*");
    private static final String DEFAULT_GROUP_ATTRIBUTE = ControlRate.class.getName() + "###____DEFAULT_GROUP_ATTRIBUTE___###";
    private final ConcurrentMap<String, Throttle> throttleMap = new ConcurrentHashMap<String, Throttle>();
    private List<PropertyDescriptor> properties;
    private Set<Relationship> relationships;
    private final AtomicLong lastThrottleClearTime = new AtomicLong(System.currentTimeMillis());

    protected void init(ProcessorInitializationContext context) {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(RATE_CONTROL_CRITERIA);
        properties.add(MAX_RATE);
        properties.add(RATE_CONTROL_ATTRIBUTE_NAME);
        properties.add(TIME_PERIOD);
        properties.add(GROUPING_ATTRIBUTE_NAME);
        this.properties = Collections.unmodifiableList(properties);
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(REL_SUCCESS);
        this.relationships = Collections.unmodifiableSet(relationships);
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.properties;
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext context) {
        Validator rateValidator;
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>(super.customValidate(context));
        switch (context.getProperty(RATE_CONTROL_CRITERIA).getValue().toLowerCase()) {
            case "data rate": {
                rateValidator = StandardValidators.DATA_SIZE_VALIDATOR;
                break;
            }
            case "attribute value": {
                rateValidator = StandardValidators.POSITIVE_LONG_VALIDATOR;
                String rateAttr = context.getProperty(RATE_CONTROL_ATTRIBUTE_NAME).getValue();
                if (rateAttr != null) break;
                validationResults.add(new ValidationResult.Builder().subject(RATE_CONTROL_ATTRIBUTE_NAME.getName()).explanation("<Rate Controlled Attribute> property must be set if using <Rate Control Criteria> of 'attribute value'").build());
                break;
            }
            default: {
                rateValidator = StandardValidators.POSITIVE_LONG_VALIDATOR;
            }
        }
        ValidationResult rateResult = rateValidator.validate("Maximum Rate", context.getProperty(MAX_RATE).getValue(), null);
        if (!rateResult.isValid()) {
            validationResults.add(rateResult);
        }
        return validationResults;
    }

    public void onPropertyModified(PropertyDescriptor descriptor, String oldValue, String newValue) {
        super.onPropertyModified(descriptor, oldValue, newValue);
        if (descriptor.equals((Object)RATE_CONTROL_CRITERIA) || descriptor.equals((Object)RATE_CONTROL_ATTRIBUTE_NAME) || descriptor.equals((Object)GROUPING_ATTRIBUTE_NAME) || descriptor.equals((Object)TIME_PERIOD)) {
            this.throttleMap.clear();
        } else if (descriptor.equals((Object)MAX_RATE)) {
            long newRate = DataUnit.DATA_SIZE_PATTERN.matcher(newValue).matches() ? DataUnit.parseDataSize((String)newValue, (DataUnit)DataUnit.B).longValue() : Long.parseLong(newValue);
            for (Throttle throttle : this.throttleMap.values()) {
                throttle.setMaxRate(newRate);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        long rateValue;
        FlowFile flowFile;
        long throttleExpirationMillis;
        long lastClearTime = this.lastThrottleClearTime.get();
        if (lastClearTime < (throttleExpirationMillis = System.currentTimeMillis() - 2L * context.getProperty(TIME_PERIOD).asTimePeriod(TimeUnit.MILLISECONDS)) && this.lastThrottleClearTime.compareAndSet(lastClearTime, System.currentTimeMillis())) {
            Iterator itr = this.throttleMap.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry entry = itr.next();
                Throttle throttle = (Throttle)entry.getValue();
                if (!throttle.tryLock()) continue;
                try {
                    if (throttle.lastUpdateTime() >= lastClearTime) continue;
                    itr.remove();
                }
                finally {
                    throttle.unlock();
                }
            }
        }
        if ((flowFile = session.get()) == null) {
            return;
        }
        ProcessorLog logger = this.getLogger();
        long seconds = context.getProperty(TIME_PERIOD).asTimePeriod(TimeUnit.SECONDS);
        String rateControlAttributeName = context.getProperty(RATE_CONTROL_ATTRIBUTE_NAME).getValue();
        switch (context.getProperty(RATE_CONTROL_CRITERIA).getValue().toLowerCase()) {
            case "data rate": {
                rateValue = flowFile.getSize();
                break;
            }
            case "flowfile count": {
                rateValue = 1L;
                break;
            }
            case "attribute value": {
                String attributeValue = flowFile.getAttribute(rateControlAttributeName);
                if (attributeValue == null) {
                    logger.error("routing {} to 'failure' because FlowFile is missing required attribute {}", new Object[]{flowFile, rateControlAttributeName});
                    session.transfer(flowFile, REL_FAILURE);
                    return;
                }
                if (!POSITIVE_LONG_PATTERN.matcher(attributeValue).matches()) {
                    logger.error("routing {} to 'failure' because FlowFile attribute {} has a value of {}, which is not a positive integer", new Object[]{flowFile, rateControlAttributeName, attributeValue});
                    session.transfer(flowFile, REL_FAILURE);
                    return;
                }
                rateValue = Long.parseLong(attributeValue);
                break;
            }
            default: {
                throw new AssertionError((Object)("<Rate Control Criteria> property set to illegal value of " + context.getProperty(RATE_CONTROL_CRITERIA).getValue()));
            }
        }
        String groupingAttributeName = context.getProperty(GROUPING_ATTRIBUTE_NAME).getValue();
        String groupName = groupingAttributeName == null ? DEFAULT_GROUP_ATTRIBUTE : flowFile.getAttribute(groupingAttributeName);
        Throttle throttle = (Throttle)this.throttleMap.get(groupName);
        if (throttle == null) {
            throttle = new Throttle((int)seconds, TimeUnit.SECONDS, logger);
            String maxRateValue = context.getProperty(MAX_RATE).getValue();
            long newRate = DataUnit.DATA_SIZE_PATTERN.matcher(maxRateValue).matches() ? DataUnit.parseDataSize((String)maxRateValue, (DataUnit)DataUnit.B).longValue() : Long.parseLong(maxRateValue);
            throttle.setMaxRate(newRate);
            this.throttleMap.put(groupName, throttle);
        }
        throttle.lock();
        try {
            if (throttle.tryAdd(rateValue)) {
                logger.info("transferring {} to 'success'", new Object[]{flowFile});
                session.transfer(flowFile, REL_SUCCESS);
            } else {
                flowFile = session.penalize(flowFile);
                session.transfer(flowFile);
            }
        }
        finally {
            throttle.unlock();
        }
    }

    private static class Throttle
    extends ReentrantLock {
        private final AtomicLong maxRate = new AtomicLong(1L);
        private final long timePeriodValue;
        private final TimeUnit timePeriodUnit;
        private final TimedBuffer<TimestampedLong> timedBuffer;
        private final ProcessorLog logger;
        private volatile long penalizationExpired;
        private volatile long lastUpdateTime;

        public Throttle(int timePeriod, TimeUnit unit, ProcessorLog logger) {
            this.timePeriodUnit = unit;
            this.timePeriodValue = timePeriod;
            this.timedBuffer = new TimedBuffer(unit, timePeriod, (EntityAccess)new RateEntityAccess());
            this.logger = logger;
        }

        public void setMaxRate(long maxRate) {
            this.maxRate.set(maxRate);
        }

        public long lastUpdateTime() {
            return this.lastUpdateTime;
        }

        public boolean tryAdd(long value) {
            long now = System.currentTimeMillis();
            if (this.penalizationExpired > now) {
                return false;
            }
            long maxRateValue = this.maxRate.get();
            TimestampedLong sum = (TimestampedLong)this.timedBuffer.getAggregateValue(TimeUnit.MILLISECONDS.convert(this.timePeriodValue, this.timePeriodUnit));
            if (sum != null && sum.getValue() >= maxRateValue) {
                this.logger.debug("current sum for throttle is {}, so not allowing rate of {} through", new Object[]{sum.getValue(), value});
                return false;
            }
            this.logger.debug("current sum for throttle is {}, so allowing rate of {} through", new Object[]{sum == null ? 0L : sum.getValue(), value});
            long transferred = ((TimestampedLong)this.timedBuffer.add((Object)new TimestampedLong(value))).getValue();
            if (transferred > maxRateValue) {
                long amountOver = transferred - maxRateValue;
                long milliDuration = TimeUnit.MILLISECONDS.convert(this.timePeriodValue, this.timePeriodUnit);
                double pct = (double)amountOver / (double)maxRateValue;
                long penalizationPeriod = (long)((double)milliDuration * pct);
                this.penalizationExpired = now + penalizationPeriod;
                this.logger.debug("allowing rate of {} through but penalizing Throttle for {} milliseconds", new Object[]{value, penalizationPeriod});
            }
            this.lastUpdateTime = now;
            return true;
        }
    }

    private static class RateEntityAccess
    implements EntityAccess<TimestampedLong> {
        private RateEntityAccess() {
        }

        public TimestampedLong aggregate(TimestampedLong oldValue, TimestampedLong toAdd) {
            if (oldValue == null && toAdd == null) {
                return new TimestampedLong(0L);
            }
            if (oldValue == null) {
                return toAdd;
            }
            if (toAdd == null) {
                return oldValue;
            }
            return new TimestampedLong(oldValue.getValue() + toAdd.getValue());
        }

        public TimestampedLong createNew() {
            return new TimestampedLong(0L);
        }

        public long getTimestamp(TimestampedLong entity) {
            return entity == null ? 0L : entity.getTimestamp();
        }
    }

    private static class TimestampedLong {
        private final Long value;
        private final long timestamp = System.currentTimeMillis();

        public TimestampedLong(Long value) {
            this.value = value;
        }

        public Long getValue() {
            return this.value;
        }

        public long getTimestamp() {
            return this.timestamp;
        }
    }
}

