/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.security.Provider;
import java.security.Security;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.annotation.behavior.EventDriven;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ProcessorLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.io.StreamCallback;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.standard.util.OpenPGPKeyBasedEncryptor;
import org.apache.nifi.processors.standard.util.OpenPGPPasswordBasedEncryptor;
import org.apache.nifi.processors.standard.util.PasswordBasedEncryptor;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.util.StopWatch;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

@EventDriven
@SideEffectFree
@SupportsBatching
@Tags(value={"encryption", "decryption", "password", "JCE", "OpenPGP", "PGP", "GPG"})
@CapabilityDescription(value="Encrypts or Decrypts a FlowFile using either symmetric encryption with a password and randomly generated salt, or asymmetric encryption using a public and secret key.")
public class EncryptContent
extends AbstractProcessor {
    public static final String ENCRYPT_MODE = "Encrypt";
    public static final String DECRYPT_MODE = "Decrypt";
    public static final PropertyDescriptor MODE = new PropertyDescriptor.Builder().name("Mode").description("Specifies whether the content should be encrypted or decrypted").required(true).allowableValues(new String[]{"Encrypt", "Decrypt"}).defaultValue("Encrypt").build();
    public static final PropertyDescriptor ENCRYPTION_ALGORITHM = new PropertyDescriptor.Builder().name("Encryption Algorithm").description("The Encryption Algorithm to use").required(true).allowableValues((Enum[])EncryptionMethod.values()).defaultValue(EncryptionMethod.MD5_256AES.name()).build();
    public static final PropertyDescriptor PASSWORD = new PropertyDescriptor.Builder().name("Password").description("The Password to use for encrypting or decrypting the data").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).sensitive(true).build();
    public static final PropertyDescriptor PUBLIC_KEYRING = new PropertyDescriptor.Builder().name("public-keyring-file").displayName("Public Keyring File").description("In a PGP encrypt mode, this keyring contains the public key of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PUBLIC_KEY_USERID = new PropertyDescriptor.Builder().name("public-key-user-id").displayName("Public Key User Id").description("In a PGP encrypt mode, this user id of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PRIVATE_KEYRING = new PropertyDescriptor.Builder().name("private-keyring-file").displayName("Private Keyring File").description("In a PGP decrypt mode, this keyring contains the private key of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PRIVATE_KEYRING_PASSPHRASE = new PropertyDescriptor.Builder().name("private-keyring-passphrase").displayName("Private Keyring Passphrase").description("In a PGP decrypt mode, this is the private keyring passphrase").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).sensitive(true).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Any FlowFile that is successfully encrypted or decrypted will be routed to success").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Any FlowFile that cannot be encrypted or decrypted will be routed to failure").build();
    private List<PropertyDescriptor> properties;
    private Set<Relationship> relationships;

    protected void init(ProcessorInitializationContext context) {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(MODE);
        properties.add(ENCRYPTION_ALGORITHM);
        properties.add(PASSWORD);
        properties.add(PUBLIC_KEYRING);
        properties.add(PUBLIC_KEY_USERID);
        properties.add(PRIVATE_KEYRING);
        properties.add(PRIVATE_KEYRING_PASSPHRASE);
        this.properties = Collections.unmodifiableList(properties);
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(REL_SUCCESS);
        relationships.add(REL_FAILURE);
        this.relationships = Collections.unmodifiableSet(relationships);
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.properties;
    }

    public static boolean isPGPAlgorithm(String algorithm) {
        return algorithm.startsWith("PGP");
    }

    public static boolean isPGPArmoredAlgorithm(String algorithm) {
        return EncryptContent.isPGPAlgorithm(algorithm) && algorithm.endsWith("ASCII-ARMOR");
    }

    protected Collection<ValidationResult> customValidate(ValidationContext context) {
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>(super.customValidate(context));
        String method = context.getProperty(ENCRYPTION_ALGORITHM).getValue();
        String algorithm = EncryptionMethod.valueOf((String)method).getAlgorithm();
        String password = context.getProperty(PASSWORD).getValue();
        if (EncryptContent.isPGPAlgorithm(algorithm)) {
            if (password == null) {
                boolean encrypt = context.getProperty(MODE).getValue().equalsIgnoreCase(ENCRYPT_MODE);
                if (encrypt) {
                    String publicKeyring = context.getProperty(PUBLIC_KEYRING).getValue();
                    String publicUserId = context.getProperty(PUBLIC_KEY_USERID).getValue();
                    if (publicKeyring == null || publicUserId == null) {
                        validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation(algorithm + " encryption without a " + PASSWORD.getDisplayName() + " requires both " + PUBLIC_KEYRING.getDisplayName() + " and " + PUBLIC_KEY_USERID.getDisplayName()).build());
                    } else {
                        try {
                            if (OpenPGPKeyBasedEncryptor.getPublicKey(publicUserId, publicKeyring) == null) {
                                validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation(PUBLIC_KEYRING.getDisplayName() + " " + publicKeyring + " does not contain user id " + publicUserId).build());
                            }
                        }
                        catch (Exception e) {
                            validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation("Invalid " + PUBLIC_KEYRING.getDisplayName() + " " + publicKeyring + " because " + e.toString()).build());
                        }
                    }
                } else {
                    String privateKeyring = context.getProperty(PRIVATE_KEYRING).getValue();
                    String keyringPassphrase = context.getProperty(PRIVATE_KEYRING_PASSPHRASE).getValue();
                    if (privateKeyring == null || keyringPassphrase == null) {
                        validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getName()).explanation(algorithm + " decryption without a " + PASSWORD.getDisplayName() + " requires both " + PRIVATE_KEYRING.getDisplayName() + " and " + PRIVATE_KEYRING_PASSPHRASE.getDisplayName()).build());
                    } else {
                        String providerName = EncryptionMethod.valueOf((String)method).getProvider();
                        try {
                            if (!OpenPGPKeyBasedEncryptor.validateKeyring(providerName, privateKeyring, keyringPassphrase.toCharArray())) {
                                validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getDisplayName()).explanation(PRIVATE_KEYRING.getDisplayName() + " " + privateKeyring + " could not be opened with the provided " + PRIVATE_KEYRING_PASSPHRASE.getDisplayName()).build());
                            }
                        }
                        catch (Exception e) {
                            validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getDisplayName()).explanation("Invalid " + PRIVATE_KEYRING.getDisplayName() + " " + privateKeyring + " because " + e.toString()).build());
                        }
                    }
                }
            }
        } else if (password == null) {
            validationResults.add(new ValidationResult.Builder().subject(PASSWORD.getName()).explanation(PASSWORD.getDisplayName() + " is required when using algorithm " + algorithm).build());
        }
        return validationResults;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        StreamCallback callback;
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        ProcessorLog logger = this.getLogger();
        String method = context.getProperty(ENCRYPTION_ALGORITHM).getValue();
        EncryptionMethod encryptionMethod = EncryptionMethod.valueOf((String)method);
        String providerName = encryptionMethod.getProvider();
        String algorithm = encryptionMethod.getAlgorithm();
        String password = context.getProperty(PASSWORD).getValue();
        boolean encrypt = context.getProperty(MODE).getValue().equalsIgnoreCase(ENCRYPT_MODE);
        try {
            Encryptor encryptor;
            if (EncryptContent.isPGPAlgorithm(algorithm)) {
                String filename = flowFile.getAttribute(CoreAttributes.FILENAME.key());
                String publicKeyring = context.getProperty(PUBLIC_KEYRING).getValue();
                String privateKeyring = context.getProperty(PRIVATE_KEYRING).getValue();
                if (encrypt && publicKeyring != null) {
                    String publicUserId = context.getProperty(PUBLIC_KEY_USERID).getValue();
                    encryptor = new OpenPGPKeyBasedEncryptor(algorithm, providerName, publicKeyring, publicUserId, null, filename);
                } else if (!encrypt && privateKeyring != null) {
                    char[] keyringPassphrase = context.getProperty(PRIVATE_KEYRING_PASSPHRASE).getValue().toCharArray();
                    encryptor = new OpenPGPKeyBasedEncryptor(algorithm, providerName, privateKeyring, null, keyringPassphrase, filename);
                } else {
                    char[] passphrase = Normalizer.normalize(password, Normalizer.Form.NFC).toCharArray();
                    encryptor = new OpenPGPPasswordBasedEncryptor(algorithm, providerName, passphrase, filename);
                }
            } else {
                char[] passphrase = Normalizer.normalize(password, Normalizer.Form.NFC).toCharArray();
                encryptor = new PasswordBasedEncryptor(algorithm, providerName, passphrase);
            }
            callback = encrypt ? encryptor.getEncryptionCallback() : encryptor.getDecryptionCallback();
        }
        catch (Exception e) {
            logger.error("Failed to initialize {}cryption algorithm because - ", new Object[]{encrypt ? "en" : "de", e});
            session.rollback();
            context.yield();
            return;
        }
        try {
            StopWatch stopWatch = new StopWatch(true);
            flowFile = session.write(flowFile, callback);
            logger.info("successfully {}crypted {}", new Object[]{encrypt ? "en" : "de", flowFile});
            session.getProvenanceReporter().modifyContent(flowFile, stopWatch.getElapsed(TimeUnit.MILLISECONDS));
            session.transfer(flowFile, REL_SUCCESS);
        }
        catch (ProcessException e) {
            logger.error("Cannot {}crypt {} - ", new Object[]{encrypt ? "en" : "de", flowFile, e});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }

    public static interface Encryptor {
        public StreamCallback getEncryptionCallback() throws Exception;

        public StreamCallback getDecryptionCallback() throws Exception;
    }
}

