/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.annotation.behavior.EventDriven;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.Stateful;
import org.apache.nifi.annotation.behavior.TriggerSerially;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.PropertyValue;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.components.state.StateMap;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;

@EventDriven
@Tags(value={"sort", "order"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@TriggerSerially
@CapabilityDescription(value="Enforces expected ordering of FlowFiles that belong to the same data group within a single node.  Although PriorityAttributePrioritizer can be used on a connection to ensure that flow files going through that connection are in priority order, depending on error-handling, branching, and other flow designs, it is possible for FlowFiles to get out-of-order. EnforceOrder can be used to enforce original ordering for those FlowFiles. [IMPORTANT] In order to take effect of EnforceOrder, FirstInFirstOutPrioritizer should be used at EVERY downstream relationship UNTIL the order of FlowFiles physically get FIXED by operation such as MergeContent or being stored to the final destination.")
@Stateful(scopes={Scope.LOCAL}, description="EnforceOrder uses following states per ordering group: '<groupId>.target' is a order number which is being waited to arrive next. When a FlowFile with a matching order arrives, or a FlowFile overtakes the FlowFile being waited for because of wait timeout, target order will be updated to (FlowFile.order + 1). '<groupId>.max is the maximum order number for a group. '<groupId>.updatedAt' is a timestamp when the order of a group was updated last time. These managed states will be removed automatically once a group is determined as inactive, see 'Inactive Timeout' for detail.")
@WritesAttributes(value={@WritesAttribute(attribute="EnforceOrder.startedAt", description="All FlowFiles going through this processor will have this attribute. This value is used to determine wait timeout."), @WritesAttribute(attribute="EnforceOrder.result", description="All FlowFiles going through this processor will have this attribute denoting which relationship it was routed to."), @WritesAttribute(attribute="EnforceOrder.detail", description="FlowFiles routed to 'failure' or 'skipped' relationship will have this attribute describing details."), @WritesAttribute(attribute="EnforceOrder.expectedOrder", description="FlowFiles routed to 'wait' or 'skipped' relationship will have this attribute denoting expected order when the FlowFile was processed.")})
public class EnforceOrder
extends AbstractProcessor {
    public static final String ATTR_STARTED_AT = "EnforceOrder.startedAt";
    public static final String ATTR_EXPECTED_ORDER = "EnforceOrder.expectedOrder";
    public static final String ATTR_RESULT = "EnforceOrder.result";
    public static final String ATTR_DETAIL = "EnforceOrder.detail";
    private static final Function<String, String> STATE_TARGET_ORDER = groupId -> groupId + ".target";
    private static final String STATE_SUFFIX_UPDATED_AT = ".updatedAt";
    private static final Function<String, String> STATE_UPDATED_AT = groupId -> groupId + STATE_SUFFIX_UPDATED_AT;
    private static final Function<String, String> STATE_MAX_ORDER = groupId -> groupId + ".max";
    public static final PropertyDescriptor GROUP_IDENTIFIER = new PropertyDescriptor.Builder().name("group-id").displayName("Group Identifier").description("EnforceOrder is capable of multiple ordering groups. 'Group Identifier' is used to determine which group a FlowFile belongs to. This property will be evaluated with each incoming FlowFile. If evaluated result is empty, the FlowFile will be routed to failure.").required(true).addValidator(StandardValidators.NON_BLANK_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).defaultValue("${filename}").build();
    public static final PropertyDescriptor ORDER_ATTRIBUTE = new PropertyDescriptor.Builder().name("order-attribute").displayName("Order Attribute").description("A name of FlowFile attribute whose value will be used to enforce order of FlowFiles within a group. If a FlowFile does not have this attribute, or its value is not an integer, the FlowFile will be routed to failure.").required(true).addValidator(StandardValidators.NON_BLANK_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    public static final PropertyDescriptor INITIAL_ORDER = new PropertyDescriptor.Builder().name("initial-order").displayName("Initial Order").description("When the first FlowFile of a group arrives, initial target order will be computed and stored in the managed state. After that, target order will start being tracked by EnforceOrder and stored in the state management store. If Expression Language is used but evaluated result was not an integer, then the FlowFile will be routed to failure, and initial order will be left unknown until consecutive FlowFiles provide a valid initial order.").required(true).addValidator(StandardValidators.NON_NEGATIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).defaultValue("0").build();
    public static final PropertyDescriptor MAX_ORDER = new PropertyDescriptor.Builder().name("maximum-order").displayName("Maximum Order").description("If specified, any FlowFiles that have larger order will be routed to failure. This property is computed only once for a given group. After a maximum order is computed, it will be persisted in the state management store and used for other FlowFiles belonging to the same group. If Expression Language is used but evaluated result was not an integer, then the FlowFile will be routed to failure, and maximum order will be left unknown until consecutive FlowFiles provide a valid maximum order.").required(false).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor WAIT_TIMEOUT = new PropertyDescriptor.Builder().name("wait-timeout").displayName("Wait Timeout").description("Indicates the duration after which waiting FlowFiles will be routed to the 'overtook' relationship.").required(true).defaultValue("10 min").addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    public static final PropertyDescriptor INACTIVE_TIMEOUT = new PropertyDescriptor.Builder().name("inactive-timeout").displayName("Inactive Timeout").description("Indicates the duration after which state for an inactive group will be cleared from managed state. Group is determined as inactive if any new incoming FlowFile has not seen for a group for specified duration. Inactive Timeout must be longer than Wait Timeout. If a FlowFile arrives late after its group is already cleared, it will be treated as a brand new group, but will never match the order since expected preceding FlowFiles are already gone. The FlowFile will eventually timeout for waiting and routed to 'overtook'. To avoid this, group states should be kept long enough, however, shorter duration would be helpful for reusing the same group identifier again.").required(true).defaultValue("30 min").addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    public static final PropertyDescriptor BATCH_COUNT = new PropertyDescriptor.Builder().name("batch-count").displayName("Batch Count").description("The maximum number of FlowFiles that EnforceOrder can process at an execution.").required(true).defaultValue("1000").addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("A FlowFile with a matching order number will be routed to this relationship.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("A FlowFiles which does not have required attributes, or fails to compute those will be routed to this relationship").build();
    public static final Relationship REL_WAIT = new Relationship.Builder().name("wait").description("A FlowFile with non matching order will be routed to this relationship").build();
    public static final Relationship REL_OVERTOOK = new Relationship.Builder().name("overtook").description("A FlowFile that waited for preceding FlowFiles longer than Wait Timeout and overtook those FlowFiles, will be routed to this relationship.").build();
    public static final Relationship REL_SKIPPED = new Relationship.Builder().name("skipped").description("A FlowFile that has an order younger than current, which means arrived too late and skipped, will be routed to this relationship.").build();
    private final Set<Relationship> relationships;

    public EnforceOrder() {
        HashSet<Relationship> rels = new HashSet<Relationship>();
        rels.add(REL_SUCCESS);
        rels.add(REL_WAIT);
        rels.add(REL_OVERTOOK);
        rels.add(REL_FAILURE);
        rels.add(REL_SKIPPED);
        this.relationships = Collections.unmodifiableSet(rels);
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        ArrayList<PropertyDescriptor> descriptors = new ArrayList<PropertyDescriptor>();
        descriptors.add(GROUP_IDENTIFIER);
        descriptors.add(ORDER_ATTRIBUTE);
        descriptors.add(INITIAL_ORDER);
        descriptors.add(MAX_ORDER);
        descriptors.add(BATCH_COUNT);
        descriptors.add(WAIT_TIMEOUT);
        descriptors.add(INACTIVE_TIMEOUT);
        return descriptors;
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>(super.customValidate(validationContext));
        Long waitTimeoutMillis = validationContext.getProperty(WAIT_TIMEOUT).asTimePeriod(TimeUnit.MICROSECONDS);
        Long inactiveTimeoutMillis = validationContext.getProperty(INACTIVE_TIMEOUT).asTimePeriod(TimeUnit.MICROSECONDS);
        if (waitTimeoutMillis >= inactiveTimeoutMillis) {
            results.add(new ValidationResult.Builder().input(validationContext.getProperty(INACTIVE_TIMEOUT).getValue()).subject(INACTIVE_TIMEOUT.getDisplayName()).explanation(String.format("%s should be longer than %s", INACTIVE_TIMEOUT.getDisplayName(), WAIT_TIMEOUT.getDisplayName())).valid(false).build());
        }
        return results;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        StateMap stateMap;
        Integer batchCount = context.getProperty(BATCH_COUNT).asInteger();
        List flowFiles = session.get(batchCount.intValue());
        if (flowFiles == null || flowFiles.isEmpty()) {
            return;
        }
        try {
            stateMap = session.getState(Scope.LOCAL);
        }
        catch (IOException e) {
            this.getLogger().error("Failed to retrieve state from StateManager due to {}" + e, (Throwable)e);
            context.yield();
            return;
        }
        OrderingContext oc = new OrderingContext(context, session);
        oc.groupStates.putAll(stateMap.toMap());
        for (FlowFile flowFile : flowFiles) {
            oc.setFlowFile(flowFile);
            if (oc.flowFile == null) break;
            if (!oc.computeGroupId() || !oc.computeOrder() || !oc.computeInitialOrder() || !oc.computeMaxOrder()) continue;
            oc.markFlowFileValid();
        }
        oc.transferFlowFiles();
        oc.cleanupInactiveStates();
        try {
            session.setState(oc.groupStates, Scope.LOCAL);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to update state due to " + e + ". Session will be rollback and processor will be yielded for a while.", e);
        }
    }

    private class OrderingContext {
        private final ComponentLog logger;
        private final ProcessSession processSession;
        private final ProcessContext processContext;
        private final String orderAttribute;
        private final Long waitTimeoutMillis;
        private final Function<FlowFile, Integer> getOrder;
        private final Map<String, String> groupStates;
        private final long now;
        private final PropertyValue groupIdentifierProperty;
        private final PropertyValue initOrderProperty;
        private final PropertyValue maxOrderProperty;
        private final Map<String, List<FlowFile>> flowFileGroups;
        private FlowFile flowFile;
        private String groupId;
        private Integer order;

        private OrderingContext(ProcessContext processContext, ProcessSession processSession) {
            this.logger = EnforceOrder.this.getLogger();
            this.groupStates = new HashMap<String, String>();
            this.now = System.currentTimeMillis();
            this.flowFileGroups = new TreeMap<String, List<FlowFile>>();
            this.processContext = processContext;
            this.processSession = processSession;
            this.orderAttribute = processContext.getProperty(ORDER_ATTRIBUTE).getValue();
            this.waitTimeoutMillis = processContext.getProperty(WAIT_TIMEOUT).asTimePeriod(TimeUnit.MILLISECONDS);
            this.getOrder = flowFile -> Integer.parseInt(flowFile.getAttribute(this.orderAttribute));
            this.groupIdentifierProperty = processContext.getProperty(GROUP_IDENTIFIER);
            this.initOrderProperty = processContext.getProperty(INITIAL_ORDER);
            this.maxOrderProperty = processContext.getProperty(MAX_ORDER);
        }

        private void setFlowFile(FlowFile flowFile) {
            this.flowFile = flowFile;
            this.groupId = null;
            this.order = null;
        }

        private boolean computeGroupId() {
            this.groupId = this.groupIdentifierProperty.evaluateAttributeExpressions(this.flowFile).getValue();
            if (StringUtils.isBlank((CharSequence)this.groupId)) {
                this.transferToFailure(this.flowFile, "Failed to get Group Identifier.");
                return false;
            }
            return true;
        }

        private boolean computeOrder() {
            try {
                this.order = this.getOrder.apply(this.flowFile);
            }
            catch (NumberFormatException e) {
                this.transferToFailure(this.flowFile, "Failed to parse order attribute due to " + e, e);
                return false;
            }
            return true;
        }

        private boolean computeMaxOrder() {
            if (this.maxOrderProperty.isSet()) {
                Integer maxOrder;
                String maxOrderStr = this.groupStates.computeIfAbsent((String)STATE_MAX_ORDER.apply(this.groupId), k -> this.maxOrderProperty.evaluateAttributeExpressions(this.flowFile).getValue());
                if (StringUtils.isBlank((CharSequence)maxOrderStr)) {
                    this.transferToFailure(this.flowFile, String.format("%s was specified but result was empty.", MAX_ORDER.getDisplayName()));
                    return false;
                }
                try {
                    maxOrder = Integer.parseInt(maxOrderStr);
                }
                catch (NumberFormatException e) {
                    String msg = String.format("Failed to get Maximum Order for group [%s] due to %s", this.groupId, e);
                    this.transferToFailure(this.flowFile, msg, e);
                    return false;
                }
                if (this.order > maxOrder) {
                    String msg = String.format("Order (%d) is greater than the Maximum Order (%d) for Group [%s]", this.order, maxOrder, this.groupId);
                    this.transferToFailure(this.flowFile, msg);
                    return false;
                }
            }
            return true;
        }

        private boolean computeInitialOrder() {
            String stateKeyOrder = (String)STATE_TARGET_ORDER.apply(this.groupId);
            try {
                AtomicReference computedInitOrder = new AtomicReference();
                this.groupStates.computeIfAbsent(stateKeyOrder, k -> {
                    String initOrderStr = this.initOrderProperty.evaluateAttributeExpressions(this.flowFile).getValue();
                    Integer.parseInt(initOrderStr);
                    computedInitOrder.set(initOrderStr);
                    return initOrderStr;
                });
                if (!StringUtils.isBlank((CharSequence)((CharSequence)computedInitOrder.get()))) {
                    this.groupStates.put((String)STATE_UPDATED_AT.apply(this.groupId), String.valueOf(this.now));
                }
            }
            catch (NumberFormatException e) {
                String msg = String.format("Failed to get Initial Order for Group [%s] due to %s", this.groupId, e);
                this.transferToFailure(this.flowFile, msg, e);
                return false;
            }
            return true;
        }

        private void markFlowFileValid() {
            List groupedFlowFiles = this.flowFileGroups.computeIfAbsent(this.groupId, k -> new ArrayList());
            FlowFile validFlowFile = StringUtils.isBlank((CharSequence)this.flowFile.getAttribute(EnforceOrder.ATTR_STARTED_AT)) ? this.processSession.putAttribute(this.flowFile, EnforceOrder.ATTR_STARTED_AT, String.valueOf(this.now)) : this.flowFile;
            groupedFlowFiles.add(validFlowFile);
        }

        private void transferFlowFiles() {
            this.flowFileGroups.entrySet().stream().filter(entry -> !((List)entry.getValue()).isEmpty()).map(entry -> {
                List groupedFlowFiles = (List)entry.getValue();
                groupedFlowFiles.sort(Comparator.comparing(this.getOrder));
                return entry;
            }).forEach(entry -> {
                String groupId = (String)entry.getKey();
                String stateKeyOrder = (String)STATE_TARGET_ORDER.apply(groupId);
                int previousTargetOrder = Integer.parseInt(this.groupStates.get(stateKeyOrder));
                AtomicInteger targetOrder = new AtomicInteger(previousTargetOrder);
                List groupedFlowFiles = (List)entry.getValue();
                String maxOrderStr = this.groupStates.get(STATE_MAX_ORDER.apply(groupId));
                groupedFlowFiles.forEach(f -> {
                    boolean isMaxOrder;
                    Integer order = this.getOrder.apply((FlowFile)f);
                    boolean bl = isMaxOrder = !StringUtils.isBlank((CharSequence)maxOrderStr) && order.equals(Integer.parseInt(maxOrderStr));
                    if (order.intValue() == targetOrder.get()) {
                        this.transferResult((FlowFile)f, REL_SUCCESS, null, null);
                        if (!isMaxOrder) {
                            targetOrder.incrementAndGet();
                        }
                    } else if (order > targetOrder.get()) {
                        if (this.now - Long.parseLong(f.getAttribute(EnforceOrder.ATTR_STARTED_AT)) > this.waitTimeoutMillis) {
                            this.transferResult((FlowFile)f, REL_OVERTOOK, null, targetOrder.get());
                            targetOrder.set(isMaxOrder ? order : order + 1);
                        } else {
                            this.transferResult((FlowFile)f, REL_WAIT, null, targetOrder.get());
                        }
                    } else {
                        String msg = String.format("Skipped, FlowFile order was %d but current target is %d", order, targetOrder.get());
                        this.logger.warn(msg + ". {}", new Object[]{f});
                        this.transferResult((FlowFile)f, REL_SKIPPED, msg, targetOrder.get());
                    }
                });
                if (previousTargetOrder != targetOrder.get()) {
                    this.groupStates.put(stateKeyOrder, String.valueOf(targetOrder.get()));
                    this.groupStates.put((String)STATE_UPDATED_AT.apply(groupId), String.valueOf(this.now));
                }
            });
        }

        private void transferResult(FlowFile flowFile, Relationship result, String detail, Integer expectedOrder) {
            HashMap<String, String> attributes = new HashMap<String, String>();
            attributes.put(EnforceOrder.ATTR_RESULT, result.getName());
            if (expectedOrder != null) {
                attributes.put(EnforceOrder.ATTR_EXPECTED_ORDER, expectedOrder.toString());
            }
            if (!StringUtils.isBlank((CharSequence)detail)) {
                attributes.put(EnforceOrder.ATTR_DETAIL, detail);
            }
            FlowFile resultFlowFile = this.processSession.putAllAttributes(flowFile, attributes);
            if (expectedOrder == null) {
                resultFlowFile = this.processSession.removeAttribute(resultFlowFile, EnforceOrder.ATTR_EXPECTED_ORDER);
            }
            if (detail == null) {
                resultFlowFile = this.processSession.removeAttribute(resultFlowFile, EnforceOrder.ATTR_DETAIL);
            }
            this.processSession.transfer(resultFlowFile, result);
        }

        private void transferToFailure(FlowFile flowFile, String message) {
            this.transferToFailure(flowFile, message, null);
        }

        private void transferToFailure(FlowFile flowFile, String message, Throwable cause) {
            if (cause != null) {
                EnforceOrder.this.getLogger().warn(message + " {}", new Object[]{flowFile, cause});
            } else {
                EnforceOrder.this.getLogger().warn(message + " {}", new Object[]{flowFile});
            }
            this.transferResult(flowFile, REL_FAILURE, message, null);
        }

        private void cleanupInactiveStates() {
            Long inactiveTimeout = this.processContext.getProperty(INACTIVE_TIMEOUT).asTimePeriod(TimeUnit.MILLISECONDS);
            List<String> inactiveGroups = this.groupStates.keySet().stream().filter(k -> k.endsWith(EnforceOrder.STATE_SUFFIX_UPDATED_AT) && this.now - Long.parseLong(this.groupStates.get(k)) > inactiveTimeout).map(k -> k.substring(0, k.length() - EnforceOrder.STATE_SUFFIX_UPDATED_AT.length())).collect(Collectors.toList());
            inactiveGroups.forEach(groupId -> {
                this.groupStates.remove(STATE_TARGET_ORDER.apply(groupId));
                this.groupStates.remove(STATE_UPDATED_AT.apply(groupId));
                this.groupStates.remove(STATE_MAX_ORDER.apply(groupId));
            });
        }
    }
}

