/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.io.IOUtils;
import org.apache.nifi.annotation.behavior.EventDriven;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.dbcp.DBCPService;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.io.InputStreamCallback;
import org.apache.nifi.processor.io.OutputStreamCallback;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.standard.util.JdbcCommon;
import org.apache.nifi.util.StopWatch;

@EventDriven
@InputRequirement(value=InputRequirement.Requirement.INPUT_ALLOWED)
@Tags(value={"sql", "select", "jdbc", "query", "database"})
@CapabilityDescription(value="Execute provided SQL select query. Query result will be converted to Avro format. Streaming is used so arbitrarily large result sets are supported. This processor can be scheduled to run on a timer, or cron expression, using the standard scheduling methods, or it can be triggered by an incoming FlowFile. If it is triggered by an incoming FlowFile, then attributes of that FlowFile will be available when evaluating the select query. FlowFile attribute 'executesql.row.count' indicates how many rows were selected.")
@WritesAttributes(value={@WritesAttribute(attribute="executesql.row.count", description="Contains the number of rows returned in the select query"), @WritesAttribute(attribute="executesql.query.duration", description="Duration of the query in milliseconds")})
public class ExecuteSQL
extends AbstractProcessor {
    public static final String RESULT_ROW_COUNT = "executesql.row.count";
    public static final String RESULT_QUERY_DURATION = "executesql.query.duration";
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Successfully created FlowFile from SQL query result set.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("SQL query execution failed. Incoming FlowFile will be penalized and routed to this relationship").build();
    private final Set<Relationship> relationships;
    public static final PropertyDescriptor DBCP_SERVICE = new PropertyDescriptor.Builder().name("Database Connection Pooling Service").description("The Controller Service that is used to obtain connection to database").required(true).identifiesControllerService(DBCPService.class).build();
    public static final PropertyDescriptor SQL_SELECT_QUERY = new PropertyDescriptor.Builder().name("SQL select query").description("The SQL select query to execute. The query can be empty, a constant value, or built from attributes using Expression Language. If this property is specified, it will be used regardless of the content of incoming flowfiles. If this property is empty, the content of the incoming flow file is expected to contain a valid SQL select query, to be issued by the processor to the database. Note that Expression Language is not evaluated for flow file contents.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(true).build();
    public static final PropertyDescriptor QUERY_TIMEOUT = new PropertyDescriptor.Builder().name("Max Wait Time").description("The maximum amount of time allowed for a running SQL select query  , zero means there is no limit. Max time less than 1 second will be equal to zero.").defaultValue("0 seconds").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).sensitive(false).build();
    private final List<PropertyDescriptor> propDescriptors;

    public ExecuteSQL() {
        HashSet<Relationship> r = new HashSet<Relationship>();
        r.add(REL_SUCCESS);
        r.add(REL_FAILURE);
        this.relationships = Collections.unmodifiableSet(r);
        ArrayList<PropertyDescriptor> pds = new ArrayList<PropertyDescriptor>();
        pds.add(DBCP_SERVICE);
        pds.add(SQL_SELECT_QUERY);
        pds.add(QUERY_TIMEOUT);
        pds.add(JdbcCommon.NORMALIZE_NAMES_FOR_AVRO);
        pds.add(JdbcCommon.USE_AVRO_LOGICAL_TYPES);
        pds.add(JdbcCommon.DEFAULT_PRECISION);
        pds.add(JdbcCommon.DEFAULT_SCALE);
        this.propDescriptors = Collections.unmodifiableList(pds);
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.propDescriptors;
    }

    @OnScheduled
    public void setup(ProcessContext context) {
        if (!context.getProperty(SQL_SELECT_QUERY).isSet() && !context.hasIncomingConnection()) {
            String errorString = "Either the Select Query must be specified or there must be an incoming connection providing flowfile(s) containing a SQL select query";
            this.getLogger().error("Either the Select Query must be specified or there must be an incoming connection providing flowfile(s) containing a SQL select query");
            throw new ProcessException("Either the Select Query must be specified or there must be an incoming connection providing flowfile(s) containing a SQL select query");
        }
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        String selectQuery;
        FlowFile fileToProcess = null;
        if (context.hasIncomingConnection() && (fileToProcess = session.get()) == null && context.hasNonLoopConnection()) {
            return;
        }
        ComponentLog logger = this.getLogger();
        DBCPService dbcpService = (DBCPService)context.getProperty(DBCP_SERVICE).asControllerService(DBCPService.class);
        Integer queryTimeout = context.getProperty(QUERY_TIMEOUT).asTimePeriod(TimeUnit.SECONDS).intValue();
        final boolean convertNamesForAvro = context.getProperty(JdbcCommon.NORMALIZE_NAMES_FOR_AVRO).asBoolean();
        final Boolean useAvroLogicalTypes = context.getProperty(JdbcCommon.USE_AVRO_LOGICAL_TYPES).asBoolean();
        final Integer defaultPrecision = context.getProperty(JdbcCommon.DEFAULT_PRECISION).evaluateAttributeExpressions().asInteger();
        final Integer defaultScale = context.getProperty(JdbcCommon.DEFAULT_SCALE).evaluateAttributeExpressions().asInteger();
        StopWatch stopWatch = new StopWatch(true);
        if (context.getProperty(SQL_SELECT_QUERY).isSet()) {
            selectQuery = context.getProperty(SQL_SELECT_QUERY).evaluateAttributeExpressions(fileToProcess).getValue();
        } else {
            final StringBuilder queryContents = new StringBuilder();
            session.read(fileToProcess, new InputStreamCallback(){

                public void process(InputStream in) throws IOException {
                    queryContents.append(IOUtils.toString((InputStream)in));
                }
            });
            selectQuery = queryContents.toString();
        }
        int resultCount = 0;
        try (Connection con = dbcpService.getConnection();
             final Statement st = con.createStatement();){
            st.setQueryTimeout(queryTimeout);
            logger.debug("Executing query {}", new Object[]{selectQuery});
            boolean results = st.execute(selectQuery);
            while (results) {
                FlowFile resultSetFF;
                if (fileToProcess == null) {
                    resultSetFF = session.create();
                } else {
                    resultSetFF = session.create(fileToProcess);
                    resultSetFF = session.putAllAttributes(resultSetFF, fileToProcess.getAttributes());
                }
                final AtomicLong nrOfRows = new AtomicLong(0L);
                resultSetFF = session.write(resultSetFF, new OutputStreamCallback(){

                    public void process(OutputStream out) throws IOException {
                        try {
                            ResultSet resultSet = st.getResultSet();
                            JdbcCommon.AvroConversionOptions options = JdbcCommon.AvroConversionOptions.builder().convertNames(convertNamesForAvro).useLogicalTypes(useAvroLogicalTypes).defaultPrecision(defaultPrecision).defaultScale(defaultScale).build();
                            nrOfRows.set(JdbcCommon.convertToAvroStream(resultSet, out, options, null));
                        }
                        catch (SQLException e) {
                            throw new ProcessException((Throwable)e);
                        }
                    }
                });
                long duration = stopWatch.getElapsed(TimeUnit.MILLISECONDS);
                resultSetFF = session.putAttribute(resultSetFF, RESULT_ROW_COUNT, String.valueOf(nrOfRows.get()));
                resultSetFF = session.putAttribute(resultSetFF, RESULT_QUERY_DURATION, String.valueOf(duration));
                resultSetFF = session.putAttribute(resultSetFF, CoreAttributes.MIME_TYPE.key(), "application/avro-binary");
                logger.info("{} contains {} Avro records; transferring to 'success'", new Object[]{resultSetFF, nrOfRows.get()});
                session.getProvenanceReporter().modifyContent(resultSetFF, "Retrieved " + nrOfRows.get() + " rows", duration);
                session.transfer(resultSetFF, REL_SUCCESS);
                ++resultCount;
                try {
                    results = st.getMoreResults();
                }
                catch (SQLException ex) {
                    results = false;
                }
            }
            if (fileToProcess != null) {
                if (resultCount > 0) {
                    session.remove(fileToProcess);
                } else {
                    fileToProcess = session.write(fileToProcess, new OutputStreamCallback(){

                        public void process(OutputStream out) throws IOException {
                            JdbcCommon.createEmptyAvroStream(out);
                        }
                    });
                    session.transfer(fileToProcess, REL_SUCCESS);
                }
            }
        }
        catch (SQLException | ProcessException e) {
            if (fileToProcess == null) {
                logger.error("Unable to execute SQL select query {} due to {}. No FlowFile to route to failure", new Object[]{selectQuery, e});
                context.yield();
            }
            if (context.hasIncomingConnection()) {
                logger.error("Unable to execute SQL select query {} for {} due to {}; routing to failure", new Object[]{selectQuery, fileToProcess, e});
                fileToProcess = session.penalize(fileToProcess);
            } else {
                logger.error("Unable to execute SQL select query {} due to {}; routing to failure", new Object[]{selectQuery, e});
                context.yield();
            }
            session.transfer(fileToProcess, REL_FAILURE);
        }
    }
}

