/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.nio.charset.StandardCharsets;
import java.security.Provider;
import java.security.Security;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.annotation.behavior.EventDriven;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.SystemResource;
import org.apache.nifi.annotation.behavior.SystemResourceConsideration;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.io.StreamCallback;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.security.util.EncryptionMethod;
import org.apache.nifi.security.util.KeyDerivationFunction;
import org.apache.nifi.security.util.crypto.CipherUtility;
import org.apache.nifi.security.util.crypto.KeyedEncryptor;
import org.apache.nifi.security.util.crypto.OpenPGPKeyBasedEncryptor;
import org.apache.nifi.security.util.crypto.OpenPGPPasswordBasedEncryptor;
import org.apache.nifi.security.util.crypto.PasswordBasedEncryptor;
import org.apache.nifi.util.StopWatch;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

@EventDriven
@SideEffectFree
@SupportsBatching
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"encryption", "decryption", "password", "JCE", "OpenPGP", "PGP", "GPG"})
@CapabilityDescription(value="Encrypts or Decrypts a FlowFile using either symmetric encryption with a password and randomly generated salt, or asymmetric encryption using a public and secret key.")
@SystemResourceConsideration(resource=SystemResource.CPU)
public class EncryptContent
extends AbstractProcessor {
    public static final String ENCRYPT_MODE = "Encrypt";
    public static final String DECRYPT_MODE = "Decrypt";
    private static final String WEAK_CRYPTO_ALLOWED_NAME = "allowed";
    private static final String WEAK_CRYPTO_NOT_ALLOWED_NAME = "not-allowed";
    public static final PropertyDescriptor MODE = new PropertyDescriptor.Builder().name("Mode").description("Specifies whether the content should be encrypted or decrypted").required(true).allowableValues(new String[]{"Encrypt", "Decrypt"}).defaultValue("Encrypt").build();
    public static final PropertyDescriptor KEY_DERIVATION_FUNCTION = new PropertyDescriptor.Builder().name("key-derivation-function").displayName("Key Derivation Function").description("Specifies the key derivation function to generate the key from the password (and salt)").required(true).allowableValues(EncryptContent.buildKeyDerivationFunctionAllowableValues()).defaultValue(KeyDerivationFunction.BCRYPT.name()).build();
    public static final PropertyDescriptor ENCRYPTION_ALGORITHM = new PropertyDescriptor.Builder().name("Encryption Algorithm").description("The Encryption Algorithm to use").required(true).allowableValues(EncryptContent.buildEncryptionMethodAllowableValues()).defaultValue(EncryptionMethod.MD5_128AES.name()).build();
    public static final PropertyDescriptor PASSWORD = new PropertyDescriptor.Builder().name("Password").description("The Password to use for encrypting or decrypting the data").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).sensitive(true).build();
    public static final PropertyDescriptor PUBLIC_KEYRING = new PropertyDescriptor.Builder().name("public-keyring-file").displayName("Public Keyring File").description("In a PGP encrypt mode, this keyring contains the public key of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PUBLIC_KEY_USERID = new PropertyDescriptor.Builder().name("public-key-user-id").displayName("Public Key User Id").description("In a PGP encrypt mode, this user id of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PRIVATE_KEYRING = new PropertyDescriptor.Builder().name("private-keyring-file").displayName("Private Keyring File").description("In a PGP decrypt mode, this keyring contains the private key of the recipient").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor PRIVATE_KEYRING_PASSPHRASE = new PropertyDescriptor.Builder().name("private-keyring-passphrase").displayName("Private Keyring Passphrase").description("In a PGP decrypt mode, this is the private keyring passphrase").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).sensitive(true).build();
    public static final PropertyDescriptor RAW_KEY_HEX = new PropertyDescriptor.Builder().name("raw-key-hex").displayName("Raw Key (hexadecimal)").description("In keyed encryption, this is the raw key, encoded in hexadecimal").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).sensitive(true).build();
    public static final PropertyDescriptor ALLOW_WEAK_CRYPTO = new PropertyDescriptor.Builder().name("allow-weak-crypto").displayName("Allow insecure cryptographic modes").description("Overrides the default behavior to prevent unsafe combinations of encryption algorithms and short passwords on JVMs with limited strength cryptographic jurisdiction policies").required(true).allowableValues(EncryptContent.buildWeakCryptoAllowableValues()).defaultValue(EncryptContent.buildDefaultWeakCryptoAllowableValue().getValue()).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Any FlowFile that is successfully encrypted or decrypted will be routed to success").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Any FlowFile that cannot be encrypted or decrypted will be routed to failure").build();
    private List<PropertyDescriptor> properties;
    private Set<Relationship> relationships;

    private static AllowableValue[] buildKeyDerivationFunctionAllowableValues() {
        KeyDerivationFunction[] keyDerivationFunctions = KeyDerivationFunction.values();
        ArrayList<AllowableValue> allowableValues = new ArrayList<AllowableValue>(keyDerivationFunctions.length);
        for (KeyDerivationFunction kdf : keyDerivationFunctions) {
            allowableValues.add(new AllowableValue(kdf.name(), kdf.getName(), kdf.getDescription()));
        }
        return allowableValues.toArray(new AllowableValue[0]);
    }

    private static AllowableValue[] buildEncryptionMethodAllowableValues() {
        EncryptionMethod[] encryptionMethods = EncryptionMethod.values();
        ArrayList<AllowableValue> allowableValues = new ArrayList<AllowableValue>(encryptionMethods.length);
        for (EncryptionMethod em : encryptionMethods) {
            allowableValues.add(new AllowableValue(em.name(), em.name(), em.toString()));
        }
        return allowableValues.toArray(new AllowableValue[0]);
    }

    private static AllowableValue[] buildWeakCryptoAllowableValues() {
        ArrayList<AllowableValue> allowableValues = new ArrayList<AllowableValue>();
        allowableValues.add(new AllowableValue(WEAK_CRYPTO_ALLOWED_NAME, "Allowed", "Operation will not be blocked and no alerts will be presented when unsafe combinations of encryption algorithms and passwords are provided"));
        allowableValues.add(EncryptContent.buildDefaultWeakCryptoAllowableValue());
        return allowableValues.toArray(new AllowableValue[0]);
    }

    private static AllowableValue buildDefaultWeakCryptoAllowableValue() {
        return new AllowableValue(WEAK_CRYPTO_NOT_ALLOWED_NAME, "Not Allowed", "When set, operation will be blocked and alerts will be presented to the user if unsafe combinations of encryption algorithms and passwords are provided on a JVM with limited strength crypto. To fix this, see the Admin Guide.");
    }

    protected void init(ProcessorInitializationContext context) {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(MODE);
        properties.add(KEY_DERIVATION_FUNCTION);
        properties.add(ENCRYPTION_ALGORITHM);
        properties.add(ALLOW_WEAK_CRYPTO);
        properties.add(PASSWORD);
        properties.add(RAW_KEY_HEX);
        properties.add(PUBLIC_KEYRING);
        properties.add(PUBLIC_KEY_USERID);
        properties.add(PRIVATE_KEYRING);
        properties.add(PRIVATE_KEYRING_PASSPHRASE);
        this.properties = Collections.unmodifiableList(properties);
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(REL_SUCCESS);
        relationships.add(REL_FAILURE);
        this.relationships = Collections.unmodifiableSet(relationships);
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.properties;
    }

    public static boolean isPGPAlgorithm(String algorithm) {
        return algorithm.startsWith("PGP");
    }

    public static boolean isPGPArmoredAlgorithm(String algorithm) {
        return EncryptContent.isPGPAlgorithm(algorithm) && algorithm.endsWith("ASCII-ARMOR");
    }

    protected Collection<ValidationResult> customValidate(ValidationContext context) {
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>(super.customValidate(context));
        String methodValue = context.getProperty(ENCRYPTION_ALGORITHM).getValue();
        EncryptionMethod encryptionMethod = EncryptionMethod.valueOf((String)methodValue);
        String algorithm = encryptionMethod.getAlgorithm();
        String password = context.getProperty(PASSWORD).getValue();
        KeyDerivationFunction kdf = KeyDerivationFunction.valueOf((String)context.getProperty(KEY_DERIVATION_FUNCTION).getValue());
        String keyHex = context.getProperty(RAW_KEY_HEX).getValue();
        if (EncryptContent.isPGPAlgorithm(algorithm)) {
            boolean encrypt = context.getProperty(MODE).getValue().equalsIgnoreCase(ENCRYPT_MODE);
            String publicKeyring = context.getProperty(PUBLIC_KEYRING).getValue();
            String publicUserId = context.getProperty(PUBLIC_KEY_USERID).getValue();
            String privateKeyring = context.getProperty(PRIVATE_KEYRING).getValue();
            String privateKeyringPassphrase = context.getProperty(PRIVATE_KEYRING_PASSPHRASE).evaluateAttributeExpressions().getValue();
            validationResults.addAll(this.validatePGP(encryptionMethod, password, encrypt, publicKeyring, publicUserId, privateKeyring, privateKeyringPassphrase));
        } else if (encryptionMethod.isKeyedCipher()) {
            validationResults.addAll(this.validateKeyed(encryptionMethod, kdf, keyHex));
        } else {
            boolean allowWeakCrypto = context.getProperty(ALLOW_WEAK_CRYPTO).getValue().equalsIgnoreCase(WEAK_CRYPTO_ALLOWED_NAME);
            validationResults.addAll(this.validatePBE(encryptionMethod, kdf, password, allowWeakCrypto));
        }
        return validationResults;
    }

    private List<ValidationResult> validatePGP(EncryptionMethod encryptionMethod, String password, boolean encrypt, String publicKeyring, String publicUserId, String privateKeyring, String privateKeyringPassphrase) {
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>();
        if (password == null) {
            if (encrypt) {
                if (publicKeyring == null || publicUserId == null) {
                    validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation(encryptionMethod.getAlgorithm() + " encryption without a " + PASSWORD.getDisplayName() + " requires both " + PUBLIC_KEYRING.getDisplayName() + " and " + PUBLIC_KEY_USERID.getDisplayName()).build());
                } else {
                    try {
                        if (OpenPGPKeyBasedEncryptor.getPublicKey(publicUserId, publicKeyring) == null) {
                            validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation(PUBLIC_KEYRING.getDisplayName() + " " + publicKeyring + " does not contain user id " + publicUserId).build());
                        }
                    }
                    catch (Exception e) {
                        validationResults.add(new ValidationResult.Builder().subject(PUBLIC_KEYRING.getDisplayName()).explanation("Invalid " + PUBLIC_KEYRING.getDisplayName() + " " + publicKeyring + " because " + e.toString()).build());
                    }
                }
            } else if (privateKeyring == null || privateKeyringPassphrase == null) {
                validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getName()).explanation(encryptionMethod.getAlgorithm() + " decryption without a " + PASSWORD.getDisplayName() + " requires both " + PRIVATE_KEYRING.getDisplayName() + " and " + PRIVATE_KEYRING_PASSPHRASE.getDisplayName()).build());
            } else {
                String providerName = encryptionMethod.getProvider();
                try {
                    if (!OpenPGPKeyBasedEncryptor.validateKeyring(providerName, privateKeyring, privateKeyringPassphrase.toCharArray())) {
                        validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getDisplayName()).explanation(PRIVATE_KEYRING.getDisplayName() + " " + privateKeyring + " could not be opened with the provided " + PRIVATE_KEYRING_PASSPHRASE.getDisplayName()).build());
                    }
                }
                catch (Exception e) {
                    validationResults.add(new ValidationResult.Builder().subject(PRIVATE_KEYRING.getDisplayName()).explanation("Invalid " + PRIVATE_KEYRING.getDisplayName() + " " + privateKeyring + " because " + e.toString()).build());
                }
            }
        }
        return validationResults;
    }

    private List<ValidationResult> validatePBE(EncryptionMethod encryptionMethod, KeyDerivationFunction kdf, String password, boolean allowWeakCrypto) {
        int minimumSafePasswordLength;
        boolean limitedStrengthCrypto;
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>();
        boolean bl = limitedStrengthCrypto = !PasswordBasedEncryptor.supportsUnlimitedStrength();
        if (StringUtils.isEmpty((CharSequence)password)) {
            validationResults.add(new ValidationResult.Builder().subject(PASSWORD.getName()).explanation(PASSWORD.getDisplayName() + " is required when using algorithm " + encryptionMethod.getAlgorithm()).build());
            return validationResults;
        }
        int passwordBytesLength = password.getBytes(StandardCharsets.UTF_8).length;
        if (!allowWeakCrypto && passwordBytesLength < (minimumSafePasswordLength = PasswordBasedEncryptor.getMinimumSafePasswordLength())) {
            validationResults.add(new ValidationResult.Builder().subject(PASSWORD.getName()).explanation("Password length less than " + minimumSafePasswordLength + " characters is potentially unsafe. See Admin Guide.").build());
        }
        if (limitedStrengthCrypto) {
            boolean passwordLongerThanLimit;
            if (encryptionMethod.isUnlimitedStrength()) {
                validationResults.add(new ValidationResult.Builder().subject(ENCRYPTION_ALGORITHM.getName()).explanation(encryptionMethod.name() + " (" + encryptionMethod.getAlgorithm() + ") is not supported by this JVM due to lacking JCE Unlimited Strength Jurisdiction Policy files. See Admin Guide.").build());
            }
            boolean bl2 = passwordLongerThanLimit = !CipherUtility.passwordLengthIsValidForAlgorithmOnLimitedStrengthCrypto((int)passwordBytesLength, (EncryptionMethod)encryptionMethod);
            if (passwordLongerThanLimit) {
                int maxPasswordLength = CipherUtility.getMaximumPasswordLengthForAlgorithmOnLimitedStrengthCrypto((EncryptionMethod)encryptionMethod);
                validationResults.add(new ValidationResult.Builder().subject(PASSWORD.getName()).explanation("Password length greater than " + maxPasswordLength + " characters is not supported by this JVM due to lacking JCE Unlimited Strength Jurisdiction Policy files. See Admin Guide.").build());
            }
        }
        List<String> kdfsForPBECipher = this.getKDFsForPBECipher(encryptionMethod);
        if (kdf == null || !kdfsForPBECipher.contains(kdf.name())) {
            String displayName = KEY_DERIVATION_FUNCTION.getDisplayName();
            validationResults.add(new ValidationResult.Builder().subject(displayName).explanation(displayName + " is required to be " + StringUtils.join(kdfsForPBECipher, (String)", ") + " when using algorithm " + encryptionMethod.getAlgorithm() + ". See Admin Guide.").build());
        }
        return validationResults;
    }

    private List<ValidationResult> validateKeyed(EncryptionMethod encryptionMethod, KeyDerivationFunction kdf, String keyHex) {
        boolean limitedStrengthCrypto;
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>();
        boolean bl = limitedStrengthCrypto = !PasswordBasedEncryptor.supportsUnlimitedStrength();
        if (limitedStrengthCrypto && encryptionMethod.isUnlimitedStrength()) {
            validationResults.add(new ValidationResult.Builder().subject(ENCRYPTION_ALGORITHM.getName()).explanation(encryptionMethod.name() + " (" + encryptionMethod.getAlgorithm() + ") is not supported by this JVM due to lacking JCE Unlimited Strength Jurisdiction Policy files. See Admin Guide.").build());
        }
        int allowedKeyLength = PasswordBasedEncryptor.getMaxAllowedKeyLength(ENCRYPTION_ALGORITHM.getName());
        if (StringUtils.isEmpty((CharSequence)keyHex)) {
            validationResults.add(new ValidationResult.Builder().subject(RAW_KEY_HEX.getName()).explanation(RAW_KEY_HEX.getDisplayName() + " is required when using algorithm " + encryptionMethod.getAlgorithm() + ". See Admin Guide.").build());
        } else {
            byte[] keyBytes = new byte[]{};
            try {
                keyBytes = Hex.decodeHex((char[])keyHex.toCharArray());
            }
            catch (DecoderException e) {
                validationResults.add(new ValidationResult.Builder().subject(RAW_KEY_HEX.getName()).explanation("Key must be valid hexadecimal string. See Admin Guide.").build());
            }
            if (keyBytes.length * 8 > allowedKeyLength) {
                validationResults.add(new ValidationResult.Builder().subject(RAW_KEY_HEX.getName()).explanation("Key length greater than " + allowedKeyLength + " bits is not supported by this JVM due to lacking JCE Unlimited Strength Jurisdiction Policy files. See Admin Guide.").build());
            }
            if (!CipherUtility.isValidKeyLengthForAlgorithm((int)(keyBytes.length * 8), (String)encryptionMethod.getAlgorithm())) {
                List validKeyLengths = CipherUtility.getValidKeyLengthsForAlgorithm((String)encryptionMethod.getAlgorithm());
                validationResults.add(new ValidationResult.Builder().subject(RAW_KEY_HEX.getName()).explanation("Key must be valid length [" + StringUtils.join((Iterable)validKeyLengths, (String)", ") + "]. See Admin Guide.").build());
            }
        }
        List<String> kdfsForKeyedCipher = this.getKDFsForKeyedCipher();
        if (kdf == null || !kdfsForKeyedCipher.contains(kdf.name())) {
            validationResults.add(new ValidationResult.Builder().subject(KEY_DERIVATION_FUNCTION.getName()).explanation(KEY_DERIVATION_FUNCTION.getDisplayName() + " is required to be " + StringUtils.join(kdfsForKeyedCipher, (String)", ") + " when using algorithm " + encryptionMethod.getAlgorithm()).build());
        }
        return validationResults;
    }

    private List<String> getKDFsForKeyedCipher() {
        ArrayList<String> kdfsForKeyedCipher = new ArrayList<String>();
        kdfsForKeyedCipher.add(KeyDerivationFunction.NONE.name());
        for (KeyDerivationFunction k : KeyDerivationFunction.values()) {
            if (!k.isStrongKDF()) continue;
            kdfsForKeyedCipher.add(k.name());
        }
        return kdfsForKeyedCipher;
    }

    private List<String> getKDFsForPBECipher(EncryptionMethod encryptionMethod) {
        ArrayList<String> kdfsForPBECipher = new ArrayList<String>();
        for (KeyDerivationFunction k : KeyDerivationFunction.values()) {
            if (!k.isStrongKDF() && !k.equals((Object)KeyDerivationFunction.NONE)) {
                kdfsForPBECipher.add(k.name());
                continue;
            }
            if (!encryptionMethod.isCompatibleWithStrongKDFs() || !k.isStrongKDF()) continue;
            kdfsForPBECipher.add(k.name());
        }
        return kdfsForPBECipher;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        StreamCallback callback;
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        ComponentLog logger = this.getLogger();
        String method = context.getProperty(ENCRYPTION_ALGORITHM).getValue();
        EncryptionMethod encryptionMethod = EncryptionMethod.valueOf((String)method);
        String providerName = encryptionMethod.getProvider();
        String algorithm = encryptionMethod.getAlgorithm();
        String password = context.getProperty(PASSWORD).getValue();
        KeyDerivationFunction kdf = KeyDerivationFunction.valueOf((String)context.getProperty(KEY_DERIVATION_FUNCTION).getValue());
        boolean encrypt = context.getProperty(MODE).getValue().equalsIgnoreCase(ENCRYPT_MODE);
        try {
            Encryptor encryptor;
            if (EncryptContent.isPGPAlgorithm(algorithm)) {
                String filename = flowFile.getAttribute(CoreAttributes.FILENAME.key());
                String publicKeyring = context.getProperty(PUBLIC_KEYRING).getValue();
                String privateKeyring = context.getProperty(PRIVATE_KEYRING).getValue();
                if (encrypt && publicKeyring != null) {
                    String publicUserId = context.getProperty(PUBLIC_KEY_USERID).getValue();
                    encryptor = new OpenPGPKeyBasedEncryptor(algorithm, providerName, publicKeyring, publicUserId, null, filename);
                } else if (!encrypt && privateKeyring != null) {
                    char[] keyringPassphrase = context.getProperty(PRIVATE_KEYRING_PASSPHRASE).evaluateAttributeExpressions().getValue().toCharArray();
                    encryptor = new OpenPGPKeyBasedEncryptor(algorithm, providerName, privateKeyring, null, keyringPassphrase, filename);
                } else {
                    char[] passphrase = Normalizer.normalize(password, Normalizer.Form.NFC).toCharArray();
                    encryptor = new OpenPGPPasswordBasedEncryptor(algorithm, providerName, passphrase, filename);
                }
            } else if (kdf.equals((Object)KeyDerivationFunction.NONE)) {
                String keyHex = context.getProperty(RAW_KEY_HEX).getValue();
                encryptor = new KeyedEncryptor(encryptionMethod, Hex.decodeHex((char[])keyHex.toCharArray()));
            } else {
                char[] passphrase = Normalizer.normalize(password, Normalizer.Form.NFC).toCharArray();
                encryptor = new PasswordBasedEncryptor(encryptionMethod, passphrase, kdf);
            }
            callback = encrypt ? encryptor.getEncryptionCallback() : encryptor.getDecryptionCallback();
        }
        catch (Exception e) {
            logger.error("Failed to initialize {}cryption algorithm because - ", new Object[]{encrypt ? "en" : "de", e});
            session.rollback();
            context.yield();
            return;
        }
        try {
            StopWatch stopWatch = new StopWatch(true);
            flowFile = session.write(flowFile, callback);
            logger.info("successfully {}crypted {}", new Object[]{encrypt ? "en" : "de", flowFile});
            session.getProvenanceReporter().modifyContent(flowFile, stopWatch.getElapsed(TimeUnit.MILLISECONDS));
            session.transfer(flowFile, REL_SUCCESS);
        }
        catch (ProcessException e) {
            logger.error("Cannot {}crypt {} - ", new Object[]{encrypt ? "en" : "de", flowFile, e});
            session.transfer(flowFile, REL_FAILURE);
        }
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }

    public static interface Encryptor {
        public StreamCallback getEncryptionCallback() throws Exception;

        public StreamCallback getDecryptionCallback() throws Exception;
    }
}

