/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.nifi.annotation.behavior.DefaultRunDuration;
import org.apache.nifi.annotation.behavior.DynamicProperty;
import org.apache.nifi.annotation.behavior.DynamicRelationship;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.behavior.TriggerWhenAnyDestinationAvailable;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.util.StandardValidators;

@SideEffectFree
@SupportsBatching(defaultDuration=DefaultRunDuration.TWENTY_FIVE_MILLIS)
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@TriggerWhenAnyDestinationAvailable
@Tags(value={"distribute", "load balance", "route", "round robin", "weighted"})
@CapabilityDescription(value="Distributes FlowFiles to downstream processors based on a Distribution Strategy. If using the Round Robin strategy, the default is to assign each destination a weighting of 1 (evenly distributed). However, optional properties can be added to the change this; adding a property with the name '5' and value '10' means that the relationship with name '5' will be receive 10 FlowFiles in each iteration instead of 1.")
@DynamicProperty(name="The relationship name (positive number)", value="The relationship Weight (positive number)", description="Adding a property with the name '5' and value '10' means that the relationship with name '5' will receive 10 FlowFiles in each iteration instead of 1.")
@DynamicRelationship(name="A number 1..<Number Of Relationships>", description="FlowFiles are sent to this relationship per the <Distribution Strategy>")
@WritesAttributes(value={@WritesAttribute(attribute="distribute.load.relationship", description="The name of the specific relationship the FlowFile has been routed through")})
public class DistributeLoad
extends AbstractProcessor {
    public static final String ROUND_ROBIN = "round robin";
    public static final String NEXT_AVAILABLE = "next available";
    public static final String OVERFLOW = "overflow";
    public static final AllowableValue STRATEGY_ROUND_ROBIN = new AllowableValue("round robin", "round robin", "Relationship selection is evenly distributed in a round robin fashion; all relationships must be available.");
    public static final AllowableValue STRATEGY_NEXT_AVAILABLE = new AllowableValue("next available", "next available", "Relationship selection is distributed across all available relationships in order of their weight; at least one relationship must be available.");
    public static final AllowableValue STRATEGY_OVERFLOW = new AllowableValue("overflow", "overflow", "Relationship selection is the first available relationship without further distribution among all relationships; at least one relationship must be available.");
    public static final PropertyDescriptor NUM_RELATIONSHIPS = new PropertyDescriptor.Builder().name("Number of Relationships").description("Determines the number of Relationships to which the load should be distributed").required(true).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).defaultValue("1").build();
    public static final PropertyDescriptor DISTRIBUTION_STRATEGY = new PropertyDescriptor.Builder().name("Distribution Strategy").description("Determines how the load will be distributed. Relationship weight is in numeric order where '1' has the greatest weight.").required(true).allowableValues(new AllowableValue[]{STRATEGY_ROUND_ROBIN, STRATEGY_NEXT_AVAILABLE, STRATEGY_OVERFLOW}).defaultValue("round robin").build();
    public static final String RELATIONSHIP_ATTRIBUTE = "distribute.load.relationship";
    private List<PropertyDescriptor> properties;
    private final AtomicReference<Set<Relationship>> relationshipsRef = new AtomicReference();
    private final AtomicReference<DistributionStrategy> strategyRef = new AtomicReference<RoundRobinStrategy>(new RoundRobinStrategy());
    private final AtomicReference<List<Relationship>> weightedRelationshipListRef = new AtomicReference();
    private final AtomicBoolean doSetProps = new AtomicBoolean(true);

    protected void init(ProcessorInitializationContext context) {
        HashSet<Relationship> relationships = new HashSet<Relationship>();
        relationships.add(DistributeLoad.createRelationship(1));
        this.relationshipsRef.set(Collections.unmodifiableSet(relationships));
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(NUM_RELATIONSHIPS);
        properties.add(DISTRIBUTION_STRATEGY);
        this.properties = Collections.unmodifiableList(properties);
    }

    private static Relationship createRelationship(int num) {
        return new Relationship.Builder().name(String.valueOf(num)).description("Where to route flowfiles for this relationship index").build();
    }

    public Set<Relationship> getRelationships() {
        return this.relationshipsRef.get();
    }

    public void onPropertyModified(PropertyDescriptor descriptor, String oldValue, String newValue) {
        if (descriptor.equals((Object)NUM_RELATIONSHIPS)) {
            HashSet<Relationship> relationships = new HashSet<Relationship>();
            for (int i = 1; i <= Integer.parseInt(newValue); ++i) {
                relationships.add(DistributeLoad.createRelationship(i));
            }
            this.relationshipsRef.set(Collections.unmodifiableSet(relationships));
        } else if (descriptor.equals((Object)DISTRIBUTION_STRATEGY)) {
            switch (newValue.toLowerCase()) {
                case "round robin": {
                    this.strategyRef.set(new RoundRobinStrategy());
                    break;
                }
                case "next available": {
                    this.strategyRef.set(new NextAvailableStrategy());
                    break;
                }
                case "overflow": {
                    this.strategyRef.set(new OverflowStrategy());
                    break;
                }
                default: {
                    throw new IllegalStateException("Invalid distribution strategy");
                }
            }
            this.doSetProps.set(true);
        }
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        if (this.doSetProps.getAndSet(false)) {
            ArrayList<PropertyDescriptor> props = new ArrayList<PropertyDescriptor>();
            props.add(NUM_RELATIONSHIPS);
            props.add(DISTRIBUTION_STRATEGY);
            this.properties = Collections.unmodifiableList(props);
        }
        return this.properties;
    }

    protected PropertyDescriptor getSupportedDynamicPropertyDescriptor(String propertyDescriptorName) {
        int numRelationships = this.relationshipsRef.get().size();
        try {
            int value = Integer.parseInt(propertyDescriptorName);
            if (value <= 0 || value > numRelationships) {
                return new PropertyDescriptor.Builder().addValidator((Validator)new InvalidPropertyNameValidator(propertyDescriptorName)).name(propertyDescriptorName).build();
            }
        }
        catch (NumberFormatException e) {
            return new PropertyDescriptor.Builder().addValidator((Validator)new InvalidPropertyNameValidator(propertyDescriptorName)).name(propertyDescriptorName).build();
        }
        return new PropertyDescriptor.Builder().addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).name(propertyDescriptorName).dynamic(true).build();
    }

    @OnScheduled
    public void createWeightedList(ProcessContext context) {
        LinkedHashMap<Integer, Integer> weightings = new LinkedHashMap<Integer, Integer>();
        int numRelationships = context.getProperty(NUM_RELATIONSHIPS).asInteger();
        for (int i = 1; i <= numRelationships; ++i) {
            weightings.put(i, 1);
        }
        for (PropertyDescriptor propDesc : context.getProperties().keySet()) {
            if (this.properties.contains(propDesc)) continue;
            int relationship = Integer.parseInt(propDesc.getName());
            int weighting = context.getProperty(propDesc).asInteger();
            weightings.put(relationship, weighting);
        }
        this.updateWeightedRelationships(weightings);
    }

    private void updateWeightedRelationships(Map<Integer, Integer> weightings) {
        ArrayList<Relationship> relationshipList = new ArrayList<Relationship>();
        for (Map.Entry<Integer, Integer> entry : weightings.entrySet()) {
            String relationshipName = String.valueOf(entry.getKey());
            Relationship relationship = new Relationship.Builder().name(relationshipName).build();
            for (int i = 0; i < entry.getValue(); ++i) {
                relationshipList.add(relationship);
            }
        }
        this.weightedRelationshipListRef.set(Collections.unmodifiableList(relationshipList));
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        boolean allDestinationsAvailable;
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        DistributionStrategy strategy = this.strategyRef.get();
        Set available = context.getAvailableRelationships();
        int numRelationships = context.getProperty(NUM_RELATIONSHIPS).asInteger();
        boolean bl = allDestinationsAvailable = available.size() == numRelationships;
        if (!allDestinationsAvailable && strategy.requiresAllDestinationsAvailable()) {
            session.rollback();
            context.yield();
            return;
        }
        Relationship relationship = strategy.mapToRelationship(context, flowFile);
        if (relationship == null) {
            session.rollback();
            context.yield();
            return;
        }
        session.putAttribute(flowFile, RELATIONSHIP_ATTRIBUTE, relationship.getName());
        session.transfer(flowFile, relationship);
        session.getProvenanceReporter().route(flowFile, relationship);
    }

    private class RoundRobinStrategy
    implements DistributionStrategy {
        private final AtomicLong counter = new AtomicLong(0L);

        private RoundRobinStrategy() {
        }

        @Override
        public Relationship mapToRelationship(ProcessContext context, FlowFile flowFile) {
            List<Relationship> relationshipList = DistributeLoad.this.weightedRelationshipListRef.get();
            long counterValue = this.counter.getAndIncrement();
            int idx = (int)(counterValue % (long)relationshipList.size());
            Relationship relationship = relationshipList.get(idx);
            return relationship;
        }

        @Override
        public boolean requiresAllDestinationsAvailable() {
            return true;
        }
    }

    private class NextAvailableStrategy
    implements DistributionStrategy {
        private final AtomicLong counter = new AtomicLong(0L);

        private NextAvailableStrategy() {
        }

        @Override
        public Relationship mapToRelationship(ProcessContext context, FlowFile flowFile) {
            List<Relationship> relationshipList = DistributeLoad.this.weightedRelationshipListRef.get();
            int numRelationships = relationshipList.size();
            boolean foundFreeRelationship = false;
            Relationship relationship = null;
            int attempts = 0;
            while (!foundFreeRelationship) {
                long counterValue = this.counter.getAndIncrement();
                int idx = (int)(counterValue % (long)numRelationships);
                relationship = relationshipList.get(idx);
                foundFreeRelationship = context.getAvailableRelationships().contains(relationship);
                if (++attempts % numRelationships != 0 || foundFreeRelationship) continue;
                return null;
            }
            return relationship;
        }

        @Override
        public boolean requiresAllDestinationsAvailable() {
            return false;
        }
    }

    private class OverflowStrategy
    implements DistributionStrategy {
        private OverflowStrategy() {
        }

        @Override
        public Relationship mapToRelationship(ProcessContext context, FlowFile flowFile) {
            List<Relationship> relationshipList = DistributeLoad.this.weightedRelationshipListRef.get();
            int numRelationships = relationshipList.size();
            boolean foundFreeRelationship = false;
            Relationship relationship = null;
            Set availableRelationships = context.getAvailableRelationships();
            int weightedIndex = 0;
            while (!foundFreeRelationship) {
                relationship = relationshipList.get(weightedIndex);
                foundFreeRelationship = availableRelationships.contains(relationship);
                if (++weightedIndex % numRelationships != 0 || foundFreeRelationship) continue;
                return null;
            }
            return relationship;
        }

        @Override
        public boolean requiresAllDestinationsAvailable() {
            return false;
        }
    }

    private static class InvalidPropertyNameValidator
    implements Validator {
        private final String propertyName;

        public InvalidPropertyNameValidator(String propertyName) {
            this.propertyName = propertyName;
        }

        public ValidationResult validate(String subject, String input, ValidationContext validationContext) {
            return new ValidationResult.Builder().subject("Property Name").input(this.propertyName).explanation("Property Name must be a positive integer between 1 and the number of relationships (inclusive)").valid(false).build();
        }
    }

    private static interface DistributionStrategy {
        public Relationship mapToRelationship(ProcessContext var1, FlowFile var2);

        public boolean requiresAllDestinationsAvailable();
    }
}

