/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.ReadsAttribute;
import org.apache.nifi.annotation.behavior.Restricted;
import org.apache.nifi.annotation.behavior.Restriction;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.RequiredPermission;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.standard.FetchFile;
import org.apache.nifi.processors.standard.GetFile;
import org.apache.nifi.util.StopWatch;

@SupportsBatching
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"put", "local", "copy", "archive", "files", "filesystem"})
@CapabilityDescription(value="Writes the contents of a FlowFile to the local file system")
@SeeAlso(value={FetchFile.class, GetFile.class})
@ReadsAttribute(attribute="filename", description="The filename to use when writing the FlowFile to disk.")
@Restricted(restrictions={@Restriction(requiredPermission=RequiredPermission.WRITE_FILESYSTEM, explanation="Provides operator the ability to write to any file that NiFi has access to.")})
public class PutFile
extends AbstractProcessor {
    public static final String REPLACE_RESOLUTION = "replace";
    public static final String IGNORE_RESOLUTION = "ignore";
    public static final String FAIL_RESOLUTION = "fail";
    public static final String FILE_MODIFY_DATE_ATTRIBUTE = "file.lastModifiedTime";
    public static final String FILE_MODIFY_DATE_ATTR_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ");
    public static final Pattern RWX_PATTERN = Pattern.compile("^([r-][w-])([x-])([r-][w-])([x-])([r-][w-])([x-])$");
    public static final Pattern NUM_PATTERN = Pattern.compile("^[0-7]{3}$");
    private static final Validator PERMISSIONS_VALIDATOR = new Validator(){

        public ValidationResult validate(String subject, String input, ValidationContext context) {
            ValidationResult.Builder vr = new ValidationResult.Builder();
            if (context.isExpressionLanguagePresent(input)) {
                return new ValidationResult.Builder().subject(subject).input(input).explanation("Expression Language Present").valid(true).build();
            }
            if (RWX_PATTERN.matcher(input).matches() || NUM_PATTERN.matcher(input).matches()) {
                return vr.valid(true).build();
            }
            return vr.valid(false).subject(subject).input(input).explanation("This must be expressed in rwxr-x--- form or octal triplet form.").build();
        }
    };
    public static final PropertyDescriptor DIRECTORY = new PropertyDescriptor.Builder().name("Directory").description("The directory to which files should be written. You may use expression language such as /aa/bb/${path}").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor MAX_DESTINATION_FILES = new PropertyDescriptor.Builder().name("Maximum File Count").description("Specifies the maximum number of files that can exist in the output directory").required(false).addValidator(StandardValidators.INTEGER_VALIDATOR).build();
    public static final PropertyDescriptor CONFLICT_RESOLUTION = new PropertyDescriptor.Builder().name("Conflict Resolution Strategy").description("Indicates what should happen when a file with the same name already exists in the output directory").required(true).defaultValue("fail").allowableValues(new String[]{"replace", "ignore", "fail"}).build();
    public static final PropertyDescriptor CHANGE_LAST_MODIFIED_TIME = new PropertyDescriptor.Builder().name("Last Modified Time").description("Sets the lastModifiedTime on the output file to the value of this attribute.  Format must be yyyy-MM-dd'T'HH:mm:ssZ.  You may also use expression language such as ${file.lastModifiedTime}.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor CHANGE_PERMISSIONS = new PropertyDescriptor.Builder().name("Permissions").description("Sets the permissions on the output file to the value of this attribute.  Format must be either UNIX rwxrwxrwx with a - in place of denied permissions (e.g. rw-r--r--) or an octal number (e.g. 644).  You may also use expression language such as ${file.permissions}.").required(false).addValidator(PERMISSIONS_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor CHANGE_OWNER = new PropertyDescriptor.Builder().name("Owner").description("Sets the owner on the output file to the value of this attribute.  You may also use expression language such as ${file.owner}. Note on many operating systems Nifi must be running as a super-user to have the permissions to set the file owner.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor CHANGE_GROUP = new PropertyDescriptor.Builder().name("Group").description("Sets the group on the output file to the value of this attribute.  You may also use expression language such as ${file.group}.").required(false).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor CREATE_DIRS = new PropertyDescriptor.Builder().name("Create Missing Directories").description("If true, then missing destination directories will be created. If false, flowfiles are penalized and sent to failure.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").build();
    public static final int MAX_FILE_LOCK_ATTEMPTS = 10;
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Files that have been successfully written to the output directory are transferred to this relationship").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Files that could not be written to the output directory for some reason are transferred to this relationship").build();
    private List<PropertyDescriptor> properties;
    private Set<Relationship> relationships;

    protected void init(ProcessorInitializationContext context) {
        HashSet<Relationship> procRels = new HashSet<Relationship>();
        procRels.add(REL_SUCCESS);
        procRels.add(REL_FAILURE);
        this.relationships = Collections.unmodifiableSet(procRels);
        ArrayList<PropertyDescriptor> supDescriptors = new ArrayList<PropertyDescriptor>();
        supDescriptors.add(DIRECTORY);
        supDescriptors.add(CONFLICT_RESOLUTION);
        supDescriptors.add(CREATE_DIRS);
        supDescriptors.add(MAX_DESTINATION_FILES);
        supDescriptors.add(CHANGE_LAST_MODIFIED_TIME);
        supDescriptors.add(CHANGE_PERMISSIONS);
        supDescriptors.add(CHANGE_OWNER);
        supDescriptors.add(CHANGE_GROUP);
        this.properties = Collections.unmodifiableList(supDescriptors);
    }

    public Set<Relationship> getRelationships() {
        return this.relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.properties;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        StopWatch stopWatch = new StopWatch(true);
        Path configuredRootDirPath = Paths.get(context.getProperty(DIRECTORY).evaluateAttributeExpressions(flowFile).getValue(), new String[0]);
        String conflictResponse = context.getProperty(CONFLICT_RESOLUTION).getValue();
        Integer maxDestinationFiles = context.getProperty(MAX_DESTINATION_FILES).asInteger();
        ComponentLog logger = this.getLogger();
        Path tempDotCopyFile = null;
        try {
            long numFiles;
            Path dotCopyFile;
            PosixFileAttributeView view;
            Path rootDirPath = configuredRootDirPath.toAbsolutePath();
            String filename = flowFile.getAttribute(CoreAttributes.FILENAME.key());
            Path tempCopyFile = rootDirPath.resolve("." + filename);
            Path copyFile = rootDirPath.resolve(filename);
            String permissions = context.getProperty(CHANGE_PERMISSIONS).evaluateAttributeExpressions(flowFile).getValue();
            String owner = context.getProperty(CHANGE_OWNER).evaluateAttributeExpressions(flowFile).getValue();
            String group = context.getProperty(CHANGE_GROUP).evaluateAttributeExpressions(flowFile).getValue();
            if (!Files.exists(rootDirPath, new LinkOption[0])) {
                if (context.getProperty(CREATE_DIRS).asBoolean().booleanValue()) {
                    boolean chGroup;
                    Path existing = rootDirPath;
                    while (!Files.exists(existing, new LinkOption[0])) {
                        existing = existing.getParent();
                    }
                    if (permissions != null && !permissions.trim().isEmpty()) {
                        try {
                            String perms = this.stringPermissions(permissions, true);
                            if (!perms.isEmpty()) {
                                Files.createDirectories(rootDirPath, PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString(perms)));
                            }
                            Files.createDirectories(rootDirPath, new FileAttribute[0]);
                        }
                        catch (Exception e) {
                            flowFile = session.penalize(flowFile);
                            session.transfer(flowFile, REL_FAILURE);
                            logger.error("Could not set create directory with permissions {}", new Object[]{permissions, e});
                            return;
                        }
                    } else {
                        Files.createDirectories(rootDirPath, new FileAttribute[0]);
                    }
                    boolean chOwner = owner != null && !owner.trim().isEmpty();
                    boolean bl = chGroup = group != null && !group.trim().isEmpty();
                    if (chOwner || chGroup) {
                        Path currentPath = rootDirPath;
                        while (!currentPath.equals(existing)) {
                            UserPrincipalLookupService lookupService;
                            if (chOwner) {
                                try {
                                    lookupService = currentPath.getFileSystem().getUserPrincipalLookupService();
                                    Files.setOwner(currentPath, lookupService.lookupPrincipalByName(owner));
                                }
                                catch (Exception e) {
                                    logger.warn("Could not set directory owner to {}", new Object[]{owner, e});
                                }
                            }
                            if (chGroup) {
                                try {
                                    lookupService = currentPath.getFileSystem().getUserPrincipalLookupService();
                                    view = Files.getFileAttributeView(currentPath, PosixFileAttributeView.class, new LinkOption[0]);
                                    view.setGroup(lookupService.lookupPrincipalByGroupName(group));
                                }
                                catch (Exception e) {
                                    logger.warn("Could not set file group to {}", new Object[]{group, e});
                                }
                            }
                            currentPath = currentPath.getParent();
                        }
                    }
                } else {
                    flowFile = session.penalize(flowFile);
                    session.transfer(flowFile, REL_FAILURE);
                    logger.error("Penalizing {} and routing to 'failure' because the output directory {} does not exist and Processor is configured not to create missing directories", new Object[]{flowFile, rootDirPath});
                    return;
                }
            }
            tempDotCopyFile = dotCopyFile = tempCopyFile;
            Path finalCopyFile = copyFile;
            Path finalCopyFileDir = finalCopyFile.getParent();
            if (Files.exists(finalCopyFileDir, new LinkOption[0]) && maxDestinationFiles != null && (numFiles = this.getFilesNumberInFolder(finalCopyFileDir, filename)) >= (long)maxDestinationFiles.intValue()) {
                flowFile = session.penalize(flowFile);
                logger.warn("Penalizing {} and routing to 'failure' because the output directory {} has {} files, which exceeds the configured maximum number of files", new Object[]{flowFile, finalCopyFileDir, numFiles});
                session.transfer(flowFile, REL_FAILURE);
                return;
            }
            if (Files.exists(finalCopyFile, new LinkOption[0])) {
                switch (conflictResponse) {
                    case "replace": {
                        Files.delete(finalCopyFile);
                        logger.info("Deleted {} as configured in order to replace with the contents of {}", new Object[]{finalCopyFile, flowFile});
                        break;
                    }
                    case "ignore": {
                        session.transfer(flowFile, REL_SUCCESS);
                        logger.info("Transferring {} to success because file with same name already exists", new Object[]{flowFile});
                        return;
                    }
                    case "fail": {
                        flowFile = session.penalize(flowFile);
                        logger.warn("Penalizing {} and routing to failure as configured because file with the same name already exists", new Object[]{flowFile});
                        session.transfer(flowFile, REL_FAILURE);
                        return;
                    }
                }
            }
            session.exportTo(flowFile, dotCopyFile, false);
            String lastModifiedTime = context.getProperty(CHANGE_LAST_MODIFIED_TIME).evaluateAttributeExpressions(flowFile).getValue();
            if (lastModifiedTime != null && !lastModifiedTime.trim().isEmpty()) {
                try {
                    OffsetDateTime fileModifyTime = OffsetDateTime.parse(lastModifiedTime, DATE_TIME_FORMATTER);
                    dotCopyFile.toFile().setLastModified(fileModifyTime.toInstant().toEpochMilli());
                }
                catch (Exception e) {
                    logger.warn("Could not set file lastModifiedTime to {}", new Object[]{lastModifiedTime, e});
                }
            }
            if (permissions != null && !permissions.trim().isEmpty()) {
                try {
                    String perms = this.stringPermissions(permissions, false);
                    if (!perms.isEmpty()) {
                        Files.setPosixFilePermissions(dotCopyFile, PosixFilePermissions.fromString(perms));
                    }
                }
                catch (Exception e) {
                    logger.warn("Could not set file permissions to {}", new Object[]{permissions, e});
                }
            }
            if (owner != null && !owner.trim().isEmpty()) {
                try {
                    UserPrincipalLookupService lookupService = dotCopyFile.getFileSystem().getUserPrincipalLookupService();
                    Files.setOwner(dotCopyFile, lookupService.lookupPrincipalByName(owner));
                }
                catch (Exception e) {
                    logger.warn("Could not set file owner to {}", new Object[]{owner, e});
                }
            }
            if (group != null && !group.trim().isEmpty()) {
                try {
                    UserPrincipalLookupService lookupService = dotCopyFile.getFileSystem().getUserPrincipalLookupService();
                    view = Files.getFileAttributeView(dotCopyFile, PosixFileAttributeView.class, new LinkOption[0]);
                    view.setGroup(lookupService.lookupPrincipalByGroupName(group));
                }
                catch (Exception e) {
                    logger.warn("Could not set file group to {}", new Object[]{group, e});
                }
            }
            boolean renamed = false;
            for (int i = 0; i < 10; ++i) {
                if (dotCopyFile.toFile().renameTo(finalCopyFile.toFile())) {
                    renamed = true;
                    break;
                }
                Thread.sleep(100L);
            }
            if (!renamed) {
                if (Files.exists(dotCopyFile, new LinkOption[0]) && dotCopyFile.toFile().delete()) {
                    logger.debug("Deleted dot copy file {}", new Object[]{dotCopyFile});
                }
                throw new ProcessException("Could not rename: " + String.valueOf(dotCopyFile));
            }
            logger.info("Produced copy of {} at location {}", new Object[]{flowFile, finalCopyFile});
            session.getProvenanceReporter().send(flowFile, finalCopyFile.toFile().toURI().toString(), stopWatch.getElapsed(TimeUnit.MILLISECONDS));
            session.transfer(flowFile, REL_SUCCESS);
        }
        catch (Throwable t) {
            if (tempDotCopyFile != null) {
                try {
                    Files.deleteIfExists(tempDotCopyFile);
                }
                catch (Exception e) {
                    logger.error("Unable to remove temporary file {}", new Object[]{tempDotCopyFile, e});
                }
            }
            flowFile = session.penalize(flowFile);
            logger.error("Penalizing {} and transferring to failure", new Object[]{flowFile, t});
            session.transfer(flowFile, REL_FAILURE);
        }
    }

    private long getFilesNumberInFolder(Path folder, String filename) {
        String[] filesInFolder = folder.toFile().list();
        return Arrays.stream(filesInFolder).filter(eachFilename -> !eachFilename.equals(filename)).count();
    }

    protected String stringPermissions(String perms, boolean directory) {
        String permissions = "";
        Matcher rwx = RWX_PATTERN.matcher(perms);
        if (rwx.matches()) {
            if (directory) {
                StringBuilder permBuilder = new StringBuilder();
                permBuilder.append("$1");
                permBuilder.append(rwx.group(1).equals("--") ? "$2" : "x");
                permBuilder.append("$3");
                permBuilder.append(rwx.group(3).equals("--") ? "$4" : "x");
                permBuilder.append("$5");
                permBuilder.append(rwx.group(5).equals("--") ? "$6" : "x");
                permissions = rwx.replaceAll(permBuilder.toString());
            } else {
                permissions = perms;
            }
        } else if (NUM_PATTERN.matcher(perms).matches()) {
            try {
                int number = Integer.parseInt(perms, 8);
                StringBuilder permBuilder = new StringBuilder();
                if ((number & 0x100) > 0) {
                    permBuilder.append('r');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 0x80) > 0) {
                    permBuilder.append('w');
                } else {
                    permBuilder.append('-');
                }
                if (directory || (number & 0x40) > 0) {
                    permBuilder.append('x');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 0x20) > 0) {
                    permBuilder.append('r');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 0x10) > 0) {
                    permBuilder.append('w');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 8) > 0) {
                    permBuilder.append('x');
                } else if (directory && (number & 0x30) > 0) {
                    permBuilder.append('x');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 4) > 0) {
                    permBuilder.append('r');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 2) > 0) {
                    permBuilder.append('w');
                } else {
                    permBuilder.append('-');
                }
                if ((number & 1) > 0) {
                    permBuilder.append('x');
                } else if (directory && (number & 6) > 0) {
                    permBuilder.append('x');
                } else {
                    permBuilder.append('-');
                }
                permissions = permBuilder.toString();
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return permissions;
    }
}

