/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.toolkit.tls.util;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.toolkit.tls.util.OutputStreamFactory;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.pkcs.RSAPrivateKey;
import org.bouncycastle.asn1.pkcs.RSAPublicKey;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.asn1.x509.ExtensionsGenerator;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.openssl.PEMException;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaMiscPEMGenerator;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequest;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TlsHelper {
    private static final Logger logger = LoggerFactory.getLogger(TlsHelper.class);
    private static final int DEFAULT_MAX_ALLOWED_KEY_LENGTH = 128;
    public static final String JCE_URL = "http://www.oracle.com/technetwork/java/javase/downloads/jce8-download-2133166.html";
    public static final String ILLEGAL_KEY_SIZE = "illegal key size";
    private static boolean isUnlimitedStrengthCryptographyEnabled;
    private static boolean isVerbose;

    private static void logTruncationWarning(File file) {
        String fileToString = file.toString();
        String fileName = file.getName();
        logger.warn("**********************************************************************************");
        logger.warn("                                    WARNING!!!!");
        logger.warn("**********************************************************************************");
        logger.warn("Unlimited JCE Policy is not installed which means we cannot utilize a");
        logger.warn("PKCS12 password longer than 7 characters.");
        logger.warn("Autogenerated password has been reduced to 7 characters.");
        logger.warn("");
        logger.warn("Please strongly consider installing Unlimited JCE Policy at");
        logger.warn(JCE_URL);
        logger.warn("");
        logger.warn("Another alternative is to add a stronger password with the openssl tool to the");
        logger.warn("resulting client certificate: " + fileToString);
        logger.warn("");
        logger.warn("openssl pkcs12 -in '" + fileToString + "' -out '/tmp/" + fileName + "'");
        logger.warn("openssl pkcs12 -export -in '/tmp/" + fileName + "' -out '" + fileToString + "'");
        logger.warn("rm -f '/tmp/" + fileName + "'");
        logger.warn("");
        logger.warn("**********************************************************************************");
    }

    private TlsHelper() {
    }

    public static boolean isUnlimitedStrengthCryptographyEnabled() {
        return isUnlimitedStrengthCryptographyEnabled;
    }

    public static String writeKeyStore(KeyStore keyStore, OutputStreamFactory outputStreamFactory, File file, String password, boolean generatedPassword) throws IOException, GeneralSecurityException {
        try (OutputStream fileOutputStream = outputStreamFactory.create(file);){
            keyStore.store(fileOutputStream, password.toCharArray());
        }
        catch (IOException e) {
            if (e.getMessage().toLowerCase().contains(ILLEGAL_KEY_SIZE) && !TlsHelper.isUnlimitedStrengthCryptographyEnabled()) {
                if (generatedPassword) {
                    file.delete();
                    String truncatedPassword = password.substring(0, 7);
                    try (OutputStream fileOutputStream2 = outputStreamFactory.create(file);){
                        keyStore.store(fileOutputStream2, truncatedPassword.toCharArray());
                    }
                    TlsHelper.logTruncationWarning(file);
                    return truncatedPassword;
                }
                throw new GeneralSecurityException("Specified password for " + file + " too long to work without unlimited JCE policy installed." + System.lineSeparator() + "Please see " + JCE_URL);
            }
            throw e;
        }
        return password;
    }

    private static KeyPairGenerator createKeyPairGenerator(String algorithm, int keySize) throws NoSuchAlgorithmException {
        KeyPairGenerator instance = KeyPairGenerator.getInstance(algorithm);
        instance.initialize(keySize);
        return instance;
    }

    public static byte[] calculateHMac(String token, PublicKey publicKey) throws GeneralSecurityException {
        if (token == null) {
            throw new IllegalArgumentException("Token cannot be null");
        }
        byte[] tokenBytes = token.getBytes(StandardCharsets.UTF_8);
        if (tokenBytes.length < 16) {
            throw new GeneralSecurityException("Token does not meet minimum size of 16 bytes.");
        }
        SecretKeySpec keySpec = new SecretKeySpec(tokenBytes, "RAW");
        Mac mac = Mac.getInstance("Hmac-SHA256", "BC");
        mac.init(keySpec);
        return mac.doFinal(TlsHelper.getKeyIdentifier(publicKey));
    }

    public static byte[] getKeyIdentifier(PublicKey publicKey) throws NoSuchAlgorithmException {
        return new JcaX509ExtensionUtils().createSubjectKeyIdentifier(publicKey).getKeyIdentifier();
    }

    public static String pemEncodeJcaObject(Object object) throws IOException {
        StringWriter writer = new StringWriter();
        try (PemWriter pemWriter = new PemWriter((Writer)writer);){
            pemWriter.writeObject((PemObjectGenerator)new JcaMiscPEMGenerator(object));
        }
        return writer.toString();
    }

    public static JcaPKCS10CertificationRequest parseCsr(String pemEncodedCsr) throws IOException {
        try (PEMParser pemParser = new PEMParser((Reader)new StringReader(pemEncodedCsr));){
            Object o = pemParser.readObject();
            if (!PKCS10CertificationRequest.class.isInstance(o)) {
                throw new IOException("Expecting instance of " + PKCS10CertificationRequest.class + " but got " + o);
            }
            JcaPKCS10CertificationRequest jcaPKCS10CertificationRequest = new JcaPKCS10CertificationRequest((PKCS10CertificationRequest)o);
            return jcaPKCS10CertificationRequest;
        }
    }

    public static X509Certificate parseCertificate(Reader pemEncodedCertificate) throws IOException, CertificateException {
        return new JcaX509CertificateConverter().setProvider("BC").getCertificate(TlsHelper.parsePem(X509CertificateHolder.class, pemEncodedCertificate));
    }

    public static KeyPair parseKeyPairFromReader(Reader pemKeyPairReader) throws IOException {
        Throwable throwable = null;
        try (PEMParser pemParser = new PEMParser(pemKeyPairReader);){
            Object parsedObject = pemParser.readObject();
            if (PrivateKeyInfo.class.isInstance(parsedObject)) {
                if (TlsHelper.isVerbose()) {
                    logger.info("Provided private key is in PKCS #8 format");
                }
                PEMKeyPair keyPair = TlsHelper.convertPrivateKeyFromPKCS8ToPKCS1((PrivateKeyInfo)parsedObject);
                KeyPair keyPair2 = TlsHelper.getKeyPair(keyPair);
                return keyPair2;
            }
            if (PEMKeyPair.class.isInstance(parsedObject)) {
                KeyPair keyPair = TlsHelper.getKeyPair((PEMKeyPair)parsedObject);
                return keyPair;
            }
            try {
                logger.warn("Expected one of %s or %s but got %s", new Object[]{PrivateKeyInfo.class, PEMKeyPair.class, parsedObject.getClass()});
                throw new IOException("Expected private key in PKCS #1 or PKCS #8 unencrypted format");
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
        }
    }

    private static KeyPair getKeyPair(PEMKeyPair keyPair) throws PEMException {
        return new JcaPEMKeyConverter().setProvider("BC").getKeyPair(keyPair);
    }

    private static PEMKeyPair convertPrivateKeyFromPKCS8ToPKCS1(PrivateKeyInfo privateKeyInfo) throws IOException {
        ASN1Encodable asn1PrivateKey = privateKeyInfo.parsePrivateKey();
        RSAPrivateKey keyStruct = RSAPrivateKey.getInstance((Object)asn1PrivateKey);
        RSAPublicKey pubSpec = new RSAPublicKey(keyStruct.getModulus(), keyStruct.getPublicExponent());
        AlgorithmIdentifier algId = new AlgorithmIdentifier(PKCSObjectIdentifiers.rsaEncryption, (ASN1Encodable)DERNull.INSTANCE);
        if (TlsHelper.isVerbose()) {
            logger.info("Converted private key from PKCS #8 to PKCS #1 RSA private key");
        }
        return new PEMKeyPair(new SubjectPublicKeyInfo(algId, (ASN1Encodable)pubSpec), new PrivateKeyInfo(algId, (ASN1Encodable)keyStruct));
    }

    public static <T> T parsePem(Class<T> clazz, Reader pemReader) throws IOException {
        try (PEMParser pemParser = new PEMParser(pemReader);){
            Object object = pemParser.readObject();
            if (!clazz.isInstance(object)) {
                throw new IOException("Expected " + clazz + " but got " + object.getClass());
            }
            Object object2 = object;
            return (T)object2;
        }
    }

    public static KeyPair generateKeyPair(String algorithm, int keySize) throws NoSuchAlgorithmException {
        return TlsHelper.createKeyPairGenerator(algorithm, keySize).generateKeyPair();
    }

    public static JcaPKCS10CertificationRequest generateCertificationRequest(String requestedDn, String domainAlternativeNames, KeyPair keyPair, String signingAlgorithm) throws OperatorCreationException {
        JcaPKCS10CertificationRequestBuilder jcaPKCS10CertificationRequestBuilder = new JcaPKCS10CertificationRequestBuilder(new X500Name(requestedDn), keyPair.getPublic());
        try {
            jcaPKCS10CertificationRequestBuilder.addAttribute(PKCSObjectIdentifiers.pkcs_9_at_extensionRequest, (ASN1Encodable)TlsHelper.createDomainAlternativeNamesExtensions(domainAlternativeNames, requestedDn));
        }
        catch (IOException e) {
            throw new OperatorCreationException("Error while adding " + domainAlternativeNames + " as Subject Alternative Name.", (Throwable)e);
        }
        JcaContentSignerBuilder jcaContentSignerBuilder = new JcaContentSignerBuilder(signingAlgorithm);
        return new JcaPKCS10CertificationRequest(jcaPKCS10CertificationRequestBuilder.build(jcaContentSignerBuilder.build(keyPair.getPrivate())));
    }

    public static Extensions createDomainAlternativeNamesExtensions(String domainAlternativeNames, String requestedDn) throws IOException {
        ArrayList<GeneralName> namesList = new ArrayList<GeneralName>();
        try {
            String cn = IETFUtils.valueToString((ASN1Encodable)new X500Name(requestedDn).getRDNs(BCStyle.CN)[0].getFirst().getValue());
            namesList.add(new GeneralName(2, cn));
        }
        catch (Exception e) {
            throw new IOException("Failed to extract CN from request DN: " + requestedDn, e);
        }
        if (StringUtils.isNotBlank((CharSequence)domainAlternativeNames)) {
            for (String alternativeName : domainAlternativeNames.split(",")) {
                namesList.add(new GeneralName(2, alternativeName));
            }
        }
        GeneralNames subjectAltNames = new GeneralNames(namesList.toArray(new GeneralName[0]));
        ExtensionsGenerator extGen = new ExtensionsGenerator();
        extGen.addExtension(Extension.subjectAlternativeName, false, (ASN1Encodable)subjectAltNames);
        return extGen.generate();
    }

    public static final String escapeFilename(String filename) {
        return filename.replaceAll("[^\\w\\.\\-\\=]+", "_");
    }

    public static boolean verifyCertificateSignature(X509Certificate certificate, List<X509Certificate> signingCertificates) {
        String certificateDisplayInfo = TlsHelper.getCertificateDisplayInfo(certificate);
        if (TlsHelper.isVerbose()) {
            logger.info("Verifying the certificate signature for " + certificateDisplayInfo);
        }
        boolean signatureMatches = false;
        for (X509Certificate signingCert : signingCertificates) {
            String signingCertDisplayInfo = TlsHelper.getCertificateDisplayInfo(signingCert);
            try {
                if (TlsHelper.isVerbose()) {
                    logger.info("Attempting to verify certificate " + certificateDisplayInfo + " signature with " + signingCertDisplayInfo);
                }
                PublicKey pub = signingCert.getPublicKey();
                certificate.verify(pub);
                if (TlsHelper.isVerbose()) {
                    logger.info("Certificate was signed by " + signingCertDisplayInfo);
                }
                signatureMatches = true;
                break;
            }
            catch (Exception e) {
                if (!TlsHelper.isVerbose()) continue;
                logger.warn("Certificate " + certificateDisplayInfo + " not signed by " + signingCertDisplayInfo + " [" + e.getLocalizedMessage() + "]");
            }
        }
        return signatureMatches;
    }

    private static String getCertificateDisplayInfo(X509Certificate certificate) {
        return certificate.getSubjectX500Principal().getName();
    }

    private static boolean isVerbose() {
        return isVerbose;
    }

    static {
        isVerbose = true;
        try {
            isUnlimitedStrengthCryptographyEnabled = Cipher.getMaxAllowedKeyLength("AES") > 128;
        }
        catch (NoSuchAlgorithmException e) {
            isUnlimitedStrengthCryptographyEnabled = false;
        }
    }
}

