/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.x509.ocsp;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.io.FileInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.net.URI;
import java.security.KeyStore;
import java.security.NoSuchProviderException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.framework.security.util.SslContextFactory;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.web.security.x509.ocsp.CertificateStatusException;
import org.apache.nifi.web.security.x509.ocsp.OcspRequest;
import org.apache.nifi.web.security.x509.ocsp.OcspStatus;
import org.apache.nifi.web.util.WebUtils;
import org.bouncycastle.ocsp.BasicOCSPResp;
import org.bouncycastle.ocsp.CertificateID;
import org.bouncycastle.ocsp.CertificateStatus;
import org.bouncycastle.ocsp.OCSPException;
import org.bouncycastle.ocsp.OCSPReq;
import org.bouncycastle.ocsp.OCSPReqGenerator;
import org.bouncycastle.ocsp.OCSPResp;
import org.bouncycastle.ocsp.RevokedStatus;
import org.bouncycastle.ocsp.SingleResp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OcspCertificateValidator {
    private static final Logger logger = LoggerFactory.getLogger(OcspCertificateValidator.class);
    private static final String HTTPS = "https";
    private static final String CONTENT_TYPE_HEADER = "Content-Type";
    private static final String OCSP_REQUEST_CONTENT_TYPE = "application/ocsp-request";
    private static final int CONNECT_TIMEOUT = 10000;
    private static final int READ_TIMEOUT = 10000;
    private URI validationAuthorityURI;
    private Client client;
    private Map<String, X509Certificate> trustedCAs;
    private LoadingCache<OcspRequest, OcspStatus> ocspCache;

    public OcspCertificateValidator(NiFiProperties properties) {
        String rawValidationAuthorityUrl = properties.getProperty("nifi.security.ocsp.responder.url");
        if (StringUtils.isNotBlank((CharSequence)rawValidationAuthorityUrl)) {
            try {
                this.validationAuthorityURI = URI.create(rawValidationAuthorityUrl);
                DefaultClientConfig config = new DefaultClientConfig();
                config.getProperties().put("com.sun.jersey.client.property.readTimeout", 10000);
                config.getProperties().put("com.sun.jersey.client.property.connectTimeout", 10000);
                this.client = HTTPS.equalsIgnoreCase(this.validationAuthorityURI.getScheme()) ? WebUtils.createClient((ClientConfig)config, (SSLContext)SslContextFactory.createSslContext((NiFiProperties)properties)) : WebUtils.createClient((ClientConfig)config);
                this.trustedCAs = this.getTrustedCAs(properties);
                X509Certificate ocspCertificate = this.getOcspCertificate(properties);
                if (ocspCertificate != null) {
                    this.trustedCAs.put(ocspCertificate.getSubjectX500Principal().getName(), ocspCertificate);
                }
                String rawCacheDurationDuration = properties.getUserCredentialCacheDuration();
                long cacheDurationMillis = FormatUtils.getTimeDuration((String)rawCacheDurationDuration, (TimeUnit)TimeUnit.MILLISECONDS);
                this.ocspCache = CacheBuilder.newBuilder().expireAfterWrite(cacheDurationMillis, TimeUnit.MILLISECONDS).build((CacheLoader)new CacheLoader<OcspRequest, OcspStatus>(){

                    public OcspStatus load(OcspRequest ocspRequest) throws Exception {
                        String subjectDn = ocspRequest.getSubjectCertificate().getSubjectX500Principal().getName();
                        logger.info(String.format("Validating client certificate via OCSP: <%s>", subjectDn));
                        OcspStatus ocspStatus = OcspCertificateValidator.this.getOcspStatus(ocspRequest);
                        logger.info(String.format("Client certificate status for <%s>: %s", subjectDn, ocspStatus.toString()));
                        return ocspStatus;
                    }
                });
            }
            catch (Exception e) {
                logger.error("Disabling OCSP certificate validation. Unable to load OCSP configuration: " + e, (Throwable)e);
                this.client = null;
            }
        }
    }

    private X509Certificate getOcspCertificate(NiFiProperties properties) {
        X509Certificate validationAuthorityCertificate = null;
        String validationAuthorityCertificatePath = properties.getProperty("nifi.security.ocsp.responder.certificate");
        if (StringUtils.isNotBlank((CharSequence)validationAuthorityCertificatePath)) {
            try (FileInputStream fis = new FileInputStream(validationAuthorityCertificatePath);){
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                validationAuthorityCertificate = (X509Certificate)cf.generateCertificate(fis);
            }
            catch (Exception e) {
                throw new IllegalStateException("Unable to load the validation authority certificate: " + e);
            }
        }
        return validationAuthorityCertificate;
    }

    private Map<String, X509Certificate> getTrustedCAs(NiFiProperties properties) {
        HashMap<String, X509Certificate> certificateAuthorities = new HashMap<String, X509Certificate>();
        String truststorePath = properties.getProperty("nifi.security.truststore");
        if (truststorePath == null) {
            throw new IllegalArgumentException("The truststore path is required.");
        }
        String rawTruststorePassword = properties.getProperty("nifi.security.truststorePasswd");
        char[] truststorePassword = rawTruststorePassword == null ? new char[]{} : rawTruststorePassword.toCharArray();
        try (FileInputStream fis = new FileInputStream(truststorePath);){
            KeyStore truststore = KeyStore.getInstance(KeyStore.getDefaultType());
            truststore.load(fis, truststorePassword);
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(truststore);
            for (TrustManager trustManager : trustManagerFactory.getTrustManagers()) {
                if (!(trustManager instanceof X509TrustManager)) continue;
                for (X509Certificate ca : ((X509TrustManager)trustManager).getAcceptedIssuers()) {
                    certificateAuthorities.put(ca.getSubjectX500Principal().getName(), ca);
                }
            }
        }
        catch (Exception e) {
            throw new IllegalStateException("Unable to load the configured truststore: " + e);
        }
        return certificateAuthorities;
    }

    public void validate(HttpServletRequest request) throws CertificateStatusException {
        X509Certificate[] certificates = (X509Certificate[])request.getAttribute("javax.servlet.request.X509Certificate");
        if (this.client != null && certificates != null && certificates.length > 0) {
            X509Certificate subjectCertificate = this.getSubjectCertificate(certificates);
            X509Certificate issuerCertificate = this.getIssuerCertificate(certificates);
            if (issuerCertificate == null) {
                throw new IllegalArgumentException(String.format("Unable to obtain certificate of issuer <%s> for the specified subject certificate <%s>.", subjectCertificate.getIssuerX500Principal().getName(), subjectCertificate.getSubjectX500Principal().getName()));
            }
            OcspRequest ocspRequest = new OcspRequest(subjectCertificate, issuerCertificate);
            try {
                OcspStatus ocspStatus = (OcspStatus)this.ocspCache.getUnchecked((Object)ocspRequest);
                if (OcspStatus.VerificationStatus.Verified.equals((Object)ocspStatus.getVerificationStatus()) && OcspStatus.ValidationStatus.Revoked.equals((Object)ocspStatus.getValidationStatus())) {
                    throw new CertificateStatusException(String.format("Client certificate for <%s> is revoked according to the certificate authority.", subjectCertificate.getSubjectX500Principal().getName()));
                }
            }
            catch (UncheckedExecutionException uee) {
                logger.warn(String.format("Unable to validate client certificate via OCSP: <%s>", subjectCertificate.getSubjectX500Principal().getName()), uee.getCause());
            }
        }
    }

    private X509Certificate getSubjectCertificate(X509Certificate[] certificates) {
        return certificates[0];
    }

    private X509Certificate getIssuerCertificate(X509Certificate[] certificates) {
        if (certificates.length > 1) {
            return certificates[1];
        }
        if (certificates.length == 1) {
            X509Certificate subjectCertificate = this.getSubjectCertificate(certificates);
            X500Principal issuerPrincipal = subjectCertificate.getIssuerX500Principal();
            return this.trustedCAs.get(issuerPrincipal.getName());
        }
        return null;
    }

    private OcspStatus getOcspStatus(OcspRequest ocspStatusKey) {
        X509Certificate subjectCertificate = ocspStatusKey.getSubjectCertificate();
        X509Certificate issuerCertificate = ocspStatusKey.getIssuerCertificate();
        OcspStatus ocspStatus = new OcspStatus();
        ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unknown);
        ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Unknown);
        try {
            SingleResp[] responses;
            BigInteger subjectSerialNumber = subjectCertificate.getSerialNumber();
            CertificateID certificateId = new CertificateID("1.3.14.3.2.26", issuerCertificate, subjectSerialNumber);
            OCSPReqGenerator requestGenerator = new OCSPReqGenerator();
            requestGenerator.addRequest(certificateId);
            OCSPReq ocspRequest = requestGenerator.generate();
            WebResource resource = this.client.resource(this.validationAuthorityURI);
            ClientResponse response = (ClientResponse)resource.header(CONTENT_TYPE_HEADER, (Object)OCSP_REQUEST_CONTENT_TYPE).post(ClientResponse.class, (Object)ocspRequest.getEncoded());
            if (ClientResponse.Status.OK.getStatusCode() != response.getStatusInfo().getStatusCode()) {
                logger.warn(String.format("OCSP request was unsuccessful (%s).", response.getStatus()));
                return ocspStatus;
            }
            OCSPResp ocspResponse = new OCSPResp(response.getEntityInputStream());
            switch (ocspResponse.getStatus()) {
                case 0: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Successful);
                    break;
                }
                case 2: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.InternalError);
                    break;
                }
                case 1: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.MalformedRequest);
                    break;
                }
                case 5: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.SignatureRequired);
                    break;
                }
                case 3: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.TryLater);
                    break;
                }
                case 6: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Unauthorized);
                    break;
                }
                default: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Unknown);
                }
            }
            if (ocspResponse.getStatus() != 0) {
                logger.warn(String.format("OCSP request was unsuccessful (%s).", ocspStatus.getResponseStatus().toString()));
                return ocspStatus;
            }
            Object ocspResponseObject = ocspResponse.getResponseObject();
            if (ocspResponseObject == null || !(ocspResponseObject instanceof BasicOCSPResp)) {
                logger.warn(String.format("Unexcepted OCSP response object: %s", ocspResponseObject));
                return ocspStatus;
            }
            BasicOCSPResp basicOcspResponse = (BasicOCSPResp)ocspResponse.getResponseObject();
            X509Certificate[] responderCertificates = basicOcspResponse.getCerts(null);
            if (responderCertificates.length != 1) {
                logger.warn(String.format("Unexcepted number of OCSP responder certificates: %s", responderCertificates.length));
                return ocspStatus;
            }
            X509Certificate trustedResponderCertificate = this.getTrustedResponderCertificate(responderCertificates[0], issuerCertificate);
            if (trustedResponderCertificate != null) {
                if (basicOcspResponse.verify(trustedResponderCertificate.getPublicKey(), null)) {
                    ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Verified);
                } else {
                    ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unverified);
                }
            } else {
                ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unverified);
            }
            for (SingleResp singleResponse : responses = basicOcspResponse.getResponses()) {
                CertificateID responseCertificateId = singleResponse.getCertID();
                BigInteger responseSerialNumber = responseCertificateId.getSerialNumber();
                if (!responseSerialNumber.equals(subjectSerialNumber)) continue;
                Object certStatus = singleResponse.getCertStatus();
                if (CertificateStatus.GOOD == certStatus) {
                    ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Good);
                    continue;
                }
                if (certStatus instanceof RevokedStatus) {
                    ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Revoked);
                    continue;
                }
                ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Unknown);
            }
        }
        catch (ClientHandlerException | UniformInterfaceException | IOException | NoSuchProviderException | OCSPException e) {
            logger.error(e.getMessage(), e);
        }
        return ocspStatus;
    }

    private X509Certificate getTrustedResponderCertificate(X509Certificate responderCertificate, X509Certificate issuerCertificate) {
        if (this.trustedCAs.containsKey(responderCertificate.getSubjectX500Principal().getName())) {
            return this.trustedCAs.get(responderCertificate.getSubjectX500Principal().getName());
        }
        if (responderCertificate.getIssuerX500Principal().equals(issuerCertificate.getSubjectX500Principal())) {
            return null;
        }
        return null;
    }
}

