/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.jwt;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.JwsHeader;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SignatureException;
import io.jsonwebtoken.SigningKeyResolver;
import io.jsonwebtoken.SigningKeyResolverAdapter;
import io.jsonwebtoken.UnsupportedJwtException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Calendar;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.admin.service.AdministrationException;
import org.apache.nifi.admin.service.KeyService;
import org.apache.nifi.key.Key;
import org.apache.nifi.web.security.jwt.JwtAuthenticationFilter;
import org.apache.nifi.web.security.token.LoginAuthenticationToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JwtService {
    private static final Logger logger = LoggerFactory.getLogger(JwtService.class);
    private static final SignatureAlgorithm SIGNATURE_ALGORITHM = SignatureAlgorithm.HS256;
    private static final String KEY_ID_CLAIM = "kid";
    private static final String USERNAME_CLAIM = "preferred_username";
    private final KeyService keyService;

    public JwtService(KeyService keyService) {
        this.keyService = keyService;
    }

    public String getAuthenticationFromToken(String base64EncodedToken) throws JwtException {
        try {
            Jws<Claims> jws = this.parseTokenFromBase64EncodedString(base64EncodedToken);
            if (jws == null) {
                throw new JwtException("Unable to parse token");
            }
            if (StringUtils.isEmpty((CharSequence)((Claims)jws.getBody()).getSubject())) {
                throw new JwtException("No subject available in token");
            }
            if (StringUtils.isEmpty((CharSequence)((Claims)jws.getBody()).getIssuer())) {
                throw new JwtException("No issuer available in token");
            }
            return ((Claims)jws.getBody()).getSubject();
        }
        catch (JwtException e) {
            logger.debug("The Base64 encoded JWT: " + base64EncodedToken);
            String errorMessage = "There was an error validating the JWT";
            String causeMessage = e.getLocalizedMessage();
            if (e.getCause() != null) {
                causeMessage = causeMessage + "\n\tCaused by: " + e.getCause().getLocalizedMessage();
            }
            if (logger.isDebugEnabled()) {
                logger.error("There was an error validating the JWT", (Throwable)e);
            } else {
                logger.error("There was an error validating the JWT");
                logger.error(causeMessage);
            }
            throw e;
        }
    }

    private Jws<Claims> parseTokenFromBase64EncodedString(String base64EncodedToken) throws JwtException {
        try {
            return Jwts.parser().setSigningKeyResolver((SigningKeyResolver)new SigningKeyResolverAdapter(){

                public byte[] resolveSigningKeyBytes(JwsHeader header, Claims claims) {
                    String identity = claims.getSubject();
                    Integer keyId = (Integer)claims.get(JwtService.KEY_ID_CLAIM, Integer.class);
                    Key key = JwtService.this.keyService.getKey(keyId.intValue());
                    if (key == null || key.getKey() == null) {
                        throw new UnsupportedJwtException("Unable to determine signing key for " + identity + " [kid: " + keyId + "]");
                    }
                    return key.getKey().getBytes(StandardCharsets.UTF_8);
                }
            }).parseClaimsJws(base64EncodedToken);
        }
        catch (ExpiredJwtException | MalformedJwtException | SignatureException | UnsupportedJwtException | IllegalArgumentException | AdministrationException e) {
            String errorMessage = "Unable to validate the access token.";
            throw new JwtException("Unable to validate the access token.", e);
        }
    }

    public String generateSignedToken(LoginAuthenticationToken authenticationToken) throws JwtException {
        if (authenticationToken == null) {
            throw new IllegalArgumentException("Cannot generate a JWT for a null authentication token");
        }
        Calendar expiration = Calendar.getInstance();
        expiration.setTimeInMillis(authenticationToken.getExpiration());
        Object principal = authenticationToken.getPrincipal();
        if (principal == null || StringUtils.isEmpty((CharSequence)principal.toString())) {
            String errorMessage = "Cannot generate a JWT for a token with an empty identity issued by " + authenticationToken.getIssuer();
            logger.error(errorMessage);
            throw new JwtException(errorMessage);
        }
        String identity = principal.toString();
        String username = authenticationToken.getName();
        String rawIssuer = authenticationToken.getIssuer();
        try {
            Key key = this.keyService.getOrCreateKey(identity);
            byte[] keyBytes = key.getKey().getBytes(StandardCharsets.UTF_8);
            logger.trace("Generating JWT for " + (Object)((Object)authenticationToken));
            String encodedIssuer = URLEncoder.encode(rawIssuer, "UTF-8");
            return Jwts.builder().setSubject(identity).setIssuer(encodedIssuer).setAudience(encodedIssuer).claim(USERNAME_CLAIM, (Object)username).claim(KEY_ID_CLAIM, (Object)key.getId()).setExpiration(expiration.getTime()).setIssuedAt(Calendar.getInstance().getTime()).signWith(SIGNATURE_ALGORITHM, keyBytes).compact();
        }
        catch (NullPointerException | AdministrationException e) {
            String errorMessage = "Could not retrieve the signing key for JWT for " + identity;
            logger.error(errorMessage, e);
            throw new JwtException(errorMessage, e);
        }
        catch (UnsupportedEncodingException e) {
            String errorMessage = "Could not URL encode issuer: " + rawIssuer;
            logger.error(errorMessage, (Throwable)e);
            throw new JwtException(errorMessage, (Throwable)e);
        }
    }

    public void logOut(String token) {
        Jws<Claims> claims = this.parseTokenFromBase64EncodedString(token);
        Integer keyId = (Integer)((Claims)claims.getBody()).get(KEY_ID_CLAIM, Integer.class);
        if (keyId == null) {
            throw new JwtException("The key claim (kid) was not present in the request token to log out user.");
        }
        try {
            this.keyService.deleteKey(keyId);
        }
        catch (Exception e) {
            logger.error("The key with key ID: " + keyId + " failed to be removed from the user database.");
            throw e;
        }
    }

    public void logOutUsingAuthHeader(String authorizationHeader) {
        String base64EncodedToken = JwtAuthenticationFilter.getTokenFromHeader(authorizationHeader);
        this.logOut(base64EncodedToken);
    }
}

