/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.openejb.core.security.jacc;

import org.apache.openejb.core.security.JaccProvider;

import javax.security.jacc.PolicyConfiguration;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;
import java.security.CodeSource;
import java.security.Permission;
import java.security.PermissionCollection;
import java.security.ProtectionDomain;
import java.util.HashMap;
import java.util.Map;

/**
 * @version $Rev: 1240707 $ $Date: 2012-02-05 13:08:37 +0100 (Dim, 05 fév 2012) $
 */
public class BasicJaccProvider extends JaccProvider {

    private Map<String, BasicPolicyConfiguration> configurations = new HashMap<String, BasicPolicyConfiguration>();

    private final java.security.Policy systemPolicy;

    public BasicJaccProvider() {
        systemPolicy = Policy.getPolicy();
    }

    public PolicyConfiguration getPolicyConfiguration(String contextID, boolean remove) throws PolicyContextException {
        BasicPolicyConfiguration configuration = configurations.get(contextID);

        if (configuration == null) {
            configuration = createPolicyConfiguration(contextID);
            configurations.put(contextID, configuration);
        } else {
            configuration.open(remove);
        }

        return configuration;
    }

    protected BasicPolicyConfiguration createPolicyConfiguration(String contextID) {
        return new BasicPolicyConfiguration(contextID);
    }

    public boolean inService(String contextID) throws PolicyContextException {
        PolicyConfiguration configuration = getPolicyConfiguration(contextID, false);
        return configuration.inService();
    }

    public PermissionCollection getPermissions(CodeSource codesource) {
        return systemPolicy == null ? null: systemPolicy.getPermissions(codesource);
    }

    public void refresh() {
    }

    public boolean implies(ProtectionDomain domain, Permission permission) {
        String contextID = PolicyContext.getContextID();

        if (contextID != null) {
            try {
                BasicPolicyConfiguration configuration = configurations.get(contextID);

                if (configuration == null || !configuration.inService()) return false;

                return configuration.implies(domain, permission);
            } catch (PolicyContextException e) {
            }
        }

        return (systemPolicy != null)? systemPolicy.implies(domain, permission): false;
    }
}
