/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import com.beust.jcommander.internal.Lists;
import com.carrotsearch.hppc.ObjectObjectHashMap;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.netty.buffer.ByteBuf;
import io.netty.util.concurrent.FastThreadLocal;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.bookkeeper.mledger.AsyncCallbacks;
import org.apache.bookkeeper.mledger.Entry;
import org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.bookkeeper.mledger.ManagedLedger;
import org.apache.bookkeeper.mledger.ManagedLedgerException;
import org.apache.bookkeeper.mledger.Position;
import org.apache.bookkeeper.mledger.impl.ManagedCursorImpl;
import org.apache.bookkeeper.mledger.impl.ManagedLedgerImpl;
import org.apache.bookkeeper.mledger.impl.PositionImpl;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.service.BrokerService;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.Consumer;
import org.apache.pulsar.broker.service.Producer;
import org.apache.pulsar.broker.service.Replicator;
import org.apache.pulsar.broker.service.ServerCnx;
import org.apache.pulsar.broker.service.Subscription;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.persistent.PersistentDispatcherMultipleConsumers;
import org.apache.pulsar.broker.service.persistent.PersistentReplicator;
import org.apache.pulsar.broker.service.persistent.PersistentSubscription;
import org.apache.pulsar.broker.stats.ClusterReplicationMetrics;
import org.apache.pulsar.broker.stats.NamespaceStats;
import org.apache.pulsar.broker.stats.ReplicationMetrics;
import org.apache.pulsar.client.api.MessageId;
import org.apache.pulsar.client.impl.MessageIdImpl;
import org.apache.pulsar.client.impl.MessageImpl;
import org.apache.pulsar.client.util.FutureUtil;
import org.apache.pulsar.common.api.proto.PulsarApi;
import org.apache.pulsar.common.naming.DestinationName;
import org.apache.pulsar.common.policies.data.BacklogQuota;
import org.apache.pulsar.common.policies.data.ConsumerStats;
import org.apache.pulsar.common.policies.data.PersistentTopicInternalStats;
import org.apache.pulsar.common.policies.data.PersistentTopicStats;
import org.apache.pulsar.common.policies.data.Policies;
import org.apache.pulsar.common.policies.data.PublisherStats;
import org.apache.pulsar.common.policies.data.ReplicatorStats;
import org.apache.pulsar.common.policies.data.SubscriptionStats;
import org.apache.pulsar.common.util.Codec;
import org.apache.pulsar.common.util.collections.ConcurrentOpenHashMap;
import org.apache.pulsar.common.util.collections.ConcurrentOpenHashSet;
import org.apache.pulsar.policies.data.loadbalancer.NamespaceBundleStats;
import org.apache.pulsar.utils.StatsOutputStream;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersistentTopic
implements Topic,
AsyncCallbacks.AddEntryCallback {
    private final String topic;
    private final ManagedLedger ledger;
    private final ConcurrentOpenHashSet<Producer> producers;
    private final ConcurrentOpenHashMap<String, PersistentSubscription> subscriptions;
    private final ConcurrentOpenHashMap<String, Replicator> replicators;
    private final BrokerService brokerService;
    private volatile boolean isFenced;
    protected static final AtomicLongFieldUpdater<PersistentTopic> USAGE_COUNT_UPDATER = AtomicLongFieldUpdater.newUpdater(PersistentTopic.class, "usageCount");
    private volatile long usageCount = 0L;
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    public final String replicatorPrefix;
    private static final double MESSAGE_EXPIRY_THRESHOLD = 1.5;
    private static final long POLICY_UPDATE_FAILURE_RETRY_TIME_SECONDS = 60L;
    public static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSSZ").withZone(ZoneId.systemDefault());
    private volatile long lastActive;
    private volatile boolean hasBatchMessagePublished = false;
    private static final FastThreadLocal<TopicStats> threadLocalTopicStats = new FastThreadLocal<TopicStats>(){

        protected TopicStats initialValue() {
            return new TopicStats();
        }
    };
    private static final Logger log = LoggerFactory.getLogger(PersistentTopic.class);

    public PersistentTopic(String topic, ManagedLedger ledger, BrokerService brokerService) {
        this.topic = topic;
        this.ledger = ledger;
        this.brokerService = brokerService;
        this.producers = new ConcurrentOpenHashSet();
        this.subscriptions = new ConcurrentOpenHashMap();
        this.replicators = new ConcurrentOpenHashMap();
        this.isFenced = false;
        this.replicatorPrefix = brokerService.pulsar().getConfiguration().getReplicatorPrefix();
        USAGE_COUNT_UPDATER.set(this, 0L);
        for (ManagedCursor cursor : ledger.getCursors()) {
            if (cursor.getName().startsWith(this.replicatorPrefix)) {
                String localCluster = brokerService.pulsar().getConfiguration().getClusterName();
                String remoteCluster = PersistentReplicator.getRemoteCluster(cursor.getName());
                this.replicators.put((Object)remoteCluster, (Object)new PersistentReplicator(this, cursor, localCluster, remoteCluster, brokerService));
                continue;
            }
            String subscriptionName = Codec.decode((String)cursor.getName());
            this.subscriptions.put((Object)subscriptionName, (Object)new PersistentSubscription(this, subscriptionName, cursor));
            ((PersistentSubscription)this.subscriptions.get((Object)subscriptionName)).deactivateCursor();
        }
        this.lastActive = System.nanoTime();
    }

    @Override
    public void publishMessage(ByteBuf headersAndPayload, Topic.PublishCallback callback) {
        this.ledger.asyncAddEntry(headersAndPayload, (AsyncCallbacks.AddEntryCallback)this, (Object)callback);
    }

    public void addComplete(Position pos, Object ctx) {
        Topic.PublishCallback callback = (Topic.PublishCallback)ctx;
        PositionImpl position = (PositionImpl)pos;
        callback.completed(null, position.getLedgerId(), position.getEntryId());
    }

    public void addFailed(ManagedLedgerException exception, Object ctx) {
        Topic.PublishCallback callback = (Topic.PublishCallback)ctx;
        log.error("[{}] Failed to persist msg in store: {}", (Object)this.topic, (Object)exception.getMessage());
        if (exception instanceof ManagedLedgerException.ManagedLedgerTerminatedException) {
            callback.completed(new BrokerServiceException.TopicTerminatedException(exception), -1L, -1L);
        } else {
            callback.completed(new BrokerServiceException.PersistenceException(exception), -1L, -1L);
        }
        if (exception instanceof ManagedLedgerException.ManagedLedgerFencedException) {
            this.close();
        }
    }

    @Override
    public void addProducer(Producer producer) throws BrokerServiceException {
        Preconditions.checkArgument((producer.getTopic() == this ? 1 : 0) != 0);
        this.lock.readLock().lock();
        try {
            if (this.isFenced) {
                log.warn("[{}] Attempting to add producer to a fenced topic", (Object)this.topic);
                throw new BrokerServiceException.TopicFencedException("Topic is temporarily unavailable");
            }
            if (this.ledger.isTerminated()) {
                log.warn("[{}] Attempting to add producer to a terminated topic", (Object)this.topic);
                throw new BrokerServiceException.TopicTerminatedException("Topic was already terminated");
            }
            if (log.isDebugEnabled()) {
                log.debug("[{}] {} Got request to create producer ", (Object)this.topic, (Object)producer.getProducerName());
            }
            if (!this.producers.add((Object)producer)) {
                throw new BrokerServiceException.NamingException("Producer with name '" + producer.getProducerName() + "' is already connected to topic");
            }
            USAGE_COUNT_UPDATER.incrementAndGet(this);
            if (log.isDebugEnabled()) {
                log.debug("[{}] [{}] Added producer -- count: {}", new Object[]{this.topic, producer.getProducerName(), USAGE_COUNT_UPDATER.get(this)});
            }
            this.startReplProducers();
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private boolean hasLocalProducers() {
        AtomicBoolean foundLocal = new AtomicBoolean(false);
        this.producers.forEach(producer -> {
            if (!producer.isRemote()) {
                foundLocal.set(true);
            }
        });
        return foundLocal.get();
    }

    private boolean hasRemoteProducers() {
        AtomicBoolean foundRemote = new AtomicBoolean(false);
        this.producers.forEach(producer -> {
            if (producer.isRemote()) {
                foundRemote.set(true);
            }
        });
        return foundRemote.get();
    }

    public void startReplProducers() {
        try {
            Policies policies = (Policies)this.brokerService.pulsar().getConfigurationCache().policiesCache().get(AdminResource.path("policies", DestinationName.get((String)this.topic).getNamespace())).orElseThrow(() -> new KeeperException.NoNodeException());
            if (policies.replication_clusters != null) {
                TreeSet configuredClusters = Sets.newTreeSet((Iterable)policies.replication_clusters);
                this.replicators.forEach((region, replicator) -> {
                    if (configuredClusters.contains(region)) {
                        replicator.startProducer();
                    }
                });
            }
        }
        catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Error getting policies while starting repl-producers {}", (Object)this.topic, (Object)e.getMessage());
            }
            this.replicators.forEach((region, replicator) -> replicator.startProducer());
        }
    }

    public CompletableFuture<Void> stopReplProducers() {
        List closeFutures = Lists.newArrayList();
        this.replicators.forEach((region, replicator) -> {
            boolean bl = closeFutures.add(replicator.disconnect());
        });
        return FutureUtil.waitForAll((List)closeFutures);
    }

    private synchronized CompletableFuture<Void> closeReplProducersIfNoBacklog() {
        List closeFutures = Lists.newArrayList();
        this.replicators.forEach((region, replicator) -> {
            boolean bl = closeFutures.add(replicator.disconnect(true));
        });
        return FutureUtil.waitForAll((List)closeFutures);
    }

    @Override
    public void removeProducer(Producer producer) {
        Preconditions.checkArgument((producer.getTopic() == this ? 1 : 0) != 0);
        if (this.producers.remove((Object)producer)) {
            USAGE_COUNT_UPDATER.decrementAndGet(this);
            if (log.isDebugEnabled()) {
                log.debug("[{}] [{}] Removed producer -- count: {}", new Object[]{this.topic, producer.getProducerName(), USAGE_COUNT_UPDATER.get(this)});
            }
            this.lastActive = System.nanoTime();
        }
    }

    @Override
    public CompletableFuture<Consumer> subscribe(ServerCnx cnx, String subscriptionName, long consumerId, PulsarApi.CommandSubscribe.SubType subType, int priorityLevel, String consumerName, boolean isDurable, MessageId startMessageId) {
        CompletableFuture<Consumer> future = new CompletableFuture<Consumer>();
        if (StringUtils.isBlank((CharSequence)subscriptionName)) {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Empty subscription name", (Object)this.topic);
            }
            future.completeExceptionally(new BrokerServiceException.NamingException("Empty subscription name"));
            return future;
        }
        if (this.hasBatchMessagePublished && !cnx.isBatchMessageCompatibleVersion()) {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Consumer doesn't support batch-message {}", (Object)this.topic, (Object)subscriptionName);
            }
            future.completeExceptionally(new BrokerServiceException.UnsupportedVersionException("Consumer doesn't support batch-message"));
            return future;
        }
        if (subscriptionName.startsWith(this.replicatorPrefix)) {
            log.warn("[{}] Failed to create subscription for {}", (Object)this.topic, (Object)subscriptionName);
            future.completeExceptionally(new BrokerServiceException.NamingException("Subscription with reserved subscription name attempted"));
            return future;
        }
        this.lock.readLock().lock();
        try {
            if (this.isFenced) {
                log.warn("[{}] Attempting to subscribe to a fenced topic", (Object)this.topic);
                future.completeExceptionally(new BrokerServiceException.TopicFencedException("Topic is temporarily unavailable"));
                CompletableFuture<Consumer> completableFuture = future;
                return completableFuture;
            }
            USAGE_COUNT_UPDATER.incrementAndGet(this);
            if (log.isDebugEnabled()) {
                log.debug("[{}] [{}] [{}] Added consumer -- count: {}", new Object[]{this.topic, subscriptionName, consumerName, USAGE_COUNT_UPDATER.get(this)});
            }
        }
        finally {
            this.lock.readLock().unlock();
        }
        CompletableFuture<Subscription> subscriptionFuture = isDurable ? this.getDurableSubscription(subscriptionName) : this.getNonDurableSubscription(subscriptionName, startMessageId);
        int maxUnackedMessages = isDurable ? this.brokerService.pulsar().getConfiguration().getMaxUnackedMessagesPerConsumer() : 0;
        ((CompletableFuture)subscriptionFuture.thenAccept(subscription -> {
            try {
                Consumer consumer = new Consumer((Subscription)subscription, subType, consumerId, priorityLevel, consumerName, maxUnackedMessages, cnx, cnx.getRole());
                subscription.addConsumer(consumer);
                if (!cnx.isActive()) {
                    consumer.close();
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] [{}] [{}] Subscribe failed -- count: {}", new Object[]{this.topic, subscriptionName, consumer.consumerName(), USAGE_COUNT_UPDATER.get(this)});
                    }
                    future.completeExceptionally(new BrokerServiceException("Connection was closed while the opening the cursor "));
                } else {
                    log.info("[{}][{}] Created new subscription for {}", new Object[]{this.topic, subscriptionName, consumerId});
                    future.complete(consumer);
                }
            }
            catch (BrokerServiceException e) {
                if (e instanceof BrokerServiceException.ConsumerBusyException) {
                    log.warn("[{}][{}] Consumer {} {} already connected", new Object[]{this.topic, subscriptionName, consumerId, consumerName});
                } else if (e instanceof BrokerServiceException.SubscriptionBusyException) {
                    log.warn("[{}][{}] {}", new Object[]{this.topic, subscriptionName, e.getMessage()});
                }
                USAGE_COUNT_UPDATER.decrementAndGet(this);
                future.completeExceptionally(e);
            }
        })).exceptionally(ex -> {
            log.warn("[{}] Failed to create subscription for {}: ", new Object[]{this.topic, subscriptionName, ex.getMessage()});
            USAGE_COUNT_UPDATER.decrementAndGet(this);
            future.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)ex));
            return null;
        });
        return future;
    }

    private CompletableFuture<Subscription> getDurableSubscription(final String subscriptionName) {
        final CompletableFuture<Subscription> subscriptionFuture = new CompletableFuture<Subscription>();
        this.ledger.asyncOpenCursor(Codec.encode((String)subscriptionName), new AsyncCallbacks.OpenCursorCallback(){

            public void openCursorComplete(ManagedCursor cursor, Object ctx) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}][{}] Opened cursor", (Object)PersistentTopic.this.topic, (Object)subscriptionName);
                }
                subscriptionFuture.complete((Subscription)PersistentTopic.this.subscriptions.computeIfAbsent((Object)subscriptionName, name -> new PersistentSubscription(PersistentTopic.this, subscriptionName, cursor)));
            }

            public void openCursorFailed(ManagedLedgerException exception, Object ctx) {
                log.warn("[{}] Failed to create subscription for {}", (Object)PersistentTopic.this.topic, (Object)subscriptionName);
                USAGE_COUNT_UPDATER.decrementAndGet(PersistentTopic.this);
                subscriptionFuture.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)exception));
            }
        }, null);
        return subscriptionFuture;
    }

    private CompletableFuture<? extends Subscription> getNonDurableSubscription(String subscriptionName, MessageId startMessageId) {
        CompletableFuture<Subscription> subscriptionFuture = new CompletableFuture<Subscription>();
        Subscription subscription = (Subscription)this.subscriptions.computeIfAbsent((Object)subscriptionName, name -> {
            MessageIdImpl msgId = startMessageId != null ? (MessageIdImpl)startMessageId : (MessageIdImpl)MessageId.latest;
            PositionImpl startPosition = new PositionImpl(msgId.getLedgerId(), msgId.getEntryId());
            ManagedCursor cursor = null;
            try {
                cursor = this.ledger.newNonDurableCursor((Position)startPosition);
            }
            catch (ManagedLedgerException e) {
                subscriptionFuture.completeExceptionally(e);
            }
            return new PersistentSubscription(this, subscriptionName, cursor);
        });
        if (!subscriptionFuture.isDone()) {
            subscriptionFuture.complete(subscription);
        } else {
            this.subscriptions.remove((Object)subscriptionName);
        }
        return subscriptionFuture;
    }

    @Override
    public CompletableFuture<Subscription> createSubscription(String subscriptionName) {
        return this.getDurableSubscription(subscriptionName);
    }

    @Override
    public CompletableFuture<Void> unsubscribe(final String subscriptionName) {
        final CompletableFuture<Void> unsubscribeFuture = new CompletableFuture<Void>();
        this.ledger.asyncDeleteCursor(Codec.encode((String)subscriptionName), new AsyncCallbacks.DeleteCursorCallback(){

            public void deleteCursorComplete(Object ctx) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}][{}] Cursor deleted successfully", (Object)PersistentTopic.this.topic, (Object)subscriptionName);
                }
                PersistentTopic.this.subscriptions.remove((Object)subscriptionName);
                unsubscribeFuture.complete(null);
                PersistentTopic.this.lastActive = System.nanoTime();
            }

            public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}][{}] Error deleting cursor for subscription", new Object[]{PersistentTopic.this.topic, subscriptionName, exception});
                }
                unsubscribeFuture.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)exception));
            }
        }, null);
        return unsubscribeFuture;
    }

    void removeSubscription(String subscriptionName) {
        this.subscriptions.remove((Object)subscriptionName);
    }

    @Override
    public CompletableFuture<Void> delete() {
        return this.delete(false);
    }

    private CompletableFuture<Void> delete(boolean failIfHasSubscriptions) {
        final CompletableFuture<Void> deleteFuture = new CompletableFuture<Void>();
        this.lock.writeLock().lock();
        try {
            if (this.isFenced) {
                log.warn("[{}] Topic is already being closed or deleted", (Object)this.topic);
                deleteFuture.completeExceptionally(new BrokerServiceException.TopicFencedException("Topic is already fenced"));
                CompletableFuture<Void> completableFuture = deleteFuture;
                return completableFuture;
            }
            if (USAGE_COUNT_UPDATER.get(this) == 0L) {
                this.isFenced = true;
                List futures = Lists.newArrayList();
                if (failIfHasSubscriptions) {
                    if (!this.subscriptions.isEmpty()) {
                        this.isFenced = false;
                        deleteFuture.completeExceptionally(new BrokerServiceException.TopicBusyException("Topic has subscriptions"));
                        CompletableFuture<Void> completableFuture = deleteFuture;
                        return completableFuture;
                    }
                } else {
                    this.subscriptions.forEach((s, sub) -> {
                        boolean bl = futures.add(sub.delete());
                    });
                }
                FutureUtil.waitForAll((List)futures).whenComplete((v, ex) -> {
                    if (ex != null) {
                        log.error("[{}] Error deleting topic", (Object)this.topic, ex);
                        this.isFenced = false;
                        deleteFuture.completeExceptionally((Throwable)ex);
                    } else {
                        this.ledger.asyncDelete(new AsyncCallbacks.DeleteLedgerCallback(){

                            public void deleteLedgerComplete(Object ctx) {
                                PersistentTopic.this.brokerService.removeTopicFromCache(PersistentTopic.this.topic);
                                log.info("[{}] Topic deleted", (Object)PersistentTopic.this.topic);
                                deleteFuture.complete(null);
                            }

                            public void deleteLedgerFailed(ManagedLedgerException exception, Object ctx) {
                                PersistentTopic.this.isFenced = false;
                                log.error("[{}] Error deleting topic", (Object)PersistentTopic.this.topic, (Object)exception);
                                deleteFuture.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)exception));
                            }
                        }, null);
                    }
                });
            } else {
                deleteFuture.completeExceptionally(new BrokerServiceException.TopicBusyException("Topic has " + USAGE_COUNT_UPDATER.get(this) + " connected producers/consumers"));
            }
        }
        finally {
            this.lock.writeLock().unlock();
        }
        return deleteFuture;
    }

    @Override
    public CompletableFuture<Void> close() {
        CompletableFuture<Void> closeFuture;
        block4: {
            closeFuture = new CompletableFuture<Void>();
            this.lock.writeLock().lock();
            try {
                if (!this.isFenced) {
                    this.isFenced = true;
                    break block4;
                }
                log.warn("[{}] Topic is already being closed or deleted", (Object)this.topic);
                closeFuture.completeExceptionally(new BrokerServiceException.TopicFencedException("Topic is already fenced"));
                CompletableFuture<Void> completableFuture = closeFuture;
                return completableFuture;
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
        List futures = Lists.newArrayList();
        this.replicators.forEach((cluster, replicator) -> {
            boolean bl = futures.add(replicator.disconnect());
        });
        this.producers.forEach(producer -> {
            boolean bl = futures.add(producer.disconnect());
        });
        this.subscriptions.forEach((s, sub) -> {
            boolean bl = futures.add(sub.disconnect());
        });
        ((CompletableFuture)FutureUtil.waitForAll((List)futures).thenRun(() -> this.ledger.asyncClose(new AsyncCallbacks.CloseCallback(){

            public void closeComplete(Object ctx) {
                PersistentTopic.this.brokerService.removeTopicFromCache(PersistentTopic.this.topic);
                log.info("[{}] Topic closed", (Object)PersistentTopic.this.topic);
                closeFuture.complete(null);
            }

            public void closeFailed(ManagedLedgerException exception, Object ctx) {
                log.error("[{}] Failed to close managed ledger, proceeding anyway.", (Object)PersistentTopic.this.topic, (Object)exception);
                PersistentTopic.this.brokerService.removeTopicFromCache(PersistentTopic.this.topic);
                closeFuture.complete(null);
            }
        }, null))).exceptionally(exception -> {
            log.error("[{}] Error closing topic", (Object)this.topic, exception);
            this.isFenced = false;
            closeFuture.completeExceptionally((Throwable)exception);
            return null;
        });
        return closeFuture;
    }

    private CompletableFuture<Void> checkReplicationAndRetryOnFailure() {
        CompletableFuture<Void> result = new CompletableFuture<Void>();
        ((CompletableFuture)this.checkReplication().thenAccept(res -> {
            log.info("[{}] Policies updated successfully", (Object)this.topic);
            result.complete(null);
        })).exceptionally(th -> {
            log.error("[{}] Policies update failed {}, scheduled retry in {} seconds", new Object[]{this.topic, th.getMessage(), 60L, th});
            this.brokerService.executor().schedule(this::checkReplicationAndRetryOnFailure, 60L, TimeUnit.SECONDS);
            result.completeExceptionally((Throwable)th);
            return null;
        });
        return result;
    }

    @Override
    public CompletableFuture<Void> checkReplication() {
        DestinationName name = DestinationName.get((String)this.topic);
        if (!name.isGlobal()) {
            return CompletableFuture.completedFuture(null);
        }
        if (log.isDebugEnabled()) {
            log.debug("[{}] Checking replication status", (Object)name);
        }
        Policies policies = null;
        try {
            policies = (Policies)this.brokerService.pulsar().getConfigurationCache().policiesCache().get(AdminResource.path("policies", name.getNamespace())).orElseThrow(() -> new KeeperException.NoNodeException());
        }
        catch (Exception e) {
            CompletableFuture<Void> future = new CompletableFuture<Void>();
            future.completeExceptionally(new BrokerServiceException.ServerMetadataException(e));
            return future;
        }
        int newMessageTTLinSeconds = policies.message_ttl_in_seconds;
        TreeSet configuredClusters = policies.replication_clusters != null ? Sets.newTreeSet((Iterable)policies.replication_clusters) : Collections.emptySet();
        String localCluster = this.brokerService.pulsar().getConfiguration().getClusterName();
        List futures = Lists.newArrayList();
        for (String cluster2 : configuredClusters) {
            if (cluster2.equals(localCluster) || this.replicators.containsKey((Object)cluster2)) continue;
            futures.add(this.startReplicator(cluster2));
        }
        this.replicators.forEach((cluster, replicator) -> {
            ((PersistentReplicator)replicator).updateMessageTTL(newMessageTTLinSeconds);
            if (!cluster.equals(localCluster) && !configuredClusters.contains(cluster)) {
                futures.add(this.removeReplicator((String)cluster));
            }
        });
        return FutureUtil.waitForAll((List)futures);
    }

    @Override
    public void checkMessageExpiry() {
        block3: {
            DestinationName name = DestinationName.get((String)this.topic);
            try {
                Policies policies = (Policies)this.brokerService.pulsar().getConfigurationCache().policiesCache().get(AdminResource.path("policies", name.getNamespace())).orElseThrow(() -> new KeeperException.NoNodeException());
                if (policies.message_ttl_in_seconds != 0) {
                    this.subscriptions.forEach((subName, sub) -> sub.expireMessages(policies.message_ttl_in_seconds));
                    this.replicators.forEach((region, replicator) -> ((PersistentReplicator)replicator).expireMessages(policies.message_ttl_in_seconds));
                }
            }
            catch (Exception exception) {
                if (!log.isDebugEnabled()) break block3;
                log.debug("[{}] Error getting policies", (Object)this.topic);
            }
        }
    }

    CompletableFuture<Void> startReplicator(final String remoteCluster) {
        log.info("[{}] Starting replicator to remote: {}", (Object)this.topic, (Object)remoteCluster);
        final CompletableFuture<Void> future = new CompletableFuture<Void>();
        String name = PersistentReplicator.getReplicatorName(this.replicatorPrefix, remoteCluster);
        this.ledger.asyncOpenCursor(name, new AsyncCallbacks.OpenCursorCallback(){

            public void openCursorComplete(ManagedCursor cursor, Object ctx) {
                String localCluster = PersistentTopic.this.brokerService.pulsar().getConfiguration().getClusterName();
                PersistentTopic.this.replicators.computeIfAbsent((Object)remoteCluster, r -> new PersistentReplicator(PersistentTopic.this, cursor, localCluster, remoteCluster, PersistentTopic.this.brokerService));
                future.complete(null);
            }

            public void openCursorFailed(ManagedLedgerException exception, Object ctx) {
                future.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)exception));
            }
        }, null);
        return future;
    }

    CompletableFuture<Void> removeReplicator(final String remoteCluster) {
        log.info("[{}] Removing replicator to {}", (Object)this.topic, (Object)remoteCluster);
        final CompletableFuture<Void> future = new CompletableFuture<Void>();
        final String name = PersistentReplicator.getReplicatorName(this.replicatorPrefix, remoteCluster);
        ((CompletableFuture)((Replicator)this.replicators.get((Object)remoteCluster)).disconnect().thenRun(() -> this.ledger.asyncDeleteCursor(name, new AsyncCallbacks.DeleteCursorCallback(){

            public void deleteCursorComplete(Object ctx) {
                PersistentTopic.this.replicators.remove((Object)remoteCluster);
                future.complete(null);
            }

            public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                log.error("[{}] Failed to delete cursor {} {}", new Object[]{PersistentTopic.this.topic, name, exception.getMessage(), exception});
                future.completeExceptionally(new BrokerServiceException.PersistenceException((Throwable)exception));
            }
        }, null))).exceptionally(e -> {
            log.error("[{}] Failed to close replication producer {} {}", new Object[]{this.topic, name, e.getMessage(), e});
            future.completeExceptionally((Throwable)e);
            return null;
        });
        return future;
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add("topic", (Object)this.topic).toString();
    }

    @Override
    public ConcurrentOpenHashSet<Producer> getProducers() {
        return this.producers;
    }

    public ConcurrentOpenHashMap<String, PersistentSubscription> getSubscriptions() {
        return this.subscriptions;
    }

    @Override
    public PersistentSubscription getSubscription(String subscriptionName) {
        return (PersistentSubscription)this.subscriptions.get((Object)subscriptionName);
    }

    public BrokerService getBrokerService() {
        return this.brokerService;
    }

    public ConcurrentOpenHashMap<String, Replicator> getReplicators() {
        return this.replicators;
    }

    public Replicator getPersistentReplicator(String remoteCluster) {
        return (Replicator)this.replicators.get((Object)remoteCluster);
    }

    @Override
    public String getName() {
        return this.topic;
    }

    public ManagedLedger getManagedLedger() {
        return this.ledger;
    }

    @Override
    public void updateRates(NamespaceStats nsStats, NamespaceBundleStats bundleStats, StatsOutputStream destStatsStream, ClusterReplicationMetrics replStats, String namespace) {
        TopicStats topicStats = (TopicStats)threadLocalTopicStats.get();
        topicStats.reset();
        this.replicators.forEach((region, replicator) -> replicator.updateRates());
        nsStats.producerCount = (int)((long)nsStats.producerCount + this.producers.size());
        bundleStats.producerCount = (int)((long)bundleStats.producerCount + this.producers.size());
        destStatsStream.startObject(this.topic);
        this.producers.forEach(producer -> {
            producer.updateRates();
            PublisherStats publisherStats = producer.getStats();
            topicStats.aggMsgRateIn += publisherStats.msgRateIn;
            topicStats.aggMsgThroughputIn += publisherStats.msgThroughputIn;
            if (producer.isRemote()) {
                topicStats.remotePublishersStats.put((Object)producer.getRemoteCluster(), (Object)publisherStats);
            }
        });
        destStatsStream.startList("publishers");
        destStatsStream.endList();
        destStatsStream.startObject("replication");
        nsStats.replicatorCount += topicStats.remotePublishersStats.size();
        this.replicators.forEach((cluster, replicator) -> {
            ((PersistentReplicator)replicator).updateCursorState();
            ReplicatorStats rStat = replicator.getStats();
            PublisherStats pubStats = (PublisherStats)topicStats.remotePublishersStats.get((Object)replicator.getRemoteCluster());
            if (pubStats != null) {
                rStat.msgRateIn = pubStats.msgRateIn;
                rStat.msgThroughputIn = pubStats.msgThroughputIn;
                rStat.inboundConnection = pubStats.address;
                rStat.inboundConnectedSince = pubStats.connectedSince;
            }
            topicStats.aggMsgRateOut += rStat.msgRateOut;
            topicStats.aggMsgThroughputOut += rStat.msgThroughputOut;
            destStatsStream.startObject((String)cluster);
            destStatsStream.writePair("connected", rStat.connected);
            destStatsStream.writePair("msgRateExpired", rStat.msgRateExpired);
            destStatsStream.writePair("msgRateIn", rStat.msgRateIn);
            destStatsStream.writePair("msgRateOut", rStat.msgRateOut);
            destStatsStream.writePair("msgThroughputIn", rStat.msgThroughputIn);
            destStatsStream.writePair("msgThroughputOut", rStat.msgThroughputOut);
            destStatsStream.writePair("replicationBacklog", rStat.replicationBacklog);
            destStatsStream.writePair("replicationDelayInSeconds", rStat.replicationDelayInSeconds);
            destStatsStream.writePair("inboundConnection", rStat.inboundConnection);
            destStatsStream.writePair("inboundConnectedSince", rStat.inboundConnectedSince);
            destStatsStream.writePair("outboundConnection", rStat.outboundConnection);
            destStatsStream.writePair("outboundConnectedSince", rStat.outboundConnectedSince);
            destStatsStream.endObject();
            namespaceStats.msgReplBacklog += (double)rStat.replicationBacklog;
            if (replStats.isMetricsEnabled()) {
                String namespaceClusterKey = replStats.getKeyName(namespace, (String)cluster);
                ReplicationMetrics replicationMetrics = replStats.get(namespaceClusterKey);
                boolean update = false;
                if (replicationMetrics == null) {
                    replicationMetrics = ReplicationMetrics.get();
                    update = true;
                }
                replicationMetrics.connected = replicationMetrics.connected + (rStat.connected ? 1 : 0);
                replicationMetrics.msgRateOut += rStat.msgRateOut;
                replicationMetrics.msgThroughputOut += rStat.msgThroughputOut;
                replicationMetrics.msgReplBacklog += (double)rStat.replicationBacklog;
                if (update) {
                    replStats.put(namespaceClusterKey, replicationMetrics);
                }
            }
        });
        destStatsStream.endObject();
        destStatsStream.startObject("subscriptions");
        nsStats.subsCount = (int)((long)nsStats.subsCount + this.subscriptions.size());
        this.subscriptions.forEach((subscriptionName, subscription) -> {
            double subMsgRateOut = 0.0;
            double subMsgThroughputOut = 0.0;
            double subMsgRateRedeliver = 0.0;
            try {
                destStatsStream.startObject((String)subscriptionName);
                T[] consumers = ((org.apache.pulsar.utils.CopyOnWriteArrayList)subscription.getConsumers()).array();
                namespaceStats.consumerCount += consumers.length;
                namespaceBundleStats.consumerCount += consumers.length;
                destStatsStream.startList("consumers");
                T[] TArray = consumers;
                int n = consumers.length;
                int n2 = 0;
                while (n2 < n) {
                    Object consumerObj = TArray[n2];
                    Consumer consumer = (Consumer)consumerObj;
                    consumer.updateRates();
                    ConsumerStats consumerStats = consumer.getStats();
                    subMsgRateOut += consumerStats.msgRateOut;
                    subMsgThroughputOut += consumerStats.msgThroughputOut;
                    subMsgRateRedeliver += consumerStats.msgRateRedeliver;
                    destStatsStream.startObject();
                    destStatsStream.writePair("address", consumerStats.address);
                    destStatsStream.writePair("consumerName", consumerStats.consumerName);
                    destStatsStream.writePair("availablePermits", consumerStats.availablePermits);
                    destStatsStream.writePair("connectedSince", consumerStats.connectedSince);
                    destStatsStream.writePair("msgRateOut", consumerStats.msgRateOut);
                    destStatsStream.writePair("msgThroughputOut", consumerStats.msgThroughputOut);
                    destStatsStream.writePair("msgRateRedeliver", consumerStats.msgRateRedeliver);
                    if (PulsarApi.CommandSubscribe.SubType.Shared.equals((Object)subscription.getType())) {
                        destStatsStream.writePair("unackedMessages", consumerStats.unackedMessages);
                        destStatsStream.writePair("blockedConsumerOnUnackedMsgs", consumerStats.blockedConsumerOnUnackedMsgs);
                    }
                    if (consumerStats.clientVersion != null) {
                        destStatsStream.writePair("clientVersion", consumerStats.clientVersion);
                    }
                    destStatsStream.endObject();
                    ++n2;
                }
                destStatsStream.endList();
                destStatsStream.writePair("msgBacklog", subscription.getNumberOfEntriesInBacklog());
                destStatsStream.writePair("msgRateExpired", subscription.getExpiredMessageRate());
                destStatsStream.writePair("msgRateOut", subMsgRateOut);
                destStatsStream.writePair("msgThroughputOut", subMsgThroughputOut);
                destStatsStream.writePair("msgRateRedeliver", subMsgRateRedeliver);
                destStatsStream.writePair("type", subscription.getTypeString());
                if (PulsarApi.CommandSubscribe.SubType.Shared.equals((Object)subscription.getType()) && subscription.getDispatcher() instanceof PersistentDispatcherMultipleConsumers) {
                    PersistentDispatcherMultipleConsumers dispatcher = (PersistentDispatcherMultipleConsumers)subscription.getDispatcher();
                    destStatsStream.writePair("blockedSubscriptionOnUnackedMsgs", dispatcher.isBlockedDispatcherOnUnackedMsgs());
                    destStatsStream.writePair("unackedMessages", dispatcher.getTotalUnackedMessages());
                }
                destStatsStream.endObject();
                topicStats.aggMsgRateOut += subMsgRateOut;
                topicStats.aggMsgThroughputOut += subMsgThroughputOut;
                namespaceStats.msgBacklog += (double)subscription.getNumberOfEntriesInBacklog();
            }
            catch (Exception e) {
                log.error("Got exception when creating consumer stats for subscription {}: {}", new Object[]{subscriptionName, e.getMessage(), e});
            }
        });
        destStatsStream.endObject();
        topicStats.averageMsgSize = topicStats.aggMsgRateIn == 0.0 ? 0.0 : topicStats.aggMsgThroughputIn / topicStats.aggMsgRateIn;
        destStatsStream.writePair("producerCount", this.producers.size());
        destStatsStream.writePair("averageMsgSize", topicStats.averageMsgSize);
        destStatsStream.writePair("msgRateIn", topicStats.aggMsgRateIn);
        destStatsStream.writePair("msgRateOut", topicStats.aggMsgRateOut);
        destStatsStream.writePair("msgThroughputIn", topicStats.aggMsgThroughputIn);
        destStatsStream.writePair("msgThroughputOut", topicStats.aggMsgThroughputOut);
        destStatsStream.writePair("storageSize", this.ledger.getEstimatedBacklogSize());
        destStatsStream.writePair("pendingAddEntriesCount", ((ManagedLedgerImpl)this.ledger).getPendingAddEntriesCount());
        nsStats.msgRateIn += topicStats.aggMsgRateIn;
        nsStats.msgRateOut += topicStats.aggMsgRateOut;
        nsStats.msgThroughputIn += topicStats.aggMsgThroughputIn;
        nsStats.msgThroughputOut += topicStats.aggMsgThroughputOut;
        nsStats.storageSize += (double)this.ledger.getEstimatedBacklogSize();
        bundleStats.msgRateIn += topicStats.aggMsgRateIn;
        bundleStats.msgRateOut += topicStats.aggMsgRateOut;
        bundleStats.msgThroughputIn += topicStats.aggMsgThroughputIn;
        bundleStats.msgThroughputOut += topicStats.aggMsgThroughputOut;
        bundleStats.cacheSize += ((ManagedLedgerImpl)this.ledger).getCacheSize();
        destStatsStream.endObject();
    }

    @Override
    public PersistentTopicStats getStats() {
        PersistentTopicStats stats = new PersistentTopicStats();
        ObjectObjectHashMap remotePublishersStats = new ObjectObjectHashMap();
        this.producers.forEach(producer -> {
            PublisherStats publisherStats = producer.getStats();
            persistentTopicStats.msgRateIn += publisherStats.msgRateIn;
            persistentTopicStats.msgThroughputIn += publisherStats.msgThroughputIn;
            if (producer.isRemote()) {
                remotePublishersStats.put((Object)producer.getRemoteCluster(), (Object)publisherStats);
            } else {
                persistentTopicStats.publishers.add(publisherStats);
            }
        });
        stats.averageMsgSize = stats.msgRateIn == 0.0 ? 0.0 : stats.msgThroughputIn / stats.msgRateIn;
        this.subscriptions.forEach((name, subscription) -> {
            SubscriptionStats subStats = subscription.getStats();
            persistentTopicStats.msgRateOut += subStats.msgRateOut;
            persistentTopicStats.msgThroughputOut += subStats.msgThroughputOut;
            persistentTopicStats.subscriptions.put(name, subStats);
        });
        this.replicators.forEach((cluster, replicator) -> {
            ReplicatorStats replicatorStats = replicator.getStats();
            PublisherStats pubStats = (PublisherStats)remotePublishersStats.get((Object)replicator.getRemoteCluster());
            if (pubStats != null) {
                replicatorStats.msgRateIn = pubStats.msgRateIn;
                replicatorStats.msgThroughputIn = pubStats.msgThroughputIn;
                replicatorStats.inboundConnection = pubStats.address;
                replicatorStats.inboundConnectedSince = pubStats.connectedSince;
            }
            persistentTopicStats.msgRateOut += replicatorStats.msgRateOut;
            persistentTopicStats.msgThroughputOut += replicatorStats.msgThroughputOut;
            persistentTopicStats.replication.put(replicator.getRemoteCluster(), replicatorStats);
        });
        stats.storageSize = this.ledger.getEstimatedBacklogSize();
        return stats;
    }

    @Override
    public PersistentTopicInternalStats getInternalStats() {
        PersistentTopicInternalStats stats = new PersistentTopicInternalStats();
        ManagedLedgerImpl ml = (ManagedLedgerImpl)this.ledger;
        stats.entriesAddedCounter = ml.getEntriesAddedCounter();
        stats.numberOfEntries = ml.getNumberOfEntries();
        stats.totalSize = ml.getTotalSize();
        stats.currentLedgerEntries = ml.getCurrentLedgerEntries();
        stats.currentLedgerSize = ml.getCurrentLedgerSize();
        stats.lastLedgerCreatedTimestamp = DATE_FORMAT.format(Instant.ofEpochMilli(ml.getLastLedgerCreatedTimestamp()));
        if (ml.getLastLedgerCreationFailureTimestamp() != 0L) {
            stats.lastLedgerCreationFailureTimestamp = DATE_FORMAT.format(Instant.ofEpochMilli(ml.getLastLedgerCreationFailureTimestamp()));
        }
        stats.waitingCursorsCount = ml.getWaitingCursorsCount();
        stats.pendingAddEntriesCount = ml.getPendingAddEntriesCount();
        stats.lastConfirmedEntry = ml.getLastConfirmedEntry().toString();
        stats.state = ml.getState().toString();
        stats.ledgers = Lists.newArrayList();
        ml.getLedgersInfo().forEach((id, li) -> {
            PersistentTopicInternalStats.LedgerInfo info = new PersistentTopicInternalStats.LedgerInfo();
            info.ledgerId = li.getLedgerId();
            info.entries = li.getEntries();
            info.size = li.getSize();
            persistentTopicInternalStats.ledgers.add(info);
        });
        stats.cursors = Maps.newTreeMap();
        ml.getCursors().forEach(c -> {
            ManagedCursorImpl cursor = (ManagedCursorImpl)c;
            PersistentTopicInternalStats.CursorStats cs = new PersistentTopicInternalStats.CursorStats();
            cs.markDeletePosition = cursor.getMarkDeletedPosition().toString();
            cs.readPosition = cursor.getReadPosition().toString();
            cs.waitingReadOp = cursor.hasPendingReadRequest();
            cs.pendingReadOps = cursor.getPendingReadOpsCount();
            cs.messagesConsumedCounter = cursor.getMessagesConsumedCounter();
            cs.cursorLedger = cursor.getCursorLedger();
            cs.cursorLedgerLastEntry = cursor.getCursorLedgerLastEntry();
            cs.individuallyDeletedMessages = cursor.getIndividuallyDeletedMessages();
            cs.lastLedgerSwitchTimestamp = DATE_FORMAT.format(Instant.ofEpochMilli(cursor.getLastLedgerSwitchTimestamp()));
            cs.state = cursor.getState();
            persistentTopicInternalStats.cursors.put(cursor.getName(), cs);
        });
        return stats;
    }

    public long getBacklogSize() {
        return this.ledger.getEstimatedBacklogSize();
    }

    public boolean isActive() {
        if (DestinationName.get((String)this.topic).isGlobal()) {
            return !this.subscriptions.isEmpty() || this.hasLocalProducers();
        }
        return USAGE_COUNT_UPDATER.get(this) != 0L || !this.subscriptions.isEmpty();
    }

    @Override
    public void checkGC(int gcIntervalInSeconds) {
        if (this.isActive()) {
            this.lastActive = System.nanoTime();
        } else if (System.nanoTime() - this.lastActive > TimeUnit.SECONDS.toNanos(gcIntervalInSeconds)) {
            CompletableFuture replCloseFuture = new CompletableFuture();
            if (DestinationName.get((String)this.topic).isGlobal()) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Global topic inactive for {} seconds, closing repl producers.", (Object)this.topic, (Object)gcIntervalInSeconds);
                }
                ((CompletableFuture)this.closeReplProducersIfNoBacklog().thenRun(() -> {
                    if (this.hasRemoteProducers()) {
                        if (log.isDebugEnabled()) {
                            log.debug("[{}] Global topic has connected remote producers. Not a candidate for GC", (Object)this.topic);
                        }
                        replCloseFuture.completeExceptionally(new BrokerServiceException.TopicBusyException("Topic has connected remote producers"));
                    } else {
                        log.info("[{}] Global topic inactive for {} seconds, closed repl producers", (Object)this.topic, (Object)gcIntervalInSeconds);
                        replCloseFuture.complete(null);
                    }
                })).exceptionally(e -> {
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Global topic has replication backlog. Not a candidate for GC", (Object)this.topic);
                    }
                    replCloseFuture.completeExceptionally(e.getCause());
                    return null;
                });
            } else {
                replCloseFuture.complete(null);
            }
            ((CompletableFuture)((CompletableFuture)replCloseFuture.thenCompose(v -> this.delete(true))).thenRun(() -> log.info("[{}] Topic deleted successfully due to inactivity", (Object)this.topic))).exceptionally(e -> {
                if (e.getCause() instanceof BrokerServiceException.TopicBusyException) {
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Did not delete busy topic: {}", (Object)this.topic, (Object)e.getCause().getMessage());
                    }
                } else {
                    log.warn("[{}] Inactive topic deletion failed", (Object)this.topic, e);
                }
                return null;
            });
        }
    }

    @Override
    public CompletableFuture<Void> onPoliciesUpdate(Policies data) {
        this.producers.forEach(Producer::checkPermissions);
        this.subscriptions.forEach((subName, sub) -> ((CopyOnWriteArrayList)sub.getConsumers()).forEach(Consumer::checkPermissions));
        this.checkMessageExpiry();
        return this.checkReplicationAndRetryOnFailure();
    }

    @Override
    public BacklogQuota getBacklogQuota() {
        DestinationName destination = DestinationName.get((String)this.getName());
        String namespace = destination.getNamespace();
        String policyPath = AdminResource.path("policies", namespace);
        BacklogQuota backlogQuota = this.brokerService.getBacklogQuotaManager().getBacklogQuota(namespace, policyPath);
        return backlogQuota;
    }

    @Override
    public boolean isBacklogQuotaExceeded(String producerName) {
        BacklogQuota backlogQuota = this.getBacklogQuota();
        if (backlogQuota != null) {
            BacklogQuota.RetentionPolicy retentionPolicy = backlogQuota.getPolicy();
            if ((retentionPolicy == BacklogQuota.RetentionPolicy.producer_request_hold || retentionPolicy == BacklogQuota.RetentionPolicy.producer_exception) && this.brokerService.isBacklogExceeded(this)) {
                log.info("[{}] Backlog quota exceeded. Cannot create producer [{}]", (Object)this.getName(), (Object)producerName);
                return true;
            }
            return false;
        }
        return false;
    }

    public CompletableFuture<MessageId> terminate() {
        final CompletableFuture<MessageId> future = new CompletableFuture<MessageId>();
        this.ledger.asyncTerminate(new AsyncCallbacks.TerminateCallback(){

            public void terminateComplete(Position lastCommittedPosition, Object ctx) {
                PersistentTopic.this.producers.forEach(Producer::disconnect);
                PersistentTopic.this.subscriptions.forEach((name, sub) -> sub.topicTerminated());
                PositionImpl lastPosition = (PositionImpl)lastCommittedPosition;
                MessageIdImpl messageId = new MessageIdImpl(lastPosition.getLedgerId(), lastPosition.getEntryId(), -1);
                log.info("[{}] Topic terminated at {}", (Object)PersistentTopic.this.getName(), (Object)messageId);
                future.complete(messageId);
            }

            public void terminateFailed(ManagedLedgerException exception, Object ctx) {
                future.completeExceptionally((Throwable)exception);
            }
        }, null);
        return future;
    }

    public boolean isOldestMessageExpired(ManagedCursor cursor, long messageTTLInSeconds) {
        MessageImpl msg = null;
        Entry entry = null;
        boolean isOldestMessageExpired = false;
        try {
            try {
                entry = cursor.getNthEntry(1, ManagedCursor.IndividualDeletedEntries.Include);
                if (entry != null) {
                    msg = MessageImpl.deserialize((ByteBuf)entry.getDataBuffer());
                    isOldestMessageExpired = messageTTLInSeconds != 0L && System.currentTimeMillis() > msg.getPublishTime() + TimeUnit.SECONDS.toMillis((long)((double)messageTTLInSeconds * 1.5));
                }
            }
            catch (Exception e) {
                log.warn("[{}] Error while getting the oldest message", (Object)this.topic, (Object)e);
                if (entry != null) {
                    entry.release();
                }
                if (msg != null) {
                    msg.recycle();
                }
            }
        }
        finally {
            if (entry != null) {
                entry.release();
            }
            if (msg != null) {
                msg.recycle();
            }
        }
        return isOldestMessageExpired;
    }

    public CompletableFuture<Void> clearBacklog() {
        log.info("[{}] Clearing backlog on all cursors in the topic.", (Object)this.topic);
        List futures = Lists.newArrayList();
        List cursors = this.getSubscriptions().keys();
        cursors.addAll(this.getReplicators().keys());
        for (String cursor : cursors) {
            futures.add(this.clearBacklog(cursor));
        }
        return FutureUtil.waitForAll((List)futures);
    }

    public CompletableFuture<Void> clearBacklog(String cursorName) {
        log.info("[{}] Clearing backlog for cursor {} in the topic.", (Object)this.topic, (Object)cursorName);
        PersistentSubscription sub = this.getSubscription(cursorName);
        if (sub != null) {
            return sub.clearBacklog();
        }
        PersistentReplicator repl = (PersistentReplicator)this.getPersistentReplicator(cursorName);
        if (repl != null) {
            return repl.clearBacklog();
        }
        return FutureUtil.failedFuture((Throwable)new BrokerServiceException("Cursor not found"));
    }

    public void markBatchMessagePublished() {
        this.hasBatchMessagePublished = true;
    }

    private static class TopicStats {
        public double averageMsgSize;
        public double aggMsgRateIn;
        public double aggMsgThroughputIn;
        public double aggMsgRateOut;
        public double aggMsgThroughputOut;
        public final ObjectObjectHashMap<String, PublisherStats> remotePublishersStats = new ObjectObjectHashMap();

        public TopicStats() {
            this.reset();
        }

        public void reset() {
            this.averageMsgSize = 0.0;
            this.aggMsgRateIn = 0.0;
            this.aggMsgThroughputIn = 0.0;
            this.aggMsgRateOut = 0.0;
            this.aggMsgThroughputOut = 0.0;
            this.remotePublishersStats.clear();
        }
    }
}

