/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import io.netty.buffer.ByteBuf;
import io.netty.util.Recycler;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import org.apache.bookkeeper.mledger.util.Rate;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.ServerCnx;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.nonpersistent.NonPersistentTopic;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.checksum.utils.Crc32cChecksum;
import org.apache.pulsar.common.api.Commands;
import org.apache.pulsar.common.api.proto.PulsarApi;
import org.apache.pulsar.common.naming.DestinationName;
import org.apache.pulsar.common.policies.data.NonPersistentPublisherStats;
import org.apache.pulsar.common.policies.data.PublisherStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Producer {
    private final Topic topic;
    private final ServerCnx cnx;
    private final String producerName;
    private final long producerId;
    private final String appId;
    private Rate msgIn;
    private final Rate msgDrop;
    private volatile long pendingPublishAcks = 0L;
    private static final AtomicLongFieldUpdater<Producer> pendingPublishAcksUpdater = AtomicLongFieldUpdater.newUpdater(Producer.class, "pendingPublishAcks");
    private boolean isClosed = false;
    private final CompletableFuture<Void> closeFuture;
    private final PublisherStats stats;
    private final boolean isRemote;
    private final String remoteCluster;
    private final boolean isNonPersistentTopic;
    private static final Logger log = LoggerFactory.getLogger(Producer.class);

    public Producer(Topic topic, ServerCnx cnx, long producerId, String producerName, String appId) {
        this.topic = topic;
        this.cnx = cnx;
        this.producerId = producerId;
        this.producerName = (String)Preconditions.checkNotNull((Object)producerName);
        this.closeFuture = new CompletableFuture();
        this.appId = appId;
        this.msgIn = new Rate();
        this.isNonPersistentTopic = topic instanceof NonPersistentTopic;
        this.msgDrop = this.isNonPersistentTopic ? new Rate() : null;
        this.stats = this.isNonPersistentTopic ? new NonPersistentPublisherStats() : new PublisherStats();
        this.stats.address = cnx.clientAddress().toString();
        this.stats.connectedSince = PersistentTopic.DATE_FORMAT.format(Instant.now());
        this.stats.clientVersion = cnx.getClientVersion();
        this.stats.producerName = producerName;
        this.stats.producerId = producerId;
        this.isRemote = producerName.startsWith(cnx.getBrokerService().pulsar().getConfiguration().getReplicatorPrefix());
        this.remoteCluster = this.isRemote ? producerName.split("\\.")[2] : null;
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.producerName});
    }

    public boolean equals(Object obj) {
        if (obj instanceof Producer) {
            Producer other = (Producer)obj;
            return Objects.equal((Object)this.producerName, (Object)other.producerName) && Objects.equal((Object)this.topic, (Object)other.topic);
        }
        return false;
    }

    public void publishMessage(long producerId, long sequenceId, ByteBuf headersAndPayload, long batchSize) {
        if (this.isClosed) {
            this.cnx.ctx().channel().eventLoop().execute(() -> {
                this.cnx.ctx().writeAndFlush((Object)Commands.newSendError((long)producerId, (long)sequenceId, (PulsarApi.ServerError)PulsarApi.ServerError.PersistenceError, (String)"Producer is closed"));
                this.cnx.completedSendOperation(this.isNonPersistentTopic);
            });
            return;
        }
        if (!this.verifyChecksum(headersAndPayload)) {
            this.cnx.ctx().channel().eventLoop().execute(() -> {
                this.cnx.ctx().writeAndFlush((Object)Commands.newSendError((long)producerId, (long)sequenceId, (PulsarApi.ServerError)PulsarApi.ServerError.ChecksumError, (String)"Checksum failed on the broker"));
                this.cnx.completedSendOperation(this.isNonPersistentTopic);
            });
            return;
        }
        this.startPublishOperation();
        this.topic.publishMessage(headersAndPayload, MessagePublishContext.get(this, sequenceId, this.msgIn, headersAndPayload.readableBytes(), batchSize));
    }

    private boolean verifyChecksum(ByteBuf headersAndPayload) {
        if (Commands.hasChecksum((ByteBuf)headersAndPayload)) {
            int checksum = Commands.readChecksum((ByteBuf)headersAndPayload).intValue();
            int readerIndex = headersAndPayload.readerIndex();
            try {
                long computedChecksum = Crc32cChecksum.computeChecksum((ByteBuf)headersAndPayload);
                if ((long)checksum == computedChecksum) {
                    return true;
                }
                log.error("[{}] [{}] Failed to verify checksum", (Object)this.topic, (Object)this.producerName);
                return false;
            }
            finally {
                headersAndPayload.readerIndex(readerIndex);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("[{}] [{}] Payload does not have checksum to verify", (Object)this.topic, (Object)this.producerName);
        }
        return true;
    }

    private void startPublishOperation() {
        pendingPublishAcksUpdater.lazySet(this, this.pendingPublishAcks + 1L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void publishOperationCompleted() {
        long newPendingPublishAcks = this.pendingPublishAcks - 1L;
        pendingPublishAcksUpdater.lazySet(this, newPendingPublishAcks);
        if (newPendingPublishAcks == 0L && !this.closeFuture.isDone()) {
            Producer producer = this;
            synchronized (producer) {
                if (this.isClosed && !this.closeFuture.isDone()) {
                    this.closeNow();
                }
            }
        }
    }

    public void recordMessageDrop(int batchSize) {
        if (this.isNonPersistentTopic) {
            this.msgDrop.recordEvent((long)batchSize);
        }
    }

    public long getLastSequenceId() {
        if (this.isNonPersistentTopic) {
            return -1L;
        }
        return ((PersistentTopic)this.topic).getLastPublishedSequenceId(this.producerName);
    }

    public Topic getTopic() {
        return this.topic;
    }

    public String getProducerName() {
        return this.producerName;
    }

    public long getProducerId() {
        return this.producerId;
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add("topic", (Object)this.topic).add("client", (Object)this.cnx.clientAddress()).add("producerName", (Object)this.producerName).add("producerId", this.producerId).toString();
    }

    public synchronized CompletableFuture<Void> close() {
        if (log.isDebugEnabled()) {
            log.debug("Closing producer {} -- isClosed={}", (Object)this, (Object)this.isClosed);
        }
        if (!this.isClosed) {
            this.isClosed = true;
            if (log.isDebugEnabled()) {
                log.debug("Trying to close producer {} -- cnxIsActive: {} -- pendingPublishAcks: {}", new Object[]{this, this.cnx.isActive(), this.pendingPublishAcks});
            }
            if (!this.cnx.isActive() || this.pendingPublishAcks == 0L) {
                this.closeNow();
            }
        }
        return this.closeFuture;
    }

    void closeNow() {
        this.topic.removeProducer(this);
        this.cnx.removedProducer(this);
        if (log.isDebugEnabled()) {
            log.debug("Removed producer: {}", (Object)this);
        }
        this.closeFuture.complete(null);
    }

    public CompletableFuture<Void> disconnect() {
        if (!this.closeFuture.isDone()) {
            log.info("Disconnecting producer: {}", (Object)this);
            this.cnx.ctx().executor().execute(() -> {
                this.cnx.closeProducer(this);
                this.closeNow();
            });
        }
        return this.closeFuture;
    }

    public void updateRates() {
        this.msgIn.calculateRate();
        this.stats.msgRateIn = this.msgIn.getRate();
        this.stats.msgThroughputIn = this.msgIn.getValueRate();
        this.stats.averageMsgSize = this.msgIn.getAverageValue();
        if (this.isNonPersistentTopic) {
            this.msgDrop.calculateRate();
            ((NonPersistentPublisherStats)this.stats).msgDropRate = this.msgDrop.getRate();
        }
    }

    public boolean isRemote() {
        return this.isRemote;
    }

    public String getRemoteCluster() {
        return this.remoteCluster;
    }

    public PublisherStats getStats() {
        return this.stats;
    }

    public boolean isNonPersistentTopic() {
        return this.isNonPersistentTopic;
    }

    @VisibleForTesting
    long getPendingPublishAcks() {
        return this.pendingPublishAcks;
    }

    public void checkPermissions() {
        DestinationName destination = DestinationName.get((String)this.topic.getName());
        if (this.cnx.getBrokerService().getAuthorizationManager() != null) {
            try {
                if (this.cnx.getBrokerService().getAuthorizationManager().canProduce(destination, this.appId)) {
                    return;
                }
            }
            catch (Exception e) {
                log.warn("[{}] Get unexpected error while autorizing [{}]  {}", new Object[]{this.appId, this.topic.getName(), e.getMessage(), e});
            }
            log.info("[{}] is not allowed to produce from destination [{}] anymore", (Object)this.appId, (Object)this.topic.getName());
            this.disconnect();
        }
    }

    private static final class MessagePublishContext
    implements Topic.PublishContext,
    Runnable {
        private Producer producer;
        private long sequenceId;
        private long ledgerId;
        private long entryId;
        private Rate rateIn;
        private int msgSize;
        private long batchSize;
        private String originalProducerName;
        private long originalSequenceId;
        private final Recycler.Handle recyclerHandle;
        private static final Recycler<MessagePublishContext> RECYCLER = new Recycler<MessagePublishContext>(){

            protected MessagePublishContext newObject(Recycler.Handle handle) {
                return new MessagePublishContext(handle);
            }
        };

        @Override
        public String getProducerName() {
            return this.producer.getProducerName();
        }

        @Override
        public long getSequenceId() {
            return this.sequenceId;
        }

        @Override
        public void setOriginalProducerName(String originalProducerName) {
            this.originalProducerName = originalProducerName;
        }

        @Override
        public void setOriginalSequenceId(long originalSequenceId) {
            this.originalSequenceId = originalSequenceId;
        }

        @Override
        public String getOriginalProducerName() {
            return this.originalProducerName;
        }

        @Override
        public long getOriginalSequenceId() {
            return this.originalSequenceId;
        }

        @Override
        public void completed(Exception exception, long ledgerId, long entryId) {
            if (exception != null) {
                PulsarApi.ServerError serverError = exception instanceof BrokerServiceException.TopicTerminatedException ? PulsarApi.ServerError.TopicTerminatedError : PulsarApi.ServerError.PersistenceError;
                this.producer.cnx.ctx().channel().eventLoop().execute(() -> {
                    this.producer.cnx.ctx().writeAndFlush((Object)Commands.newSendError((long)this.producer.producerId, (long)this.sequenceId, (PulsarApi.ServerError)serverError, (String)exception.getMessage()));
                    this.producer.cnx.completedSendOperation(this.producer.isNonPersistentTopic);
                    this.producer.publishOperationCompleted();
                });
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] [{}] [{}] triggered send callback. cnx {}, sequenceId {}", new Object[]{this.producer.topic, this.producer.producerName, this.producer.producerId, this.producer.cnx.clientAddress(), this.sequenceId});
                }
                this.ledgerId = ledgerId;
                this.entryId = entryId;
                this.producer.cnx.ctx().channel().eventLoop().execute((Runnable)this);
            }
        }

        @Override
        public void run() {
            if (log.isDebugEnabled()) {
                log.debug("[{}] [{}] [{}] Persisted message. cnx {}, sequenceId {}", new Object[]{this.producer.topic, this.producer.producerName, this.producer.producerId, this.producer.cnx, this.sequenceId});
            }
            this.rateIn.recordMultipleEvents(this.batchSize, (long)this.msgSize);
            this.producer.cnx.ctx().writeAndFlush((Object)Commands.newSendReceipt((long)this.producer.producerId, (long)this.sequenceId, (long)this.ledgerId, (long)this.entryId), this.producer.cnx.ctx().voidPromise());
            this.producer.cnx.completedSendOperation(this.producer.isNonPersistentTopic);
            this.producer.publishOperationCompleted();
            this.recycle();
        }

        static MessagePublishContext get(Producer producer, long sequenceId, Rate rateIn, int msgSize, long batchSize) {
            MessagePublishContext callback = (MessagePublishContext)RECYCLER.get();
            callback.producer = producer;
            callback.sequenceId = sequenceId;
            callback.rateIn = rateIn;
            callback.msgSize = msgSize;
            callback.batchSize = batchSize;
            callback.originalProducerName = null;
            callback.originalSequenceId = -1L;
            return callback;
        }

        private MessagePublishContext(Recycler.Handle recyclerHandle) {
            this.recyclerHandle = recyclerHandle;
        }

        public void recycle() {
            this.producer = null;
            this.sequenceId = -1L;
            this.rateIn = null;
            this.msgSize = 0;
            this.ledgerId = -1L;
            this.entryId = -1L;
            this.batchSize = 0L;
            RECYCLER.recycle((Object)this, this.recyclerHandle);
        }
    }
}

