/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.admin;

import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.google.common.collect.Maps;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.web.RestException;
import org.apache.pulsar.common.naming.NamedEntity;
import org.apache.pulsar.common.policies.data.ClusterData;
import org.apache.pulsar.common.policies.data.FailureDomain;
import org.apache.pulsar.common.policies.data.NamespaceIsolationData;
import org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicies;
import org.apache.pulsar.common.util.ObjectMapperFactory;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Path(value="/clusters")
@Api(value="/clusters", description="Cluster admin apis", tags={"clusters"})
@Produces(value={"application/json"})
public class Clusters
extends AdminResource {
    private static final Logger log = LoggerFactory.getLogger(Clusters.class);

    @GET
    @ApiOperation(value="Get the list of all the Pulsar clusters.", response=String.class, responseContainer="Set")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission")})
    public Set<String> getClusters() throws Exception {
        try {
            return this.clustersListCache().get();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters list", (Object)this.clientAppId(), (Object)e);
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}")
    @ApiOperation(value="Get the configuration data for the specified cluster.", response=ClusterData.class)
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Cluster doesn't exist")})
    public ClusterData getCluster(@PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        try {
            return (ClusterData)this.clustersCache().get(Clusters.path("clusters", cluster)).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Cluster does not exist"));
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster {}", new Object[]{this.clientAppId(), cluster, e});
            if (e instanceof RestException) {
                throw (RestException)((Object)e);
            }
            throw new RestException(e);
        }
    }

    @PUT
    @Path(value="/{cluster}")
    @ApiOperation(value="Provisions a new cluster. This operation requires Pulsar super-user privileges.", notes="The name cannot contain '/' characters.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been created"), @ApiResponse(code=403, message="You don't have admin permission to create the cluster"), @ApiResponse(code=409, message="Cluster already exists"), @ApiResponse(code=412, message="Cluster name is not valid")})
    public void createCluster(@PathParam(value="cluster") String cluster, ClusterData clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        try {
            NamedEntity.checkName((String)cluster);
            this.zkCreate(Clusters.path("clusters", cluster), Clusters.jsonMapper().writeValueAsBytes((Object)clusterData));
            log.info("[{}] Created cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {
            log.warn("[{}] Failed to create already existing cluster {}", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.CONFLICT, "Cluster already exist");
        }
        catch (IllegalArgumentException e) {
            log.warn("[{}] Failed to create cluster with invalid name {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster name is not valid");
        }
        catch (Exception e) {
            log.error("[{}] Failed to create cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}")
    @ApiOperation(value="Update the configuration for a cluster.", notes="This operation requires Pulsar super-user privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated"), @ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Cluster doesn't exist")})
    public void updateCluster(@PathParam(value="cluster") String cluster, ClusterData clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        try {
            String clusterPath = Clusters.path("clusters", cluster);
            Stat nodeStat = new Stat();
            byte[] content = this.globalZk().getData(clusterPath, null, nodeStat);
            ClusterData currentClusterData = null;
            if (content.length > 0) {
                currentClusterData = (ClusterData)Clusters.jsonMapper().readValue(content, ClusterData.class);
                currentClusterData.update(clusterData);
            } else {
                currentClusterData = clusterData;
            }
            this.globalZk().setData(clusterPath, Clusters.jsonMapper().writeValueAsBytes((Object)currentClusterData), nodeStat.getVersion());
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Updated cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/peers")
    @ApiOperation(value="Update peer-cluster-list for a cluster.", notes="This operation requires Pulsar super-user privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated"), @ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=412, message="Peer cluster doesn't exist"), @ApiResponse(code=404, message="Cluster doesn't exist")})
    public void setPeerClusterNames(@PathParam(value="cluster") String cluster, LinkedHashSet<String> peerClusterNames) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        if (peerClusterNames != null && !peerClusterNames.isEmpty()) {
            for (String peerCluster : peerClusterNames) {
                try {
                    if (cluster.equalsIgnoreCase(peerCluster)) {
                        throw new RestException(Response.Status.PRECONDITION_FAILED, String.valueOf(cluster) + " itself can't be part of peer-list");
                    }
                    this.clustersCache().get(Clusters.path("clusters", peerCluster)).orElseThrow(() -> new RestException(Response.Status.PRECONDITION_FAILED, "Peer cluster " + peerCluster + " does not exist"));
                }
                catch (RestException e) {
                    log.warn("[{}] Peer cluster doesn't exist from {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw e;
                }
                catch (Exception e) {
                    log.warn("[{}] Failed to validate peer-cluster list {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw new RestException(e);
                }
            }
        }
        try {
            String clusterPath = Clusters.path("clusters", cluster);
            Stat nodeStat = new Stat();
            byte[] content = this.globalZk().getData(clusterPath, null, nodeStat);
            ClusterData currentClusterData = (ClusterData)Clusters.jsonMapper().readValue(content, ClusterData.class);
            currentClusterData.setPeerClusterNames(peerClusterNames);
            this.globalZk().setData(clusterPath, Clusters.jsonMapper().writeValueAsBytes((Object)currentClusterData), nodeStat.getVersion());
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Successfully added peer-cluster {} for {}", new Object[]{this.clientAppId(), peerClusterNames, cluster});
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}")
    @ApiOperation(value="Delete an existing cluster")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated"), @ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Cluster doesn't exist"), @ApiResponse(code=412, message="Cluster is not empty")})
    public void deleteCluster(@PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        boolean isClusterUsed = false;
        try {
            for (String property : this.globalZk().getChildren(Clusters.path("policies"), false)) {
                if (this.globalZk().exists(Clusters.path("policies", property, cluster), false) == null) continue;
                if (this.globalZk().getChildren(Clusters.path("policies", property, cluster), false).isEmpty()) continue;
                isClusterUsed = true;
                break;
            }
            String path = Clusters.path("clusters", cluster, "namespaceIsolationPolicies");
            Optional nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(path);
            if (nsIsolationPolicies.isPresent()) {
                if (((NamespaceIsolationPolicies)nsIsolationPolicies.get()).getPolicies().isEmpty()) {
                    this.globalZk().delete(path, -1);
                    this.namespaceIsolationPoliciesCache().invalidate(path);
                } else {
                    isClusterUsed = true;
                }
            }
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster usage {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        if (isClusterUsed) {
            log.warn("[{}] Failed to delete cluster {} - Cluster not empty", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster not empty");
        }
        try {
            String clusterPath = Clusters.path("clusters", cluster);
            this.deleteFailureDomain(clusterPath);
            this.globalZk().delete(clusterPath, -1);
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Deleted cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to delete cluster {} - Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    private void deleteFailureDomain(String clusterPath) {
        try {
            String failureDomain = Clusters.joinPath(clusterPath, "failureDomain");
            if (this.globalZk().exists(failureDomain, false) == null) {
                return;
            }
            for (String domain : this.globalZk().getChildren(failureDomain, false)) {
                String domainPath = Clusters.joinPath(failureDomain, domain);
                this.globalZk().delete(domainPath, -1);
            }
            this.globalZk().delete(failureDomain, -1);
            this.failureDomainCache().clear();
            this.failureDomainListCache().clear();
        }
        catch (Exception e) {
            log.warn("Failed to delete failure-domain under cluster {}", (Object)clusterPath);
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies")
    @ApiOperation(value="Get the namespace isolation policies assigned in the cluster", response=NamespaceIsolationData.class, responseContainer="Map")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Cluster doesn't exist")})
    public Map<String, NamespaceIsolationData> getNamespaceIsolationPolicies(@PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        if (!this.clustersCache().get(Clusters.path("clusters", cluster)).isPresent()) {
            throw new RestException(Response.Status.NOT_FOUND, "Cluster " + cluster + " does not exist.");
        }
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = (NamespaceIsolationPolicies)this.namespaceIsolationPoliciesCache().get(Clusters.path("clusters", cluster, "namespaceIsolationPolicies")).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            return nsIsolationPolicies.getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Get a single namespace isolation policy assigned in the cluster", response=NamespaceIsolationData.class)
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Policy doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist")})
    public NamespaceIsolationData getNamespaceIsolationPolicy(@PathParam(value="cluster") String cluster, @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = (NamespaceIsolationPolicies)this.namespaceIsolationPoliciesCache().get(Clusters.path("clusters", cluster, "namespaceIsolationPolicies")).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            if (!nsIsolationPolicies.getPolicies().containsKey(policyName)) {
                log.info("[{}] Cannot find NamespaceIsolationPolicy {} for cluster {}", (Object)policyName, (Object)cluster);
                throw new RestException(Response.Status.NOT_FOUND, "Cannot find NamespaceIsolationPolicy " + policyName + " for cluster " + cluster);
            }
            return (NamespaceIsolationData)nsIsolationPolicies.getPolicies().get(policyName);
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Set namespace isolation policy")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or plicy is read only"), @ApiResponse(code=412, message="Cluster doesn't exist")})
    public void setNamespaceIsolationPolicy(@PathParam(value="cluster") String cluster, @PathParam(value="policyName") String policyName, NamespaceIsolationData policyData) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        try {
            policyData.validate();
            String nsIsolationPolicyPath = Clusters.path("clusters", cluster, "namespaceIsolationPolicies");
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(nsIsolationPolicyPath).orElseGet(() -> {
                try {
                    this.createZnodeIfNotExist(nsIsolationPolicyPath, Optional.of(Collections.emptyMap()));
                    return new NamespaceIsolationPolicies();
                }
                catch (InterruptedException | KeeperException e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.setPolicy(policyName, policyData);
            this.globalZk().setData(nsIsolationPolicyPath, Clusters.jsonMapper().writeValueAsBytes((Object)nsIsolationPolicies.getPolicies()), -1);
            this.namespaceIsolationPoliciesCache().invalidate(nsIsolationPolicyPath);
        }
        catch (IllegalArgumentException iae) {
            log.info("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}. Input data is invalid", new Object[]{this.clientAppId(), cluster, policyName, iae});
            String jsonInput = ObjectMapperFactory.create().writeValueAsString((Object)policyData);
            throw new RestException(Response.Status.BAD_REQUEST, "Invalid format of input policy data. policy: " + policyName + "; data: " + jsonInput);
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to update clusters/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            throw new RestException(e);
        }
    }

    private boolean createZnodeIfNotExist(String path, Optional<Object> value) throws KeeperException, InterruptedException {
        if (this.globalZk().exists(path, false) == null) {
            try {
                ZkUtils.createFullPathOptimistic((ZooKeeper)this.globalZk(), (String)path, (byte[])(value.isPresent() ? Clusters.jsonMapper().writeValueAsBytes(value.get()) : null), ZooDefs.Ids.OPEN_ACL_UNSAFE, (CreateMode)CreateMode.PERSISTENT);
                return true;
            }
            catch (KeeperException.NodeExistsException nodeExistsException) {
                if (log.isDebugEnabled()) {
                    log.debug("Other broker preempted the full path [{}] already. Continue...", (Object)path);
                }
            }
            catch (JsonGenerationException jsonGenerationException) {
            }
            catch (JsonMappingException jsonMappingException) {
            }
            catch (IOException iOException) {}
        }
        return false;
    }

    @DELETE
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Delete namespace isolation policy")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or plicy is read only"), @ApiResponse(code=412, message="Cluster doesn't exist")})
    public void deleteNamespaceIsolationPolicy(@PathParam(value="cluster") String cluster, @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        try {
            String nsIsolationPolicyPath = Clusters.path("clusters", cluster, "namespaceIsolationPolicies");
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(nsIsolationPolicyPath).orElseGet(() -> {
                try {
                    this.createZnodeIfNotExist(nsIsolationPolicyPath, Optional.of(Collections.emptyMap()));
                    return new NamespaceIsolationPolicies();
                }
                catch (InterruptedException | KeeperException e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.deletePolicy(policyName);
            this.globalZk().setData(nsIsolationPolicyPath, Clusters.jsonMapper().writeValueAsBytes((Object)nsIsolationPolicies.getPolicies()), -1);
            this.namespaceIsolationPoliciesCache().invalidate(nsIsolationPolicyPath);
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to update brokers/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update brokers/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Set cluster's failure Domain")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=409, message="Broker already exist into other domain"), @ApiResponse(code=404, message="Cluster doesn't exist")})
    public void setFailureDomain(@PathParam(value="cluster") String cluster, @PathParam(value="domainName") String domainName, FailureDomain domain) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validateBrokerExistsInOtherDomain(cluster, domainName, domain);
        try {
            String domainPath = Clusters.joinPath(this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT, domainName);
            if (this.createZnodeIfNotExist(domainPath, Optional.ofNullable(domain))) {
                this.failureDomainListCache().clear();
            } else {
                this.globalZk().setData(domainPath, Clusters.jsonMapper().writeValueAsBytes((Object)domain), -1);
                this.failureDomainCache().invalidate(domainPath);
            }
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Failed to update domain {}. clusters {}  Does not exist", new Object[]{this.clientAppId(), cluster, domainName});
            throw new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/domainName/{}", new Object[]{this.clientAppId(), cluster, domainName, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/failureDomains")
    @ApiOperation(value="Get the cluster failure domains", response=FailureDomain.class, responseContainer="Map")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission")})
    public Map<String, FailureDomain> getFailureDomains(@PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        HashMap domains = Maps.newHashMap();
        try {
            String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
            for (String domainName : this.failureDomainListCache().get()) {
                try {
                    Optional domain = this.failureDomainCache().get(Clusters.joinPath(failureDomainRootPath, domainName));
                    if (!domain.isPresent()) continue;
                    domains.put(domainName, (FailureDomain)domain.get());
                }
                catch (Exception e) {
                    log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                }
            }
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failure-domain is not configured for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            return Collections.emptyMap();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get failure-domains for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        return domains;
    }

    @GET
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Get a domain in a cluster", response=FailureDomain.class)
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="Domain doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist")})
    public FailureDomain getDomain(@PathParam(value="cluster") String cluster, @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
            return (FailureDomain)this.failureDomainCache().get(Clusters.joinPath(failureDomainRootPath, domainName)).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist"));
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get domain {} for cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Delete cluster's failure omain")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or plicy is read only"), @ApiResponse(code=412, message="Cluster doesn't exist")})
    public void deleteFailureDomain(@PathParam(value="cluster") String cluster, @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            String domainPath = Clusters.joinPath(this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT, domainName);
            this.globalZk().delete(domainPath, -1);
            this.failureDomainCache().invalidate(domainPath);
            this.failureDomainListCache().clear();
        }
        catch (KeeperException.NoNodeException noNodeException) {
            log.warn("[{}] Domain {} does not exist in {}", new Object[]{this.clientAppId(), domainName, cluster});
            throw new RestException(Response.Status.NOT_FOUND, "Domain-name " + domainName + " or cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete domain {} in cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    private void validateBrokerExistsInOtherDomain(String cluster, String inputDomainName, FailureDomain inputDomain) {
        if (inputDomain != null && inputDomain.brokers != null) {
            try {
                String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
                for (String domainName : this.failureDomainListCache().get()) {
                    if (inputDomainName.equals(domainName)) continue;
                    try {
                        List duplicateBrokers;
                        Optional domain = this.failureDomainCache().get(Clusters.joinPath(failureDomainRootPath, domainName));
                        if (!domain.isPresent() || ((FailureDomain)domain.get()).brokers == null || (duplicateBrokers = ((Stream)((FailureDomain)domain.get()).brokers.stream().parallel()).filter(inputDomain.brokers::contains).collect(Collectors.toList())).isEmpty()) continue;
                        throw new RestException(Response.Status.CONFLICT, duplicateBrokers + " already exist into " + domainName);
                    }
                    catch (Exception e) {
                        if (e instanceof RestException) {
                            throw e;
                        }
                        log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                    }
                }
            }
            catch (KeeperException.NoNodeException e) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Domain is not configured for cluster", (Object)this.clientAppId(), (Object)e);
                }
            }
            catch (Exception e) {
                log.error("[{}] Failed to get domains for cluster {}", (Object)this.clientAppId(), (Object)e);
                throw new RestException(e);
            }
        }
    }
}

