/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.scurrilous.circe.checksum.Crc32cIntChecksum;
import io.netty.buffer.ByteBuf;
import io.netty.util.Recycler;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.ServerCnx;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.TransportCnx;
import org.apache.pulsar.broker.service.nonpersistent.NonPersistentTopic;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.client.api.transaction.TxnID;
import org.apache.pulsar.common.api.proto.MessageMetadata;
import org.apache.pulsar.common.api.proto.ProducerAccessMode;
import org.apache.pulsar.common.api.proto.ServerError;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.policies.data.stats.NonPersistentPublisherStatsImpl;
import org.apache.pulsar.common.policies.data.stats.PublisherStatsImpl;
import org.apache.pulsar.common.protocol.Commands;
import org.apache.pulsar.common.protocol.schema.SchemaVersion;
import org.apache.pulsar.common.stats.Rate;
import org.apache.pulsar.common.util.DateFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Producer {
    private final Topic topic;
    private final TransportCnx cnx;
    private final String producerName;
    private final long epoch;
    private final boolean userProvidedProducerName;
    private final long producerId;
    private final String appId;
    private Rate msgIn;
    private Rate chunkedMessageRate;
    private final Rate msgDrop;
    private volatile long pendingPublishAcks = 0L;
    private static final AtomicLongFieldUpdater<Producer> pendingPublishAcksUpdater = AtomicLongFieldUpdater.newUpdater(Producer.class, "pendingPublishAcks");
    private boolean isClosed = false;
    private final CompletableFuture<Void> closeFuture;
    private final PublisherStatsImpl stats;
    private final boolean isRemote;
    private final String remoteCluster;
    private final boolean isNonPersistentTopic;
    private final boolean isEncrypted;
    private final ProducerAccessMode accessMode;
    private Optional<Long> topicEpoch;
    private final Map<String, String> metadata;
    private final SchemaVersion schemaVersion;
    private final String clientAddress;
    private final AtomicBoolean isDisconnecting = new AtomicBoolean(false);
    private static final Logger log = LoggerFactory.getLogger(Producer.class);

    public Producer(Topic topic, TransportCnx cnx, long producerId, String producerName, String appId, boolean isEncrypted, Map<String, String> metadata, SchemaVersion schemaVersion, long epoch, boolean userProvidedProducerName, ProducerAccessMode accessMode, Optional<Long> topicEpoch, boolean supportsPartialProducer) {
        ServiceConfiguration serviceConf = cnx.getBrokerService().pulsar().getConfiguration();
        this.topic = topic;
        this.cnx = cnx;
        this.producerId = producerId;
        this.producerName = (String)Preconditions.checkNotNull((Object)producerName);
        this.userProvidedProducerName = userProvidedProducerName;
        this.epoch = epoch;
        this.closeFuture = new CompletableFuture();
        this.appId = appId;
        this.msgIn = new Rate();
        this.chunkedMessageRate = new Rate();
        this.isNonPersistentTopic = topic instanceof NonPersistentTopic;
        this.msgDrop = this.isNonPersistentTopic ? new Rate() : null;
        this.metadata = metadata != null ? metadata : Collections.emptyMap();
        Object object = this.stats = this.isNonPersistentTopic ? new NonPersistentPublisherStatsImpl() : new PublisherStatsImpl();
        if (cnx.hasHAProxyMessage()) {
            this.stats.setAddress(cnx.getHAProxyMessage().sourceAddress() + ":" + cnx.getHAProxyMessage().sourcePort());
        } else {
            this.stats.setAddress(cnx.clientAddress().toString());
        }
        this.stats.setConnectedSince(DateFormatter.now());
        this.stats.setClientVersion(cnx.getClientVersion());
        this.stats.setProducerName(producerName);
        this.stats.producerId = producerId;
        if (serviceConf.isAggregatePublisherStatsByProducerName() && this.stats.getProducerName() != null) {
            this.stats.setSupportsPartialProducer(supportsPartialProducer);
        } else {
            this.stats.setSupportsPartialProducer(false);
        }
        this.stats.metadata = this.metadata;
        this.stats.accessMode = Commands.convertProducerAccessMode((ProducerAccessMode)accessMode);
        String replicatorPrefix = serviceConf.getReplicatorPrefix() + ".";
        this.isRemote = producerName.startsWith(replicatorPrefix);
        this.remoteCluster = this.parseRemoteClusterName(producerName, this.isRemote, replicatorPrefix);
        this.isEncrypted = isEncrypted;
        this.schemaVersion = schemaVersion;
        this.accessMode = accessMode;
        this.topicEpoch = topicEpoch;
        this.clientAddress = cnx.clientSourceAddress();
    }

    private String parseRemoteClusterName(String producerName, boolean isRemote, String replicatorPrefix) {
        if (isRemote) {
            String clusterName = producerName.substring(replicatorPrefix.length());
            return clusterName.contains("-->") ? clusterName.split("-->")[0] : clusterName;
        }
        return null;
    }

    public boolean isSuccessorTo(Producer other) {
        return Objects.equals(this.producerName, other.producerName) && Objects.equals(this.topic, other.topic) && this.producerId == other.producerId && Objects.equals(this.cnx, other.cnx) && other.getEpoch() < this.epoch;
    }

    public void publishMessage(long producerId, long sequenceId, ByteBuf headersAndPayload, long batchSize, boolean isChunked, boolean isMarker) {
        if (this.checkAndStartPublish(producerId, sequenceId, headersAndPayload, batchSize)) {
            this.publishMessageToTopic(headersAndPayload, sequenceId, batchSize, isChunked, isMarker);
        }
    }

    public void publishMessage(long producerId, long lowestSequenceId, long highestSequenceId, ByteBuf headersAndPayload, long batchSize, boolean isChunked, boolean isMarker) {
        if (lowestSequenceId > highestSequenceId) {
            this.cnx.execute(() -> {
                this.cnx.getCommandSender().sendSendError(producerId, highestSequenceId, ServerError.MetadataError, "Invalid lowest or highest sequence id");
                this.cnx.completedSendOperation(this.isNonPersistentTopic, headersAndPayload.readableBytes());
            });
            return;
        }
        if (this.checkAndStartPublish(producerId, highestSequenceId, headersAndPayload, batchSize)) {
            this.publishMessageToTopic(headersAndPayload, lowestSequenceId, highestSequenceId, batchSize, isChunked, isMarker);
        }
    }

    public boolean checkAndStartPublish(long producerId, long sequenceId, ByteBuf headersAndPayload, long batchSize) {
        if (this.isClosed) {
            this.cnx.execute(() -> {
                this.cnx.getCommandSender().sendSendError(producerId, sequenceId, ServerError.PersistenceError, "Producer is closed");
                this.cnx.completedSendOperation(this.isNonPersistentTopic, headersAndPayload.readableBytes());
            });
            return false;
        }
        if (!this.verifyChecksum(headersAndPayload)) {
            this.cnx.execute(() -> {
                this.cnx.getCommandSender().sendSendError(producerId, sequenceId, ServerError.ChecksumError, "Checksum failed on the broker");
                this.cnx.completedSendOperation(this.isNonPersistentTopic, headersAndPayload.readableBytes());
            });
            return false;
        }
        if (this.topic.isEncryptionRequired()) {
            headersAndPayload.markReaderIndex();
            MessageMetadata msgMetadata = Commands.parseMessageMetadata((ByteBuf)headersAndPayload);
            headersAndPayload.resetReaderIndex();
            int encryptionKeysCount = msgMetadata.getEncryptionKeysCount();
            if (encryptionKeysCount < 1) {
                log.warn("[{}] Messages must be encrypted", (Object)this.getTopic().getName());
                this.cnx.execute(() -> {
                    this.cnx.getCommandSender().sendSendError(producerId, sequenceId, ServerError.MetadataError, "Messages must be encrypted");
                    this.cnx.completedSendOperation(this.isNonPersistentTopic, headersAndPayload.readableBytes());
                });
                return false;
            }
        }
        this.startPublishOperation((int)batchSize, headersAndPayload.readableBytes());
        return true;
    }

    private void publishMessageToTopic(ByteBuf headersAndPayload, long sequenceId, long batchSize, boolean isChunked, boolean isMarker) {
        this.topic.publishMessage(headersAndPayload, MessagePublishContext.get(this, sequenceId, this.msgIn, headersAndPayload.readableBytes(), batchSize, isChunked, System.nanoTime(), isMarker));
    }

    private void publishMessageToTopic(ByteBuf headersAndPayload, long lowestSequenceId, long highestSequenceId, long batchSize, boolean isChunked, boolean isMarker) {
        this.topic.publishMessage(headersAndPayload, MessagePublishContext.get(this, lowestSequenceId, highestSequenceId, this.msgIn, headersAndPayload.readableBytes(), batchSize, isChunked, System.nanoTime(), isMarker));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean verifyChecksum(ByteBuf headersAndPayload) {
        if (Commands.hasChecksum((ByteBuf)headersAndPayload)) {
            int readerIndex = headersAndPayload.readerIndex();
            try {
                int checksum = Commands.readChecksum((ByteBuf)headersAndPayload);
                long computedChecksum = Crc32cIntChecksum.computeChecksum((ByteBuf)headersAndPayload);
                if ((long)checksum == computedChecksum) {
                    boolean bl = true;
                    return bl;
                }
                log.error("[{}] [{}] Failed to verify checksum", (Object)this.topic, (Object)this.producerName);
                boolean bl = false;
                return bl;
            }
            finally {
                headersAndPayload.readerIndex(readerIndex);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("[{}] [{}] Payload does not have checksum to verify", (Object)this.topic, (Object)this.producerName);
        }
        return true;
    }

    private void startPublishOperation(int batchSize, long msgSize) {
        pendingPublishAcksUpdater.lazySet(this, this.pendingPublishAcks + 1L);
        this.getTopic().incrementPublishCount(batchSize, msgSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void publishOperationCompleted() {
        long newPendingPublishAcks = this.pendingPublishAcks - 1L;
        pendingPublishAcksUpdater.lazySet(this, newPendingPublishAcks);
        if (newPendingPublishAcks == 0L && !this.closeFuture.isDone()) {
            Producer producer = this;
            synchronized (producer) {
                if (this.isClosed && !this.closeFuture.isDone()) {
                    this.closeNow(true);
                }
            }
        }
    }

    public void recordMessageDrop(int batchSize) {
        if (this.isNonPersistentTopic) {
            this.msgDrop.recordEvent((long)batchSize);
        }
    }

    public long getLastSequenceId() {
        if (this.isNonPersistentTopic) {
            return -1L;
        }
        return ((PersistentTopic)this.topic).getLastPublishedSequenceId(this.producerName);
    }

    public TransportCnx getCnx() {
        return this.cnx;
    }

    public Topic getTopic() {
        return this.topic;
    }

    public String getProducerName() {
        return this.producerName;
    }

    public long getProducerId() {
        return this.producerId;
    }

    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("topic", (Object)this.topic).add("client", (Object)this.cnx.clientAddress()).add("producerName", (Object)this.producerName).add("producerId", this.producerId).toString();
    }

    public synchronized CompletableFuture<Void> close(boolean removeFromTopic) {
        if (log.isDebugEnabled()) {
            log.debug("Closing producer {} -- isClosed={}", (Object)this, (Object)this.isClosed);
        }
        if (!this.isClosed) {
            this.isClosed = true;
            if (log.isDebugEnabled()) {
                log.debug("Trying to close producer {} -- cnxIsActive: {} -- pendingPublishAcks: {}", new Object[]{this, this.cnx.isActive(), this.pendingPublishAcks});
            }
            if (!this.cnx.isActive() || this.pendingPublishAcks == 0L) {
                this.closeNow(removeFromTopic);
            }
        }
        return this.closeFuture;
    }

    public void closeNow(boolean removeFromTopic) {
        if (removeFromTopic) {
            this.topic.removeProducer(this);
        }
        this.cnx.removedProducer(this);
        if (log.isDebugEnabled()) {
            log.debug("Removed producer: {}", (Object)this);
        }
        this.closeFuture.complete(null);
        this.isDisconnecting.set(false);
    }

    public CompletableFuture<Void> disconnect() {
        if (!this.closeFuture.isDone() && this.isDisconnecting.compareAndSet(false, true)) {
            log.info("Disconnecting producer: {}", (Object)this);
            this.cnx.execute(() -> {
                this.cnx.closeProducer(this);
                this.closeNow(true);
            });
        }
        return this.closeFuture;
    }

    public void updateRates() {
        this.msgIn.calculateRate();
        this.chunkedMessageRate.calculateRate();
        this.stats.msgRateIn = this.msgIn.getRate();
        this.stats.msgThroughputIn = this.msgIn.getValueRate();
        this.stats.averageMsgSize = this.msgIn.getAverageValue();
        this.stats.chunkedMessageRate = this.chunkedMessageRate.getRate();
        if (this.chunkedMessageRate.getCount() > 0L && this.topic instanceof PersistentTopic) {
            ((PersistentTopic)this.topic).msgChunkPublished = true;
        }
        if (this.isNonPersistentTopic) {
            this.msgDrop.calculateRate();
            ((NonPersistentPublisherStatsImpl)this.stats).msgDropRate = this.msgDrop.getRate();
        }
    }

    public void updateRates(int numOfMessages, long msgSizeInBytes) {
        this.msgIn.recordMultipleEvents((long)numOfMessages, msgSizeInBytes);
    }

    public boolean isRemote() {
        return this.isRemote;
    }

    public String getRemoteCluster() {
        return this.remoteCluster;
    }

    public PublisherStatsImpl getStats() {
        return this.stats;
    }

    public boolean isNonPersistentTopic() {
        return this.isNonPersistentTopic;
    }

    public long getEpoch() {
        return this.epoch;
    }

    public boolean isUserProvidedProducerName() {
        return this.userProvidedProducerName;
    }

    @VisibleForTesting
    long getPendingPublishAcks() {
        return this.pendingPublishAcks;
    }

    public CompletableFuture<Void> checkPermissionsAsync() {
        TopicName topicName = TopicName.get((String)this.topic.getName());
        if (this.cnx.getBrokerService().getAuthorizationService() != null) {
            return this.cnx.getBrokerService().getAuthorizationService().canProduceAsync(topicName, this.appId, this.cnx.getAuthenticationData()).handle((ok, ex) -> {
                if (ex != null) {
                    log.warn("[{}] Get unexpected error while autorizing [{}]  {}", new Object[]{this.appId, this.topic.getName(), ex.getMessage(), ex});
                }
                if (ok == null || !ok.booleanValue()) {
                    log.info("[{}] is not allowed to produce on topic [{}] anymore", (Object)this.appId, (Object)this.topic.getName());
                    this.disconnect();
                }
                return null;
            });
        }
        return CompletableFuture.completedFuture(null);
    }

    public void checkEncryption() {
        if (this.topic.isEncryptionRequired() && !this.isEncrypted) {
            log.info("[{}] [{}] Unencrypted producer is not allowed to produce on topic [{}] anymore", new Object[]{this.producerId, this.producerName, this.topic.getName()});
            this.disconnect();
        }
    }

    public void publishTxnMessage(TxnID txnID, long producerId, long sequenceId, long highSequenceId, ByteBuf headersAndPayload, long batchSize, boolean isChunked, boolean isMarker) {
        this.checkAndStartPublish(producerId, sequenceId, headersAndPayload, batchSize);
        this.topic.publishTxnMessage(txnID, headersAndPayload, MessagePublishContext.get(this, sequenceId, highSequenceId, this.msgIn, headersAndPayload.readableBytes(), batchSize, isChunked, System.nanoTime(), isMarker));
    }

    public SchemaVersion getSchemaVersion() {
        return this.schemaVersion;
    }

    public ProducerAccessMode getAccessMode() {
        return this.accessMode;
    }

    public Optional<Long> getTopicEpoch() {
        return this.topicEpoch;
    }

    public String getClientAddress() {
        return this.clientAddress;
    }

    public boolean isDisconnecting() {
        return this.isDisconnecting.get();
    }

    private static final class MessagePublishContext
    implements Topic.PublishContext,
    Runnable {
        Map<String, Object> propertyMap;
        private Producer producer;
        private long sequenceId;
        private long ledgerId;
        private long entryId;
        private Rate rateIn;
        private int msgSize;
        private long batchSize;
        private boolean chunked;
        private boolean isMarker;
        private long startTimeNs;
        private String originalProducerName;
        private long originalSequenceId;
        private long highestSequenceId;
        private long originalHighestSequenceId;
        private final Recycler.Handle<MessagePublishContext> recyclerHandle;
        private static final Recycler<MessagePublishContext> RECYCLER = new Recycler<MessagePublishContext>(){

            protected MessagePublishContext newObject(Recycler.Handle<MessagePublishContext> handle) {
                return new MessagePublishContext(handle);
            }
        };

        @Override
        public String getProducerName() {
            return this.producer.getProducerName();
        }

        @Override
        public long getSequenceId() {
            return this.sequenceId;
        }

        @Override
        public boolean isChunked() {
            return this.chunked;
        }

        @Override
        public void setProperty(String propertyName, Object value) {
            if (this.propertyMap == null) {
                this.propertyMap = new HashMap<String, Object>();
            }
            this.propertyMap.put(propertyName, value);
        }

        @Override
        public Object getProperty(String propertyName) {
            if (this.propertyMap != null) {
                return this.propertyMap.get(propertyName);
            }
            return null;
        }

        @Override
        public long getHighestSequenceId() {
            return this.highestSequenceId;
        }

        @Override
        public void setOriginalProducerName(String originalProducerName) {
            this.originalProducerName = originalProducerName;
        }

        @Override
        public void setOriginalSequenceId(long originalSequenceId) {
            this.originalSequenceId = originalSequenceId;
        }

        @Override
        public String getOriginalProducerName() {
            return this.originalProducerName;
        }

        @Override
        public long getOriginalSequenceId() {
            return this.originalSequenceId;
        }

        @Override
        public void setOriginalHighestSequenceId(long originalHighestSequenceId) {
            this.originalHighestSequenceId = originalHighestSequenceId;
        }

        @Override
        public long getOriginalHighestSequenceId() {
            return this.originalHighestSequenceId;
        }

        @Override
        public void completed(Exception exception, long ledgerId, long entryId) {
            if (exception != null) {
                ServerError serverError = this.getServerError(exception);
                this.producer.cnx.execute(() -> {
                    if (!(exception instanceof BrokerServiceException.TopicClosedException)) {
                        long callBackSequenceId = Math.max(this.highestSequenceId, this.sequenceId);
                        this.producer.cnx.getCommandSender().sendSendError(this.producer.producerId, callBackSequenceId, serverError, exception.getMessage());
                    }
                    this.producer.cnx.completedSendOperation(this.producer.isNonPersistentTopic, this.msgSize);
                    this.producer.publishOperationCompleted();
                    this.recycle();
                });
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] [{}] [{}] triggered send callback. cnx {}, sequenceId {}", new Object[]{this.producer.topic, this.producer.producerName, this.producer.producerId, this.producer.cnx.clientAddress(), this.sequenceId});
                }
                this.ledgerId = ledgerId;
                this.entryId = entryId;
                this.producer.cnx.execute(this);
            }
        }

        private ServerError getServerError(Exception exception) {
            ServerError serverError = exception instanceof BrokerServiceException.TopicTerminatedException ? ServerError.TopicTerminatedError : (exception instanceof BrokerServiceException.NotAllowedException ? ServerError.NotAllowedError : ServerError.PersistenceError);
            return serverError;
        }

        @Override
        public void run() {
            if (log.isDebugEnabled()) {
                log.debug("[{}] [{}] [{}] Persisted message. cnx {}, sequenceId {}", new Object[]{this.producer.topic, this.producer.producerName, this.producer.producerId, this.producer.cnx, this.sequenceId});
            }
            this.rateIn.recordMultipleEvents(this.batchSize, (long)this.msgSize);
            this.producer.topic.recordAddLatency(System.nanoTime() - this.startTimeNs, TimeUnit.NANOSECONDS);
            this.producer.cnx.getCommandSender().sendSendReceiptResponse(this.producer.producerId, this.sequenceId, this.highestSequenceId, this.ledgerId, this.entryId);
            this.producer.cnx.completedSendOperation(this.producer.isNonPersistentTopic, this.msgSize);
            if (this.chunked) {
                this.producer.chunkedMessageRate.recordEvent();
            }
            this.producer.publishOperationCompleted();
            if (this.producer.cnx.getBrokerService().getInterceptor() != null) {
                this.producer.cnx.getBrokerService().getInterceptor().messageProduced((ServerCnx)this.producer.cnx, this.producer, this.startTimeNs, this.ledgerId, this.entryId, this);
            }
            this.recycle();
        }

        static MessagePublishContext get(Producer producer, long sequenceId, Rate rateIn, int msgSize, long batchSize, boolean chunked, long startTimeNs, boolean isMarker) {
            MessagePublishContext callback = (MessagePublishContext)RECYCLER.get();
            callback.producer = producer;
            callback.sequenceId = sequenceId;
            callback.rateIn = rateIn;
            callback.msgSize = msgSize;
            callback.batchSize = batchSize;
            callback.chunked = chunked;
            callback.originalProducerName = null;
            callback.originalSequenceId = -1L;
            callback.startTimeNs = startTimeNs;
            callback.isMarker = isMarker;
            if (callback.propertyMap != null) {
                callback.propertyMap.clear();
            }
            return callback;
        }

        static MessagePublishContext get(Producer producer, long lowestSequenceId, long highestSequenceId, Rate rateIn, int msgSize, long batchSize, boolean chunked, long startTimeNs, boolean isMarker) {
            MessagePublishContext callback = (MessagePublishContext)RECYCLER.get();
            callback.producer = producer;
            callback.sequenceId = lowestSequenceId;
            callback.highestSequenceId = highestSequenceId;
            callback.rateIn = rateIn;
            callback.msgSize = msgSize;
            callback.batchSize = batchSize;
            callback.originalProducerName = null;
            callback.originalSequenceId = -1L;
            callback.startTimeNs = startTimeNs;
            callback.chunked = chunked;
            callback.isMarker = isMarker;
            if (callback.propertyMap != null) {
                callback.propertyMap.clear();
            }
            return callback;
        }

        @Override
        public long getNumberOfMessages() {
            return this.batchSize;
        }

        @Override
        public boolean isMarkerMessage() {
            return this.isMarker;
        }

        private MessagePublishContext(Recycler.Handle<MessagePublishContext> recyclerHandle) {
            this.recyclerHandle = recyclerHandle;
        }

        public void recycle() {
            this.producer = null;
            this.sequenceId = -1L;
            this.highestSequenceId = -1L;
            this.originalSequenceId = -1L;
            this.originalHighestSequenceId = -1L;
            this.rateIn = null;
            this.msgSize = 0;
            this.ledgerId = -1L;
            this.entryId = -1L;
            this.batchSize = 0L;
            this.startTimeNs = -1L;
            this.chunked = false;
            this.isMarker = false;
            if (this.propertyMap != null) {
                this.propertyMap.clear();
            }
            this.recyclerHandle.recycle((Object)this);
        }
    }
}

