/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.admin.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.Example;
import io.swagger.annotations.ExampleProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.container.AsyncResponse;
import javax.ws.rs.container.Suspended;
import javax.ws.rs.core.Response;
import org.apache.pulsar.broker.resources.ClusterResources;
import org.apache.pulsar.broker.web.PulsarWebResource;
import org.apache.pulsar.broker.web.RestException;
import org.apache.pulsar.client.admin.Namespaces;
import org.apache.pulsar.common.naming.NamedEntity;
import org.apache.pulsar.common.policies.data.BrokerNamespaceIsolationData;
import org.apache.pulsar.common.policies.data.BrokerNamespaceIsolationDataImpl;
import org.apache.pulsar.common.policies.data.ClusterData;
import org.apache.pulsar.common.policies.data.ClusterDataImpl;
import org.apache.pulsar.common.policies.data.FailureDomainImpl;
import org.apache.pulsar.common.policies.data.NamespaceIsolationData;
import org.apache.pulsar.common.policies.data.NamespaceIsolationDataImpl;
import org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicies;
import org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicyImpl;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.common.util.ObjectMapperFactory;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClustersBase
extends PulsarWebResource {
    private static final Logger log = LoggerFactory.getLogger(ClustersBase.class);

    @GET
    @ApiOperation(value="Get the list of all the Pulsar clusters.", response=String.class, responseContainer="Set")
    @ApiResponses(value={@ApiResponse(code=200, message="Return a list of clusters."), @ApiResponse(code=500, message="Internal server error.")})
    public Set<String> getClusters() throws Exception {
        try {
            Set<String> clusters = this.clusterResources().list().stream().filter(cluster -> !"global".equals(cluster)).collect(Collectors.toSet());
            return clusters;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters list", (Object)this.clientAppId(), (Object)e);
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}")
    @ApiOperation(value="Get the configuration for the specified cluster.", response=ClusterDataImpl.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=200, message="Return the cluster data.", response=ClusterDataImpl.class), @ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public ClusterData getCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        try {
            return (ClusterData)this.clusterResources().getCluster(cluster).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Cluster does not exist"));
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster {}", new Object[]{this.clientAppId(), cluster, e});
            if (e instanceof RestException) {
                throw (RestException)((Object)e);
            }
            throw new RestException(e);
        }
    }

    @PUT
    @Path(value="/{cluster}")
    @ApiOperation(value="Create a new cluster.", notes="This operation requires Pulsar superuser privileges, and the name cannot contain the '/' characters.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been created."), @ApiResponse(code=403, message="You don't have admin permission to create the cluster."), @ApiResponse(code=409, message="Cluster already exists."), @ApiResponse(code=412, message="Cluster name is not valid."), @ApiResponse(code=500, message="Internal server error.")})
    public void createCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The cluster data", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="{\n   'serviceUrl': 'http://pulsar.example.com:8080',\n   'brokerServiceUrl': 'pulsar://pulsar.example.com:6651',\n}")})) ClusterDataImpl clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        if (clusterData == null) {
            throw new RestException(Response.Status.BAD_REQUEST, "cluster data is required");
        }
        try {
            NamedEntity.checkName((String)cluster);
            if (this.clusterResources().getCluster(cluster).isPresent()) {
                log.warn("[{}] Failed to create already existing cluster {}", (Object)this.clientAppId(), (Object)cluster);
                throw new RestException(Response.Status.CONFLICT, "Cluster already exists");
            }
            this.clusterResources().createCluster(cluster, (ClusterData)clusterData);
            log.info("[{}] Created cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (IllegalArgumentException e) {
            log.warn("[{}] Failed to create cluster with invalid name {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster name is not valid");
        }
        catch (Exception e) {
            log.error("[{}] Failed to create cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}")
    @ApiOperation(value="Update the configuration for a cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void updateCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The cluster data", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="{\n   'serviceUrl': 'http://pulsar.example.com:8080',\n   'brokerServiceUrl': 'pulsar://pulsar.example.com:6651'\n}")})) ClusterDataImpl clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        try {
            this.clusterResources().updateCluster(cluster, old -> clusterData);
            log.info("[{}] Updated cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (MetadataStoreException.NotFoundException e) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/peers")
    @ApiOperation(value="Update peer-cluster-list for a cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=412, message="Peer cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setPeerClusterNames(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The list of peer cluster names", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="[\n   'cluster-a',\n   'cluster-b'\n]")})) LinkedHashSet<String> peerClusterNames) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        if (peerClusterNames != null && !peerClusterNames.isEmpty()) {
            for (String peerCluster : peerClusterNames) {
                try {
                    if (cluster.equalsIgnoreCase(peerCluster)) {
                        throw new RestException(Response.Status.PRECONDITION_FAILED, cluster + " itself can't be part of peer-list");
                    }
                    this.clusterResources().getCluster(peerCluster).orElseThrow(() -> new RestException(Response.Status.PRECONDITION_FAILED, "Peer cluster " + peerCluster + " does not exist"));
                }
                catch (RestException e) {
                    log.warn("[{}] Peer cluster doesn't exist from {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw e;
                }
                catch (Exception e) {
                    log.warn("[{}] Failed to validate peer-cluster list {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw new RestException(e);
                }
            }
        }
        try {
            this.clusterResources().updateCluster(cluster, old -> old.clone().peerClusterNames(peerClusterNames).build());
            log.info("[{}] Successfully added peer-cluster {} for {}", new Object[]{this.clientAppId(), peerClusterNames, cluster});
        }
        catch (MetadataStoreException.NotFoundException e) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/peers")
    @ApiOperation(value="Get the peer-cluster data for the specified cluster.", response=String.class, responseContainer="Set", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public Set<String> getPeerCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        try {
            ClusterData clusterData = (ClusterData)this.clusterResources().getCluster(cluster).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Cluster does not exist"));
            return clusterData.getPeerClusterNames();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}")
    @ApiOperation(value="Delete an existing cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been deleted."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=412, message="Cluster is not empty."), @ApiResponse(code=500, message="Internal server error.")})
    public void deleteCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        boolean isClusterUsed = false;
        try {
            isClusterUsed = this.pulsar().getPulsarResources().getClusterResources().isClusterUsed(cluster);
            Optional nsIsolationPolicies = this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster);
            if (nsIsolationPolicies.isPresent()) {
                if (((NamespaceIsolationPolicies)nsIsolationPolicies.get()).getPolicies().isEmpty()) {
                    this.namespaceIsolationPolicies().deleteIsolationData(cluster);
                } else {
                    isClusterUsed = true;
                }
            }
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster usage {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        if (isClusterUsed) {
            log.warn("[{}] Failed to delete cluster {} - Cluster not empty", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster not empty");
        }
        try {
            this.clusterResources().getFailureDomainResources().deleteFailureDomains(cluster);
            this.clusterResources().deleteCluster(cluster);
            log.info("[{}] Deleted cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (MetadataStoreException.NotFoundException e) {
            log.warn("[{}] Failed to delete cluster {} - Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies")
    @ApiOperation(value="Get the namespace isolation policies assigned to the cluster.", response=NamespaceIsolationDataImpl.class, responseContainer="Map", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public Map<String, ? extends NamespaceIsolationData> getNamespaceIsolationPolicies(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        if (!this.clusterResources().clusterExists(cluster)) {
            throw new RestException(Response.Status.NOT_FOUND, "Cluster " + cluster + " does not exist.");
        }
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = (NamespaceIsolationPolicies)this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            return nsIsolationPolicies.getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Get the single namespace isolation policy assigned to the cluster.", response=NamespaceIsolationDataImpl.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public NamespaceIsolationData getNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The name of the namespace isolation policy", required=true) @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = (NamespaceIsolationPolicies)this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            if (!nsIsolationPolicies.getPolicies().containsKey(policyName)) {
                log.info("[{}] Cannot find NamespaceIsolationPolicy {} for cluster {}", new Object[]{this.clientAppId(), policyName, cluster});
                throw new RestException(Response.Status.NOT_FOUND, "Cannot find NamespaceIsolationPolicy " + policyName + " for cluster " + cluster);
            }
            return (NamespaceIsolationData)nsIsolationPolicies.getPolicies().get(policyName);
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/brokers")
    @ApiOperation(value="Get list of brokers with namespace-isolation policies attached to them.", response=BrokerNamespaceIsolationDataImpl.class, responseContainer="set", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Namespace-isolation policies not found."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public List<BrokerNamespaceIsolationData> getBrokersWithNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        Map nsPolicies;
        Set<String> availableBrokers;
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            availableBrokers = this.pulsar().getLoadManager().get().getAvailableBrokers();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get list of brokers in cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        try {
            Optional nsPoliciesResult = this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster);
            if (!nsPoliciesResult.isPresent()) {
                throw new RestException(Response.Status.NOT_FOUND, "namespace-isolation policies not found for " + cluster);
            }
            nsPolicies = ((NamespaceIsolationPolicies)nsPoliciesResult.get()).getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get namespace isolation-policies {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        return availableBrokers.stream().map(broker -> {
            BrokerNamespaceIsolationData.Builder brokerIsolationData = BrokerNamespaceIsolationData.builder().brokerName(broker);
            if (nsPolicies != null) {
                ArrayList namespaceRegexes = new ArrayList();
                nsPolicies.forEach((name, policyData) -> {
                    NamespaceIsolationPolicyImpl nsPolicyImpl = new NamespaceIsolationPolicyImpl(policyData);
                    if (nsPolicyImpl.isPrimaryBroker(broker) || nsPolicyImpl.isSecondaryBroker(broker)) {
                        namespaceRegexes.addAll(policyData.getNamespaces());
                        if (nsPolicyImpl.isPrimaryBroker(broker)) {
                            brokerIsolationData.primary(true);
                        }
                    }
                });
                brokerIsolationData.namespaceRegex(namespaceRegexes);
            }
            return brokerIsolationData.build();
        }).collect(Collectors.toList());
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/brokers/{broker}")
    @ApiOperation(value="Get a broker with namespace-isolation policies attached to it.", response=BrokerNamespaceIsolationDataImpl.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Namespace-isolation policies/ Broker not found."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public BrokerNamespaceIsolationData getBrokerWithNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The broker name (<broker-hostname>:<web-service-port>)", required=true, example="broker1:8080") @PathParam(value="broker") String broker) {
        Map nsPolicies;
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            Optional nsPoliciesResult = this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster);
            if (!nsPoliciesResult.isPresent()) {
                throw new RestException(Response.Status.NOT_FOUND, "namespace-isolation policies not found for " + cluster);
            }
            nsPolicies = ((NamespaceIsolationPolicies)nsPoliciesResult.get()).getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get namespace isolation-policies {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        BrokerNamespaceIsolationData.Builder brokerIsolationData = BrokerNamespaceIsolationData.builder().brokerName(broker);
        if (nsPolicies != null) {
            ArrayList namespaceRegexes = new ArrayList();
            nsPolicies.forEach((name, policyData) -> {
                NamespaceIsolationPolicyImpl nsPolicyImpl = new NamespaceIsolationPolicyImpl(policyData);
                boolean isPrimary = nsPolicyImpl.isPrimaryBroker(broker);
                if (isPrimary || nsPolicyImpl.isSecondaryBroker(broker)) {
                    namespaceRegexes.addAll(policyData.getNamespaces());
                    brokerIsolationData.primary(isPrimary);
                    brokerIsolationData.policyName(name);
                }
            });
            brokerIsolationData.namespaceRegex(namespaceRegexes);
        }
        return brokerIsolationData.build();
    }

    @POST
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Set namespace isolation policy.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=400, message="Namespace isolation policy data is invalid."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Namespace isolation policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setNamespaceIsolationPolicy(@Suspended AsyncResponse asyncResponse, @ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The namespace isolation policy name", required=true) @PathParam(value="policyName") String policyName, @ApiParam(value="The namespace isolation policy data", required=true) NamespaceIsolationDataImpl policyData) {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        String jsonInput = null;
        try {
            policyData.validate();
            jsonInput = ObjectMapperFactory.create().writeValueAsString((Object)policyData);
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster).orElseGet(() -> {
                try {
                    this.namespaceIsolationPolicies().setIsolationDataWithCreate(cluster, p -> Collections.emptyMap());
                    return new NamespaceIsolationPolicies();
                }
                catch (Exception e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.setPolicy(policyName, (NamespaceIsolationData)policyData);
            this.namespaceIsolationPolicies().setIsolationData(cluster, old -> nsIsolationPolicies.getPolicies());
            if (!this.pulsar().getConfiguration().isEnableNamespaceIsolationUpdateOnTime()) {
                asyncResponse.resume((Object)Response.noContent().build());
                return;
            }
            this.filterAndUnloadMatchedNameSpaces(asyncResponse, policyData);
        }
        catch (IllegalArgumentException iae) {
            log.info("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}. Input data is invalid", new Object[]{this.clientAppId(), cluster, policyName, iae});
            asyncResponse.resume((Throwable)((Object)new RestException(Response.Status.BAD_REQUEST, "Invalid format of input policy data. policy: " + policyName + "; data: " + jsonInput)));
        }
        catch (MetadataStoreException.NotFoundException nne) {
            log.warn("[{}] Failed to update clusters/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            asyncResponse.resume((Throwable)((Object)new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist")));
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            asyncResponse.resume((Throwable)((Object)new RestException(e)));
        }
    }

    private void filterAndUnloadMatchedNameSpaces(AsyncResponse asyncResponse, NamespaceIsolationDataImpl policyData) throws Exception {
        Namespaces namespaces = this.pulsar().getAdminClient().namespaces();
        ArrayList nssToUnload = Lists.newArrayList();
        this.pulsar().getAdminClient().tenants().getTenantsAsync().whenComplete((tenants, ex) -> {
            if (ex != null) {
                log.error("[{}] Failed to get tenants when setNamespaceIsolationPolicy.", (Object)this.clientAppId(), ex);
                return;
            }
            AtomicInteger tenantsNumber = new AtomicInteger(tenants.size());
            tenants.forEach(tenant -> namespaces.getNamespacesAsync(tenant).whenComplete((nss, e) -> {
                int leftTenantsToHandle = tenantsNumber.decrementAndGet();
                if (e != null) {
                    log.error("[{}] Failed to get namespaces for tenant {} when setNamespaceIsolationPolicy.", new Object[]{this.clientAppId(), tenant, e});
                    if (leftTenantsToHandle == 0) {
                        this.unloadMatchedNamespacesList(asyncResponse, nssToUnload, namespaces);
                    }
                    return;
                }
                AtomicInteger nssNumber = new AtomicInteger(nss.size());
                nss.forEach(namespaceName -> {
                    int leftNssToHandle = nssNumber.decrementAndGet();
                    if (policyData.getNamespaces().stream().anyMatch(nsnameRegex -> namespaceName.matches((String)nsnameRegex))) {
                        nssToUnload.add(namespaceName);
                    }
                    if (leftNssToHandle == 0 && leftTenantsToHandle == 0) {
                        this.unloadMatchedNamespacesList(asyncResponse, nssToUnload, namespaces);
                    }
                });
            }));
        });
    }

    private void unloadMatchedNamespacesList(AsyncResponse asyncResponse, List<String> nssToUnload, Namespaces namespaces) {
        if (nssToUnload.size() == 0) {
            asyncResponse.resume((Object)Response.noContent().build());
            return;
        }
        List futures = nssToUnload.stream().map(namespaceName -> namespaces.unloadAsync(namespaceName)).collect(Collectors.toList());
        FutureUtil.waitForAll(futures).whenComplete((result, exception) -> {
            if (exception != null) {
                log.error("[{}] Failed to unload namespace while setNamespaceIsolationPolicy.", (Object)this.clientAppId(), exception);
                asyncResponse.resume((Throwable)((Object)new RestException((Throwable)exception)));
                return;
            }
            try {
                this.pulsar().getLoadManager().get().writeLoadReportOnZookeeper(true);
            }
            catch (Exception e) {
                log.warn("[{}] Failed to writeLoadReportOnZookeeper.", (Object)this.clientAppId(), (Object)e);
            }
            asyncResponse.resume((Object)Response.noContent().build());
        });
    }

    @DELETE
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Delete namespace isolation policy.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or policies are read only."), @ApiResponse(code=404, message="Namespace isolation policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void deleteNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The namespace isolation policy name", required=true) @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPolicies().getIsolationDataPolicies(cluster).orElseGet(() -> {
                try {
                    this.namespaceIsolationPolicies().setIsolationDataWithCreate(cluster, p -> Collections.emptyMap());
                    return new NamespaceIsolationPolicies();
                }
                catch (Exception e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.deletePolicy(policyName);
            this.namespaceIsolationPolicies().setIsolationData(cluster, old -> nsIsolationPolicies.getPolicies());
        }
        catch (MetadataStoreException.NotFoundException nne) {
            log.warn("[{}] Failed to update brokers/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update brokers/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Set the failure domain of the cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Failure domain doesn't exist."), @ApiResponse(code=409, message="Broker already exists in another domain."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setFailureDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName, @ApiParam(value="The configuration data of a failure domain", required=true) FailureDomainImpl domain) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validateBrokerExistsInOtherDomain(cluster, domainName, domain);
        try {
            this.clusterResources().getFailureDomainResources().setFailureDomainWithCreate(cluster, domainName, old -> domain);
        }
        catch (MetadataStoreException.NotFoundException nne) {
            log.warn("[{}] Failed to update domain {}. clusters {}  Does not exist", new Object[]{this.clientAppId(), cluster, domainName});
            throw new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/domainName/{}", new Object[]{this.clientAppId(), cluster, domainName, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/failureDomains")
    @ApiOperation(value="Get the cluster failure domains.", response=FailureDomainImpl.class, responseContainer="Map", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=500, message="Internal server error")})
    public Map<String, FailureDomainImpl> getFailureDomains(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        HashMap domains = Maps.newHashMap();
        try {
            ClusterResources.FailureDomainResources fdr = this.clusterResources().getFailureDomainResources();
            for (String domainName : fdr.listFailureDomains(cluster)) {
                try {
                    Optional domain = fdr.getFailureDomain(cluster, domainName);
                    domain.ifPresent(failureDomain -> domains.put(domainName, failureDomain));
                }
                catch (Exception e) {
                    log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                }
            }
        }
        catch (MetadataStoreException.NotFoundException e) {
            log.warn("[{}] Failure-domain is not configured for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            return Collections.emptyMap();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get failure-domains for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        return domains;
    }

    @GET
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Get a domain in a cluster", response=FailureDomainImpl.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="FailureDomain doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist"), @ApiResponse(code=500, message="Internal server error")})
    public FailureDomainImpl getDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            return (FailureDomainImpl)this.clusterResources().getFailureDomainResources().getFailureDomain(cluster, domainName).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist"));
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get domain {} for cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Delete the failure domain of the cluster", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or policy is read only"), @ApiResponse(code=404, message="FailureDomain doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist"), @ApiResponse(code=500, message="Internal server error")})
    public void deleteFailureDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            this.clusterResources().getFailureDomainResources().deleteFailureDomain(cluster, domainName);
        }
        catch (MetadataStoreException.NotFoundException nne) {
            log.warn("[{}] Domain {} does not exist in {}", new Object[]{this.clientAppId(), domainName, cluster});
            throw new RestException(Response.Status.NOT_FOUND, "Domain-name " + domainName + " or cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete domain {} in cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    private void validateBrokerExistsInOtherDomain(String cluster, String inputDomainName, FailureDomainImpl inputDomain) {
        if (inputDomain != null && inputDomain.brokers != null) {
            try {
                for (String domainName : this.clusterResources().getFailureDomainResources().listFailureDomains(cluster)) {
                    if (inputDomainName.equals(domainName)) continue;
                    try {
                        Optional domain = this.clusterResources().getFailureDomainResources().getFailureDomain(cluster, domainName);
                        if (!domain.isPresent() || ((FailureDomainImpl)domain.get()).brokers == null) continue;
                        List duplicateBrokers = ((Stream)((FailureDomainImpl)domain.get()).brokers.stream().parallel()).filter(inputDomain.brokers::contains).collect(Collectors.toList());
                        if (duplicateBrokers.isEmpty()) continue;
                        throw new RestException(Response.Status.CONFLICT, duplicateBrokers + " already exists in " + domainName);
                    }
                    catch (Exception e) {
                        if (e instanceof RestException) {
                            throw e;
                        }
                        log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                    }
                }
            }
            catch (MetadataStoreException.NotFoundException e) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Domain is not configured for cluster", (Object)this.clientAppId(), (Object)e);
                }
            }
            catch (Exception e) {
                log.error("[{}] Failed to get domains for cluster {}", (Object)this.clientAppId(), (Object)e);
                throw new RestException(e);
            }
        }
    }
}

