/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.ea.agentloader.AgentLoader;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Optional;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.proto.BookieServer;
import org.apache.bookkeeper.replication.AutoRecoveryMain;
import org.apache.bookkeeper.stats.StatsProvider;
import org.apache.bookkeeper.util.ReflectionUtils;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.ServiceConfigurationUtils;
import org.apache.pulsar.common.configuration.PulsarConfigurationLoader;
import org.apache.pulsar.functions.worker.WorkerConfig;
import org.apache.pulsar.functions.worker.WorkerService;
import org.aspectj.weaver.loadtime.Agent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.bridge.SLF4JBridgeHandler;

public class PulsarBrokerStarter {
    private static final Logger log = LoggerFactory.getLogger(PulsarBrokerStarter.class);

    private static ServiceConfiguration loadConfig(String configFile) throws Exception {
        SLF4JBridgeHandler.removeHandlersForRootLogger();
        SLF4JBridgeHandler.install();
        ServiceConfiguration config = (ServiceConfiguration)PulsarConfigurationLoader.create((InputStream)new FileInputStream(configFile), ServiceConfiguration.class);
        PulsarConfigurationLoader.isComplete((Object)config);
        return config;
    }

    private static ServerConfiguration readBookieConfFile(String bookieConfigFile) throws IllegalArgumentException {
        ServerConfiguration bookieConf = new ServerConfiguration();
        try {
            bookieConf.loadConf(new File(bookieConfigFile).toURI().toURL());
            bookieConf.validate();
            log.info("Using bookie configuration file {}", (Object)bookieConfigFile);
        }
        catch (MalformedURLException e) {
            log.error("Could not open configuration file: {}", (Object)bookieConfigFile, (Object)e);
            throw new IllegalArgumentException("Could not open configuration file");
        }
        catch (ConfigurationException e) {
            log.error("Malformed configuration file: {}", (Object)bookieConfigFile, (Object)e);
            throw new IllegalArgumentException("Malformed configuration file");
        }
        return bookieConf;
    }

    private static boolean argsContains(String[] args, String arg) {
        return Arrays.asList(args).contains(arg);
    }

    public static void main(String[] args) throws Exception {
        Thread.setDefaultUncaughtExceptionHandler((thread, exception) -> log.error("Uncaught exception in thread {}: {}", new Object[]{thread.getName(), exception.getMessage(), exception}));
        AgentLoader.loadAgentClass((String)Agent.class.getName(), null);
        BrokerStarter starter = new BrokerStarter(args);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> starter.shutdown()));
        try {
            starter.start();
        }
        catch (Exception e) {
            log.error("Failed to start pulsar service.", (Throwable)e);
            Runtime.getRuntime().halt(1);
        }
        starter.join();
    }

    private static class BrokerStarter {
        private final ServiceConfiguration brokerConfig;
        private final PulsarService pulsarService;
        private final BookieServer bookieServer;
        private final AutoRecoveryMain autoRecoveryMain;
        private final StatsProvider bookieStatsProvider;
        private final ServerConfiguration bookieConfig;
        private final WorkerService functionsWorkerService;

        BrokerStarter(String[] args) throws Exception {
            StarterArguments starterArguments = new StarterArguments();
            JCommander jcommander = new JCommander((Object)starterArguments);
            jcommander.setProgramName("PulsarBrokerStarter");
            jcommander.parse(args);
            if (starterArguments.help) {
                jcommander.usage();
                System.exit(-1);
            }
            if (StringUtils.isBlank((CharSequence)starterArguments.brokerConfigFile)) {
                jcommander.usage();
                throw new IllegalArgumentException("Need to specify a configuration file for broker");
            }
            this.brokerConfig = PulsarBrokerStarter.loadConfig(starterArguments.brokerConfigFile);
            if (starterArguments.runFunctionsWorker || this.brokerConfig.isFunctionsWorkerEnabled()) {
                WorkerConfig workerConfig = StringUtils.isBlank((CharSequence)starterArguments.fnWorkerConfigFile) ? new WorkerConfig() : WorkerConfig.load((String)starterArguments.fnWorkerConfigFile);
                boolean useTls = workerConfig.isUseTls();
                String localhost = "127.0.0.1";
                String pulsarServiceUrl = useTls ? PulsarService.brokerUrlTls(localhost, this.brokerConfig.getBrokerServicePortTls()) : PulsarService.brokerUrl(localhost, this.brokerConfig.getBrokerServicePort());
                String webServiceUrl = useTls ? PulsarService.webAddressTls(localhost, this.brokerConfig.getWebServicePortTls()) : PulsarService.webAddress(localhost, this.brokerConfig.getWebServicePort());
                workerConfig.setPulsarServiceUrl(pulsarServiceUrl);
                workerConfig.setPulsarWebServiceUrl(webServiceUrl);
                String hostname = ServiceConfigurationUtils.getDefaultOrConfiguredAddress((String)this.brokerConfig.getAdvertisedAddress());
                workerConfig.setWorkerHostname(hostname);
                workerConfig.setWorkerPort(this.brokerConfig.getWebServicePort());
                workerConfig.setWorkerId("c-" + this.brokerConfig.getClusterName() + "-fw-" + hostname + "-" + workerConfig.getWorkerPort());
                this.functionsWorkerService = new WorkerService(workerConfig);
            } else {
                this.functionsWorkerService = null;
            }
            this.pulsarService = new PulsarService(this.brokerConfig, Optional.ofNullable(this.functionsWorkerService));
            if (!PulsarBrokerStarter.argsContains(args, "-rb") && !PulsarBrokerStarter.argsContains(args, "--run-bookie")) {
                Preconditions.checkState((!starterArguments.runBookie ? 1 : 0) != 0, (Object)"runBookie should be false if has no argument specified");
                starterArguments.runBookie = this.brokerConfig.isEnableRunBookieTogether();
            }
            if (!PulsarBrokerStarter.argsContains(args, "-ra") && !PulsarBrokerStarter.argsContains(args, "--run-bookie-autorecovery")) {
                Preconditions.checkState((!starterArguments.runBookieAutoRecovery ? 1 : 0) != 0, (Object)"runBookieAutoRecovery should be false if has no argument specified");
                starterArguments.runBookieAutoRecovery = this.brokerConfig.isEnableRunBookieAutoRecoveryTogether();
            }
            if ((starterArguments.runBookie || starterArguments.runBookieAutoRecovery) && StringUtils.isBlank((CharSequence)starterArguments.bookieConfigFile)) {
                jcommander.usage();
                throw new IllegalArgumentException("No configuration file for Bookie");
            }
            if (starterArguments.runBookie || starterArguments.runBookieAutoRecovery) {
                Preconditions.checkState((boolean)StringUtils.isNotBlank((CharSequence)starterArguments.bookieConfigFile), (Object)"No configuration file for Bookie");
                this.bookieConfig = PulsarBrokerStarter.readBookieConfFile(starterArguments.bookieConfigFile);
                Class statsProviderClass = this.bookieConfig.getStatsProviderClass();
                this.bookieStatsProvider = (StatsProvider)ReflectionUtils.newInstance((Class)statsProviderClass);
            } else {
                this.bookieConfig = null;
                this.bookieStatsProvider = null;
            }
            if (starterArguments.runBookie) {
                Preconditions.checkNotNull((Object)this.bookieConfig, (Object)"No ServerConfiguration for Bookie");
                Preconditions.checkNotNull((Object)this.bookieStatsProvider, (Object)"No Stats Provider for Bookie");
                this.bookieServer = new BookieServer(this.bookieConfig, this.bookieStatsProvider.getStatsLogger(""));
            } else {
                this.bookieServer = null;
            }
            if (starterArguments.runBookieAutoRecovery) {
                Preconditions.checkNotNull((Object)this.bookieConfig, (Object)"No ServerConfiguration for Bookie Autorecovery");
                this.autoRecoveryMain = new AutoRecoveryMain(this.bookieConfig);
            } else {
                this.autoRecoveryMain = null;
            }
        }

        public void start() throws Exception {
            if (this.bookieStatsProvider != null) {
                this.bookieStatsProvider.start((Configuration)this.bookieConfig);
                log.info("started bookieStatsProvider.");
            }
            if (this.bookieServer != null) {
                this.bookieServer.start();
                log.info("started bookieServer.");
            }
            if (this.autoRecoveryMain != null) {
                this.autoRecoveryMain.start();
                log.info("started bookie autoRecoveryMain.");
            }
            this.pulsarService.start();
            log.info("PulsarService started.");
        }

        public void join() throws InterruptedException {
            this.pulsarService.waitUntilClosed();
            if (this.bookieServer != null) {
                this.bookieServer.join();
            }
            if (this.autoRecoveryMain != null) {
                this.autoRecoveryMain.join();
            }
        }

        public void shutdown() {
            if (this.functionsWorkerService != null) {
                this.functionsWorkerService.stop();
                log.info("Shut down functions worker service successfully.");
            }
            this.pulsarService.getShutdownService().run();
            log.info("Shut down broker service successfully.");
            if (this.bookieStatsProvider != null) {
                this.bookieStatsProvider.stop();
                log.info("Shut down bookieStatsProvider successfully.");
            }
            if (this.bookieServer != null) {
                this.bookieServer.shutdown();
                log.info("Shut down bookieServer successfully.");
            }
            if (this.autoRecoveryMain != null) {
                this.autoRecoveryMain.shutdown();
                log.info("Shut down autoRecoveryMain successfully.");
            }
        }
    }

    @VisibleForTesting
    private static class StarterArguments {
        @Parameter(names={"-c", "--broker-conf"}, description="Configuration file for Broker")
        private String brokerConfigFile = String.valueOf(Paths.get("", new String[0]).toAbsolutePath().normalize().toString()) + "/conf/broker.conf";
        @Parameter(names={"-rb", "--run-bookie"}, description="Run Bookie together with Broker")
        private boolean runBookie = false;
        @Parameter(names={"-ra", "--run-bookie-autorecovery"}, description="Run Bookie Autorecovery together with broker")
        private boolean runBookieAutoRecovery = false;
        @Parameter(names={"-bc", "--bookie-conf"}, description="Configuration file for Bookie")
        private String bookieConfigFile = String.valueOf(Paths.get("", new String[0]).toAbsolutePath().normalize().toString()) + "/conf/bookkeeper.conf";
        @Parameter(names={"-rfw", "--run-functions-worker"}, description="Run functions worker with Broker")
        private boolean runFunctionsWorker = false;
        @Parameter(names={"-fwc", "--functions-worker-conf"}, description="Configuration file for Functions Worker")
        private String fnWorkerConfigFile = String.valueOf(Paths.get("", new String[0]).toAbsolutePath().normalize().toString()) + "/conf/functions_worker.yml";
        @Parameter(names={"-h", "--help"}, description="Show this help message")
        private boolean help = false;

        private StarterArguments() {
        }
    }
}

