/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.validation.constraints.NotNull;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.LedgerEntry;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.client.api.BKException;
import org.apache.bookkeeper.client.api.DigestType;
import org.apache.bookkeeper.util.ZkUtils;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.service.schema.LongSchemaVersion;
import org.apache.pulsar.broker.service.schema.SchemaStorage;
import org.apache.pulsar.broker.service.schema.SchemaStorageFormat;
import org.apache.pulsar.broker.service.schema.StoredSchema;
import org.apache.pulsar.common.schema.SchemaVersion;
import org.apache.pulsar.zookeeper.ZooKeeperCache;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BookkeeperSchemaStorage
implements SchemaStorage {
    private static final Logger log = LoggerFactory.getLogger(BookkeeperSchemaStorage.class);
    private static final String SchemaPath = "/schemas";
    private static final List<ACL> Acl = ZooDefs.Ids.OPEN_ACL_UNSAFE;
    private static final byte[] LedgerPassword = "".getBytes();
    private final PulsarService pulsar;
    private final ZooKeeper zooKeeper;
    private final ZooKeeperCache localZkCache;
    private final ServiceConfiguration config;
    private BookKeeper bookKeeper;
    private final ConcurrentMap<String, CompletableFuture<StoredSchema>> readSchemaOperations = new ConcurrentHashMap<String, CompletableFuture<StoredSchema>>();

    @VisibleForTesting
    BookkeeperSchemaStorage(PulsarService pulsar) {
        this.pulsar = pulsar;
        this.localZkCache = pulsar.getLocalZkCache();
        this.zooKeeper = this.localZkCache.getZooKeeper();
        this.config = pulsar.getConfiguration();
    }

    @VisibleForTesting
    public void init() throws KeeperException, InterruptedException {
        try {
            if (this.zooKeeper.exists(SchemaPath, false) == null) {
                this.zooKeeper.create(SchemaPath, new byte[0], Acl, CreateMode.PERSISTENT);
            }
        }
        catch (KeeperException.NodeExistsException nodeExistsException) {}
    }

    @Override
    public void start() throws IOException {
        this.bookKeeper = this.pulsar.getBookKeeperClientFactory().create(this.pulsar.getConfiguration(), this.pulsar.getZkClient());
    }

    @Override
    public CompletableFuture<SchemaVersion> put(String key, byte[] value, byte[] hash) {
        return this.putSchemaIfAbsent(key, value, hash).thenApply(LongSchemaVersion::new);
    }

    @Override
    public CompletableFuture<StoredSchema> get(String key, SchemaVersion version) {
        if (version == SchemaVersion.Latest) {
            return this.getSchema(key);
        }
        LongSchemaVersion longVersion = (LongSchemaVersion)version;
        return this.getSchema(key, longVersion.getVersion());
    }

    @Override
    public CompletableFuture<SchemaVersion> delete(String key) {
        return this.deleteSchema(key).thenApply(LongSchemaVersion::new);
    }

    @NotNull
    private CompletableFuture<StoredSchema> getSchema(String schemaId) {
        return this.readSchemaOperations.computeIfAbsent(schemaId, key -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Fetching schema from store", (Object)schemaId);
            }
            CompletableFuture future = new CompletableFuture();
            ((CompletableFuture)this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(locator -> {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Got schema locator {}", (Object)schemaId, locator);
                }
                if (!locator.isPresent()) {
                    return CompletableFuture.completedFuture(null);
                }
                SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
                return this.readSchemaEntry(schemaLocator.getInfo().getPosition()).thenApply(entry -> new StoredSchema(entry.getSchemaData().toByteArray(), new LongSchemaVersion(schemaLocator.getInfo().getVersion())));
            })).handleAsync((res, ex) -> {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Get operation completed. res={} -- ex={}", new Object[]{schemaId, res, ex});
                }
                this.readSchemaOperations.remove(schemaId, future);
                if (ex != null) {
                    future.completeExceptionally((Throwable)ex);
                } else {
                    future.complete(res);
                }
                return null;
            });
            return future;
        });
    }

    @Override
    public SchemaVersion versionFromBytes(byte[] version) {
        ByteBuffer bb = ByteBuffer.wrap(version);
        return new LongSchemaVersion(bb.getLong());
    }

    @Override
    public void close() throws Exception {
        if (Objects.nonNull(this.bookKeeper)) {
            this.bookKeeper.close();
        }
    }

    @NotNull
    private CompletableFuture<StoredSchema> getSchema(String schemaId, long version) {
        if (log.isDebugEnabled()) {
            log.debug("[{}] Get schema - version: {}", (Object)schemaId, (Object)version);
        }
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(locator -> {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Get schema - version: {} - locator: {}", new Object[]{schemaId, version, locator});
            }
            if (!locator.isPresent()) {
                return CompletableFuture.completedFuture(null);
            }
            SchemaStorageFormat.SchemaLocator schemaLocator = ((LocatorEntry)locator.get()).locator;
            if (version > schemaLocator.getInfo().getVersion()) {
                return CompletableFuture.completedFuture(null);
            }
            return this.findSchemaEntryByVersion(schemaLocator.getIndexList(), version).thenApply(entry -> new StoredSchema(entry.getSchemaData().toByteArray(), new LongSchemaVersion(version)));
        });
    }

    @NotNull
    private CompletableFuture<Long> putSchema(String schemaId, byte[] data, byte[] hash) {
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(optLocatorEntry -> {
            if (optLocatorEntry.isPresent()) {
                return this.addNewSchemaEntryToStore(((LocatorEntry)optLocatorEntry.get()).locator.getIndexList(), data).thenCompose(position -> this.updateSchemaLocator(schemaId, (LocatorEntry)optLocatorEntry.get(), (SchemaStorageFormat.PositionInfo)position, hash));
            }
            CompletableFuture future = new CompletableFuture();
            ((CompletableFuture)this.createNewSchema(schemaId, data, hash).thenAccept(version -> {
                boolean bl = future.complete(version);
            })).exceptionally(ex -> {
                if (ex.getCause() instanceof KeeperException.NodeExistsException) {
                    ((CompletableFuture)this.putSchema(schemaId, data, hash).thenAccept(version -> {
                        boolean bl = future.complete(version);
                    })).exceptionally(ex2 -> {
                        future.completeExceptionally((Throwable)ex2);
                        return null;
                    });
                } else {
                    future.completeExceptionally((Throwable)ex);
                }
                return null;
            });
            return future;
        });
    }

    @NotNull
    private CompletableFuture<Long> putSchemaIfAbsent(String schemaId, byte[] data, byte[] hash) {
        return this.getSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId)).thenCompose(optLocatorEntry -> {
            if (optLocatorEntry.isPresent()) {
                SchemaStorageFormat.SchemaLocator locator = ((LocatorEntry)optLocatorEntry.get()).locator;
                byte[] storedHash = locator.getInfo().getHash().toByteArray();
                if (storedHash.length > 0 && Arrays.equals(storedHash, hash)) {
                    return CompletableFuture.completedFuture(locator.getInfo().getVersion());
                }
                if (log.isDebugEnabled()) {
                    log.debug("[{}] findSchemaEntryByHash - hash={}", (Object)schemaId, (Object)hash);
                }
                return this.findSchemaEntryByHash(locator.getIndexList(), hash).thenCompose(version -> {
                    if (Objects.isNull(version)) {
                        return this.addNewSchemaEntryToStore(locator.getIndexList(), data).thenCompose(position -> this.updateSchemaLocator(schemaId, (LocatorEntry)optLocatorEntry.get(), (SchemaStorageFormat.PositionInfo)position, hash));
                    }
                    return CompletableFuture.completedFuture(version);
                });
            }
            CompletableFuture future = new CompletableFuture();
            ((CompletableFuture)this.createNewSchema(schemaId, data, hash).thenAccept(version -> {
                boolean bl = future.complete(version);
            })).exceptionally(ex -> {
                if (ex.getCause() instanceof KeeperException.NodeExistsException) {
                    ((CompletableFuture)this.putSchemaIfAbsent(schemaId, data, hash).thenAccept(version -> {
                        boolean bl = future.complete(version);
                    })).exceptionally(ex2 -> {
                        future.completeExceptionally((Throwable)ex2);
                        return null;
                    });
                } else {
                    future.completeExceptionally((Throwable)ex);
                }
                return null;
            });
            return future;
        });
    }

    private CompletableFuture<Long> createNewSchema(String schemaId, byte[] data, byte[] hash) {
        SchemaStorageFormat.IndexEntry emptyIndex = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(0L).setHash(ByteString.copyFrom((byte[])hash)).setPosition(SchemaStorageFormat.PositionInfo.newBuilder().setEntryId(-1L).setLedgerId(-1L)).build();
        return this.addNewSchemaEntryToStore(Collections.singletonList(emptyIndex), data).thenCompose(position -> {
            SchemaStorageFormat.IndexEntry info = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(0L).setPosition((SchemaStorageFormat.PositionInfo)position).setHash(ByteString.copyFrom((byte[])hash)).build();
            return this.createSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId), SchemaStorageFormat.SchemaLocator.newBuilder().setInfo(info).addAllIndex(Lists.newArrayList((Object[])new SchemaStorageFormat.IndexEntry[]{info})).build()).thenApply(ignore -> 0L);
        });
    }

    @NotNull
    private CompletableFuture<Long> deleteSchema(String schemaId) {
        return this.getSchema(schemaId).thenCompose(schemaAndVersion -> {
            if (Objects.isNull(schemaAndVersion)) {
                return CompletableFuture.completedFuture(null);
            }
            return this.putSchema(schemaId, new byte[0], new byte[0]);
        });
    }

    @NotNull
    private static String getSchemaPath(String schemaId) {
        return "/schemas/" + schemaId;
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.PositionInfo> addNewSchemaEntryToStore(List<SchemaStorageFormat.IndexEntry> index, byte[] data) {
        SchemaStorageFormat.SchemaEntry schemaEntry = Functions.newSchemaEntry(index, data);
        return this.createLedger().thenCompose(ledgerHandle -> this.addEntry((LedgerHandle)ledgerHandle, schemaEntry).thenApply(entryId -> Functions.newPositionInfo(ledgerHandle.getId(), entryId)));
    }

    @NotNull
    private CompletableFuture<Long> updateSchemaLocator(String schemaId, LocatorEntry locatorEntry, SchemaStorageFormat.PositionInfo position, byte[] hash) {
        long nextVersion = locatorEntry.locator.getInfo().getVersion() + 1L;
        SchemaStorageFormat.SchemaLocator locator = locatorEntry.locator;
        SchemaStorageFormat.IndexEntry info = SchemaStorageFormat.IndexEntry.newBuilder().setVersion(nextVersion).setPosition(position).setHash(ByteString.copyFrom((byte[])hash)).build();
        return this.updateSchemaLocator(BookkeeperSchemaStorage.getSchemaPath(schemaId), SchemaStorageFormat.SchemaLocator.newBuilder().setInfo(info).addAllIndex(Iterables.concat(locator.getIndexList(), (Iterable)Lists.newArrayList((Object[])new SchemaStorageFormat.IndexEntry[]{info}))).build(), locatorEntry.zkZnodeVersion).thenApply(ignore -> nextVersion);
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.SchemaEntry> findSchemaEntryByVersion(List<SchemaStorageFormat.IndexEntry> index, long version) {
        if (index.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        SchemaStorageFormat.IndexEntry lowest = index.get(0);
        if (version < lowest.getVersion()) {
            return this.readSchemaEntry(lowest.getPosition()).thenCompose(entry -> this.findSchemaEntryByVersion(entry.getIndexList(), version));
        }
        for (SchemaStorageFormat.IndexEntry entry2 : index) {
            if (entry2.getVersion() == version) {
                return this.readSchemaEntry(entry2.getPosition());
            }
            if (entry2.getVersion() > version) break;
        }
        return CompletableFuture.completedFuture(null);
    }

    @NotNull
    private CompletableFuture<Long> findSchemaEntryByHash(List<SchemaStorageFormat.IndexEntry> index, byte[] hash) {
        if (index.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        for (SchemaStorageFormat.IndexEntry entry2 : index) {
            if (!Arrays.equals(entry2.getHash().toByteArray(), hash)) continue;
            return CompletableFuture.completedFuture(entry2.getVersion());
        }
        if (index.get(0).getPosition().getLedgerId() == -1L) {
            return CompletableFuture.completedFuture(null);
        }
        return this.readSchemaEntry(index.get(0).getPosition()).thenCompose(entry -> this.findSchemaEntryByHash(entry.getIndexList(), hash));
    }

    @NotNull
    private CompletableFuture<SchemaStorageFormat.SchemaEntry> readSchemaEntry(SchemaStorageFormat.PositionInfo position) {
        if (log.isDebugEnabled()) {
            log.debug("Reading schema entry from {}", (Object)position);
        }
        return ((CompletableFuture)this.openLedger(position.getLedgerId()).thenCompose(ledger -> Functions.getLedgerEntry(ledger, position.getEntryId()).thenCompose(entry -> this.closeLedger((LedgerHandle)ledger).thenApply(ignore -> entry)))).thenCompose(Functions::parseSchemaEntry);
    }

    @NotNull
    private CompletableFuture<Void> updateSchemaLocator(String id, SchemaStorageFormat.SchemaLocator schema, int version) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.zooKeeper.setData(id, schema.toByteArray(), version, (rc, path, ctx, stat) -> {
            KeeperException.Code code = KeeperException.Code.get(rc);
            if (code != KeeperException.Code.OK) {
                future.completeExceptionally(KeeperException.create(code));
            } else {
                future.complete(null);
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<LocatorEntry> createSchemaLocator(String id, SchemaStorageFormat.SchemaLocator locator) {
        CompletableFuture<LocatorEntry> future = new CompletableFuture<LocatorEntry>();
        ZkUtils.asyncCreateFullPathOptimistic((ZooKeeper)this.zooKeeper, (String)id, (byte[])locator.toByteArray(), Acl, (CreateMode)CreateMode.PERSISTENT, (rc, path, ctx, name) -> {
            KeeperException.Code code = KeeperException.Code.get(rc);
            if (code != KeeperException.Code.OK) {
                future.completeExceptionally(KeeperException.create(code));
            } else {
                future.complete(new LocatorEntry(locator, 0));
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<Optional<LocatorEntry>> getSchemaLocator(String schema) {
        return this.localZkCache.getEntryAsync(schema, (ZooKeeperCache.Deserializer)new SchemaLocatorDeserializer()).thenApply(optional -> optional.map(entry -> new LocatorEntry((SchemaStorageFormat.SchemaLocator)entry.getKey(), ((Stat)entry.getValue()).getVersion())));
    }

    @NotNull
    private CompletableFuture<Long> addEntry(LedgerHandle ledgerHandle, SchemaStorageFormat.SchemaEntry entry) {
        CompletableFuture<Long> future = new CompletableFuture<Long>();
        ledgerHandle.asyncAddEntry(entry.toByteArray(), (rc, handle, entryId, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to add entry", rc, ledgerHandle.getId(), -1L));
            } else {
                future.complete(entryId);
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<LedgerHandle> createLedger() {
        CompletableFuture<LedgerHandle> future = new CompletableFuture<LedgerHandle>();
        this.bookKeeper.asyncCreateLedger(this.config.getManagedLedgerDefaultEnsembleSize(), this.config.getManagedLedgerDefaultWriteQuorum(), this.config.getManagedLedgerDefaultAckQuorum(), BookKeeper.DigestType.fromApiDigestType((DigestType)this.config.getManagedLedgerDigestType()), LedgerPassword, (rc, handle, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to create ledger", rc, -1L, -1L));
            } else {
                future.complete(handle);
            }
        }, null, Collections.emptyMap());
        return future;
    }

    @NotNull
    private CompletableFuture<LedgerHandle> openLedger(Long ledgerId) {
        CompletableFuture<LedgerHandle> future = new CompletableFuture<LedgerHandle>();
        this.bookKeeper.asyncOpenLedger(ledgerId.longValue(), BookKeeper.DigestType.fromApiDigestType((DigestType)this.config.getManagedLedgerDigestType()), LedgerPassword, (rc, handle, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to open ledger", rc, ledgerId, -1L));
            } else {
                future.complete(handle);
            }
        }, null);
        return future;
    }

    @NotNull
    private CompletableFuture<Void> closeLedger(LedgerHandle ledgerHandle) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        ledgerHandle.asyncClose((rc, handle, ctx) -> {
            if (rc != 0) {
                future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to close ledger", rc, ledgerHandle.getId(), -1L));
            } else {
                future.complete(null);
            }
        }, null);
        return future;
    }

    public static Exception bkException(String operation, int rc, long ledgerId, long entryId) {
        String message = String.valueOf(BKException.getMessage((int)rc)) + " -  ledger=" + ledgerId;
        if (entryId != -1L) {
            message = String.valueOf(message) + " - entry=" + entryId;
        }
        return new IOException(message);
    }

    static interface Functions {
        public static CompletableFuture<LedgerEntry> getLedgerEntry(LedgerHandle ledger, long entry) {
            CompletableFuture<LedgerEntry> future = new CompletableFuture<LedgerEntry>();
            ledger.asyncReadEntries(entry, entry, (rc, handle, entries, ctx) -> {
                if (rc != 0) {
                    future.completeExceptionally(BookkeeperSchemaStorage.bkException("Failed to read entry", rc, ledger.getId(), entry));
                } else {
                    future.complete((LedgerEntry)entries.nextElement());
                }
            }, null);
            return future;
        }

        public static CompletableFuture<SchemaStorageFormat.SchemaEntry> parseSchemaEntry(LedgerEntry ledgerEntry) {
            CompletableFuture<SchemaStorageFormat.SchemaEntry> result = new CompletableFuture<SchemaStorageFormat.SchemaEntry>();
            try {
                result.complete(SchemaStorageFormat.SchemaEntry.parseFrom(ledgerEntry.getEntry()));
            }
            catch (IOException e) {
                result.completeExceptionally(e);
            }
            return result;
        }

        public static SchemaStorageFormat.SchemaEntry newSchemaEntry(List<SchemaStorageFormat.IndexEntry> index, byte[] data) {
            return SchemaStorageFormat.SchemaEntry.newBuilder().setSchemaData(ByteString.copyFrom((byte[])data)).addAllIndex(index).build();
        }

        public static SchemaStorageFormat.PositionInfo newPositionInfo(long ledgerId, long entryId) {
            return SchemaStorageFormat.PositionInfo.newBuilder().setLedgerId(ledgerId).setEntryId(entryId).build();
        }
    }

    static class LocatorEntry {
        final SchemaStorageFormat.SchemaLocator locator;
        final Integer zkZnodeVersion;

        LocatorEntry(SchemaStorageFormat.SchemaLocator locator, Integer zkZnodeVersion) {
            this.locator = locator;
            this.zkZnodeVersion = zkZnodeVersion;
        }
    }

    static class SchemaLocatorDeserializer
    implements ZooKeeperCache.Deserializer<SchemaStorageFormat.SchemaLocator> {
        SchemaLocatorDeserializer() {
        }

        public SchemaStorageFormat.SchemaLocator deserialize(String key, byte[] content) throws Exception {
            return SchemaStorageFormat.SchemaLocator.parseFrom(content);
        }
    }
}

