/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.loadbalance.impl;

import com.sun.management.OperatingSystemMXBean;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.loadbalance.BrokerHostUsage;
import org.apache.pulsar.policies.data.loadbalancer.ResourceUsage;
import org.apache.pulsar.policies.data.loadbalancer.SystemResourceUsage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LinuxBrokerHostUsageImpl
implements BrokerHostUsage {
    private final int hostUsageCheckIntervalMin;
    private long lastCollection;
    private double lastTotalNicUsageTx;
    private double lastTotalNicUsageRx;
    private CpuStat lastCpuStat;
    private OperatingSystemMXBean systemBean;
    private SystemResourceUsage usage;
    private final Optional<Double> overrideBrokerNicSpeedGbps;
    private static final Logger LOG = LoggerFactory.getLogger(LinuxBrokerHostUsageImpl.class);

    public LinuxBrokerHostUsageImpl(PulsarService pulsar) {
        this.hostUsageCheckIntervalMin = pulsar.getConfiguration().getLoadBalancerHostUsageCheckIntervalMinutes();
        this.systemBean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
        this.lastCollection = 0L;
        this.usage = new SystemResourceUsage();
        this.overrideBrokerNicSpeedGbps = pulsar.getConfiguration().getLoadBalancerOverrideBrokerNicSpeedGbps();
        pulsar.getLoadManagerExecutor().scheduleAtFixedRate(this::calculateBrokerHostUsage, 0L, this.hostUsageCheckIntervalMin, TimeUnit.MINUTES);
    }

    @Override
    public SystemResourceUsage getBrokerHostUsage() {
        return this.usage;
    }

    private void calculateBrokerHostUsage() {
        List<String> nics = this.getNics();
        double totalNicLimit = this.getTotalNicLimitKbps(nics);
        double totalNicUsageTx = this.getTotalNicUsageTxKb(nics);
        double totalNicUsageRx = this.getTotalNicUsageRxKb(nics);
        double totalCpuLimit = this.getTotalCpuLimit();
        CpuStat cpuStat = this.getTotalCpuUsage();
        SystemResourceUsage usage = new SystemResourceUsage();
        long now = System.currentTimeMillis();
        if (this.lastCollection == 0L) {
            usage.setMemory(this.getMemUsage());
            usage.setBandwidthIn(new ResourceUsage(0.0, totalNicLimit));
            usage.setBandwidthOut(new ResourceUsage(0.0, totalNicLimit));
            usage.setCpu(new ResourceUsage(0.0, totalCpuLimit));
        } else {
            double elapsedSeconds = (double)(now - this.lastCollection) / 1000.0;
            double nicUsageTx = (totalNicUsageTx - this.lastTotalNicUsageTx) / elapsedSeconds;
            double nicUsageRx = (totalNicUsageRx - this.lastTotalNicUsageRx) / elapsedSeconds;
            if (cpuStat != null && this.lastCpuStat != null) {
                long cpuTimeDiff = cpuStat.getTotalTime() - this.lastCpuStat.getTotalTime();
                long cpuUsageDiff = cpuStat.getUsage() - this.lastCpuStat.getUsage();
                double cpuUsage = (double)cpuUsageDiff / (double)cpuTimeDiff * totalCpuLimit;
                usage.setCpu(new ResourceUsage(cpuUsage, totalCpuLimit));
            }
            usage.setMemory(this.getMemUsage());
            usage.setBandwidthIn(new ResourceUsage(nicUsageRx, totalNicLimit));
            usage.setBandwidthOut(new ResourceUsage(nicUsageTx, totalNicLimit));
        }
        this.lastTotalNicUsageTx = totalNicUsageTx;
        this.lastTotalNicUsageRx = totalNicUsageRx;
        this.lastCpuStat = cpuStat;
        this.lastCollection = System.currentTimeMillis();
        this.usage = usage;
    }

    private double getTotalCpuLimit() {
        return 100 * Runtime.getRuntime().availableProcessors();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private CpuStat getTotalCpuUsage() {
        try (Stream<String> stream = Files.lines(Paths.get("/proc/stat", new String[0]));){
            String[] words = stream.findFirst().get().split("\\s+");
            long total = Arrays.stream(words).filter(s -> !s.contains("cpu")).mapToLong(Long::parseLong).sum();
            long idle = Long.parseLong(words[4]);
            CpuStat cpuStat = new CpuStat(total, total - idle);
            return cpuStat;
        }
        catch (IOException e) {
            LOG.error("Failed to read CPU usage from /proc/stat", (Throwable)e);
            return null;
        }
    }

    private ResourceUsage getMemUsage() {
        double total = (double)this.systemBean.getTotalPhysicalMemorySize() / 1048576.0;
        double free = (double)this.systemBean.getFreePhysicalMemorySize() / 1048576.0;
        return new ResourceUsage(total - free, total);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private List<String> getNics() {
        try (Stream<Path> stream = Files.list(Paths.get("/sys/class/net/", new String[0]));){
            List<String> list = stream.filter(this::isPhysicalNic).map(path -> path.getFileName().toString()).collect(Collectors.toList());
            return list;
        }
        catch (IOException e) {
            LOG.error("Failed to find NICs", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private boolean isPhysicalNic(Path path) {
        if (!path.toString().contains("/virtual/")) {
            try {
                Files.readAllBytes(path.resolve("speed"));
                return true;
            }
            catch (Exception e) {
                return false;
            }
        }
        return false;
    }

    private Path getNicSpeedPath(String nic) {
        return Paths.get(String.format("/sys/class/net/%s/speed", nic), new String[0]);
    }

    private double getTotalNicLimitKbps(List<String> nics) {
        if (this.overrideBrokerNicSpeedGbps.isPresent()) {
            return this.overrideBrokerNicSpeedGbps.get() * (double)nics.size() * 1024.0 * 1024.0;
        }
        return nics.stream().mapToDouble(s -> {
            try {
                return Double.parseDouble(new String(Files.readAllBytes(this.getNicSpeedPath((String)s))));
            }
            catch (IOException e) {
                LOG.error("Failed to read speed for nic " + s, (Throwable)e);
                return 0.0;
            }
        }).sum() * 1024.0;
    }

    private Path getNicTxPath(String nic) {
        return Paths.get(String.format("/sys/class/net/%s/statistics/tx_bytes", nic), new String[0]);
    }

    private Path getNicRxPath(String nic) {
        return Paths.get(String.format("/sys/class/net/%s/statistics/rx_bytes", nic), new String[0]);
    }

    private double getTotalNicUsageRxKb(List<String> nics) {
        return nics.stream().mapToDouble(s -> {
            try {
                return Double.parseDouble(new String(Files.readAllBytes(this.getNicRxPath((String)s))));
            }
            catch (IOException e) {
                LOG.error("Failed to read rx_bytes for NIC " + s, (Throwable)e);
                return 0.0;
            }
        }).sum() * 8.0 / 1024.0;
    }

    private double getTotalNicUsageTxKb(List<String> nics) {
        return nics.stream().mapToDouble(s -> {
            try {
                return Double.parseDouble(new String(Files.readAllBytes(this.getNicTxPath((String)s))));
            }
            catch (IOException e) {
                LOG.error("Failed to read tx_bytes for NIC " + s, (Throwable)e);
                return 0.0;
            }
        }).sum() * 8.0 / 1024.0;
    }

    private class CpuStat {
        private long totalTime;
        private long usage;

        CpuStat(long totalTime, long usage) {
            this.totalTime = totalTime;
            this.usage = usage;
        }

        long getTotalTime() {
            return this.totalTime;
        }

        long getUsage() {
            return this.usage;
        }
    }
}

