/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.namespace;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.hash.Hashing;
import io.netty.channel.EventLoopGroup;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.pulsar.broker.PulsarServerException;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.loadbalance.LoadManager;
import org.apache.pulsar.broker.loadbalance.ResourceUnit;
import org.apache.pulsar.broker.lookup.LookupResult;
import org.apache.pulsar.broker.namespace.NamespaceEphemeralData;
import org.apache.pulsar.broker.namespace.OwnedBundle;
import org.apache.pulsar.broker.namespace.OwnershipCache;
import org.apache.pulsar.broker.namespace.ServiceUnitZkUtils;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.nonpersistent.NonPersistentTopic;
import org.apache.pulsar.broker.web.PulsarWebResource;
import org.apache.pulsar.client.admin.PulsarAdmin;
import org.apache.pulsar.client.api.ClientBuilder;
import org.apache.pulsar.client.api.PulsarClient;
import org.apache.pulsar.client.impl.ClientBuilderImpl;
import org.apache.pulsar.client.impl.PulsarClientImpl;
import org.apache.pulsar.client.impl.conf.ClientConfigurationData;
import org.apache.pulsar.common.api.proto.PulsarApi;
import org.apache.pulsar.common.lookup.data.LookupData;
import org.apache.pulsar.common.naming.NamespaceBundle;
import org.apache.pulsar.common.naming.NamespaceBundleFactory;
import org.apache.pulsar.common.naming.NamespaceBundles;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.ServiceUnitId;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.policies.NamespaceIsolationPolicy;
import org.apache.pulsar.common.policies.data.BrokerAssignment;
import org.apache.pulsar.common.policies.data.BundlesData;
import org.apache.pulsar.common.policies.data.ClusterData;
import org.apache.pulsar.common.policies.data.LocalPolicies;
import org.apache.pulsar.common.policies.data.NamespaceOwnershipStatus;
import org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicies;
import org.apache.pulsar.common.util.Codec;
import org.apache.pulsar.common.util.ObjectMapperFactory;
import org.apache.pulsar.common.util.collections.ConcurrentOpenHashMap;
import org.apache.pulsar.policies.data.loadbalancer.ServiceLookupData;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NamespaceService {
    private static final Logger LOG = LoggerFactory.getLogger(NamespaceService.class);
    private final ServiceConfiguration config;
    private final AtomicReference<LoadManager> loadManager;
    private final PulsarService pulsar;
    private final OwnershipCache ownershipCache;
    private final NamespaceBundleFactory bundleFactory;
    private int uncountedNamespaces;
    private final String host;
    private static final int BUNDLE_SPLIT_RETRY_LIMIT = 7;
    public static final String SLA_NAMESPACE_PROPERTY = "sla-monitor";
    public static final Pattern HEARTBEAT_NAMESPACE_PATTERN = Pattern.compile("pulsar/[^/]+/([^:]+:\\d+)");
    public static final Pattern SLA_NAMESPACE_PATTERN = Pattern.compile("sla-monitor/[^/]+/([^:]+:\\d+)");
    public static final String HEARTBEAT_NAMESPACE_FMT = "pulsar/%s/%s:%s";
    public static final String SLA_NAMESPACE_FMT = "sla-monitor/%s/%s:%s";
    public static final String NAMESPACE_ISOLATION_POLICIES = "namespaceIsolationPolicies";
    private final ConcurrentOpenHashMap<ClusterData, PulsarClientImpl> namespaceClients;
    private final ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> findingBundlesAuthoritative = new ConcurrentOpenHashMap();
    private final ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> findingBundlesNotAuthoritative = new ConcurrentOpenHashMap();

    public NamespaceService(PulsarService pulsar) {
        this.pulsar = pulsar;
        this.host = pulsar.getAdvertisedAddress();
        this.config = pulsar.getConfiguration();
        this.loadManager = pulsar.getLoadManager();
        ServiceUnitZkUtils.initZK(pulsar.getLocalZkCache().getZooKeeper(), pulsar.getSafeBrokerServiceUrl());
        this.bundleFactory = new NamespaceBundleFactory(pulsar, Hashing.crc32());
        this.ownershipCache = new OwnershipCache(pulsar, this.bundleFactory);
        this.namespaceClients = new ConcurrentOpenHashMap();
    }

    public CompletableFuture<Optional<LookupResult>> getBrokerServiceUrlAsync(TopicName topic, boolean authoritative) {
        return this.getBundleAsync(topic).thenCompose(bundle -> this.findBrokerServiceUrl((NamespaceBundle)bundle, authoritative, false));
    }

    public CompletableFuture<NamespaceBundle> getBundleAsync(TopicName topic) {
        return this.bundleFactory.getBundlesAsync(topic.getNamespaceObject()).thenApply(bundles -> bundles.findBundle(topic));
    }

    public Optional<NamespaceBundle> getBundleIfPresent(TopicName topicName) throws Exception {
        Optional<NamespaceBundles> bundles = this.bundleFactory.getBundlesIfPresent(topicName.getNamespaceObject());
        return bundles.map(b -> b.findBundle(topicName));
    }

    public NamespaceBundle getBundle(TopicName topicName) throws Exception {
        return this.bundleFactory.getBundles(topicName.getNamespaceObject()).findBundle(topicName);
    }

    public int getBundleCount(NamespaceName namespace) throws Exception {
        return this.bundleFactory.getBundles(namespace).size();
    }

    private NamespaceBundle getFullBundle(NamespaceName fqnn) throws Exception {
        return this.bundleFactory.getFullBundle(fqnn);
    }

    public Optional<URL> getWebServiceUrl(ServiceUnitId suName, boolean authoritative, boolean isRequestHttps, boolean readOnly) throws Exception {
        if (suName instanceof TopicName) {
            TopicName name = (TopicName)suName;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Getting web service URL of topic: {} - auth: {}", (Object)name, (Object)authoritative);
            }
            return this.internalGetWebServiceUrl(this.getBundle(name), authoritative, isRequestHttps, readOnly).get();
        }
        if (suName instanceof NamespaceName) {
            return this.internalGetWebServiceUrl(this.getFullBundle((NamespaceName)suName), authoritative, isRequestHttps, readOnly).get();
        }
        if (suName instanceof NamespaceBundle) {
            return this.internalGetWebServiceUrl((NamespaceBundle)suName, authoritative, isRequestHttps, readOnly).get();
        }
        throw new IllegalArgumentException("Unrecognized class of NamespaceBundle: " + suName.getClass().getName());
    }

    private CompletableFuture<Optional<URL>> internalGetWebServiceUrl(NamespaceBundle bundle, boolean authoritative, boolean isRequestHttps, boolean readOnly) {
        return this.findBrokerServiceUrl(bundle, authoritative, readOnly).thenApply(lookupResult -> {
            if (lookupResult.isPresent()) {
                try {
                    LookupData lookupData = ((LookupResult)lookupResult.get()).getLookupData();
                    String redirectUrl = isRequestHttps ? lookupData.getHttpUrlTls() : lookupData.getHttpUrl();
                    return Optional.of(new URL(redirectUrl));
                }
                catch (Exception e) {
                    LOG.warn("internalGetWebServiceUrl [{}]", (Object)e.getMessage(), (Object)e);
                }
            }
            return Optional.empty();
        });
    }

    public void registerBootstrapNamespaces() throws PulsarServerException {
        if (this.registerNamespace(NamespaceService.getHeartbeatNamespace(this.host, this.config), true)) {
            ++this.uncountedNamespaces;
            LOG.info("added heartbeat namespace name in local cache: ns={}", (Object)NamespaceService.getHeartbeatNamespace(this.host, this.config));
        }
        for (String namespace : this.config.getBootstrapNamespaces()) {
            if (!this.registerNamespace(namespace, false)) continue;
            LOG.info("added bootstrap namespace name in local cache: ns={}", (Object)namespace);
        }
    }

    private boolean registerNamespace(String namespace, boolean ensureOwned) throws PulsarServerException {
        String myUrl = this.pulsar.getSafeBrokerServiceUrl();
        try {
            NamespaceName nsname = NamespaceName.get((String)namespace);
            String otherUrl = null;
            NamespaceBundle nsFullBundle = null;
            nsFullBundle = this.bundleFactory.getFullBundle(nsname);
            otherUrl = this.ownershipCache.tryAcquiringOwnership(nsFullBundle).get().getNativeUrl();
            if (myUrl.equals(otherUrl)) {
                if (nsFullBundle != null) {
                    this.pulsar.loadNamespaceTopics(nsFullBundle);
                }
                return true;
            }
            String msg = String.format("namespace already owned by other broker : ns=%s expected=%s actual=%s", namespace, myUrl, otherUrl);
            if (!ensureOwned) {
                LOG.info(msg);
                return false;
            }
            throw new IllegalStateException(msg);
        }
        catch (Exception e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new PulsarServerException((Throwable)e);
        }
    }

    private CompletableFuture<Optional<LookupResult>> findBrokerServiceUrl(NamespaceBundle bundle, boolean authoritative, boolean readOnly) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("findBrokerServiceUrl: {} - read-only: {}", (Object)bundle, (Object)readOnly);
        }
        ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> targetMap = authoritative ? this.findingBundlesAuthoritative : this.findingBundlesNotAuthoritative;
        return (CompletableFuture)targetMap.computeIfAbsent((Object)bundle, k -> {
            CompletableFuture future = new CompletableFuture();
            ((CompletableFuture)this.ownershipCache.getOwnerAsync(bundle).thenAccept(nsData -> {
                if (!nsData.isPresent()) {
                    if (readOnly) {
                        future.complete(Optional.empty());
                    } else {
                        this.pulsar.getExecutor().execute(() -> this.searchForCandidateBroker(bundle, future, authoritative));
                    }
                } else if (((NamespaceEphemeralData)nsData.get()).isDisabled()) {
                    future.completeExceptionally(new IllegalStateException(String.format("Namespace bundle %s is being unloaded", bundle)));
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Namespace bundle {} already owned by {} ", (Object)bundle, nsData);
                    }
                    future.complete(Optional.of(new LookupResult((NamespaceEphemeralData)nsData.get())));
                }
            })).exceptionally(exception -> {
                LOG.warn("Failed to check owner for bundle {}: {}", new Object[]{bundle, exception.getMessage(), exception});
                future.completeExceptionally((Throwable)exception);
                return null;
            });
            future.whenComplete((r, t) -> this.pulsar.getExecutor().execute(() -> {
                CompletableFuture cfr_ignored_0 = (CompletableFuture)targetMap.remove((Object)bundle);
            }));
            return future;
        });
    }

    private void searchForCandidateBroker(NamespaceBundle bundle, CompletableFuture<Optional<LookupResult>> lookupFuture, boolean authoritative) {
        String candidateBroker = null;
        try {
            String broker;
            candidateBroker = NamespaceService.checkHeartbeatNamespace(bundle);
            if (candidateBroker == null && (broker = NamespaceService.getSLAMonitorBrokerName(bundle)) != null && this.isBrokerActive(broker)) {
                candidateBroker = broker;
            }
            if (candidateBroker == null) {
                if (!this.loadManager.get().isCentralized() || this.pulsar.getLeaderElectionService().isLeader() || !this.isBrokerActive(this.pulsar.getLeaderElectionService().getCurrentLeader().getServiceUrl())) {
                    Optional<String> availableBroker = this.getLeastLoadedFromLoadManager(bundle);
                    if (!availableBroker.isPresent()) {
                        lookupFuture.complete(Optional.empty());
                        return;
                    }
                    candidateBroker = availableBroker.get();
                } else {
                    candidateBroker = authoritative ? this.pulsar.getSafeWebServiceAddress() : this.pulsar.getLeaderElectionService().getCurrentLeader().getServiceUrl();
                }
            }
        }
        catch (Exception e) {
            LOG.warn("Error when searching for candidate broker to acquire {}: {}", new Object[]{bundle, e.getMessage(), e});
            lookupFuture.completeExceptionally(e);
            return;
        }
        try {
            Preconditions.checkNotNull((Object)candidateBroker);
            if (this.pulsar.getSafeWebServiceAddress().equals(candidateBroker)) {
                String policyPath = AdminResource.path("policies", bundle.getNamespaceObject().toString());
                this.pulsar.getConfigurationCache().policiesCache().invalidate(policyPath);
                ((CompletableFuture)this.ownershipCache.tryAcquiringOwnership(bundle).thenAccept(ownerInfo -> {
                    if (ownerInfo.isDisabled()) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Namespace bundle {} is currently being unloaded", (Object)bundle);
                        }
                        lookupFuture.completeExceptionally(new IllegalStateException(String.format("Namespace bundle %s is currently being unloaded", bundle)));
                    } else {
                        this.pulsar.loadNamespaceTopics(bundle);
                        lookupFuture.complete(Optional.of(new LookupResult((NamespaceEphemeralData)ownerInfo)));
                    }
                })).exceptionally(exception -> {
                    LOG.warn("Failed to acquire ownership for namespace bundle {}: ", new Object[]{bundle, exception.getMessage(), exception});
                    lookupFuture.completeExceptionally(new PulsarServerException("Failed to acquire ownership for namespace bundle " + bundle, exception));
                    return null;
                });
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Redirecting to broker {} to acquire ownership of bundle {}", (Object)candidateBroker, (Object)bundle);
                }
                ((CompletableFuture)this.createLookupResult(candidateBroker).thenAccept(lookupResult -> lookupFuture.complete(Optional.of(lookupResult)))).exceptionally(ex -> {
                    lookupFuture.completeExceptionally((Throwable)ex);
                    return null;
                });
            }
        }
        catch (Exception e) {
            LOG.warn("Error in trying to acquire namespace bundle ownership for {}: {}", new Object[]{bundle, e.getMessage(), e});
            lookupFuture.completeExceptionally(e);
        }
    }

    protected CompletableFuture<LookupResult> createLookupResult(String candidateBroker) throws Exception {
        CompletableFuture<LookupResult> lookupFuture = new CompletableFuture<LookupResult>();
        try {
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)candidateBroker), (Object)("Lookup broker can't be null " + candidateBroker));
            URI uri = new URI(candidateBroker);
            String path = String.format("%s/%s:%s", "/loadbalance/brokers", uri.getHost(), uri.getPort());
            ((CompletableFuture)this.pulsar.getLocalZkCache().getDataAsync(path, this.pulsar.getLoadManager().get().getLoadReportDeserializer()).thenAccept(reportData -> {
                if (reportData.isPresent()) {
                    ServiceLookupData lookupData = (ServiceLookupData)reportData.get();
                    lookupFuture.complete(new LookupResult(lookupData.getWebServiceUrl(), lookupData.getWebServiceUrlTls(), lookupData.getPulsarServiceUrl(), lookupData.getPulsarServiceUrlTls()));
                } else {
                    lookupFuture.completeExceptionally(new KeeperException.NoNodeException(path));
                }
            })).exceptionally(ex -> {
                lookupFuture.completeExceptionally((Throwable)ex);
                return null;
            });
        }
        catch (Exception e) {
            lookupFuture.completeExceptionally(e);
        }
        return lookupFuture;
    }

    private boolean isBrokerActive(String candidateBroker) throws KeeperException, InterruptedException {
        Set activeNativeBrokers = this.pulsar.getLocalZkCache().getChildren("/loadbalance/brokers");
        for (String brokerHostPort : activeNativeBrokers) {
            if (!candidateBroker.equals("http://" + brokerHostPort)) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Broker {} found for SLA Monitoring Namespace", (Object)brokerHostPort);
            }
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Broker not found for SLA Monitoring Namespace {}", (Object)(candidateBroker + ":" + this.config.getWebServicePort().get()));
        }
        return false;
    }

    private Optional<String> getLeastLoadedFromLoadManager(ServiceUnitId serviceUnit) throws Exception {
        Optional<ResourceUnit> leastLoadedBroker = this.loadManager.get().getLeastLoaded(serviceUnit);
        if (!leastLoadedBroker.isPresent()) {
            LOG.warn("No broker is available for {}", (Object)serviceUnit);
            return Optional.empty();
        }
        String lookupAddress = leastLoadedBroker.get().getResourceId();
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} : redirecting to the least loaded broker, lookup address={}", (Object)this.pulsar.getSafeWebServiceAddress(), (Object)lookupAddress);
        }
        return Optional.of(lookupAddress);
    }

    public void unloadNamespaceBundle(NamespaceBundle bundle) throws Exception {
        this.unloadNamespaceBundle(bundle, 5L, TimeUnit.MINUTES);
    }

    public void unloadNamespaceBundle(NamespaceBundle bundle, long timeout, TimeUnit timeoutUnit) throws Exception {
        ((OwnedBundle)Preconditions.checkNotNull((Object)this.ownershipCache.getOwnedBundle(bundle))).handleUnloadRequest(this.pulsar, timeout, timeoutUnit);
    }

    public CompletableFuture<Boolean> isNamespaceBundleOwned(NamespaceBundle bundle) {
        String bundlePath = ServiceUnitZkUtils.path(bundle);
        CompletableFuture<Boolean> isExistFuture = new CompletableFuture<Boolean>();
        this.pulsar.getLocalZkCache().getZooKeeper().exists(bundlePath, false, (rc, path, ctx, stat) -> {
            if (rc == KeeperException.Code.OK.intValue()) {
                isExistFuture.complete(true);
            } else if (rc == KeeperException.Code.NONODE.intValue()) {
                isExistFuture.complete(false);
            } else {
                isExistFuture.completeExceptionally((Throwable)KeeperException.create((int)rc));
            }
        }, null);
        return isExistFuture;
    }

    public Map<String, NamespaceOwnershipStatus> getOwnedNameSpacesStatus() throws Exception {
        NamespaceIsolationPolicies nsIsolationPolicies = this.getLocalNamespaceIsolationPolicies();
        HashMap<String, NamespaceOwnershipStatus> ownedNsStatus = new HashMap<String, NamespaceOwnershipStatus>();
        for (OwnedBundle nsObj : this.ownershipCache.getOwnedBundles().values()) {
            NamespaceOwnershipStatus nsStatus = this.getNamespaceOwnershipStatus(nsObj, nsIsolationPolicies.getPolicyByNamespace(nsObj.getNamespaceBundle().getNamespaceObject()));
            ownedNsStatus.put(nsObj.getNamespaceBundle().toString(), nsStatus);
        }
        return ownedNsStatus;
    }

    private NamespaceOwnershipStatus getNamespaceOwnershipStatus(OwnedBundle nsObj, NamespaceIsolationPolicy nsIsolationPolicy) {
        NamespaceOwnershipStatus nsOwnedStatus = new NamespaceOwnershipStatus(BrokerAssignment.shared, false, nsObj.isActive());
        if (nsIsolationPolicy == null) {
            return nsOwnedStatus;
        }
        nsOwnedStatus.is_controlled = true;
        if (nsIsolationPolicy.isPrimaryBroker(this.pulsar.getAdvertisedAddress())) {
            nsOwnedStatus.broker_assignment = BrokerAssignment.primary;
        } else if (nsIsolationPolicy.isSecondaryBroker(this.pulsar.getAdvertisedAddress())) {
            nsOwnedStatus.broker_assignment = BrokerAssignment.secondary;
        }
        return nsOwnedStatus;
    }

    private NamespaceIsolationPolicies getLocalNamespaceIsolationPolicies() throws Exception {
        String localCluster = this.pulsar.getConfiguration().getClusterName();
        return this.pulsar.getConfigurationCache().namespaceIsolationPoliciesCache().get(AdminResource.path("clusters", localCluster, NAMESPACE_ISOLATION_POLICIES)).orElseGet(() -> new NamespaceIsolationPolicies());
    }

    public boolean isNamespaceBundleDisabled(NamespaceBundle bundle) throws Exception {
        try {
            CompletableFuture<Optional<NamespaceEphemeralData>> nsDataFuture = this.ownershipCache.getOwnerAsync(bundle);
            if (nsDataFuture != null) {
                Optional nsData = nsDataFuture.getNow(null);
                if (nsData != null && nsData.isPresent()) {
                    return ((NamespaceEphemeralData)nsData.get()).isDisabled();
                }
                return false;
            }
            return false;
        }
        catch (Exception e) {
            LOG.warn("Exception in getting ownership info for service unit {}: {}", new Object[]{bundle, e.getMessage(), e});
            return false;
        }
    }

    public CompletableFuture<Void> splitAndOwnBundle(NamespaceBundle bundle, boolean unload) throws Exception {
        CompletableFuture<Void> unloadFuture = new CompletableFuture<Void>();
        AtomicInteger counter = new AtomicInteger(7);
        this.splitAndOwnBundleOnceAndRetry(bundle, unload, counter, unloadFuture);
        return unloadFuture;
    }

    void splitAndOwnBundleOnceAndRetry(NamespaceBundle bundle, boolean unload, AtomicInteger counter, CompletableFuture<Void> unloadFuture) {
        CompletableFuture updateFuture = new CompletableFuture();
        Pair<NamespaceBundles, List<NamespaceBundle>> splittedBundles = this.bundleFactory.splitBundles(bundle, 2);
        if (splittedBundles != null) {
            Preconditions.checkNotNull((Object)splittedBundles.getLeft());
            Preconditions.checkNotNull((Object)splittedBundles.getRight());
            Preconditions.checkArgument((((List)splittedBundles.getRight()).size() == 2 ? 1 : 0) != 0, (Object)"bundle has to be split in two bundles");
            NamespaceName nsname = bundle.getNamespaceObject();
            if (LOG.isDebugEnabled()) {
                LOG.debug("[{}] splitAndOwnBundleOnce: {}, counter: {},  2 bundles: {}, {}", new Object[]{nsname.toString(), bundle.getBundleRange(), counter.get(), splittedBundles != null ? ((NamespaceBundle)((List)splittedBundles.getRight()).get(0)).getBundleRange() : "null splittedBundles", splittedBundles != null ? ((NamespaceBundle)((List)splittedBundles.getRight()).get(1)).getBundleRange() : "null splittedBundles"});
            }
            try {
                for (NamespaceBundle sBundle : (List)splittedBundles.getRight()) {
                    Preconditions.checkNotNull(this.ownershipCache.tryAcquiringOwnership(sBundle));
                }
                this.updateNamespaceBundles(nsname, (NamespaceBundles)splittedBundles.getLeft(), (rc, path, zkCtx, stat) -> {
                    if (rc == KeeperException.Code.OK.intValue()) {
                        this.bundleFactory.invalidateBundleCache(bundle.getNamespaceObject());
                        updateFuture.complete(splittedBundles.getRight());
                    } else if (rc == KeeperException.Code.BADVERSION.intValue()) {
                        KeeperException keeperException = KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc));
                        String msg = String.format("failed to update namespace policies [%s], NamespaceBundle: %s due to %s, counter: %d", nsname.toString(), bundle.getBundleRange(), keeperException.getMessage(), counter.get());
                        LOG.warn(msg);
                        updateFuture.completeExceptionally(new BrokerServiceException.ServerMetadataException((Throwable)keeperException));
                    } else {
                        String msg = String.format("failed to update namespace policies [%s], NamespaceBundle: %s due to %s", nsname.toString(), bundle.getBundleRange(), KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc)).getMessage());
                        LOG.warn(msg);
                        updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg));
                    }
                });
            }
            catch (Exception e) {
                String msg = String.format("failed to acquire ownership of split bundle for namespace [%s], %s", nsname.toString(), e.getMessage());
                LOG.warn(msg, (Throwable)e);
                updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg));
            }
        } else {
            String msg = String.format("bundle %s not found under namespace", bundle.toString());
            LOG.warn(msg);
            updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg));
        }
        updateFuture.whenCompleteAsync((r, t) -> {
            if (t != null) {
                if (t instanceof BrokerServiceException.ServerMetadataException && counter.decrementAndGet() >= 0) {
                    this.pulsar.getOrderedExecutor().execute(() -> this.splitAndOwnBundleOnceAndRetry(bundle, unload, counter, unloadFuture));
                } else {
                    String msg2 = String.format(" %s not success update nsBundles, counter %d, reason %s", bundle.toString(), counter.get(), t.getMessage());
                    LOG.warn(msg2);
                    unloadFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg2));
                }
                return;
            }
            try {
                this.getOwnershipCache().updateBundleState(bundle, false);
                this.pulsar.getBrokerService().refreshTopicToStatsMaps(bundle);
                this.loadManager.get().setLoadReportForceUpdateFlag();
                if (unload) {
                    r.forEach(splitBundle -> {
                        try {
                            this.unloadNamespaceBundle((NamespaceBundle)splitBundle);
                        }
                        catch (Exception e) {
                            LOG.warn("Failed to unload split bundle {}", splitBundle, (Object)e);
                            throw new RuntimeException("Failed to unload split bundle " + splitBundle, e);
                        }
                    });
                }
                unloadFuture.complete(null);
            }
            catch (Exception e) {
                String msg1 = String.format("failed to disable bundle %s under namespace [%s] with error %s", bundle.getNamespaceObject().toString(), bundle.toString(), e.getMessage());
                LOG.warn(msg1, (Throwable)e);
                unloadFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg1));
            }
        }, (Executor)this.pulsar.getOrderedExecutor());
    }

    private void updateNamespaceBundles(NamespaceName nsname, NamespaceBundles nsBundles, AsyncCallback.StatCallback callback) throws Exception {
        Preconditions.checkNotNull((Object)nsname);
        Preconditions.checkNotNull((Object)nsBundles);
        String path = PulsarWebResource.joinPath("/admin/local-policies", nsname.toString());
        Optional policies = this.pulsar.getLocalZkCacheService().policiesCache().get(path);
        if (!policies.isPresent()) {
            this.pulsar.getLocalZkCacheService().createPolicies(path, false).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        }
        long version = nsBundles.getVersion();
        LocalPolicies local = new LocalPolicies();
        local.bundles = NamespaceBundleFactory.getBundlesData(nsBundles);
        byte[] data = ObjectMapperFactory.getThreadLocal().writeValueAsBytes((Object)local);
        this.pulsar.getLocalZkCache().getZooKeeper().setData(path, data, Math.toIntExact(version), callback, null);
        this.pulsar.getLocalZkCacheService().policiesCache().invalidate(path);
    }

    public OwnershipCache getOwnershipCache() {
        return this.ownershipCache;
    }

    public int getTotalServiceUnitsLoaded() {
        return this.ownershipCache.getOwnedBundles().size() - this.uncountedNamespaces;
    }

    public Set<NamespaceBundle> getOwnedServiceUnits() {
        return this.ownershipCache.getOwnedBundles().values().stream().map(OwnedBundle::getNamespaceBundle).collect(Collectors.toSet());
    }

    public boolean isServiceUnitOwned(ServiceUnitId suName) throws Exception {
        if (suName instanceof TopicName) {
            return this.isTopicOwned((TopicName)suName);
        }
        if (suName instanceof NamespaceName) {
            return this.isNamespaceOwned((NamespaceName)suName);
        }
        if (suName instanceof NamespaceBundle) {
            return this.ownershipCache.isNamespaceBundleOwned((NamespaceBundle)suName);
        }
        throw new IllegalArgumentException("Invalid class of NamespaceBundle: " + suName.getClass().getName());
    }

    public boolean isServiceUnitActive(TopicName topicName) {
        try {
            return this.ownershipCache.getOwnedBundle(this.getBundle(topicName)).isActive();
        }
        catch (Exception e) {
            LOG.warn("Unable to find OwnedBundle for topic - [{}]", (Object)topicName);
            return false;
        }
    }

    private boolean isNamespaceOwned(NamespaceName fqnn) throws Exception {
        return this.ownershipCache.getOwnedBundle(this.getFullBundle(fqnn)) != null;
    }

    private CompletableFuture<Boolean> isTopicOwnedAsync(TopicName topic) {
        return this.getBundleAsync(topic).thenApply(bundle -> this.ownershipCache.isNamespaceBundleOwned((NamespaceBundle)bundle));
    }

    private boolean isTopicOwned(TopicName topicName) throws Exception {
        Optional<NamespaceBundle> bundle = this.getBundleIfPresent(topicName);
        if (!bundle.isPresent()) {
            return false;
        }
        return this.ownershipCache.getOwnedBundle(bundle.get()) != null;
    }

    public void removeOwnedServiceUnit(NamespaceName nsName) throws Exception {
        this.ownershipCache.removeOwnership(this.getFullBundle(nsName)).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        this.bundleFactory.invalidateBundleCache(nsName);
    }

    public void removeOwnedServiceUnit(NamespaceBundle nsBundle) throws Exception {
        this.ownershipCache.removeOwnership(nsBundle).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        this.bundleFactory.invalidateBundleCache(nsBundle.getNamespaceObject());
    }

    public void removeOwnedServiceUnits(NamespaceName nsName, BundlesData bundleData) throws Exception {
        this.ownershipCache.removeOwnership(this.bundleFactory.getBundles(nsName, bundleData)).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        this.bundleFactory.invalidateBundleCache(nsName);
    }

    public NamespaceBundleFactory getNamespaceBundleFactory() {
        return this.bundleFactory;
    }

    public ServiceUnitId getServiceUnitId(TopicName topicName) throws Exception {
        return this.getBundle(topicName);
    }

    public List<String> getFullListOfTopics(NamespaceName namespaceName) throws Exception {
        List<String> topics = this.getListOfPersistentTopics(namespaceName);
        topics.addAll(this.getListOfNonPersistentTopics(namespaceName));
        return topics;
    }

    public List<String> getListOfTopics(NamespaceName namespaceName, PulsarApi.CommandGetTopicsOfNamespace.Mode mode) throws Exception {
        switch (mode) {
            case ALL: {
                return this.getFullListOfTopics(namespaceName);
            }
            case NON_PERSISTENT: {
                return this.getListOfNonPersistentTopics(namespaceName);
            }
        }
        return this.getListOfPersistentTopics(namespaceName);
    }

    public List<String> getListOfPersistentTopics(NamespaceName namespaceName) throws Exception {
        ArrayList topics = Lists.newArrayList();
        try {
            String path = String.format("/managed-ledgers/%s/persistent", namespaceName);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Getting children from managed-ledgers now: {}", (Object)path);
            }
            for (String topic : this.pulsar.getLocalZkCacheService().managedLedgerListCache().get(path)) {
                topics.add(String.format("persistent://%s/%s", namespaceName, Codec.decode((String)topic)));
            }
        }
        catch (KeeperException.NoNodeException noNodeException) {
            // empty catch block
        }
        topics.sort(null);
        return topics;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<String> getListOfNonPersistentTopics(NamespaceName namespaceName) throws Exception {
        ClusterData peerClusterData;
        ArrayList topics = Lists.newArrayList();
        try {
            peerClusterData = PulsarWebResource.checkLocalOrGetPeerReplicationCluster(this.pulsar, namespaceName).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new RuntimeException("Failed to contact peer replication cluster.", e);
        }
        if (peerClusterData != null) {
            return this.getNonPersistentTopicsFromPeerCluster(peerClusterData, namespaceName);
        }
        ConcurrentOpenHashMap<String, ConcurrentOpenHashMap<String, ConcurrentOpenHashMap<String, Topic>>> concurrentOpenHashMap = this.pulsar.getBrokerService().getMultiLayerTopicMap();
        synchronized (concurrentOpenHashMap) {
            if (this.pulsar.getBrokerService().getMultiLayerTopicMap().containsKey((Object)namespaceName.toString())) {
                ((ConcurrentOpenHashMap)this.pulsar.getBrokerService().getMultiLayerTopicMap().get((Object)namespaceName.toString())).values().forEach(bundle -> bundle.forEach((topicName, topic) -> {
                    if (topic instanceof NonPersistentTopic && ((NonPersistentTopic)topic).isActive()) {
                        topics.add(topicName);
                    }
                }));
            }
        }
        topics.sort(null);
        return topics;
    }

    private List<String> getNonPersistentTopicsFromPeerCluster(ClusterData peerClusterData, NamespaceName namespace) throws Exception {
        PulsarClientImpl client = this.getNamespaceClient(peerClusterData);
        return (List)client.getLookup().getTopicsUnderNamespace(namespace, PulsarApi.CommandGetTopicsOfNamespace.Mode.NON_PERSISTENT).get();
    }

    public PulsarClientImpl getNamespaceClient(ClusterData cluster) {
        PulsarClientImpl client = (PulsarClientImpl)this.namespaceClients.get((Object)cluster);
        if (client != null) {
            return client;
        }
        return (PulsarClientImpl)this.namespaceClients.computeIfAbsent((Object)cluster, key -> {
            try {
                ClientBuilder clientBuilder = PulsarClient.builder().enableTcpNoDelay(false).statsInterval(0L, TimeUnit.SECONDS);
                if (this.pulsar.getConfiguration().isAuthenticationEnabled()) {
                    clientBuilder.authentication(this.pulsar.getConfiguration().getBrokerClientAuthenticationPlugin(), this.pulsar.getConfiguration().getBrokerClientAuthenticationParameters());
                }
                if (this.pulsar.getConfiguration().isTlsEnabled()) {
                    clientBuilder.serviceUrl(StringUtils.isNotBlank((CharSequence)cluster.getBrokerServiceUrlTls()) ? cluster.getBrokerServiceUrlTls() : cluster.getServiceUrlTls()).enableTls(true).tlsTrustCertsFilePath(this.pulsar.getConfiguration().getBrokerClientTrustCertsFilePath()).allowTlsInsecureConnection(this.pulsar.getConfiguration().isTlsAllowInsecureConnection());
                } else {
                    clientBuilder.serviceUrl(StringUtils.isNotBlank((CharSequence)cluster.getBrokerServiceUrl()) ? cluster.getBrokerServiceUrl() : cluster.getServiceUrl());
                }
                ClientConfigurationData conf = ((ClientBuilderImpl)clientBuilder).getClientConfigurationData();
                return new PulsarClientImpl(conf, (EventLoopGroup)this.pulsar.getBrokerService().executor());
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    public Optional<NamespaceEphemeralData> getOwner(NamespaceBundle bundle) throws Exception {
        return this.getOwnerAsync(bundle).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
    }

    public CompletableFuture<Optional<NamespaceEphemeralData>> getOwnerAsync(NamespaceBundle bundle) {
        return this.ownershipCache.getOwnerAsync(bundle);
    }

    public void unloadSLANamespace() throws Exception {
        PulsarAdmin adminClient = null;
        String namespaceName = NamespaceService.getSLAMonitorNamespace(this.host, this.config);
        LOG.info("Checking owner for SLA namespace {}", (Object)namespaceName);
        NamespaceBundle nsFullBundle = this.getFullBundle(NamespaceName.get((String)namespaceName));
        if (!this.getOwner(nsFullBundle).isPresent()) {
            return;
        }
        LOG.info("Trying to unload SLA namespace {}", (Object)namespaceName);
        adminClient = this.pulsar.getAdminClient();
        adminClient.namespaces().unload(namespaceName);
        LOG.info("Namespace {} unloaded successfully", (Object)namespaceName);
    }

    public static String getHeartbeatNamespace(String host, ServiceConfiguration config) {
        Integer port = null;
        if (config.getWebServicePort().isPresent()) {
            port = (Integer)config.getWebServicePort().get();
        } else if (config.getWebServicePortTls().isPresent()) {
            port = (Integer)config.getWebServicePortTls().get();
        }
        return String.format(HEARTBEAT_NAMESPACE_FMT, config.getClusterName(), host, port);
    }

    public static String getSLAMonitorNamespace(String host, ServiceConfiguration config) {
        Integer port = null;
        if (config.getWebServicePort().isPresent()) {
            port = (Integer)config.getWebServicePort().get();
        } else if (config.getWebServicePortTls().isPresent()) {
            port = (Integer)config.getWebServicePortTls().get();
        }
        return String.format(SLA_NAMESPACE_FMT, config.getClusterName(), host, port);
    }

    public static String checkHeartbeatNamespace(ServiceUnitId ns) {
        Matcher m = HEARTBEAT_NAMESPACE_PATTERN.matcher(ns.getNamespaceObject().toString());
        if (m.matches()) {
            LOG.debug("SLAMonitoring namespace matched the lookup namespace {}", (Object)ns.getNamespaceObject().toString());
            return String.format("http://%s", m.group(1));
        }
        return null;
    }

    public static String getSLAMonitorBrokerName(ServiceUnitId ns) {
        Matcher m = SLA_NAMESPACE_PATTERN.matcher(ns.getNamespaceObject().toString());
        if (m.matches()) {
            return String.format("http://%s", m.group(1));
        }
        return null;
    }

    public boolean registerSLANamespace() throws PulsarServerException {
        boolean isNameSpaceRegistered = this.registerNamespace(NamespaceService.getSLAMonitorNamespace(this.host, this.config), false);
        if (isNameSpaceRegistered) {
            ++this.uncountedNamespaces;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Added SLA Monitoring namespace name in local cache: ns={}", (Object)NamespaceService.getSLAMonitorNamespace(this.host, this.config));
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("SLA Monitoring not owned by the broker: ns={}", (Object)NamespaceService.getSLAMonitorNamespace(this.host, this.config));
        }
        return isNameSpaceRegistered;
    }

    public static enum AddressType {
        BROKER_URL,
        LOOKUP_URL;

    }
}

