/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.service.BrokerService;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.broker.web.PulsarWebResource;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.policies.data.DispatchRate;
import org.apache.pulsar.common.policies.data.Policies;
import org.apache.pulsar.common.util.RateLimiter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DispatchRateLimiter {
    private final PersistentTopic topic;
    private final String topicName;
    private final Type type;
    private final BrokerService brokerService;
    private RateLimiter dispatchRateLimiterOnMessage;
    private RateLimiter dispatchRateLimiterOnByte;
    private long subscriptionRelativeRatelimiterOnMessage;
    private long subscriptionRelativeRatelimiterOnByte;
    private static final Logger log = LoggerFactory.getLogger(DispatchRateLimiter.class);

    public DispatchRateLimiter(PersistentTopic topic, Type type) {
        this.topic = topic;
        this.topicName = topic.getName();
        this.brokerService = topic.getBrokerService();
        this.type = type;
        this.subscriptionRelativeRatelimiterOnMessage = -1L;
        this.subscriptionRelativeRatelimiterOnByte = -1L;
        this.updateDispatchRate();
    }

    public long getAvailableDispatchRateLimitOnMsg() {
        return this.dispatchRateLimiterOnMessage == null ? -1L : this.dispatchRateLimiterOnMessage.getAvailablePermits();
    }

    public boolean tryDispatchPermit(long msgPermits, long bytePermits) {
        boolean acquiredMsgPermit = msgPermits <= 0L || this.dispatchRateLimiterOnMessage == null || this.dispatchRateLimiterOnMessage.tryAcquire(msgPermits);
        boolean acquiredBytePermit = bytePermits <= 0L || this.dispatchRateLimiterOnByte == null || this.dispatchRateLimiterOnByte.tryAcquire(bytePermits);
        return acquiredMsgPermit && acquiredBytePermit;
    }

    public boolean hasMessageDispatchPermit() {
        return !(this.dispatchRateLimiterOnMessage != null && this.dispatchRateLimiterOnMessage.getAvailablePermits() <= 0L || this.dispatchRateLimiterOnByte != null && this.dispatchRateLimiterOnByte.getAvailablePermits() <= 0L);
    }

    public boolean isDispatchRateLimitingEnabled() {
        return this.dispatchRateLimiterOnMessage != null || this.dispatchRateLimiterOnByte != null;
    }

    private DispatchRate createDispatchRate() {
        long dispatchThrottlingRateInByte;
        int dispatchThrottlingRateInMsg;
        ServiceConfiguration config = this.brokerService.pulsar().getConfiguration();
        switch (this.type) {
            case TOPIC: {
                dispatchThrottlingRateInMsg = config.getDispatchThrottlingRatePerTopicInMsg();
                dispatchThrottlingRateInByte = config.getDispatchThrottlingRatePerTopicInByte();
                break;
            }
            case SUBSCRIPTION: {
                dispatchThrottlingRateInMsg = config.getDispatchThrottlingRatePerSubscriptionInMsg();
                dispatchThrottlingRateInByte = config.getDispatchThrottlingRatePerSubscriptionInByte();
                break;
            }
            case REPLICATOR: {
                dispatchThrottlingRateInMsg = config.getDispatchThrottlingRatePerReplicatorInMsg();
                dispatchThrottlingRateInByte = config.getDispatchThrottlingRatePerReplicatorInByte();
                break;
            }
            default: {
                dispatchThrottlingRateInMsg = -1;
                dispatchThrottlingRateInByte = -1L;
            }
        }
        return new DispatchRate(dispatchThrottlingRateInMsg, dispatchThrottlingRateInByte, 1, config.isDispatchThrottlingRateRelativeToPublishRate());
    }

    public void updateDispatchRate() {
        DispatchRate dispatchRate = this.getPoliciesDispatchRate(this.brokerService);
        if (dispatchRate == null) {
            dispatchRate = this.createDispatchRate();
        }
        this.updateDispatchRate(dispatchRate);
        log.info("[{}] configured {} message-dispatch rate at broker {}", new Object[]{this.topicName, this.type, dispatchRate});
    }

    public static boolean isDispatchRateNeeded(BrokerService brokerService, Optional<Policies> policies, String topicName, Type type) {
        ServiceConfiguration serviceConfig = brokerService.pulsar().getConfiguration();
        policies = policies.isPresent() ? policies : DispatchRateLimiter.getPolicies(brokerService, topicName);
        return DispatchRateLimiter.isDispatchRateNeeded(serviceConfig, policies, topicName, type);
    }

    public static boolean isDispatchRateNeeded(ServiceConfiguration serviceConfig, Optional<Policies> policies, String topicName, Type type) {
        DispatchRate dispatchRate = DispatchRateLimiter.getPoliciesDispatchRate(serviceConfig.getClusterName(), policies, type);
        if (dispatchRate == null) {
            switch (type) {
                case TOPIC: {
                    return serviceConfig.getDispatchThrottlingRatePerTopicInMsg() > 0 || serviceConfig.getDispatchThrottlingRatePerTopicInByte() > 0L;
                }
                case SUBSCRIPTION: {
                    return serviceConfig.getDispatchThrottlingRatePerSubscriptionInMsg() > 0 || serviceConfig.getDispatchThrottlingRatePerSubscriptionInByte() > 0L;
                }
                case REPLICATOR: {
                    return serviceConfig.getDispatchThrottlingRatePerReplicatorInMsg() > 0 || serviceConfig.getDispatchThrottlingRatePerReplicatorInByte() > 0L;
                }
            }
            log.error("error DispatchRateLimiter type: {} ", (Object)type);
            return false;
        }
        return true;
    }

    public void onPoliciesUpdate(Policies data) {
        DispatchRate dispatchRate;
        String cluster = this.brokerService.pulsar().getConfiguration().getClusterName();
        switch (this.type) {
            case TOPIC: {
                dispatchRate = (DispatchRate)data.topicDispatchRate.get(cluster);
                if (dispatchRate != null) break;
                dispatchRate = (DispatchRate)data.clusterDispatchRate.get(cluster);
                break;
            }
            case SUBSCRIPTION: {
                dispatchRate = (DispatchRate)data.subscriptionDispatchRate.get(cluster);
                break;
            }
            case REPLICATOR: {
                dispatchRate = (DispatchRate)data.replicatorDispatchRate.get(cluster);
                break;
            }
            default: {
                log.error("error DispatchRateLimiter type: {} ", (Object)this.type);
                dispatchRate = null;
            }
        }
        if (dispatchRate != null) {
            DispatchRate newDispatchRate = this.createDispatchRate();
            if (!DispatchRateLimiter.isDispatchRateEnabled(dispatchRate) && DispatchRateLimiter.isDispatchRateEnabled(newDispatchRate)) {
                dispatchRate = newDispatchRate;
            }
            this.updateDispatchRate(dispatchRate);
        }
    }

    public static DispatchRate getPoliciesDispatchRate(String cluster, Optional<Policies> policies, Type type) {
        return policies.map(p -> {
            DispatchRate dispatchRate;
            switch (type) {
                case TOPIC: {
                    dispatchRate = (DispatchRate)p.topicDispatchRate.get(cluster);
                    if (dispatchRate != null) break;
                    dispatchRate = (DispatchRate)p.clusterDispatchRate.get(cluster);
                    break;
                }
                case SUBSCRIPTION: {
                    dispatchRate = (DispatchRate)p.subscriptionDispatchRate.get(cluster);
                    break;
                }
                case REPLICATOR: {
                    dispatchRate = (DispatchRate)p.replicatorDispatchRate.get(cluster);
                    break;
                }
                default: {
                    log.error("error DispatchRateLimiter type: {} ", (Object)type);
                    return null;
                }
            }
            return DispatchRateLimiter.isDispatchRateEnabled(dispatchRate) ? dispatchRate : null;
        }).orElse(null);
    }

    public DispatchRate getPoliciesDispatchRate(BrokerService brokerService) {
        String cluster = brokerService.pulsar().getConfiguration().getClusterName();
        Optional<Policies> policies = DispatchRateLimiter.getPolicies(brokerService, this.topicName);
        return DispatchRateLimiter.getPoliciesDispatchRate(cluster, policies, this.type);
    }

    public static Optional<Policies> getPolicies(BrokerService brokerService, String topicName) {
        NamespaceName namespace = TopicName.get((String)topicName).getNamespaceObject();
        String path = PulsarWebResource.path("policies", namespace.toString());
        Optional policies = Optional.empty();
        try {
            policies = (Optional)brokerService.pulsar().getConfigurationCache().policiesCache().getAsync(path).get(brokerService.pulsar().getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        }
        catch (Exception e) {
            log.warn("Failed to get message-rate for {} ", (Object)topicName, (Object)e);
        }
        return policies;
    }

    public synchronized void updateDispatchRate(DispatchRate dispatchRate) {
        Supplier<Long> permitUpdaterByte;
        Supplier<Long> permitUpdaterMsg;
        log.info("setting message-dispatch-rate {}", (Object)dispatchRate);
        long msgRate = dispatchRate.dispatchThrottlingRateInMsg;
        long byteRate = dispatchRate.dispatchThrottlingRateInByte;
        long ratePeriod = dispatchRate.ratePeriodInSecond;
        Supplier<Long> supplier = permitUpdaterMsg = dispatchRate.relativeToPublishRate ? () -> this.getRelativeDispatchRateInMsg(dispatchRate) : null;
        if (msgRate > 0L) {
            if (this.dispatchRateLimiterOnMessage == null) {
                this.dispatchRateLimiterOnMessage = new RateLimiter(this.brokerService.pulsar().getExecutor(), msgRate, ratePeriod, TimeUnit.SECONDS, permitUpdaterMsg);
            } else {
                this.dispatchRateLimiterOnMessage.setRate(msgRate, (long)dispatchRate.ratePeriodInSecond, TimeUnit.SECONDS, permitUpdaterMsg);
            }
        } else if (this.dispatchRateLimiterOnMessage != null) {
            this.dispatchRateLimiterOnMessage.close();
            this.dispatchRateLimiterOnMessage = null;
        }
        Supplier<Long> supplier2 = permitUpdaterByte = dispatchRate.relativeToPublishRate ? () -> this.getRelativeDispatchRateInByte(dispatchRate) : null;
        if (byteRate > 0L) {
            if (this.dispatchRateLimiterOnByte == null) {
                this.dispatchRateLimiterOnByte = new RateLimiter(this.brokerService.pulsar().getExecutor(), byteRate, ratePeriod, TimeUnit.SECONDS, permitUpdaterByte);
            } else {
                this.dispatchRateLimiterOnByte.setRate(byteRate, (long)dispatchRate.ratePeriodInSecond, TimeUnit.SECONDS, permitUpdaterByte);
            }
        } else if (this.dispatchRateLimiterOnByte != null) {
            this.dispatchRateLimiterOnByte.close();
            this.dispatchRateLimiterOnByte = null;
        }
    }

    private long getRelativeDispatchRateInMsg(DispatchRate dispatchRate) {
        return this.topic != null && dispatchRate != null ? (long)this.topic.getLastUpdatedAvgPublishRateInMsg() + (long)dispatchRate.dispatchThrottlingRateInMsg : 0L;
    }

    private long getRelativeDispatchRateInByte(DispatchRate dispatchRate) {
        return this.topic != null && dispatchRate != null ? (long)this.topic.getLastUpdatedAvgPublishRateInByte() + dispatchRate.dispatchThrottlingRateInByte : 0L;
    }

    public long getDispatchRateOnMsg() {
        return this.dispatchRateLimiterOnMessage != null ? this.dispatchRateLimiterOnMessage.getRate() : -1L;
    }

    public long getDispatchRateOnByte() {
        return this.dispatchRateLimiterOnByte != null ? this.dispatchRateLimiterOnByte.getRate() : -1L;
    }

    private static boolean isDispatchRateEnabled(DispatchRate dispatchRate) {
        return dispatchRate != null && (dispatchRate.dispatchThrottlingRateInMsg > 0 || dispatchRate.dispatchThrottlingRateInByte > 0L);
    }

    public void close() {
        if (this.dispatchRateLimiterOnMessage != null) {
            this.dispatchRateLimiterOnMessage.close();
            this.dispatchRateLimiterOnMessage = null;
        }
        if (this.dispatchRateLimiterOnByte != null) {
            this.dispatchRateLimiterOnByte.close();
            this.dispatchRateLimiterOnByte = null;
        }
    }

    public static enum Type {
        TOPIC,
        SUBSCRIPTION,
        REPLICATOR;

    }
}

