/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.namespace;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.hash.Hashing;
import io.prometheus.client.Counter;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.broker.PulsarServerException;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.loadbalance.LeaderBroker;
import org.apache.pulsar.broker.loadbalance.LeaderElectionService;
import org.apache.pulsar.broker.loadbalance.LoadManager;
import org.apache.pulsar.broker.loadbalance.ResourceUnit;
import org.apache.pulsar.broker.lookup.LookupResult;
import org.apache.pulsar.broker.namespace.LookupOptions;
import org.apache.pulsar.broker.namespace.NamespaceBundleOwnershipListener;
import org.apache.pulsar.broker.namespace.NamespaceEphemeralData;
import org.apache.pulsar.broker.namespace.OwnedBundle;
import org.apache.pulsar.broker.namespace.OwnershipCache;
import org.apache.pulsar.broker.namespace.ServiceUnitUtils;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.nonpersistent.NonPersistentTopic;
import org.apache.pulsar.broker.stats.prometheus.metrics.Summary;
import org.apache.pulsar.broker.web.PulsarWebResource;
import org.apache.pulsar.client.admin.PulsarAdmin;
import org.apache.pulsar.client.api.ClientBuilder;
import org.apache.pulsar.client.api.PulsarClient;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.client.impl.ClientBuilderImpl;
import org.apache.pulsar.client.impl.PulsarClientImpl;
import org.apache.pulsar.client.impl.conf.ClientConfigurationData;
import org.apache.pulsar.client.internal.PropertiesUtils;
import org.apache.pulsar.common.api.proto.CommandGetTopicsOfNamespace;
import org.apache.pulsar.common.lookup.data.LookupData;
import org.apache.pulsar.common.naming.NamespaceBundle;
import org.apache.pulsar.common.naming.NamespaceBundleFactory;
import org.apache.pulsar.common.naming.NamespaceBundleSplitAlgorithm;
import org.apache.pulsar.common.naming.NamespaceBundles;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.ServiceUnitId;
import org.apache.pulsar.common.naming.TopicDomain;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.policies.NamespaceIsolationPolicy;
import org.apache.pulsar.common.policies.data.BrokerAssignment;
import org.apache.pulsar.common.policies.data.ClusterDataImpl;
import org.apache.pulsar.common.policies.data.LocalPolicies;
import org.apache.pulsar.common.policies.data.NamespaceOwnershipStatus;
import org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicies;
import org.apache.pulsar.common.util.Codec;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.common.util.ObjectMapperFactory;
import org.apache.pulsar.common.util.collections.ConcurrentOpenHashMap;
import org.apache.pulsar.metadata.api.MetadataCache;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.policies.data.loadbalancer.AdvertisedListener;
import org.apache.pulsar.policies.data.loadbalancer.LocalBrokerData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NamespaceService
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(NamespaceService.class);
    private final ServiceConfiguration config;
    private final MetadataCache<LocalPolicies> localPoliciesCache;
    private final AtomicReference<LoadManager> loadManager;
    private final PulsarService pulsar;
    private final OwnershipCache ownershipCache;
    private final MetadataCache<LocalBrokerData> localBrokerDataCache;
    private final NamespaceBundleFactory bundleFactory;
    private int uncountedNamespaces;
    private final String host;
    private static final int BUNDLE_SPLIT_RETRY_LIMIT = 7;
    public static final String SLA_NAMESPACE_PROPERTY = "sla-monitor";
    public static final Pattern HEARTBEAT_NAMESPACE_PATTERN = Pattern.compile("pulsar/[^/]+/([^:]+:\\d+)");
    public static final Pattern SLA_NAMESPACE_PATTERN = Pattern.compile("sla-monitor/[^/]+/([^:]+:\\d+)");
    public static final String HEARTBEAT_NAMESPACE_FMT = "pulsar/%s/%s:%s";
    public static final String SLA_NAMESPACE_FMT = "sla-monitor/%s/%s:%s";
    public static final String NAMESPACE_ISOLATION_POLICIES = "namespaceIsolationPolicies";
    private final ConcurrentOpenHashMap<ClusterDataImpl, PulsarClientImpl> namespaceClients;
    private final List<NamespaceBundleOwnershipListener> bundleOwnershipListeners;
    private static final Counter lookupRedirects = (Counter)Counter.build((String)"pulsar_broker_lookup_redirects", (String)"-").register();
    private static final Counter lookupFailures = (Counter)Counter.build((String)"pulsar_broker_lookup_failures", (String)"-").register();
    private static final Counter lookupAnswers = (Counter)Counter.build((String)"pulsar_broker_lookup_answers", (String)"-").register();
    private static final Summary lookupLatency = (Summary)Summary.build("pulsar_broker_lookup", "-").quantile(0.5).quantile(0.99).quantile(0.999).quantile(1.0).register();
    private final ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> findingBundlesAuthoritative = ConcurrentOpenHashMap.newBuilder().build();
    private final ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> findingBundlesNotAuthoritative = ConcurrentOpenHashMap.newBuilder().build();

    public NamespaceService(PulsarService pulsar) {
        this.pulsar = pulsar;
        this.host = pulsar.getAdvertisedAddress();
        this.config = pulsar.getConfiguration();
        this.loadManager = pulsar.getLoadManager();
        this.bundleFactory = new NamespaceBundleFactory(pulsar, Hashing.crc32());
        this.ownershipCache = new OwnershipCache(pulsar, this.bundleFactory, this);
        this.namespaceClients = ConcurrentOpenHashMap.newBuilder().build();
        this.bundleOwnershipListeners = new CopyOnWriteArrayList<NamespaceBundleOwnershipListener>();
        this.localBrokerDataCache = pulsar.getLocalMetadataStore().getMetadataCache(LocalBrokerData.class);
        this.localPoliciesCache = pulsar.getLocalMetadataStore().getMetadataCache(LocalPolicies.class);
    }

    public void initialize() {
        if (!this.getOwnershipCache().refreshSelfOwnerInfo()) {
            throw new RuntimeException("Failed to refresh self owner info.");
        }
    }

    public CompletableFuture<Optional<LookupResult>> getBrokerServiceUrlAsync(TopicName topic, LookupOptions options) {
        long startTime = System.nanoTime();
        CompletionStage future = this.getBundleAsync(topic).thenCompose(bundle -> this.findBrokerServiceUrl((NamespaceBundle)bundle, options));
        ((CompletableFuture)((CompletableFuture)future).thenAccept(optResult -> {
            lookupLatency.observe(System.nanoTime() - startTime, TimeUnit.NANOSECONDS);
            if (optResult.isPresent()) {
                if (((LookupResult)optResult.get()).isRedirect()) {
                    lookupRedirects.inc();
                } else {
                    lookupAnswers.inc();
                }
            }
        })).exceptionally(ex -> {
            lookupFailures.inc();
            return null;
        });
        return future;
    }

    public CompletableFuture<NamespaceBundle> getBundleAsync(TopicName topic) {
        return this.bundleFactory.getBundlesAsync(topic.getNamespaceObject()).thenApply(bundles -> bundles.findBundle(topic));
    }

    public Optional<NamespaceBundle> getBundleIfPresent(TopicName topicName) {
        Optional<NamespaceBundles> bundles = this.bundleFactory.getBundlesIfPresent(topicName.getNamespaceObject());
        return bundles.map(b -> b.findBundle(topicName));
    }

    public NamespaceBundle getBundle(TopicName topicName) throws Exception {
        return this.bundleFactory.getBundles(topicName.getNamespaceObject()).findBundle(topicName);
    }

    public int getBundleCount(NamespaceName namespace) throws Exception {
        return this.bundleFactory.getBundles(namespace).size();
    }

    private NamespaceBundle getFullBundle(NamespaceName fqnn) throws Exception {
        return this.bundleFactory.getFullBundle(fqnn);
    }

    private CompletableFuture<NamespaceBundle> getFullBundleAsync(NamespaceName fqnn) {
        return this.bundleFactory.getFullBundleAsync(fqnn);
    }

    public CompletableFuture<Optional<URL>> getWebServiceUrlAsync(ServiceUnitId suName, LookupOptions options) {
        if (suName instanceof TopicName) {
            TopicName name = (TopicName)suName;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Getting web service URL of topic: {} - options: {}", (Object)name, (Object)options);
            }
            return this.getBundleAsync(name).thenCompose(namespaceBundle -> this.internalGetWebServiceUrl((NamespaceBundle)namespaceBundle, options));
        }
        if (suName instanceof NamespaceName) {
            return this.getFullBundleAsync((NamespaceName)suName).thenCompose(namespaceBundle -> this.internalGetWebServiceUrl((NamespaceBundle)namespaceBundle, options));
        }
        if (suName instanceof NamespaceBundle) {
            return this.internalGetWebServiceUrl((NamespaceBundle)suName, options);
        }
        throw new IllegalArgumentException("Unrecognized class of NamespaceBundle: " + suName.getClass().getName());
    }

    public Optional<URL> getWebServiceUrl(ServiceUnitId suName, LookupOptions options) throws Exception {
        return this.getWebServiceUrlAsync(suName, options).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
    }

    private CompletableFuture<Optional<URL>> internalGetWebServiceUrl(NamespaceBundle bundle, LookupOptions options) {
        return this.findBrokerServiceUrl(bundle, options).thenApply(lookupResult -> {
            if (lookupResult.isPresent()) {
                try {
                    LookupData lookupData = ((LookupResult)lookupResult.get()).getLookupData();
                    String redirectUrl = options.isRequestHttps() ? lookupData.getHttpUrlTls() : lookupData.getHttpUrl();
                    return Optional.of(new URL(redirectUrl));
                }
                catch (Exception e) {
                    LOG.warn("internalGetWebServiceUrl [{}]", (Object)e.getMessage(), (Object)e);
                }
            }
            return Optional.empty();
        });
    }

    public void registerBootstrapNamespaces() throws PulsarServerException {
        if (this.registerNamespace(NamespaceService.getHeartbeatNamespace(this.host, this.config), true)) {
            ++this.uncountedNamespaces;
            LOG.info("added heartbeat namespace name in local cache: ns={}", (Object)NamespaceService.getHeartbeatNamespace(this.host, this.config));
        }
        for (String namespace : this.config.getBootstrapNamespaces()) {
            if (!this.registerNamespace(namespace, false)) continue;
            LOG.info("added bootstrap namespace name in local cache: ns={}", (Object)namespace);
        }
    }

    public boolean registerNamespace(String namespace, boolean ensureOwned) throws PulsarServerException {
        String myUrl = this.pulsar.getSafeBrokerServiceUrl();
        try {
            NamespaceName nsname = NamespaceName.get((String)namespace);
            String otherUrl = null;
            NamespaceBundle nsFullBundle = null;
            nsFullBundle = this.bundleFactory.getFullBundle(nsname);
            otherUrl = this.ownershipCache.tryAcquiringOwnership(nsFullBundle).get().getNativeUrl();
            if (myUrl.equals(otherUrl)) {
                if (nsFullBundle != null) {
                    this.pulsar.loadNamespaceTopics(nsFullBundle);
                }
                return true;
            }
            String msg = String.format("namespace already owned by other broker : ns=%s expected=%s actual=%s", namespace, myUrl, otherUrl);
            if (!ensureOwned) {
                LOG.info(msg);
                return false;
            }
            throw new IllegalStateException(msg);
        }
        catch (Exception e) {
            LOG.error(e.getMessage(), (Throwable)e);
            throw new PulsarServerException((Throwable)e);
        }
    }

    private CompletableFuture<Optional<LookupResult>> findBrokerServiceUrl(NamespaceBundle bundle, LookupOptions options) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("findBrokerServiceUrl: {} - options: {}", (Object)bundle, (Object)options);
        }
        ConcurrentOpenHashMap<NamespaceBundle, CompletableFuture<Optional<LookupResult>>> targetMap = options.isAuthoritative() ? this.findingBundlesAuthoritative : this.findingBundlesNotAuthoritative;
        return (CompletableFuture)targetMap.computeIfAbsent((Object)bundle, k -> {
            CompletableFuture future = new CompletableFuture();
            ((CompletableFuture)this.ownershipCache.getOwnerAsync(bundle).thenAccept(nsData -> {
                if (!nsData.isPresent()) {
                    if (options.isReadOnly()) {
                        future.complete(Optional.empty());
                    } else {
                        this.pulsar.getExecutor().execute(() -> this.searchForCandidateBroker(bundle, future, options));
                    }
                } else if (((NamespaceEphemeralData)nsData.get()).isDisabled()) {
                    future.completeExceptionally(new IllegalStateException(String.format("Namespace bundle %s is being unloaded", bundle)));
                } else {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Namespace bundle {} already owned by {} ", (Object)bundle, nsData);
                    }
                    if (options.hasAdvertisedListenerName()) {
                        AdvertisedListener listener = ((NamespaceEphemeralData)nsData.get()).getAdvertisedListeners().get(options.getAdvertisedListenerName());
                        if (listener == null) {
                            future.completeExceptionally(new PulsarServerException("the broker do not have " + options.getAdvertisedListenerName() + " listener"));
                        } else {
                            URI urlTls = listener.getBrokerServiceUrlTls();
                            future.complete(Optional.of(new LookupResult((NamespaceEphemeralData)nsData.get(), listener.getBrokerServiceUrl().toString(), urlTls == null ? null : urlTls.toString())));
                        }
                        return;
                    }
                    future.complete(Optional.of(new LookupResult((NamespaceEphemeralData)nsData.get())));
                }
            })).exceptionally(exception -> {
                LOG.warn("Failed to check owner for bundle {}: {}", new Object[]{bundle, exception.getMessage(), exception});
                future.completeExceptionally((Throwable)exception);
                return null;
            });
            future.whenComplete((r, t) -> this.pulsar.getExecutor().execute(() -> targetMap.remove((Object)bundle)));
            return future;
        });
    }

    private void searchForCandidateBroker(NamespaceBundle bundle, CompletableFuture<Optional<LookupResult>> lookupFuture, LookupOptions options) {
        if (null == this.pulsar.getLeaderElectionService()) {
            LOG.warn("The leader election has not yet been completed! NamespaceBundle[{}]", (Object)bundle);
            lookupFuture.completeExceptionally(new IllegalStateException("The leader election has not yet been completed!"));
            return;
        }
        String candidateBroker = null;
        LeaderElectionService les = this.pulsar.getLeaderElectionService();
        if (les == null) {
            LOG.warn("Leader election service isn't initialized yet. Returning empty result to lookup. NamespaceBundle[{}]", (Object)bundle);
            lookupFuture.complete(Optional.empty());
            return;
        }
        boolean authoritativeRedirect = les.isLeader();
        try {
            String broker;
            candidateBroker = NamespaceService.checkHeartbeatNamespace(bundle);
            if (candidateBroker == null && (broker = NamespaceService.getSLAMonitorBrokerName(bundle)) != null && this.isBrokerActive(broker)) {
                candidateBroker = broker;
            }
            if (candidateBroker == null) {
                Optional<LeaderBroker> currentLeader = this.pulsar.getLeaderElectionService().getCurrentLeader();
                if (options.isAuthoritative()) {
                    candidateBroker = this.pulsar.getSafeWebServiceAddress();
                } else {
                    boolean makeLoadManagerDecisionOnThisBroker;
                    LoadManager loadManager = this.loadManager.get();
                    boolean bl = makeLoadManagerDecisionOnThisBroker = !loadManager.isCentralized() || les.isLeader();
                    if (!makeLoadManagerDecisionOnThisBroker) {
                        boolean leaderBrokerActive;
                        boolean bl2 = leaderBrokerActive = currentLeader.isPresent() && this.isBrokerActive(currentLeader.get().getServiceUrl());
                        if (!leaderBrokerActive) {
                            makeLoadManagerDecisionOnThisBroker = true;
                            if (!currentLeader.isPresent()) {
                                LOG.warn("The information about the current leader broker wasn't available. Handling load manager decisions in a decentralized way. NamespaceBundle[{}]", (Object)bundle);
                            } else {
                                LOG.warn("The current leader broker {} isn't active. Handling load manager decisions in a decentralized way. NamespaceBundle[{}]", (Object)currentLeader.get(), (Object)bundle);
                            }
                        }
                    }
                    if (makeLoadManagerDecisionOnThisBroker) {
                        Optional<String> availableBroker = this.getLeastLoadedFromLoadManager(bundle);
                        if (!availableBroker.isPresent()) {
                            LOG.warn("Load manager didn't return any available broker. Returning empty result to lookup. NamespaceBundle[{}]", (Object)bundle);
                            lookupFuture.complete(Optional.empty());
                            return;
                        }
                        candidateBroker = availableBroker.get();
                        authoritativeRedirect = true;
                    } else {
                        candidateBroker = currentLeader.get().getServiceUrl();
                    }
                }
            }
        }
        catch (Exception e) {
            LOG.warn("Error when searching for candidate broker to acquire {}: {}", new Object[]{bundle, e.getMessage(), e});
            lookupFuture.completeExceptionally(e);
            return;
        }
        try {
            Preconditions.checkNotNull((Object)candidateBroker);
            if (candidateBroker.equals(this.pulsar.getSafeWebServiceAddress())) {
                String policyPath = AdminResource.path("policies", bundle.getNamespaceObject().toString());
                this.pulsar.getConfigurationCache().policiesCache().invalidate(policyPath);
                ((CompletableFuture)this.ownershipCache.tryAcquiringOwnership(bundle).thenAccept(ownerInfo -> {
                    if (ownerInfo.isDisabled()) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Namespace bundle {} is currently being unloaded", (Object)bundle);
                        }
                    } else {
                        if (options.isLoadTopicsInBundle()) {
                            this.pulsar.loadNamespaceTopics(bundle);
                        }
                        if (options.hasAdvertisedListenerName()) {
                            AdvertisedListener listener = ownerInfo.getAdvertisedListeners().get(options.getAdvertisedListenerName());
                            if (listener == null) {
                                lookupFuture.completeExceptionally(new PulsarServerException("the broker do not have " + options.getAdvertisedListenerName() + " listener"));
                                return;
                            }
                            URI urlTls = listener.getBrokerServiceUrlTls();
                            lookupFuture.complete(Optional.of(new LookupResult((NamespaceEphemeralData)ownerInfo, listener.getBrokerServiceUrl().toString(), urlTls == null ? null : urlTls.toString())));
                            return;
                        }
                        lookupFuture.complete(Optional.of(new LookupResult((NamespaceEphemeralData)ownerInfo)));
                        return;
                    }
                    lookupFuture.completeExceptionally(new IllegalStateException(String.format("Namespace bundle %s is currently being unloaded", bundle)));
                })).exceptionally(exception -> {
                    LOG.warn("Failed to acquire ownership for namespace bundle {}: {}", (Object)bundle, exception);
                    lookupFuture.completeExceptionally(new PulsarServerException("Failed to acquire ownership for namespace bundle " + bundle, exception));
                    return null;
                });
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Redirecting to broker {} to acquire ownership of bundle {}", (Object)candidateBroker, (Object)bundle);
                }
                ((CompletableFuture)this.createLookupResult(candidateBroker, authoritativeRedirect, options.getAdvertisedListenerName()).thenAccept(lookupResult -> lookupFuture.complete(Optional.of(lookupResult)))).exceptionally(ex -> {
                    lookupFuture.completeExceptionally((Throwable)ex);
                    return null;
                });
            }
        }
        catch (Exception e) {
            LOG.warn("Error in trying to acquire namespace bundle ownership for {}: {}", new Object[]{bundle, e.getMessage(), e});
            lookupFuture.completeExceptionally(e);
        }
    }

    protected CompletableFuture<LookupResult> createLookupResult(String candidateBroker, boolean authoritativeRedirect, String advertisedListenerName) {
        CompletableFuture<LookupResult> lookupFuture = new CompletableFuture<LookupResult>();
        try {
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)candidateBroker), (String)"Lookup broker can't be null %s", (Object)candidateBroker);
            String path = "/loadbalance/brokers/" + NamespaceService.parseHostAndPort(candidateBroker);
            ((CompletableFuture)this.localBrokerDataCache.get(path).thenAccept(reportData -> {
                if (reportData.isPresent()) {
                    LocalBrokerData lookupData = (LocalBrokerData)reportData.get();
                    if (StringUtils.isNotBlank((CharSequence)advertisedListenerName)) {
                        AdvertisedListener listener = (AdvertisedListener)lookupData.getAdvertisedListeners().get(advertisedListenerName);
                        if (listener == null) {
                            lookupFuture.completeExceptionally(new PulsarServerException("the broker do not have " + advertisedListenerName + " listener"));
                        } else {
                            URI urlTls = listener.getBrokerServiceUrlTls();
                            lookupFuture.complete(new LookupResult(lookupData.getWebServiceUrl(), lookupData.getWebServiceUrlTls(), listener.getBrokerServiceUrl().toString(), urlTls == null ? null : urlTls.toString(), authoritativeRedirect));
                        }
                    } else {
                        lookupFuture.complete(new LookupResult(lookupData.getWebServiceUrl(), lookupData.getWebServiceUrlTls(), lookupData.getPulsarServiceUrl(), lookupData.getPulsarServiceUrlTls(), authoritativeRedirect));
                    }
                } else {
                    lookupFuture.completeExceptionally(new MetadataStoreException.NotFoundException(path));
                }
            })).exceptionally(ex -> {
                lookupFuture.completeExceptionally((Throwable)ex);
                return null;
            });
        }
        catch (Exception e) {
            lookupFuture.completeExceptionally(e);
        }
        return lookupFuture;
    }

    private boolean isBrokerActive(String candidateBroker) {
        String candidateBrokerHostAndPort = NamespaceService.parseHostAndPort(candidateBroker);
        Set<String> availableBrokers = this.getAvailableBrokers();
        if (availableBrokers.contains(candidateBrokerHostAndPort)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Broker {} ({}) is available for.", (Object)candidateBroker, (Object)candidateBrokerHostAndPort);
            }
            return true;
        }
        LOG.warn("Broker {} ({}) couldn't be found in available brokers {}", new Object[]{candidateBroker, candidateBrokerHostAndPort, availableBrokers.stream().collect(Collectors.joining(","))});
        return false;
    }

    private static String parseHostAndPort(String candidateBroker) {
        int uriSeparatorPos = candidateBroker.indexOf("://");
        if (uriSeparatorPos == -1) {
            throw new IllegalArgumentException("'" + candidateBroker + "' isn't an URI.");
        }
        String candidateBrokerHostAndPort = candidateBroker.substring(uriSeparatorPos + 3);
        return candidateBrokerHostAndPort;
    }

    private Set<String> getAvailableBrokers() {
        try {
            return this.loadManager.get().getAvailableBrokers();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Optional<String> getLeastLoadedFromLoadManager(ServiceUnitId serviceUnit) throws Exception {
        Optional<ResourceUnit> leastLoadedBroker = this.loadManager.get().getLeastLoaded(serviceUnit);
        if (!leastLoadedBroker.isPresent()) {
            LOG.warn("No broker is available for {}", (Object)serviceUnit);
            return Optional.empty();
        }
        String lookupAddress = leastLoadedBroker.get().getResourceId();
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} : redirecting to the least loaded broker, lookup address={}", (Object)this.pulsar.getSafeWebServiceAddress(), (Object)lookupAddress);
        }
        return Optional.of(lookupAddress);
    }

    public CompletableFuture<Void> unloadNamespaceBundle(NamespaceBundle bundle) {
        return this.unloadNamespaceBundle(bundle, 5L, TimeUnit.MINUTES);
    }

    public CompletableFuture<Void> unloadNamespaceBundle(NamespaceBundle bundle, long timeout, TimeUnit timeoutUnit) {
        OwnedBundle ob = this.ownershipCache.getOwnedBundle(bundle);
        if (ob == null) {
            return FutureUtil.failedFuture((Throwable)new IllegalStateException("Bundle " + bundle + " is not currently owned"));
        }
        return ob.handleUnloadRequest(this.pulsar, timeout, timeoutUnit);
    }

    public CompletableFuture<Boolean> isNamespaceBundleOwned(NamespaceBundle bundle) {
        return this.pulsar.getLocalMetadataStore().exists(ServiceUnitUtils.path(bundle));
    }

    public Map<String, NamespaceOwnershipStatus> getOwnedNameSpacesStatus() throws Exception {
        NamespaceIsolationPolicies nsIsolationPolicies = this.getLocalNamespaceIsolationPolicies();
        HashMap<String, NamespaceOwnershipStatus> ownedNsStatus = new HashMap<String, NamespaceOwnershipStatus>();
        for (OwnedBundle nsObj : this.ownershipCache.getOwnedBundles().values()) {
            NamespaceOwnershipStatus nsStatus = this.getNamespaceOwnershipStatus(nsObj, nsIsolationPolicies.getPolicyByNamespace(nsObj.getNamespaceBundle().getNamespaceObject()));
            ownedNsStatus.put(nsObj.getNamespaceBundle().toString(), nsStatus);
        }
        return ownedNsStatus;
    }

    private NamespaceOwnershipStatus getNamespaceOwnershipStatus(OwnedBundle nsObj, NamespaceIsolationPolicy nsIsolationPolicy) {
        NamespaceOwnershipStatus nsOwnedStatus = new NamespaceOwnershipStatus(BrokerAssignment.shared, false, nsObj.isActive());
        if (nsIsolationPolicy == null) {
            return nsOwnedStatus;
        }
        nsOwnedStatus.is_controlled = true;
        if (nsIsolationPolicy.isPrimaryBroker(this.pulsar.getAdvertisedAddress())) {
            nsOwnedStatus.broker_assignment = BrokerAssignment.primary;
        } else if (nsIsolationPolicy.isSecondaryBroker(this.pulsar.getAdvertisedAddress())) {
            nsOwnedStatus.broker_assignment = BrokerAssignment.secondary;
        }
        return nsOwnedStatus;
    }

    private NamespaceIsolationPolicies getLocalNamespaceIsolationPolicies() throws Exception {
        String localCluster = this.pulsar.getConfiguration().getClusterName();
        return this.pulsar.getConfigurationCache().namespaceIsolationPoliciesCache().get(AdminResource.path("clusters", localCluster, NAMESPACE_ISOLATION_POLICIES)).orElseGet(() -> new NamespaceIsolationPolicies());
    }

    public boolean isNamespaceBundleDisabled(NamespaceBundle bundle) throws Exception {
        try {
            CompletableFuture<Optional<NamespaceEphemeralData>> nsDataFuture = this.ownershipCache.getOwnerAsync(bundle);
            if (nsDataFuture != null) {
                Optional nsData = nsDataFuture.getNow(null);
                if (nsData != null && nsData.isPresent()) {
                    return ((NamespaceEphemeralData)nsData.get()).isDisabled();
                }
                return false;
            }
            return false;
        }
        catch (Exception e) {
            LOG.warn("Exception in getting ownership info for service unit {}: {}", new Object[]{bundle, e.getMessage(), e});
            return false;
        }
    }

    public CompletableFuture<Void> splitAndOwnBundle(NamespaceBundle bundle, boolean unload, NamespaceBundleSplitAlgorithm splitAlgorithm) {
        CompletableFuture<Void> unloadFuture = new CompletableFuture<Void>();
        AtomicInteger counter = new AtomicInteger(7);
        this.splitAndOwnBundleOnceAndRetry(bundle, unload, counter, unloadFuture, splitAlgorithm);
        return unloadFuture;
    }

    void splitAndOwnBundleOnceAndRetry(NamespaceBundle bundle, boolean unload, AtomicInteger counter, CompletableFuture<Void> completionFuture, NamespaceBundleSplitAlgorithm splitAlgorithm) {
        splitAlgorithm.getSplitBoundary(this, bundle).whenComplete((splitBoundary, ex) -> {
            CompletableFuture updateFuture = new CompletableFuture();
            if (ex == null) {
                try {
                    this.bundleFactory.splitBundles(bundle, 2, (Long)splitBoundary).thenAccept(splittedBundles -> {
                        if (splittedBundles == null) {
                            String msg = String.format("bundle %s not found under namespace", bundle.toString());
                            LOG.warn(msg);
                            updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg));
                            return;
                        }
                        Preconditions.checkNotNull((Object)((NamespaceBundles)splittedBundles.getLeft()));
                        Preconditions.checkNotNull((Object)((List)splittedBundles.getRight()));
                        Preconditions.checkArgument((((List)splittedBundles.getRight()).size() == 2 ? 1 : 0) != 0, (Object)"bundle has to be split in two bundles");
                        NamespaceName nsname = bundle.getNamespaceObject();
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("[{}] splitAndOwnBundleOnce: {}, counter: {},  2 bundles: {}, {}", new Object[]{nsname.toString(), bundle.getBundleRange(), counter.get(), ((NamespaceBundle)((List)splittedBundles.getRight()).get(0)).getBundleRange(), ((NamespaceBundle)((List)splittedBundles.getRight()).get(1)).getBundleRange()});
                        }
                        try {
                            for (NamespaceBundle sBundle : (List)splittedBundles.getRight()) {
                                Preconditions.checkNotNull(this.ownershipCache.tryAcquiringOwnership(sBundle));
                            }
                            ((CompletableFuture)this.updateNamespaceBundles(nsname, (NamespaceBundles)splittedBundles.getLeft()).thenRun(() -> {
                                this.bundleFactory.invalidateBundleCache(bundle.getNamespaceObject());
                                updateFuture.complete((List)splittedBundles.getRight());
                            })).exceptionally(ex1 -> {
                                String msg = String.format("failed to update namespace policies [%s], NamespaceBundle: %s due to %s", nsname.toString(), bundle.getBundleRange(), ex1.getMessage());
                                LOG.warn(msg);
                                updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg, ex1.getCause()));
                                return null;
                            });
                        }
                        catch (Exception e) {
                            String msg = String.format("failed to acquire ownership of split bundle for namespace [%s], %s", nsname.toString(), e.getMessage());
                            LOG.warn(msg, (Throwable)e);
                            updateFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg, e));
                        }
                    });
                }
                catch (Exception e) {
                    updateFuture.completeExceptionally(e);
                }
            } else {
                updateFuture.completeExceptionally((Throwable)ex);
            }
            updateFuture.whenCompleteAsync((r, t) -> {
                if (t != null) {
                    if (t.getCause() instanceof MetadataStoreException.BadVersionException && counter.decrementAndGet() >= 0) {
                        this.pulsar.getOrderedExecutor().execute(() -> this.splitAndOwnBundleOnceAndRetry(bundle, unload, counter, completionFuture, splitAlgorithm));
                    } else if (t instanceof IllegalArgumentException) {
                        completionFuture.completeExceptionally((Throwable)t);
                    } else {
                        String msg2 = String.format(" %s not success update nsBundles, counter %d, reason %s", bundle.toString(), counter.get(), t.getMessage());
                        LOG.warn(msg2);
                        completionFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg2));
                    }
                    return;
                }
                ((CompletableFuture)this.getOwnershipCache().updateBundleState(bundle, false).thenRun(() -> {
                    this.pulsar.getBrokerService().refreshTopicToStatsMaps(bundle);
                    this.loadManager.get().setLoadReportForceUpdateFlag();
                    completionFuture.complete(null);
                    if (unload) {
                        r.forEach(this::unloadNamespaceBundle);
                    }
                })).exceptionally(e -> {
                    String msg1 = String.format("failed to disable bundle %s under namespace [%s] with error %s", bundle.getNamespaceObject().toString(), bundle.toString(), ex.getMessage());
                    LOG.warn(msg1, e);
                    completionFuture.completeExceptionally(new BrokerServiceException.ServiceUnitNotReadyException(msg1));
                    return null;
                });
            }, (Executor)this.pulsar.getOrderedExecutor());
        });
    }

    private CompletableFuture<Void> updateNamespaceBundles(NamespaceName nsname, NamespaceBundles nsBundles) {
        byte[] data;
        Preconditions.checkNotNull((Object)nsname);
        Preconditions.checkNotNull((Object)nsBundles);
        String path = PulsarWebResource.joinPath("/admin/local-policies", nsname.toString());
        LocalPolicies localPolicies = nsBundles.toLocalPolicies();
        try {
            data = ObjectMapperFactory.getThreadLocal().writeValueAsBytes((Object)localPolicies);
        }
        catch (IOException e) {
            return FutureUtil.failedFuture((Throwable)e);
        }
        return this.pulsar.getLocalMetadataStore().put(path, data, nsBundles.getVersion()).thenApply(__ -> null);
    }

    public OwnershipCache getOwnershipCache() {
        return this.ownershipCache;
    }

    public int getTotalServiceUnitsLoaded() {
        return this.ownershipCache.getOwnedBundles().size() - this.uncountedNamespaces;
    }

    public Set<NamespaceBundle> getOwnedServiceUnits() {
        return this.ownershipCache.getOwnedBundles().values().stream().map(OwnedBundle::getNamespaceBundle).collect(Collectors.toSet());
    }

    public boolean isServiceUnitOwned(ServiceUnitId suName) throws Exception {
        if (suName instanceof TopicName) {
            return this.isTopicOwnedAsync((TopicName)suName).get();
        }
        if (suName instanceof NamespaceName) {
            return this.isNamespaceOwned((NamespaceName)suName);
        }
        if (suName instanceof NamespaceBundle) {
            return this.ownershipCache.isNamespaceBundleOwned((NamespaceBundle)suName);
        }
        throw new IllegalArgumentException("Invalid class of NamespaceBundle: " + suName.getClass().getName());
    }

    public CompletableFuture<Boolean> isServiceUnitOwnedAsync(ServiceUnitId suName) {
        if (suName instanceof TopicName) {
            return this.isTopicOwnedAsync((TopicName)suName);
        }
        if (suName instanceof NamespaceName) {
            return this.isNamespaceOwnedAsync((NamespaceName)suName);
        }
        if (suName instanceof NamespaceBundle) {
            return CompletableFuture.completedFuture(this.ownershipCache.isNamespaceBundleOwned((NamespaceBundle)suName));
        }
        return FutureUtil.failedFuture((Throwable)new IllegalArgumentException("Invalid class of NamespaceBundle: " + suName.getClass().getName()));
    }

    public boolean isServiceUnitActive(TopicName topicName) {
        try {
            OwnedBundle ownedBundle = this.ownershipCache.getOwnedBundle(this.getBundle(topicName));
            if (ownedBundle == null) {
                return false;
            }
            return ownedBundle.isActive();
        }
        catch (Exception e) {
            LOG.warn("Unable to find OwnedBundle for topic - [{}]", (Object)topicName, (Object)e);
            return false;
        }
    }

    private boolean isNamespaceOwned(NamespaceName fqnn) throws Exception {
        return this.ownershipCache.getOwnedBundle(this.getFullBundle(fqnn)) != null;
    }

    private CompletableFuture<Boolean> isNamespaceOwnedAsync(NamespaceName fqnn) {
        return this.getFullBundleAsync(fqnn).thenApply(bundle -> this.ownershipCache.getOwnedBundle((NamespaceBundle)bundle) != null);
    }

    private CompletableFuture<Boolean> isTopicOwnedAsync(TopicName topic) {
        return this.getBundleAsync(topic).thenApply(bundle -> this.ownershipCache.isNamespaceBundleOwned((NamespaceBundle)bundle));
    }

    private boolean isTopicOwned(TopicName topicName) {
        Optional<NamespaceBundle> bundle = this.getBundleIfPresent(topicName);
        if (bundle.isPresent()) {
            return this.ownershipCache.getOwnedBundle(bundle.get()) != null;
        }
        ((CompletableFuture)this.getBundleAsync(topicName).thenAccept(bundle2 -> LOG.info("Succeeded in getting bundle {} for topic - [{}]", bundle2, (Object)topicName))).exceptionally(ex -> {
            LOG.warn("Failed to get bundle for topic - [{}] {}", (Object)topicName, (Object)ex.getMessage());
            return null;
        });
        return false;
    }

    public CompletableFuture<Boolean> checkTopicOwnership(TopicName topicName) {
        return this.getBundleAsync(topicName).thenCompose(bundle -> this.ownershipCache.checkOwnership((NamespaceBundle)bundle));
    }

    public void removeOwnedServiceUnit(NamespaceBundle nsBundle) throws Exception {
        this.ownershipCache.removeOwnership(nsBundle).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
        this.bundleFactory.invalidateBundleCache(nsBundle.getNamespaceObject());
    }

    protected void onNamespaceBundleOwned(NamespaceBundle bundle) {
        for (NamespaceBundleOwnershipListener bundleOwnedListener : this.bundleOwnershipListeners) {
            this.notifyNamespaceBundleOwnershipListener(bundle, bundleOwnedListener);
        }
    }

    protected void onNamespaceBundleUnload(NamespaceBundle bundle) {
        for (NamespaceBundleOwnershipListener bundleOwnedListener : this.bundleOwnershipListeners) {
            try {
                if (!bundleOwnedListener.test(bundle)) continue;
                bundleOwnedListener.unLoad(bundle);
            }
            catch (Throwable t) {
                LOG.error("Call bundle {} ownership lister error", (Object)bundle, (Object)t);
            }
        }
    }

    public void addNamespaceBundleOwnershipListener(NamespaceBundleOwnershipListener ... listeners) {
        Preconditions.checkNotNull((Object)listeners);
        for (NamespaceBundleOwnershipListener listener : listeners) {
            if (listener == null) continue;
            this.bundleOwnershipListeners.add(listener);
        }
        this.getOwnedServiceUnits().forEach(bundle -> this.notifyNamespaceBundleOwnershipListener((NamespaceBundle)bundle, listeners));
    }

    private void notifyNamespaceBundleOwnershipListener(NamespaceBundle bundle, NamespaceBundleOwnershipListener ... listeners) {
        if (listeners != null) {
            for (NamespaceBundleOwnershipListener listener : listeners) {
                try {
                    if (!listener.test(bundle)) continue;
                    listener.onLoad(bundle);
                }
                catch (Throwable t) {
                    LOG.error("Call bundle {} ownership lister error", (Object)bundle, (Object)t);
                }
            }
        }
    }

    public NamespaceBundleFactory getNamespaceBundleFactory() {
        return this.bundleFactory;
    }

    public ServiceUnitId getServiceUnitId(TopicName topicName) throws Exception {
        return this.getBundle(topicName);
    }

    public CompletableFuture<List<String>> getFullListOfTopics(NamespaceName namespaceName) {
        return this.getListOfPersistentTopics(namespaceName).thenCombine(this.getListOfNonPersistentTopics(namespaceName), (persistentTopics, nonPersistentTopics) -> ListUtils.union((List)persistentTopics, (List)nonPersistentTopics));
    }

    public CompletableFuture<List<String>> getOwnedTopicListForNamespaceBundle(NamespaceBundle bundle) {
        return ((CompletableFuture)this.getFullListOfTopics(bundle.getNamespaceObject()).thenCompose(topics -> CompletableFuture.completedFuture(topics.stream().filter(topic -> bundle.includes(TopicName.get((String)topic))).collect(Collectors.toList())))).thenCombine(this.getAllPartitions(bundle.getNamespaceObject()).thenCompose(topics -> CompletableFuture.completedFuture(topics.stream().filter(topic -> bundle.includes(TopicName.get((String)topic))).collect(Collectors.toList()))), (left, right) -> {
            for (String topic : right) {
                if (left.contains(topic)) continue;
                left.add(topic);
            }
            return left;
        });
    }

    public CompletableFuture<Boolean> checkTopicExists(TopicName topic) {
        if (topic.isPersistent()) {
            return this.pulsar.getLocalZkCacheService().managedLedgerExists(topic.getPersistenceNamingEncoding());
        }
        return this.pulsar.getBrokerService().getTopicIfExists(topic.toString()).thenApply(optTopic -> optTopic.isPresent());
    }

    public CompletableFuture<List<String>> getListOfTopics(NamespaceName namespaceName, CommandGetTopicsOfNamespace.Mode mode) {
        switch (mode) {
            case ALL: {
                return this.getFullListOfTopics(namespaceName);
            }
            case NON_PERSISTENT: {
                return this.getListOfNonPersistentTopics(namespaceName);
            }
        }
        return this.getListOfPersistentTopics(namespaceName);
    }

    public CompletableFuture<List<String>> getAllPartitions(NamespaceName namespaceName) {
        return this.getPartitions(namespaceName, TopicDomain.persistent).thenCombine(this.getPartitions(namespaceName, TopicDomain.non_persistent), ListUtils::union);
    }

    public CompletableFuture<List<String>> getPartitions(NamespaceName namespaceName, TopicDomain topicDomain) {
        String path = PulsarWebResource.path("partitioned-topics", namespaceName.toString(), topicDomain.toString());
        if (LOG.isDebugEnabled()) {
            LOG.debug("Getting children from partitioned-topics now: {}", (Object)path);
        }
        return this.pulsar.getPulsarResources().getNamespaceResources().getPartitionedTopicResources().getChildrenAsync(path).thenCompose(topics -> {
            CompletableFuture result = new CompletableFuture();
            List resultPartitions = Collections.synchronizedList(Lists.newArrayList());
            if (CollectionUtils.isNotEmpty((Collection)topics)) {
                ArrayList futures = Lists.newArrayList();
                for (String topic : topics) {
                    String partitionedTopicName = String.format("%s://%s/%s", topicDomain.value(), namespaceName.toString(), Codec.decode((String)topic));
                    CompletableFuture<List<String>> future = this.getPartitionsForTopic(TopicName.get((String)partitionedTopicName));
                    futures.add(future);
                    future.thenAccept(resultPartitions::addAll);
                }
                FutureUtil.waitForAll((List)futures).whenComplete((v, ex) -> {
                    if (ex != null) {
                        result.completeExceptionally((Throwable)ex);
                    } else {
                        result.complete(resultPartitions);
                    }
                });
            } else {
                result.complete(resultPartitions);
            }
            return result;
        });
    }

    private CompletableFuture<List<String>> getPartitionsForTopic(TopicName topicName) {
        return this.pulsar.getBrokerService().fetchPartitionedTopicMetadataAsync(topicName).thenCompose(meta -> {
            ArrayList result = Lists.newArrayList();
            for (int i = 0; i < meta.partitions; ++i) {
                result.add(topicName.getPartition(i).toString());
            }
            return CompletableFuture.completedFuture(result);
        });
    }

    public CompletableFuture<List<String>> getListOfPersistentTopics(NamespaceName namespaceName) {
        String path = String.format("/managed-ledgers/%s/persistent", namespaceName);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Getting children from managed-ledgers now: {}", (Object)path);
        }
        return this.pulsar.getLocalZkCacheService().managedLedgerListCache().getAsync(path).thenApply(znodes -> {
            ArrayList topics = Lists.newArrayList();
            for (String znode : znodes) {
                topics.add(String.format("persistent://%s/%s", namespaceName, Codec.decode((String)znode)));
            }
            topics.sort(null);
            return topics;
        });
    }

    public CompletableFuture<List<String>> getListOfNonPersistentTopics(NamespaceName namespaceName) {
        return PulsarWebResource.checkLocalOrGetPeerReplicationCluster(this.pulsar, namespaceName).thenCompose(peerClusterData -> {
            if (peerClusterData != null) {
                return this.getNonPersistentTopicsFromPeerCluster((ClusterDataImpl)peerClusterData, namespaceName);
            }
            ArrayList topics = Lists.newArrayList();
            ConcurrentOpenHashMap<String, ConcurrentOpenHashMap<String, ConcurrentOpenHashMap<String, Topic>>> concurrentOpenHashMap = this.pulsar.getBrokerService().getMultiLayerTopicMap();
            synchronized (concurrentOpenHashMap) {
                if (this.pulsar.getBrokerService().getMultiLayerTopicMap().containsKey((Object)namespaceName.toString())) {
                    ((ConcurrentOpenHashMap)this.pulsar.getBrokerService().getMultiLayerTopicMap().get((Object)namespaceName.toString())).forEach((__, bundle) -> bundle.forEach((topicName, topic) -> {
                        if (topic instanceof NonPersistentTopic && ((NonPersistentTopic)topic).isActive()) {
                            topics.add(topicName);
                        }
                    }));
                }
            }
            topics.sort(null);
            return CompletableFuture.completedFuture(topics);
        });
    }

    private CompletableFuture<List<String>> getNonPersistentTopicsFromPeerCluster(ClusterDataImpl peerClusterData, NamespaceName namespace) {
        PulsarClientImpl client = this.getNamespaceClient(peerClusterData);
        return client.getLookup().getTopicsUnderNamespace(namespace, CommandGetTopicsOfNamespace.Mode.NON_PERSISTENT);
    }

    public PulsarClientImpl getNamespaceClient(ClusterDataImpl cluster) {
        PulsarClientImpl client = (PulsarClientImpl)this.namespaceClients.get((Object)cluster);
        if (client != null) {
            return client;
        }
        return (PulsarClientImpl)this.namespaceClients.computeIfAbsent((Object)cluster, key -> {
            try {
                ClientBuilder clientBuilder = PulsarClient.builder().enableTcpNoDelay(false).statsInterval(0L, TimeUnit.SECONDS);
                clientBuilder.loadConf(PropertiesUtils.filterAndMapProperties((Properties)this.config.getProperties(), (String)"brokerClient_"));
                if (this.pulsar.getConfiguration().isAuthenticationEnabled()) {
                    clientBuilder.authentication(this.pulsar.getConfiguration().getBrokerClientAuthenticationPlugin(), this.pulsar.getConfiguration().getBrokerClientAuthenticationParameters());
                }
                if (this.pulsar.getConfiguration().isTlsEnabled()) {
                    clientBuilder.serviceUrl(StringUtils.isNotBlank((CharSequence)cluster.getBrokerServiceUrlTls()) ? cluster.getBrokerServiceUrlTls() : cluster.getServiceUrlTls()).enableTls(true).tlsTrustCertsFilePath(this.pulsar.getConfiguration().getBrokerClientTrustCertsFilePath()).allowTlsInsecureConnection(this.pulsar.getConfiguration().isTlsAllowInsecureConnection());
                } else {
                    clientBuilder.serviceUrl(StringUtils.isNotBlank((CharSequence)cluster.getBrokerServiceUrl()) ? cluster.getBrokerServiceUrl() : cluster.getServiceUrl());
                }
                ClientConfigurationData conf = ((ClientBuilderImpl)clientBuilder).getClientConfigurationData();
                return new PulsarClientImpl(conf, this.pulsar.getBrokerService().executor());
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    public Optional<NamespaceEphemeralData> getOwner(NamespaceBundle bundle) throws Exception {
        return this.getOwnerAsync(bundle).get(this.pulsar.getConfiguration().getZooKeeperOperationTimeoutSeconds(), TimeUnit.SECONDS);
    }

    public CompletableFuture<Optional<NamespaceEphemeralData>> getOwnerAsync(NamespaceBundle bundle) {
        return this.ownershipCache.getOwnerAsync(bundle);
    }

    public void unloadSLANamespace() throws Exception {
        PulsarAdmin adminClient = null;
        String namespaceName = NamespaceService.getSLAMonitorNamespace(this.host, this.config);
        LOG.info("Checking owner for SLA namespace {}", (Object)namespaceName);
        NamespaceBundle nsFullBundle = this.getFullBundle(NamespaceName.get((String)namespaceName));
        if (!this.getOwner(nsFullBundle).isPresent()) {
            return;
        }
        LOG.info("Trying to unload SLA namespace {}", (Object)namespaceName);
        adminClient = this.pulsar.getAdminClient();
        adminClient.namespaces().unload(namespaceName);
        LOG.info("Namespace {} unloaded successfully", (Object)namespaceName);
    }

    public static String getHeartbeatNamespace(String host, ServiceConfiguration config) {
        Integer port = null;
        if (config.getWebServicePort().isPresent()) {
            port = (Integer)config.getWebServicePort().get();
        } else if (config.getWebServicePortTls().isPresent()) {
            port = (Integer)config.getWebServicePortTls().get();
        }
        return String.format(HEARTBEAT_NAMESPACE_FMT, config.getClusterName(), host, port);
    }

    public static String getSLAMonitorNamespace(String host, ServiceConfiguration config) {
        Integer port = null;
        if (config.getWebServicePort().isPresent()) {
            port = (Integer)config.getWebServicePort().get();
        } else if (config.getWebServicePortTls().isPresent()) {
            port = (Integer)config.getWebServicePortTls().get();
        }
        return String.format(SLA_NAMESPACE_FMT, config.getClusterName(), host, port);
    }

    public static String checkHeartbeatNamespace(ServiceUnitId ns) {
        Matcher m = HEARTBEAT_NAMESPACE_PATTERN.matcher(ns.getNamespaceObject().toString());
        if (m.matches()) {
            LOG.debug("Heartbeat namespace matched the lookup namespace {}", (Object)ns.getNamespaceObject().toString());
            return String.format("http://%s", m.group(1));
        }
        return null;
    }

    public static String getSLAMonitorBrokerName(ServiceUnitId ns) {
        Matcher m = SLA_NAMESPACE_PATTERN.matcher(ns.getNamespaceObject().toString());
        if (m.matches()) {
            return String.format("http://%s", m.group(1));
        }
        return null;
    }

    public static boolean isSystemServiceNamespace(String namespace) {
        return HEARTBEAT_NAMESPACE_PATTERN.matcher(namespace).matches() || SLA_NAMESPACE_PATTERN.matcher(namespace).matches();
    }

    public boolean registerSLANamespace() throws PulsarServerException {
        boolean isNameSpaceRegistered = this.registerNamespace(NamespaceService.getSLAMonitorNamespace(this.host, this.config), false);
        if (isNameSpaceRegistered) {
            ++this.uncountedNamespaces;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Added SLA Monitoring namespace name in local cache: ns={}", (Object)NamespaceService.getSLAMonitorNamespace(this.host, this.config));
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("SLA Monitoring not owned by the broker: ns={}", (Object)NamespaceService.getSLAMonitorNamespace(this.host, this.config));
        }
        return isNameSpaceRegistered;
    }

    public void registerOwnedBundles() {
        ArrayList<OwnedBundle> ownedBundles = new ArrayList<OwnedBundle>(this.ownershipCache.getOwnedBundles().values());
        this.ownershipCache.invalidateLocalOwnerCache();
        ownedBundles.forEach(ownedBundle -> {
            String path = ServiceUnitUtils.path(ownedBundle.getNamespaceBundle());
            try {
                if (!this.pulsar.getLocalZkCache().checkRegNodeAndWaitExpired(path)) {
                    this.ownershipCache.tryAcquiringOwnership(ownedBundle.getNamespaceBundle());
                }
            }
            catch (Exception e) {
                try {
                    ownedBundle.handleUnloadRequest(this.pulsar, 5L, TimeUnit.MINUTES);
                }
                catch (IllegalStateException illegalStateException) {
                }
                catch (Exception ex) {
                    LOG.error("Unexpected exception occur when register owned bundle {}. Shutdown broker now !!!", (Object)ownedBundle.getNamespaceBundle(), (Object)ex);
                    this.pulsar.getShutdownService().shutdown(-1);
                }
            }
        });
    }

    @Override
    public void close() {
        this.namespaceClients.forEach((cluster, client) -> {
            try {
                client.shutdown();
            }
            catch (PulsarClientException e) {
                LOG.warn("Error shutting down namespace client for cluster {}", cluster, (Object)e);
            }
        });
    }

    public static enum AddressType {
        BROKER_URL,
        LOOKUP_URL;

    }
}

