/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.crypto;

import java.io.Closeable;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;
import java.util.Properties;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.ShortBufferException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.crypto.cipher.CryptoCipher;
import org.apache.commons.crypto.cipher.CryptoCipherFactory;
import org.apache.commons.crypto.random.CryptoRandom;
import org.apache.commons.crypto.random.CryptoRandomFactory;
import org.apache.spark.network.crypto.ClientChallenge;
import org.apache.spark.network.crypto.ServerResponse;
import org.apache.spark.network.crypto.TransportCipher;
import org.apache.spark.network.util.TransportConf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spark_project.guava.annotations.VisibleForTesting;
import org.spark_project.guava.base.Preconditions;
import org.spark_project.guava.primitives.Bytes;

class AuthEngine
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(AuthEngine.class);
    private static final BigInteger ONE = new BigInteger(new byte[]{1});
    private final byte[] appId;
    private final char[] secret;
    private final TransportConf conf;
    private final Properties cryptoConf;
    private final CryptoRandom random;
    private byte[] authNonce;
    @VisibleForTesting
    byte[] challenge;
    private TransportCipher sessionCipher;
    private CryptoCipher encryptor;
    private CryptoCipher decryptor;

    AuthEngine(String appId, String secret, TransportConf conf) throws GeneralSecurityException {
        this.appId = appId.getBytes(StandardCharsets.UTF_8);
        this.conf = conf;
        this.cryptoConf = conf.cryptoConf();
        this.secret = secret.toCharArray();
        this.random = CryptoRandomFactory.getCryptoRandom((Properties)this.cryptoConf);
    }

    ClientChallenge challenge() throws GeneralSecurityException, IOException {
        this.authNonce = this.randomBytes(this.conf.encryptionKeyLength() / 8);
        SecretKeySpec authKey = this.generateKey(this.conf.keyFactoryAlgorithm(), this.conf.keyFactoryIterations(), this.authNonce, this.conf.encryptionKeyLength());
        this.initializeForAuth(this.conf.cipherTransformation(), this.authNonce, authKey);
        this.challenge = this.randomBytes(this.conf.encryptionKeyLength() / 8);
        return new ClientChallenge(new String(this.appId, StandardCharsets.UTF_8), this.conf.keyFactoryAlgorithm(), this.conf.keyFactoryIterations(), this.conf.cipherTransformation(), this.conf.encryptionKeyLength(), this.authNonce, this.challenge(this.appId, this.authNonce, this.challenge));
    }

    ServerResponse respond(ClientChallenge clientChallenge) throws GeneralSecurityException, IOException {
        SecretKeySpec authKey = this.generateKey(clientChallenge.kdf, clientChallenge.iterations, clientChallenge.nonce, clientChallenge.keyLength);
        this.initializeForAuth(clientChallenge.cipher, clientChallenge.nonce, authKey);
        byte[] challenge = this.validateChallenge(clientChallenge.nonce, clientChallenge.challenge);
        byte[] response = this.challenge(this.appId, clientChallenge.nonce, this.rawResponse(challenge));
        byte[] sessionNonce = this.randomBytes(this.conf.encryptionKeyLength() / 8);
        byte[] inputIv = this.randomBytes(this.conf.ivLength());
        byte[] outputIv = this.randomBytes(this.conf.ivLength());
        SecretKeySpec sessionKey = this.generateKey(clientChallenge.kdf, clientChallenge.iterations, sessionNonce, clientChallenge.keyLength);
        this.sessionCipher = new TransportCipher(this.cryptoConf, clientChallenge.cipher, sessionKey, inputIv, outputIv);
        return new ServerResponse(response, this.encrypt(sessionNonce), this.encrypt(outputIv), this.encrypt(inputIv));
    }

    void validate(ServerResponse serverResponse) throws GeneralSecurityException {
        byte[] response = this.validateChallenge(this.authNonce, serverResponse.response);
        byte[] expected = this.rawResponse(this.challenge);
        Preconditions.checkArgument(Arrays.equals(expected, response));
        byte[] nonce = this.decrypt(serverResponse.nonce);
        byte[] inputIv = this.decrypt(serverResponse.inputIv);
        byte[] outputIv = this.decrypt(serverResponse.outputIv);
        SecretKeySpec sessionKey = this.generateKey(this.conf.keyFactoryAlgorithm(), this.conf.keyFactoryIterations(), nonce, this.conf.encryptionKeyLength());
        this.sessionCipher = new TransportCipher(this.cryptoConf, this.conf.cipherTransformation(), sessionKey, inputIv, outputIv);
    }

    TransportCipher sessionCipher() {
        Preconditions.checkState(this.sessionCipher != null);
        return this.sessionCipher;
    }

    @Override
    public void close() throws IOException {
        RuntimeException error = null;
        byte[] dummy = new byte[8];
        try {
            this.doCipherOp(this.encryptor, dummy, true);
        }
        catch (Exception e) {
            error = new RuntimeException(e);
        }
        try {
            this.doCipherOp(this.decryptor, dummy, true);
        }
        catch (Exception e) {
            error = new RuntimeException(e);
        }
        this.random.close();
        if (error != null) {
            throw error;
        }
    }

    @VisibleForTesting
    byte[] challenge(byte[] appId, byte[] nonce, byte[] challenge) throws GeneralSecurityException {
        return this.encrypt(Bytes.concat(appId, nonce, challenge));
    }

    @VisibleForTesting
    byte[] rawResponse(byte[] challenge) {
        BigInteger orig = new BigInteger(challenge);
        BigInteger response = orig.add(ONE);
        return response.toByteArray();
    }

    private byte[] decrypt(byte[] in) throws GeneralSecurityException {
        return this.doCipherOp(this.decryptor, in, false);
    }

    private byte[] encrypt(byte[] in) throws GeneralSecurityException {
        return this.doCipherOp(this.encryptor, in, false);
    }

    private void initializeForAuth(String cipher, byte[] nonce, SecretKeySpec key) throws GeneralSecurityException {
        byte[] iv = new byte[this.conf.ivLength()];
        System.arraycopy(nonce, 0, iv, 0, Math.min(nonce.length, iv.length));
        this.encryptor = CryptoCipherFactory.getCryptoCipher((String)cipher, (Properties)this.cryptoConf);
        this.encryptor.init(1, (Key)key, (AlgorithmParameterSpec)new IvParameterSpec(iv));
        this.decryptor = CryptoCipherFactory.getCryptoCipher((String)cipher, (Properties)this.cryptoConf);
        this.decryptor.init(2, (Key)key, (AlgorithmParameterSpec)new IvParameterSpec(iv));
    }

    private byte[] validateChallenge(byte[] nonce, byte[] encryptedChallenge) throws GeneralSecurityException {
        byte[] challenge = this.decrypt(encryptedChallenge);
        this.checkSubArray(this.appId, challenge, 0);
        this.checkSubArray(nonce, challenge, this.appId.length);
        return Arrays.copyOfRange(challenge, this.appId.length + nonce.length, challenge.length);
    }

    private SecretKeySpec generateKey(String kdf, int iterations, byte[] salt, int keyLength) throws GeneralSecurityException {
        SecretKeyFactory factory = SecretKeyFactory.getInstance(kdf);
        PBEKeySpec spec = new PBEKeySpec(this.secret, salt, iterations, keyLength);
        long start = System.nanoTime();
        SecretKey key = factory.generateSecret(spec);
        long end = System.nanoTime();
        LOG.debug("Generated key with {} iterations in {} us.", (Object)this.conf.keyFactoryIterations(), (Object)((end - start) / 1000L));
        return new SecretKeySpec(key.getEncoded(), this.conf.keyAlgorithm());
    }

    private byte[] doCipherOp(CryptoCipher cipher, byte[] in, boolean isFinal) throws GeneralSecurityException {
        Preconditions.checkState(cipher != null);
        int scale = 1;
        while (true) {
            int size = in.length * scale;
            byte[] buffer = new byte[size];
            try {
                int outSize;
                int n = outSize = isFinal ? cipher.doFinal(in, 0, in.length, buffer, 0) : cipher.update(in, 0, in.length, buffer, 0);
                if (outSize != buffer.length) {
                    byte[] output = new byte[outSize];
                    System.arraycopy(buffer, 0, output, 0, output.length);
                    return output;
                }
                return buffer;
            }
            catch (ShortBufferException e) {
                scale *= 2;
                continue;
            }
            break;
        }
    }

    private byte[] randomBytes(int count) {
        byte[] bytes = new byte[count];
        this.random.nextBytes(bytes);
        return bytes;
    }

    private void checkSubArray(byte[] test, byte[] data, int offset) {
        Preconditions.checkArgument(data.length >= test.length + offset);
        for (int i = 0; i < test.length; ++i) {
            Preconditions.checkArgument(test[i] == data[i + offset]);
        }
    }
}

