/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.sshd.common.file.nativefs;

import java.io.File;

import org.apache.sshd.common.file.FileSystemView;
import org.apache.sshd.common.file.SshFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <strong>Internal class, do not use directly.</strong>
 * 
 * File system view based on native file system. Here the root directory will be
 * user virtual root (/).
 *
 * @author <a href="http://mina.apache.org">Apache MINA Project</a>
 */
public class NativeFileSystemView implements FileSystemView {

    private final Logger LOG = LoggerFactory
    .getLogger(NativeFileSystemView.class);


    // the first and the last character will always be '/'
    // It is always with respect to the root directory.
    private String currDir;

    private String userName;

    private boolean caseInsensitive = false;

    /**
     * Constructor - internal do not use directly, use {@link NativeFileSystemFactory} instead
     */
    protected NativeFileSystemView(String userName) {
        this(userName, false);
    }

    /**
     * Constructor - internal do not use directly, use {@link NativeFileSystemFactory} instead
     */
    public NativeFileSystemView(String userName, boolean caseInsensitive) {
        if (userName == null) {
            throw new IllegalArgumentException("user can not be null");
        }

        this.caseInsensitive = caseInsensitive;

        currDir = getVirtualUserDir();
        this.userName = userName;

        // add last '/' if necessary
        LOG.debug("Native filesystem view created for user \"{}\" with root \"{}\"", userName, currDir);
    }

    public String getUserName() {
        return userName;
    }

    /**
     * Get file object.
     */
    public SshFile getFile(String file) {
        return getFile(currDir, file);
    }

    public SshFile getFile(SshFile baseDir, String file) {
        return getFile(baseDir.getAbsolutePath(), file);
    }

    protected SshFile getFile(String dir, String file) {
        // get actual file object
        String physicalName = NativeSshFile.getPhysicalName(getPhysicalUserDir(),
                dir, file, caseInsensitive);
        File fileObj = new File(physicalName);

        // strip the root directory and return
        String userFileName = physicalName.substring(getPhysicalUserDir().length() - 1);
        return createNativeSshFile(userFileName, fileObj, userName);
    }

    /**
     * Returns the physical user directory, for accessing the according files or directories.
     *  
     * @return The physical user directory.
     */
    public String getPhysicalUserDir() {
    	return "/";
	}

	/**
	 * Returns the virtual user directory.
	 * The will be shown in sftp client. It is relative to the physical user directory. 
	 * 
	 * @return The virtual user directory.
	 */
	public String getVirtualUserDir() {
		return System.getProperty("user.dir");
	}

    static boolean isJava7;
    static boolean isWindows;
    static {
        // Check java 7
        boolean j7 = false;
        try {
            ClassLoader.getSystemClassLoader().loadClass("java.nio.file.Files");
            j7 = true;
        } catch (Throwable t) {
            // Ignore
        }
        isJava7 = j7;
        // Check windows
        boolean win = false;
        try {
            win = System.getProperty("os.name").toLowerCase().contains("win");
        } catch (Throwable t) {
            // Ignore
        }
        isWindows = win;
    }

    public NativeSshFile createNativeSshFile(final String fileName2, final File fileObj, final String userName2) {
        if (isJava7 && !isWindows) {
            return new NativeSshFileNio(this, fileName2, fileObj, userName2);
        } else {
		    return new NativeSshFile(this, fileName2, fileObj, userName2);
        }
	}
}
