/*
 * Redistribution and use of this software and associated documentation
 * ("Software"), with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * 1. Redistributions of source code must retain copyright
 *    statements and notices.  Redistributions must also contain a
 *    copy of this document.
 *
 * 2. Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *
 * 3. The name "Exolab" must not be used to endorse or promote
 *    products derived from this Software without prior written
 *    permission of Intalio, Inc.  For written permission,
 *    please contact info@exolab.org.
 *
 * 4. Products derived from this Software may not be called "Exolab"
 *    nor may "Exolab" appear in their names without prior written
 *    permission of Intalio, Inc. Exolab is a registered
 *    trademark of Intalio, Inc.
 *
 * 5. Due credit should be given to the Exolab Project
 *    (http://www.exolab.org/).
 *
 * THIS SOFTWARE IS PROVIDED BY INTALIO, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * INTALIO, INC. OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright 1999-2002 (C) Intalio, Inc. All Rights Reserved.
 *
 * $Id$
 */
package org.exolab.castor.xml.validators;

/**
 * A static class for performing simple validation.
 *
 * @author <a href="mailto:kvisco@intalio.com">Keith Visco</a>
 * @version $Revision$ $Date: 2003-03-03 00:05:44 -0700 (Mon, 03 Mar 2003) $
 */
public final class ValidationUtils {

    // ----------------/
    // - Constructors -/
    // ----------------/

    /**
     * No-arg constructor.
     */
    private ValidationUtils() {
        super();
    }

    // ------------------/
    // - Public Methods -/
    // ------------------/

    /**
     * Checks the given character to determine if it is a valid CombiningChar as
     * defined by the W3C XML 1.0 Recommendation.
     * <p>
     * FIXME: This method needs to be properly implemented.
     * @param ch THe character to check.
     *
     * @return true if the given character is a CombiningChar
     */
    public static boolean isCombiningChar(final char ch) {
        // -- NOTE: THIS METHOD IS NOT COMPLETE
        return false;
    } // -- isCombiningChar

    /**
     * Checks a character to see if it is a digit or not.
     * @param ch
     *            the character to check
     * @return true if the given character is a digit
     */
    public static boolean isDigit(final char ch) {
        return Character.isDigit(ch);
    } // -- isDigit

    /**
     * Checks a character to see if it is a letter or not.
     * @param ch
     *            the character to check
     * @return true if the given character is a letter
     */
    public static boolean isLetter(final char ch) {
        return Character.isLetter(ch);
    } // -- isLetter

    /**
     * Checks all characters of the given String to determine if they
     * syntactically match the production of an NCName as defined by the W3C XML
     * Namespaces recommendation.
     *
     * @param str
     *            the String to check
     * @return true if the given String follows the Syntax of an NCName
     */
    public static boolean isNCName(final String str) {
        if (str == null || str.length() == 0) {
            return false;
        }

        char[] chars = str.toCharArray();

        // -- make sure String starts with a letter or '_'
        char ch = chars[0];
        if (!isLetter(ch) && ch != '_') {
            return false;
        }

        for (int i = 1; i < chars.length; i++) {
            if (!isNCNameChar(chars[i])) {
                return false;
            }
        }
        return true;
    } // -- isNCName

    /**
     * Checks the the given character to determine if it is a valid NCNameChar
     * as defined by the W3C XML Namespaces recommendation.
     *
     * @param ch
     *            the char to check
     * @return true if the given char is an NCNameChar
     */
    public static boolean isNCNameChar(final char ch) {
        if (isLetter(ch) || isDigit(ch)) {
            return true;
        }

        if (isExtender(ch) || isCombiningChar(ch)) {
            return true;
        }

        switch (ch) {
            case '.':
            case '-':
            case '_':
                return true;
            default:
                return false;
        }
    } // -- isNCNameChar

    /**
     * Checks all characters of the given String to determine if they
     * syntactically match the production of an NMToken.
     *
     * @param str
     *            the String to check
     * @return true if the given String follows the Syntax of an NMToken
     */
    public static boolean isNMToken(final String str) {
        if (str == null) {
            return false;
        }

        char[] chars = str.toCharArray();

        for (int i = 0; i < chars.length; i++) {
            char ch = chars[i];
            if (isLetter(ch) || isDigit(ch) || isExtender(ch) || isCombiningChar(ch)) {
                continue;
            }

            switch (ch) {
                case '.':
                case '-':
                case '_':
                case ':':
                    break;
                default:
                    return false;
            }
        }
        return true;
    } // -- isNMToken

    /**
     * Checks all characters of the given String to determine if they
     * syntactically match the production of a CDATA.
     *
     * @param str
     *            the String to check
     * @return true if the given String follows the Syntax of an NMToken
     */
    public static boolean isCDATA(final String str) {
        if (str == null) {
            return false;
        }

        char[] chars = str.toCharArray();

        for (int i = 0; i < chars.length; i++) {
            char ch = chars[i];
            switch (ch) {
                case '\r':
                case '\n':
                case '\t':
                    return false;
                default:
                    continue;
            }
        }
        return true;
    } // -- isCDATA

    /**
     * Returns true if the given character is a valid XML Extender character,
     * according to the XML 1.0 specification.
     *
     * @param ch
     *            the character to check
     * @return true if the character is a valid XML Extender character
     */
    public static boolean isExtender(final char ch) {
        if ((ch >= 0x3031) && (ch <= 0x3035)) {
            return true;
        }

        if ((ch >= 0x30FC) && (ch <= 0x30FE)) {
            return true;
        }

        switch (ch) {
            case 0x00B7:
            case 0x02D0:
            case 0x02D1:
            case 0x0387:
            case 0x0640:
            case 0x0E46:
            case 0x0EC6:
            case 0x3005:
            case 0x309D:
            case 0x309E:
                return true;
            default:
                break;
        }
        return false;
    } // -- isExtender

    /**
     * Checks all characters of the given String to determine if they
     * syntactically match the production of an QName as defined by the W3C XML
     * Namespaces recommendation.
     *
     * @param str
     *            the String to check
     * @return true if the given String follows the Syntax of an QName
     */
    public static boolean isQName(final String str) {
        if (str == null || str.length() == 0) {
            return false;
        }

        char[] chars = str.toCharArray();

        // -- make sure String starts with a letter or '_'
        char ch = chars[0];
        if (!isLetter(ch) && ch != '_') {
            return false;
        }

        for (int i = 1; i < chars.length; i++) {
            if (chars[i] == ':') {
                continue;
            }

            if (!isNCNameChar(chars[i])) {
                return false;
            }
        }
        return true;
    } //-- isQName

} //-- Validator
