/**********************************************************************
Copyright (c) 2009 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.FetchPlanState;
import org.datanucleus.state.ObjectProviderFactory;
import org.datanucleus.state.StateManagerFactory;
import org.datanucleus.state.lock.LockManager;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.Extent;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.FieldValues2;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.Type;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.types.TypeManager;

/**
 * Context of execution for persistence operations
 */
public class ExecutionContextImpl implements ExecutionContext
{
    ObjectManager om;
    public ExecutionContextImpl(ObjectManager om)
    {
        this.om = om;
    }
    
    public ObjectManager getObjectManager()
    {
        return om;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.ExecutionContext#getLockManager()
     */
    public LockManager getLockManager()
    {
        return om.getLockManager();
    }

    public Object findObjectUsingAID(Type pcClass, final FieldValues2 fv, boolean ignoreCache, boolean checkInheritance)
    {
        return om.findObjectUsingAID(pcClass.getType(), new FieldValues()
        {
            
            public FetchPlan getFetchPlanForLoading()
            {
                return fv.getFetchPlanForLoading();
            }
            
            public void fetchNonLoadedFields(StateManager sm)
            {
                fv.fetchNonLoadedFields(sm.getObjectProvider());
            }
            
            public void fetchFields(StateManager sm)
            {
                fv.fetchFields(sm.getObjectProvider());
            }
        }, ignoreCache, checkInheritance);
    }

    public ClassLoaderResolver getClassLoaderResolver()
    {
        return om.getClassLoaderResolver();
    }

    public MetaDataManager getMetaDataManager()
    {
        return om.getMetaDataManager();
    }

    public Transaction getTransaction()
    {
        return om.getTransaction();
    }

    public TypeManager getTypeManager()
    {
        return om.getOMFContext().getTypeManager();
    }
    
    /**
     * @param persist persists the object if not yet persisted. 
     */
    public ObjectProvider findObjectProvider(Object object, boolean persist)
    {
        StateManager sm = om.findStateManager(object);
        if (sm == null && persist)
        {
            int objectType = ObjectProvider.PC;
            Object object2 = om.persistObjectInternal(object, null, null, -1, objectType);
            sm = om.findStateManager(object2);
        }
        else if(sm==null)
        {
            return null;
        }
        return sm.getObjectProvider();
    }
    
    public ApiAdapter getApiAdapter()
    {
        return om.getApiAdapter();
    }

    public ObjectProvider findObjectProviderForEmbedded(Object value, ObjectProvider owner, AbstractMemberMetaData mmd)
    {
        StateManager embeddedSM = om.findStateManager(value);
        if (embeddedSM == null)
        {
            // Assign a StateManager to manage our embedded object
            embeddedSM = StateManagerFactory.newStateManagerForEmbedded(om.getExecutionContext(), value, false);
        }
        if (embeddedSM.getEmbeddedOwners() == null || embeddedSM.getEmbeddedOwners().length == 0)
        {
            int absoluteFieldNumber = owner.getClassMetaData().getMetaDataForMember(mmd.getName()).getAbsoluteFieldNumber();
            embeddedSM.getObjectProvider().addEmbeddedOwner(owner, absoluteFieldNumber);
            embeddedSM.setPcObjectType(ObjectProvider.EMBEDDED_PC);
        }
        return embeddedSM.getObjectProvider();
    }

    public ObjectProvider newObjectProvider(Object id, Object obj)
    {
        ObjectProvider sm = ObjectProviderFactory.newForPersistentClean(this, id, obj);            
        return sm;
    }    

    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd)
    {
        Class pcClass = om.getClassLoaderResolver().classForName(mmd.getClassName());
        StateManager sm = StateManagerFactory.newStateManagerForHollow( om.getExecutionContext(), pcClass, null );
        if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null)
        {
            sm.initialiseForEmbedded( sm.getObject(), true );
        }
        return sm.getObjectProvider();
    }

    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, Type effectiveType)
    {
        Class pcClass = om.getClassLoaderResolver().classForName(effectiveType.getName());
        StateManager sm = StateManagerFactory.newStateManagerForHollow( om.getExecutionContext(), pcClass, null );
        if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null)
        {
            sm.initialiseForEmbedded( sm.getObject(), true );
        }
        return sm.getObjectProvider();
    }

    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, AbstractClassMetaData effectiveTypeCmd)
    {
        Class pcClass = om.getClassLoaderResolver().classForName(effectiveTypeCmd.getFullClassName());
        StateManager sm = StateManagerFactory.newStateManagerForHollow( om.getExecutionContext(), pcClass, null );
        if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null || effectiveTypeCmd.isEmbeddedOnly())
        {
            sm.initialiseForEmbedded( sm.getObject(), true );
        }
        return sm.getObjectProvider();
    }

    public void deleteObjectInternal(Object pc)
    {
        om.deleteObjectInternal(pc);
    }
    
    /**
     * Method to persist the passed object (internally).
     * @param pc The object
     * @param ownerSM StateManager of the owner when embedded
     * @param ownerFieldNum Field number in the owner where this is embedded (or -1 if not embedded)
     * @param objectType Type of object (see org.datanucleus.StateManager, e.g StateManager.PC)
     * @return The persisted object
     */
    public Object persistObjectInternal(Object pc, ObjectProvider ownerSM, int ownerFieldNum, int objectType)
    {
        if (ownerSM != null)
        {
            StateManager sm = om.findStateManager(ownerSM.getObject());
            return om.persistObjectInternal(pc, null, sm, ownerFieldNum, objectType);
        }
        else
        {
            return om.persistObjectInternal(pc, null, null, ownerFieldNum, objectType);
        }
    }
    
    public boolean isClosed()
    {
        return om.isClosed();
    }

    public FetchPlan getFetchPlan()
    {
        return om.getFetchPlan();
    }

    public Query newQuery()
    {
        return om.newQuery();
    }

    public OMFContext getOMFContext()
    {
        return om.getOMFContext();
    }

    public StoreManager getStoreManager()
    {
        return om.getStoreManager();
    }

    public ObjectProvider findObjectProvider(Object obj)
    {
        StateManager sm = om.findStateManager(obj);
        if (sm == null)
        {
            return null;
        }
        return sm.getObjectProvider();
    }

    public Object findObject(Object id, boolean validate, boolean checkInheritance, String objectClassName)
    {
        return om.findObject(id, validate, checkInheritance, objectClassName);
    }

    public void flushInternal(boolean flushToDatastore)
    {
        om.flushInternal(flushToDatastore);
    }

    public void detachObject(Object val, FetchPlanState state)
    {
        om.detachObject(val,state);
    }

    public void deleteObjects(Object[] objs)
    {
        om.deleteObjects(objs);
    }

    public Object attachObjectCopy(Object pc, boolean sco)
    {
        return om.attachObjectCopy(pc, sco);
    }

    public Object detachObjectCopy(Object pc, FetchPlanState state)
    {
        return om.detachObjectCopy(pc, state);
    }

    public void refreshObject(Object pc)
    {
        om.refreshObject(pc);
    }

    public void evictFromTransaction(ObjectProvider sm)
    {
        StateManager sm1 = om.findStateManager(sm.getObject());
        om.evictFromTransaction(sm1);
    }

    public boolean isFlushing()
    {
        return om.isFlushing();
    }

    public boolean getIgnoreCache()
    {
        return om.getIgnoreCache();
    }

    public Integer getDatastoreReadTimeoutMillis()
    {
        return om.getDatastoreReadTimeoutMillis();
    }

    public Integer getDatastoreWriteTimeoutMillis()
    {
        return om.getDatastoreWriteTimeoutMillis();
    }

    public boolean isDelayDatastoreOperationsEnabled()
    {
        return om.isDelayDatastoreOperationsEnabled();
    }

    public void markDirty(ObjectProvider sm, boolean directUpdate)
    {
        StateManager sm1 = om.findStateManager(sm.getObject());        
        om.markDirty(sm1, directUpdate);
    }

    public Extent getExtent(Class candidateClass, boolean includeSubclasses)
    {
        return om.getExtent(candidateClass, includeSubclasses);
    }

    public void attachObject(Object pc, boolean sco)
    {
        om.attachObject(pc, sco);
    }

    public Object getObjectFromCache(Object id)
    {
        return om.getObjectFromCache(id);
    }

    public void removeObjectFromCache(Object pc, Object id)
    {
        om.removeObjectFromCache(pc, id);
    }

    /**
     * Whether an object with the specified identity exists in the cache(s).
     * Used as a check on identity (inheritance-level) validity
     * @param id The identity
     * @return Whether it exists
     */
    public boolean hasIdentityInCache(Object id)
    {
        return om.hasIdentityInCache(id);
    }

    public Object findObject(Object id, final FieldValues2 fv, Class pcClass, boolean ignoreCache)
    {
        return om.findObject(id, fv==null?null:new FieldValues()
        {
            
            public FetchPlan getFetchPlanForLoading()
            {
                return fv.getFetchPlanForLoading();
            }
            
            public void fetchNonLoadedFields(StateManager sm)
            {
                fv.fetchNonLoadedFields(sm.getObjectProvider());
            }
            
            public void fetchFields(StateManager sm)
            {
                fv.fetchFields(sm.getObjectProvider());
            }
        }, pcClass, ignoreCache);
    }

    public boolean getSerializeReadForClass(String className)
    {
        return om.getSerializeReadForClass(className);
    }

    public void hasPersistenceInformationForClass(Class cls)
    {
        om.hasPersistenceInformationForClass(cls);
    }

    public void makeObjectTransient(Object pc, FetchPlanState state)
    {
        om.makeObjectTransient(pc, state);
    }

    public boolean isInserting(Object pc)
    {
        return om.isInserting(pc);
    }

    public Object getAttachedObjectForId(Object id)
    {
        return om.getAttachedObjectForId(id);
    }
    
    public ObjectManagerFactoryImpl getObjectManagerFactory()
    {
        return om.getObjectManagerFactory();
    }
    
    public void persistObjectInternal(Object pc, final FieldValues2 preInsertChanges, int objectType)
    {
        om.persistObjectInternal(pc, preInsertChanges==null?null:new FieldValues()
        {
            
            public FetchPlan getFetchPlanForLoading()
            {
                return preInsertChanges.getFetchPlanForLoading();
            }
            
            public void fetchNonLoadedFields(StateManager sm)
            {
                preInsertChanges.fetchNonLoadedFields(sm.getObjectProvider());
            }
            
            public void fetchFields(StateManager sm)
            {
                preInsertChanges.fetchFields(sm.getObjectProvider());
            }
        }, null, -1, objectType);
    }
    
    public Object newObjectId(String className, Object pc)
    {
        return om.newObjectId(className, pc);
    }

    public Object newObjectId(Class pcClass, Object key)
    {
        return om.newObjectId(pcClass, key);
    }

    public void deleteObject(Object obj)
    {
        om.deleteObject(obj);
    }
}