/**********************************************************************
Copyright (c) 2006 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2006 Thomas Marti - Added support for configurable plugin file names
    ...
**********************************************************************/
package org.datanucleus.plugin;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.HashSet;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.ClassNotResolvedException;

/**
 * Manages the registry of Extensions and Extension Points.
 */
public class PluginManager
{
    /** plugin registry **/
    private PluginRegistry registry;

    /**
     * Constructor.
     * @param registryClassName Name of the registry
     * @param registryBundleCheck What to do on check of bundles
     * @param clr the ClassLoaderResolver
     */
    public PluginManager(String registryClassName, String registryBundleCheck, ClassLoaderResolver clr)
    {
        registry = PluginRegistryFactory.newPluginRegistry(registryClassName, registryBundleCheck, clr);
    }

    /**
     * Accessor for the PluginRegistry class name.
     * @return Name of the plugin registry
     */
    public String getRegistryClassName()
    {
        return registry.getClass().getName();
    }

    /**
     * Acessor for the ExtensionPoint
     * @param id the unique id of the extension point
     * @return null if the ExtensionPoint is not registered
     */
    public ExtensionPoint getExtensionPoint(String id)
    {
        return registry.getExtensionPoint(id);
    }

    /**
     * Acessor for the currently registed ExtensionPoints
     * @return array of ExtensionPoints
     */
    public ExtensionPoint[] getExtensionPoints()
    {
        return registry.getExtensionPoints();
    }

    /**
     * Register Extension Points declared in /org/datanucleus/plugin/plugin.xml
     */
    public void registerExtensionPoints()
    {
        registry.registerExtensionPoints();
    }

    /**
     * Register ExtensionPoints and Extensions declared in plugin files
     */
    public void registerExtensions()
    {
        registry.registerExtensions();
    }

    /**
     * Resolve constraints declared in bundle files. 
     * This must be invoked after registering all bundles.
     * Should log errors if bundles are not resolvable, or raise runtime exceptions.
     */
    public void resolveConstraints()
    {
        registry.resolveConstraints();
    }
    
    /**
     * Convenience accessor for getting the (first) ConfigurationElement for an extension (of an extension point).
     * @param extensionPointName The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for discriminator attribute
     * @return The value of the attribute
     */
    public ConfigurationElement getConfigurationElementForExtension(String extensionPointName,
            String discrimAttrName, String discrimAttrValue)
    {
        ExtensionPoint extensionPoint = getExtensionPoint(extensionPointName);
        if (extensionPoint!=null) 
        {
            Extension[] ex = extensionPoint.getExtensions();
            for (int i=0; i<ex.length; i++)
            {
                ConfigurationElement[] confElm = ex[i].getConfigurationElements();
                for (int j=0; j<confElm.length; j++)
                {
                    // Find an extension with this discriminator value
                    if (confElm[j].getAttribute(discrimAttrName) != null &&
                        confElm[j].getAttribute(discrimAttrName).equalsIgnoreCase(discrimAttrValue))
                    {
                        return confElm[j];
                    }
                }
            }
        }
        return null;
    }

    /**
     * Convenience accessor for getting the ConfigurationElement(s) for an extension (of an extension point).
     * @param extensionPointName The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for discriminator attribute
     * @return The value of the attribute
     */
    public ConfigurationElement[] getConfigurationElementsForExtension(String extensionPointName,
            String discrimAttrName, String discrimAttrValue)
    {
        ExtensionPoint extensionPoint = getExtensionPoint(extensionPointName);
        HashSet elems = new HashSet();
        if (extensionPoint != null) 
        {
            Extension[] ex = extensionPoint.getExtensions();
            for (int i=0; i<ex.length; i++)
            {
                ConfigurationElement[] confElm = ex[i].getConfigurationElements();
                for (int j=0; j<confElm.length; j++)
                {
                    if (discrimAttrName == null)
                    {
                        // No discriminator set so take all values
                        elems.add(confElm[j]);
                    }
                    else if (confElm[j].getAttribute(discrimAttrName) != null &&
                        confElm[j].getAttribute(discrimAttrName).equalsIgnoreCase(discrimAttrValue))
                    {
                        // Extension with this discriminator value
                        elems.add(confElm[j]);
                    }
                }
            }
        }
        if (elems.size() > 0)
        {
            return (ConfigurationElement[])elems.toArray(new ConfigurationElement[elems.size()]);
        }
        return null;
    }

    /**
     * Convenience accessor for getting the ConfigurationElement for an extension (of an extension point).
     * @param extensionPointName The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator1
     * @param discrimAttrValue Value for discriminator1 attribute
     * @return The value of the attribute
     */
    public ConfigurationElement getConfigurationElementForExtension(String extensionPointName,
            String[] discrimAttrName, String[] discrimAttrValue)
    {
        ExtensionPoint extensionPoint = getExtensionPoint(extensionPointName);
        if (extensionPoint!=null) 
        {
            Extension[] ex = extensionPoint.getExtensions();
            for (int i=0; i<ex.length; i++)
            {
                ConfigurationElement[] confElm = ex[i].getConfigurationElements();
                for (int j=0; j<confElm.length; j++)
                {
                    // Find an extension with this discriminator value
                    boolean equals = true;
                    for (int k=0; k<discrimAttrName.length; k++)
                    {
                        if (confElm[j].getAttribute(discrimAttrName[k]) == null && discrimAttrValue[k] != null)
                        {
                            equals = false;
                            break;
                        }
                        if (!confElm[j].getAttribute(discrimAttrName[k]).equalsIgnoreCase(discrimAttrValue[k]))
                        {
                            equals = false;
                            break;
                        }
                    }
                    if (equals)
                    {
                        return confElm[j];
                    }
                }
            }
        }
        return null;
    }

    /**
     * Convenience accessor for getting the value of an attribute for an extension (of an extension point).
     * @param extensionPoint The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for discriminator attribute
     * @param attributeName Name of the attribute whose value we want
     * @return The value of the attribute
     */
    public String getAttributeValueForExtension(String extensionPoint,
            String discrimAttrName, String discrimAttrValue, String attributeName)
    {
        ConfigurationElement elem = getConfigurationElementForExtension(extensionPoint, discrimAttrName, discrimAttrValue);
        if (elem != null)
        {
            return elem.getAttribute(attributeName);
        }
        return null;
    }

    /**
     * Convenience accessor for getting the value of an attribute for an extension (of an extension point).
     * @param extensionPoint The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for discriminator attribute
     * @param attributeName Name of the attribute whose value we want
     * @return The value(s) of the attribute
     */
    public String[] getAttributeValuesForExtension(String extensionPoint,
            String discrimAttrName, String discrimAttrValue, String attributeName)
    {
        ConfigurationElement[] elems = getConfigurationElementsForExtension(extensionPoint, discrimAttrName, discrimAttrValue);
        if (elems != null)
        {
            String[] attrValues = new String[elems.length];
            for (int i=0;i<elems.length;i++)
            {
                attrValues[i] = elems[i].getAttribute(attributeName);
            }
            return attrValues;
        }
        return null;
    }

    /**
     * Convenience accessor for getting the value of an attribute for an extension (of an extension point).
     * @param extensionPoint The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator1
     * @param discrimAttrValue Value for discriminator1 attribute
     * @param attributeName Name of the attribute whose value we want
     * @return The value of the attribute
     */
    public String getAttributeValueForExtension(String extensionPoint,
            String[] discrimAttrName, String[] discrimAttrValue, String attributeName)
    {
        ConfigurationElement elem = getConfigurationElementForExtension(extensionPoint, 
            discrimAttrName, discrimAttrValue);
        if (elem != null)
        {
            return elem.getAttribute(attributeName);
        }
        return null;
    }
    
    /**
     * Convenience accessor for getting the Class of an attribute for an extension (of an extension point).
     * @param extensionPoint The extension point
     * @param discrimAttrName Attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for discriminator attribute
     * @param attributeName Name of the attribute whose value we want
     * @return The value of the attribute
     */
    public Object createExecutableExtension(String extensionPoint,
                                            String discrimAttrName, 
                                            String discrimAttrValue, 
                                            String attributeName, 
                                            Class[] argsClass, 
                                            Object[] args) throws ClassNotFoundException, SecurityException, NoSuchMethodException, IllegalArgumentException, InstantiationException, IllegalAccessException, InvocationTargetException
    {
        ConfigurationElement elem = getConfigurationElementForExtension(extensionPoint, discrimAttrName, discrimAttrValue);
        if (elem != null)
        {
            return registry.createExecutableExtension(elem, attributeName, argsClass, args);
        }
        return null;
    }

    /**
     * Convenience accessor for getting the Class of an attribute for an extension (of an extension point).
     * @param extensionPoint The extension point
     * @param discrimAttrName First attribute on the extension to use as discriminator
     * @param discrimAttrValue Value for first discriminator attribute
     * @param attributeName Name of the attribute whose value we want
     * @return The value of the attribute
     */
    public Object createExecutableExtension(String extensionPoint,
                                            String[] discrimAttrName,
                                            String[] discrimAttrValue,
                                            String attributeName, 
                                            Class[] argsClass, 
                                            Object[] args) throws ClassNotFoundException, SecurityException, NoSuchMethodException, IllegalArgumentException, InstantiationException, IllegalAccessException, InvocationTargetException
    {
        ConfigurationElement elem = getConfigurationElementForExtension(extensionPoint, 
            discrimAttrName, discrimAttrValue);
        if (elem != null)
        {
            return registry.createExecutableExtension(elem, attributeName, argsClass, args);
        }
        return null;
    }

    /**
     * Loads a class (do not initialize)
     * @param pluginId the plugin id
     * @param className the class name
     * @return the Class
     * @throws ClassNotResolvedException 
     */
    public Class loadClass(String pluginId, String className) throws ClassNotResolvedException
    {
        try
        {
            return registry.loadClass(pluginId, className);
        }
        catch(ClassNotFoundException ex)
        {
            throw new ClassNotResolvedException(ex.getMessage(),ex);
        }
    }

    /**
     * Converts a URL that uses a user-defined protocol into a URL that uses the file protocol.
     * @param url the url to be converted
     * @return the converted URL
     * @throws IOException
     */
    public URL resolveURLAsFileURL(URL url) throws IOException
    {
        return registry.resolveURLAsFileURL(url);
    }    
}