/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store;

import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.OMFContext;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.exceptions.TransactionIsolationNotSupportedException;
import org.datanucleus.management.runtime.StoreManagerRuntime;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.plugin.Extension;
import org.datanucleus.store.connection.ConnectionManager;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.schema.StoreSchemaHandler;
import org.datanucleus.store.scostore.Store;
import org.datanucleus.store.valuegenerator.ValueGenerationManager;
import org.datanucleus.util.Localiser;

/**
 * Federation Manager orchestrates the persistence/retrieval for multiple datastores.
 * It is responsible for creating the individual StoreManager instances for the datastore(s)
 * that are being federated. Currently only manages a single StoreManager
 * NOTE : THIS IS NOT CURRENTLY USED
 */
public class FederationManager implements StoreManager
{
    /** Localisation of messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ExecutionContext.class.getClassLoader());

    /** StoreManager(s). TODO Change to have multiple StoreManagers. */
    StoreManager storeManager;

    final OMFContext omfContext;

    final ClassLoaderResolver clr;

    public FederationManager(ClassLoaderResolver clr, OMFContext omfContext)
    {
        this.omfContext = omfContext;
        this.clr = clr;

        initialiseStoreManager(clr);
    }
    
    /**
     * Gets the context for this ObjectManagerFactory
     * @return Returns the context.
     */
    public OMFContext getOMFContext()
    {
        return omfContext;
    }
    
    /**
     * Method to initialise the StoreManager used by this factory.
     * @param clr ClassLoaderResolver to use for class loading issues
     */
    protected void initialiseStoreManager(ClassLoaderResolver clr)
    {
        StoreManager srm = null;
        Extension[] exts = omfContext.getPluginManager().getExtensionPoint(
            "org.datanucleus.store_manager").getExtensions();

        // Find the StoreManager using the persistence property if specified
        String storeManagerType = omfContext.getPersistenceConfiguration().getStringProperty(
            "datanucleus.storeManagerType");
        if (storeManagerType != null)
        {
            for (int e=0; srm == null && e<exts.length; e++)
            {
                ConfigurationElement[] confElm = exts[e].getConfigurationElements();
                for (int c=0; srm == null && c<confElm.length; c++)
                {
                    String key = confElm[c].getAttribute("key");
                    if (key.equalsIgnoreCase(storeManagerType))
                    {
                        Class[] ctrArgTypes = new Class[] {ClassLoaderResolver.class, OMFContext.class};
                        Object[] ctrArgs = new Object[] {clr, omfContext};
                        try
                        {
                            srm = (StoreManager) omfContext.getPluginManager().createExecutableExtension(
                                "org.datanucleus.store_manager", "key", storeManagerType, 
                                "class-name", ctrArgTypes, ctrArgs);
                        }
                        catch (InvocationTargetException ex)
                        {
                            Throwable t = ex.getTargetException();
                            if (t instanceof RuntimeException)
                            {
                                throw (RuntimeException) t;
                            }
                            else if (t instanceof Error)
                            {
                                throw (Error) t;
                            }
                            else
                            {
                                throw new NucleusException(t.getMessage(), t).setFatal();
                            }
                        }
                        catch (Exception ex)
                        {
                            throw new NucleusException(ex.getMessage(), ex).setFatal();
                        }
                    }
                }
            }
            if (srm == null)
            {
                // No StoreManager of the specified type exists, so check plugins in CLASSPATH
                throw new NucleusUserException(LOCALISER.msg("008004", storeManagerType)).setFatal();
            }
        }

        if (srm == null)
        {
            // Try using the URL of the data source
            String url = omfContext.getPersistenceConfiguration().getStringProperty("datanucleus.ConnectionURL");
            if (url != null)
            {
                int idx = url.indexOf(':');
                if (idx > -1)
                {
                    url = url.substring(0, idx);
                }
            }

            for (int e=0; srm == null && e<exts.length; e++)
            {
                ConfigurationElement[] confElm = exts[e].getConfigurationElements();
                for (int c=0; srm == null && c<confElm.length; c++)
                {
                    String urlKey = confElm[c].getAttribute("url-key");
                    if (url == null || urlKey.equalsIgnoreCase(url))
                    {
                        // Either no URL, or url defined so take this StoreManager
                        Class[] ctrArgTypes = new Class[] {ClassLoaderResolver.class, OMFContext.class};
                        Object[] ctrArgs = new Object[] {clr, omfContext};
                        try
                        {
                            srm = (StoreManager) omfContext.getPluginManager().createExecutableExtension(
                                "org.datanucleus.store_manager", "url-key", url, 
                                "class-name", ctrArgTypes, ctrArgs);
                        }
                        catch (InvocationTargetException ex)
                        {
                            Throwable t = ex.getTargetException();
                            if (t instanceof RuntimeException)
                            {
                                throw (RuntimeException) t;
                            }
                            else if (t instanceof Error)
                            {
                                throw (Error) t;
                            }
                            else
                            {
                                throw new NucleusException(t.getMessage(), t).setFatal();
                            }
                        }
                        catch (Exception ex)
                        {
                            throw new NucleusException(ex.getMessage(), ex).setFatal();
                        }
                    }
                }
            }

            if (srm == null)
            {
                throw new NucleusUserException(LOCALISER.msg("008004", url)).setFatal();
            }
        }

        processTransactionIsolation(srm);
    }

    /**
     * Method to check the supplied transaction isolation level is consistent for this StoreManagers
     * capabilities.
     * @param srm StoreManager
     */
    protected void processTransactionIsolation(StoreManager srm)
    {
        String transactionIsolation = 
            omfContext.getPersistenceConfiguration().getStringProperty("datanucleus.transactionIsolation");
        if (transactionIsolation != null)
        {
            // Transaction isolation has been specified and we need to provide at least this level
            // Order of priority is :-
            // read-uncommitted (lowest), read-committed, repeatable-read, serializable (highest)
            Collection srmOptions = srm.getSupportedOptions();
            if (!srmOptions.contains("TransactionIsolationLevel." + transactionIsolation))
            {
                // Requested transaction isolation isn't supported by datastore so check for higher
                if (transactionIsolation.equals("read-uncommitted"))
                {
                    if (srmOptions.contains("TransactionIsolationLevel.read-committed"))
                    {
                        transactionIsolation = "read-committed";
                    }
                    else if (srmOptions.contains("TransactionIsolationLevel.repeatable-read"))
                    {
                        transactionIsolation = "serializable";
                    }
                    else if (srmOptions.contains("TransactionIsolationLevel.serializable"))
                    {
                        transactionIsolation = "repeatable-read";
                    }
                }
                else if (transactionIsolation.equals("read-committed"))
                {
                    if (srmOptions.contains("TransactionIsolationLevel.repeatable-read"))
                    {
                        transactionIsolation = "repeatable-read";
                    }
                    else if (srmOptions.contains("TransactionIsolationLevel.serializable"))
                    {
                        transactionIsolation = "serializable";
                    }
                }
                else if (transactionIsolation.equals("repeatable-read"))
                {
                    if (srmOptions.contains("TransactionIsolationLevel.serializable"))
                    {
                        transactionIsolation = "serializable";
                    }
                }
                else
                {
                    throw new TransactionIsolationNotSupportedException(transactionIsolation);
                }
            }
            // TODO Handle the updated transactionIsolation value
        }
    }

    public StoreManager getStoreManager()
    {
        return storeManager;
    }
    
    public void close()
    {
        storeManager.close();
    }

    public void addClass(String className, ClassLoaderResolver clr)
    {
        storeManager.addClass(className, clr);
    }

    public void addClasses(String[] classNames, ClassLoaderResolver clr)
    {
        storeManager.addClasses(classNames, clr);        
    }

    public ApiAdapter getApiAdapter()
    {
        return storeManager.getApiAdapter();
    }

    public Store getBackingStoreForField(ClassLoaderResolver clr, AbstractMemberMetaData fmd, Class type)
    {
        return storeManager.getBackingStoreForField(clr, fmd, type);
    }

    public String getClassNameForObjectID(Object id, ClassLoaderResolver clr, ExecutionContext ec)
    {
        return storeManager.getClassNameForObjectID(id, clr, ec);
    }

    public Date getDatastoreDate()
    {
        return storeManager.getDatastoreDate();
    }

    public Extent getExtent(ExecutionContext ec, Class c, boolean subclasses)
    {
        return storeManager.getExtent(ec, c, subclasses);
    }

    public NucleusConnection getNucleusConnection(ExecutionContext ec)
    {
        return storeManager.getNucleusConnection(ec);
    }

    public NucleusSequence getNucleusSequence(ExecutionContext ec, SequenceMetaData seqmd)
    {
        return storeManager.getNucleusSequence(ec, seqmd);
    }

    public StoreSchemaHandler getSchemaHandler()
    {
        return storeManager.getSchemaHandler();
    }

    public StorePersistenceHandler getPersistenceHandler()
    {
        return storeManager.getPersistenceHandler();
    }

    public ValueGenerationManager getValueGenerationManager()
    {
        return storeManager.getValueGenerationManager();
    }

    public StoreManagerRuntime getRuntimeManager()
    {
        return storeManager.getRuntimeManager();
    }

    public String getStoreManagerKey()
    {
        return storeManager.getStoreManagerKey();
    }

    public String getQueryCacheKey()
    {
        return storeManager.getQueryCacheKey();
    }

    public Object getStrategyValue(ExecutionContext ec, AbstractClassMetaData cmd, int absoluteFieldNumber)
    {
        return storeManager.getStrategyValue(ec, cmd, absoluteFieldNumber);
    }

    public HashSet getSubClassesForClass(String className, boolean includeDescendents, ClassLoaderResolver clr)
    {
        return storeManager.getSubClassesForClass(className, includeDescendents, clr);
    }

    public boolean isStrategyDatastoreAttributed(IdentityStrategy identityStrategy, boolean datastoreIdentityField)
    {
        return storeManager.isStrategyDatastoreAttributed(identityStrategy, datastoreIdentityField);
    }

    public String manageClassForIdentity(Object id, ClassLoaderResolver clr)
    {
        return storeManager.manageClassForIdentity(id, clr);
    }

    public boolean managesClass(String className)
    {
        return storeManager.managesClass(className);
    }

    public void notifyObjectIsOutdated(ObjectProvider sm)
    {
        storeManager.notifyObjectIsOutdated(sm);
    }

    public void printInformation(String category, PrintStream ps) throws Exception
    {
        storeManager.printInformation(category, ps);        
    }

    public void performVersionCheck(ObjectProvider sm, Object versionDatastore, VersionMetaData versionMetaData)
    {
        storeManager.performVersionCheck(sm, versionDatastore, versionMetaData);        
    }

    public void removeAllClasses(ClassLoaderResolver clr)
    {
        storeManager.removeAllClasses(clr);
    }

    public boolean supportsQueryLanguage(String language)
    {
        return storeManager.supportsQueryLanguage(language);
    }

    public boolean supportsValueStrategy(String language)
    {
        return storeManager.supportsValueStrategy(language);
    }

    public Collection getSupportedOptions()
    {
        return storeManager.getSupportedOptions();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionManager()
     */
    public ConnectionManager getConnectionManager()
    {
        return storeManager.getConnectionManager();
    }

    public ManagedConnection getConnection(ExecutionContext ec)
    {
        return storeManager.getConnection(ec);
    }
    
    public ManagedConnection getConnection(ExecutionContext ec, Map options)
    {
        return storeManager.getConnection(ec, options);
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionDriverName()
     */
    public String getConnectionDriverName()
    {
        return storeManager.getConnectionDriverName();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionURL()
     */
    public String getConnectionURL()
    {
        return storeManager.getConnectionURL();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionUserName()
     */
    public String getConnectionUserName()
    {
        return storeManager.getConnectionUserName();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionPassword()
     */
    public String getConnectionPassword()
    {
        return storeManager.getConnectionPassword();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionFactory()
     */
    public Object getConnectionFactory()
    {
        return storeManager.getConnectionFactory();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionFactory2()
     */
    public Object getConnectionFactory2()
    {
        return storeManager.getConnectionFactory2();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionFactory2Name()
     */
    public String getConnectionFactory2Name()
    {
        return storeManager.getConnectionFactory2Name();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getConnectionFactoryName()
     */
    public String getConnectionFactoryName()
    {
        return storeManager.getConnectionFactoryName();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#transactionStarted(org.datanucleus.store.ExecutionContext)
     */
    public void transactionStarted(ExecutionContext ec)
    {
        // TODO Auto-generated method stub
        
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#transactionCommitted(org.datanucleus.store.ExecutionContext)
     */
    public void transactionCommitted(ExecutionContext ec)
    {
        // TODO Auto-generated method stub
        
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#transactionRolledBack(org.datanucleus.store.ExecutionContext)
     */
    public void transactionRolledBack(ExecutionContext ec)
    {
        // TODO Auto-generated method stub
        
    }
}