/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.json;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.FetchPlan;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusObjectNotFoundException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.store.AbstractPersistenceHandler;
import org.datanucleus.store.AbstractStoreManager;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.FieldValues2;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.Type;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.json.fieldmanager.FetchFieldManager;
import org.datanucleus.store.json.fieldmanager.InsertFieldManager;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class JsonPersistenceHandler extends AbstractPersistenceHandler
{
    protected AbstractStoreManager storeMgr;

    JsonPersistenceHandler(StoreManager storeMgr)
    {
        this.storeMgr = (AbstractStoreManager)storeMgr;
    }
    
    public void close()
    {
        // nothing to do
    }

    public String getURLPath(ObjectProvider sm)
    {
        String url = getURLPath(sm.getClassMetaData());
        JSONObject jsonobj = new JSONObject();
        sm.provideFields(sm.getClassMetaData().getPKMemberPositions(), new InsertFieldManager(sm,jsonobj));
        try
        {
            url += jsonobj.get(JSONObject.getNames(jsonobj)[0]).toString();
        }
        catch (JSONException e)
        {
            throw new NucleusException(e.getMessage(), e);
        }

        return url;
    }
    
    public String getURLPath(AbstractClassMetaData acmd)
    {
        String url = acmd.getValueForExtension("url");
        if (url==null)
        {
            url = acmd.getFullClassName();
        }        
        if (!url.endsWith("/"))
        {
            url += "/";
        }
        return url;
    } 
    
    public String getURLPathForQuery(AbstractClassMetaData acmd)
    {
        String url = acmd.getValueForExtension("url");
        if (url==null)
        {
            url = acmd.getFullClassName();
        }        
        if (!url.endsWith("/"))
        {
            url += "/";
        }
        return url;
    }     
    
    public void insertObject(ObjectProvider sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        int[] fieldNumbers = sm.getClassMetaData().getAllMemberPositions();
        Map<String,String> options = new HashMap<String,String>();
        options.put(ConnectionFactoryImpl.STORE_JSON_URL, getURLPath(sm));
        options.put("Content-Type", "application/json");
        ManagedConnection mconn = storeMgr.getConnection(sm.getExecutionContext(), options);
        URLConnection conn = (URLConnection) mconn.getConnection();
        JSONObject jsonobj = new JSONObject();

        sm.provideFields(fieldNumbers, new InsertFieldManager(sm, jsonobj));
        sm.provideFields(sm.getClassMetaData().getPKMemberPositions(), new InsertFieldManager(sm, jsonobj));

        write("POST",conn.getURL().toExternalForm(),conn,jsonobj,getHeaders("POST",options));
    }

    public void updateObject(ObjectProvider sm, int[] fieldNumbers)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        Map<String,String> options = new HashMap<String,String>();
        options.put(ConnectionFactoryImpl.STORE_JSON_URL, getURLPath(sm));
        options.put("Content-Type", "application/json");        
        ManagedConnection mconn = storeMgr.getConnection(sm.getExecutionContext(), options);
        URLConnection conn = (URLConnection) mconn.getConnection();
        JSONObject jsonobj = new JSONObject();

        sm.provideFields(fieldNumbers, new InsertFieldManager(sm, jsonobj));
        sm.provideFields(sm.getClassMetaData().getPKMemberPositions(), new InsertFieldManager(sm, jsonobj));

        write("PUT",conn.getURL().toExternalForm(),conn,jsonobj,getHeaders("PUT",options));
    }

    public void deleteObject(ObjectProvider sm)
    {
        // Check if read-only so update not permitted
        storeMgr.assertReadOnlyForUpdateOfObject(sm);

        Map<String,String> options = new HashMap<String,String>();
        options.put(ConnectionFactoryImpl.STORE_JSON_URL, getURLPath(sm));
        ManagedConnection mconn = storeMgr.getConnection(sm.getExecutionContext(), options);
        URLConnection conn = (URLConnection) mconn.getConnection();
        try
        {
            HttpURLConnection http = (HttpURLConnection)conn;
            Map<String,String> headers = getHeaders("DELETE",options);
            Iterator iterator = headers.keySet().iterator();
            while(iterator.hasNext())
            {
                String key = (String) iterator.next();
                String value = (String) headers.get(key);
                http.setRequestProperty(key, value);
            }
            http.setRequestMethod("DELETE");
            http.setReadTimeout(10000);
            http.setConnectTimeout(10000);
            http.connect();
            int code = http.getResponseCode();
            if (code == 404)
            {
                throw new NucleusObjectNotFoundException();
            }
            handleHTTPErrorCode(http);
         }
        catch (IOException e)
        {
            throw new NucleusDataStoreException(e.getMessage(),e);
        }
    }

    protected void handleHTTPErrorCode(HttpURLConnection http) throws IOException
    {
        if(http.getResponseCode()>=400)
        {
            StringBuffer sb = new StringBuffer();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            byte[] buffer = new byte[1024];
            int r;
            if (http.getErrorStream()!=null)
            {
                while ((r = http.getErrorStream().read(buffer)) != -1) {
                    baos.write(buffer, 0, r);
                }
                sb.append(new String(baos.toByteArray()));        
                http.getErrorStream().close();
            }

            throw new NucleusDataStoreException("Error on URL: '"+http.getURL().toExternalForm()+"' Request Method: "+http.getRequestMethod()+" HTTP Error code: "+http.getResponseCode()+" "+http.getResponseMessage()+" error: "+sb.toString());
        }
        if(http.getResponseCode()>=300)
        {
            throw new NucleusDataStoreException("Redirect not supported. HTTP Error code: "+http.getResponseCode()+" "+http.getResponseMessage());
        }
    }
    
    public void fetchObject(ObjectProvider sm, int[] fieldNumbers)
    {
        JSONObject jsonobj = new JSONObject();
        sm.provideFields(sm.getClassMetaData().getPKMemberPositions(), new InsertFieldManager(sm,jsonobj));
        Map<String,String> options = new HashMap<String,String>();
        options.put(ConnectionFactoryImpl.STORE_JSON_URL, getURLPath(sm));
        ManagedConnection mconn = storeMgr.getConnection(sm.getExecutionContext(), options);
        URLConnection conn = (URLConnection) mconn.getConnection();
        JSONObject result = read("GET",conn.getURL().toExternalForm(),conn,getHeaders("GET",options));
        
        sm.replaceFields(fieldNumbers, new FetchFieldManager(sm,result));

    }

    public Object findObject(ExecutionContext om, Object id)
    {
        // TODO Auto-generated method stub
        return null;
    }

    public void locateObject(ObjectProvider sm)
    {
        Map<String,String> options = new HashMap<String,String>();
        options.put(ConnectionFactoryImpl.STORE_JSON_URL, getURLPath(sm));
        ManagedConnection mconn = storeMgr.getConnection(sm.getExecutionContext(), options);
        URLConnection conn = (URLConnection) mconn.getConnection();

        try
        {
            HttpURLConnection http = (HttpURLConnection)conn;
            Map<String,String> headers = getHeaders("HEAD",options);
            Iterator iterator = headers.keySet().iterator();
            while(iterator.hasNext())
            {
                String key = (String) iterator.next();
                String value = (String) headers.get(key);
                http.setRequestProperty(key, value);
            }
            http.setDoOutput(true);
            http.setRequestMethod("HEAD");
            http.setReadTimeout(10000);
            http.setConnectTimeout(10000);
            http.connect();
            int code = http.getResponseCode();
            if (code == 404)
            {
                throw new NucleusObjectNotFoundException();
            }
            handleHTTPErrorCode(http);
        }
        catch (IOException e)
        {
            throw new NucleusObjectNotFoundException(e.getMessage(),e);
        }
    }

    protected void write(String method, String requestUri, URLConnection conn, JSONObject jsonobj, Map<String, String> headers)
    {
        try
        {
            if (JSONLogger.LOGGER.isDebugEnabled())
            {
                JSONLogger.LOGGER.debug("Writing to URL "+requestUri+" content "+jsonobj.toString());
            }
            int length = jsonobj.toString().length();
            HttpURLConnection http = (HttpURLConnection)conn;
            Iterator<String> iterator = headers.keySet().iterator();
            while(iterator.hasNext())
            {
                String key = iterator.next();
                String value = headers.get(key);
                http.setRequestProperty(key, value);
            }
            http.setRequestProperty("Content-Length", ""+length);
            http.setDoOutput(true);
            http.setRequestMethod(method);
            http.setReadTimeout(10000);
            http.setConnectTimeout(10000);
            http.connect();
            OutputStream os = conn.getOutputStream();
            os.write(jsonobj.toString().getBytes());
            os.flush();
            os.close();
            handleHTTPErrorCode(http);
       }
        catch (IOException e)
        {
            throw new NucleusDataStoreException(e.getMessage(),e);
        }
    }

    protected JSONObject read(String method, String requestUri, URLConnection conn, Map headers)
    {
        try
        {
            HttpURLConnection http = (HttpURLConnection)conn;
            Iterator iterator = headers.keySet().iterator();
            while(iterator.hasNext())
            {
                String key = (String) iterator.next();
                String value = (String) headers.get(key);
                http.setRequestProperty(key, value);
            }
            //http.setDoOutput(true);
            http.setDoInput(true);
            http.setRequestMethod(method);
            http.setReadTimeout(10000);
            http.setConnectTimeout(10000);
            http.connect();
            /*
            OutputStream os = conn.getOutputStream();
            os.write(json.toString().getBytes());
            os.flush();
            os.close();
            */
            int code = http.getResponseCode();
            if (code == 404)
            {
                throw new NucleusObjectNotFoundException();
            }
            /*String msg =*/ http.getResponseMessage();
            StringBuffer sb = new StringBuffer();
            if (http.getContentLength()>0)
            {
                for( int i=0; i<http.getContentLength(); i++)
                {
                    sb.append((char)http.getInputStream().read());
                }
            }
            else
            {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                byte[] buffer = new byte[1024];
                int r;
                while ((r = http.getInputStream().read(buffer)) != -1) {
                    baos.write(buffer, 0, r);
                }
                
                sb.append(new String(baos.toByteArray()));        
            }
            http.getInputStream().close();
            return new JSONObject(sb.toString());
        }
        catch (SocketTimeoutException e)
        {
            throw new NucleusDataStoreException(e.getMessage(),e);
        }
        catch (IOException e)
        {
            throw new NucleusDataStoreException(e.getMessage(),e);
        }
        catch (JSONException e)
        {
            throw new NucleusDataStoreException(e.getMessage(),e);
        }
    }
    
    protected Map<String,String> getHeaders(String httpVerb, Map<String,String> options)
    {
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Date", CloudStorageUtils.getHTTPDate());
        String contentType = "";
        if (options.containsKey("Content-Type"))
        {
            contentType = options.get("Content-Type");
            headers.put("Content-Type", contentType);
        }
        return headers;
    }
 
    /**
     * Convenience method to get all objects of the candidate type from the specified connection.
     * @param om ObjectManager
     * @param mconn Managed Connection
     * @param candidateClass Candidate
     * @param ignoreCache Whether to ignore the cache
     * @return List of objects of the candidate type
     */
    public List getObjectsOfCandidateType(ExecutionContext om, ManagedConnection mconn, 
            Class candidateClass, boolean ignoreCache, Map options)
    {
        List results = new ArrayList();

        try
        {
            URLConnection conn = (URLConnection) mconn.getConnection();
            ClassLoaderResolver clr = om.getClassLoaderResolver();
            final AbstractClassMetaData cmd = om.getMetaDataManager().getMetaDataForClass(candidateClass, clr);

            JSONArray jsonarray;
            try
            {                
                HttpURLConnection http = (HttpURLConnection) conn;
                Map headers = getHeaders("GET", options);
                Iterator iterator = headers.keySet().iterator();
                while(iterator.hasNext())
                {
                    String key = (String) iterator.next();
                    String value = (String) headers.get(key);
                    http.setRequestProperty(key, value);
                }
                http.setDoInput(true);
                http.setRequestMethod("GET");
                http.setReadTimeout(10000);
                http.setConnectTimeout(10000);
                http.connect();
                int code = http.getResponseCode();
                if (code == 404)
                {
                    return Collections.EMPTY_LIST;
                }

                /* String msg = */http.getResponseMessage();
                StringBuffer sb = new StringBuffer();
                if (http.getContentLength() > 0)
                {
                    for (int i = 0; i < http.getContentLength(); i++)
                    {
                        sb.append((char) http.getInputStream().read());
                    }
                }
                else
                {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    byte[] buffer = new byte[1024];
                    int r;
                    while ((r = http.getInputStream().read(buffer)) != -1)
                    {
                        baos.write(buffer, 0, r);
                    }
                    sb.append(new String(baos.toByteArray()));
                }
                http.getInputStream().close();
                jsonarray = new JSONArray(sb.toString());
            }
            catch (IOException e)
            {
                throw new NucleusDataStoreException(e.getMessage(), e);
            }
            catch (JSONException e)
            {
                throw new NucleusDataStoreException(e.getMessage(), e);
            }

            for (int i = 0; i < jsonarray.length(); i++)
            {
                final JSONObject json = jsonarray.getJSONObject(i);

                results.add(om.findObjectUsingAID(new Type(clr.classForName(cmd.getFullClassName())), new FieldValues2()
                {
                    // StateManager calls the fetchFields method
                    public void fetchFields(ObjectProvider sm)
                    {
                        sm.replaceFields(cmd.getAllMemberPositions(), new FetchFieldManager(sm, json));
                    }

                    public void fetchNonLoadedFields(ObjectProvider sm)
                    {
                        sm.replaceNonLoadedFields(cmd.getAllMemberPositions(), new FetchFieldManager(sm, json));
                    }

                    public FetchPlan getFetchPlanForLoading()
                    {
                        return null;
                    }
                }, ignoreCache, true));
            }
        }
        catch (JSONException je)
        {
            throw new NucleusException(je.getMessage(), je);
        }

        return results;
    }
}