/**********************************************************************
Copyright (c) 2008 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.json.fieldmanager;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.List;

import javax.jdo.PersistenceManager;
import javax.jdo.spi.Detachable;
import javax.jdo.spi.JDOImplHelper;
import javax.jdo.spi.PersistenceCapable;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.OMFContext;
import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.fieldmanager.FieldManager;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.TypeConversionHelper;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * FieldManager for fetching from JSON.
 */
public class FetchFieldManager implements FieldManager
{
    ObjectProvider sm;
    JSONObject result;

    public FetchFieldManager(ObjectProvider sm, JSONObject result)
    {
        this.sm = sm;
        this.result = result;
    }

    public String fetchStringField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if (result.isNull(fieldName))
        {
            return null;
        }
        try
        {
            return result.getString(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return null;
        }
    }

    public short fetchShortField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return (short) result.getInt(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public Object fetchObjectField(int fieldNumber)
    {
        AbstractMemberMetaData mmd = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return null;
        }
        try
        {
            Object value = result.get(fieldName);
            if (value instanceof JSONObject)
            {
                return getObjectFromJSONObject((JSONObject) value, mmd.getType().getName(), sm.getExecutionContext());
            }
            else if (value instanceof JSONArray)
            {
                return fetchJSONArray((JSONArray)value, fieldNumber, sm.getExecutionContext(), sm.getClassMetaData());
            }
            else 
            {
                return TypeConversionHelper.convertTo(result.get(fieldName), mmd.getType());
            }
        }
        catch (JSONException e)
        {
            throw new NucleusException(e.getMessage(), e);
        }
        
    }
    private List fetchJSONArray(JSONArray array,  int position, ExecutionContext om, AbstractClassMetaData cmd) throws JSONException
    {
        List elements = new ArrayList();
        for( int i=0; i<array.length(); i++)
        {
            if( array.isNull(i) )
            {
                elements.add(null);
            }
            else
            {
                Object value = array.get(i);
                
                if( value instanceof JSONObject)
                {
                    elements.add(getObjectFromJSONObject((JSONObject)value, ((JSONObject)value).getString("class"), om));
                }
                else if( value instanceof JSONArray)
                {
                    elements.add(fetchJSONArray((JSONArray)value,position, om, cmd));
                }
                else
                {
                    elements.add(TypeConversionHelper.convertTo(value, cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getType()));
                }
            }
        }
        return elements;
    }
    
    public long fetchLongField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return result.getLong(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public int fetchIntField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return result.getInt(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public float fetchFloatField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return (float) result.getDouble(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public double fetchDoubleField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return result.getDouble(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public char fetchCharField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            return result.getString(fieldName).charAt(0);
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public byte fetchByteField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return 0;
        }
        try
        {
            String str = result.getString(fieldName);
            return new Byte(str).byteValue();
        }
        catch (JSONException e)
        {
            //ignore
            return 0;
        }
    }

    public boolean fetchBooleanField(int fieldNumber)
    {
        String fieldName = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
        if( result.isNull(fieldName) )
        {
            return false;
        }
        try
        {
            return result.getBoolean(fieldName);
        }
        catch (JSONException e)
        {
            //ignore
            return false;
        }
    }

    public void storeStringField(int fieldNumber, String value)
    {
        // TODO Auto-generated method stub
    }

    public void storeShortField(int fieldNumber, short value)
    {
        // TODO Auto-generated method stub
    }

    public void storeObjectField(int fieldNumber, Object value)
    {
        // TODO Auto-generated method stub
    }

    public void storeLongField(int fieldNumber, long value)
    {
        // TODO Auto-generated method stub
    }

    public void storeIntField(int fieldNumber, int value)
    {
        // TODO Auto-generated method stub
    }

    public void storeFloatField(int fieldNumber, float value)
    {
        // TODO Auto-generated method stub
    }

    public void storeDoubleField(int fieldNumber, double value)
    {
        // TODO Auto-generated method stub
    }

    public void storeCharField(int fieldNumber, char value)
    {
        // TODO Auto-generated method stub
    }

    public void storeByteField(int fieldNumber, byte value)
    {
        // TODO Auto-generated method stub
    }

    public void storeBooleanField(int fieldNumber, boolean value)
    {
        // TODO Auto-generated method stub
    }
    
    /**
     * @param jsonobj JSON Object
     * @param className The class name
     * @param om The object manager
     * @return The object for the supplied JSON object
     * @throws ClassNotResolvedException when the class is not found
     * @throws {@link JDOFatalUserException} many situations, such as for google appengine when trying the set primary
     * key of a PC_NEW
     */
    protected Object getObjectFromJSONObject(final JSONObject jsonobj, String className, ExecutionContext om)
    {
        OMFContext ctx = om.getOMFContext();

        ClassLoaderResolver clr = om.getClassLoaderResolver();
        Class cls = clr.classForName(className, true);
        AbstractClassMetaData cmd = ctx.getMetaDataManager().getMetaDataForClass(cls, clr);

        if (cmd == null)
        {
            return getObject(clr, jsonobj, cls);
        }

        // TODO Remove this JDO-specific way of getting field values. All other plugins do it correctly
        LocalSM sm1 = new LocalSM(cls);
        int[] fieldNumbers = cmd.getAllMemberPositions();
        sm1.replaceFields(fieldNumbers, new LocalFieldManager(jsonobj, null, cmd, om));
        Object obj = sm1.getObject();
        try
        {
            Object id = om.getApiAdapter().getNewApplicationIdentityObjectId(sm1.getObject(), cmd);
            obj = om.findObject(id, false, true, null);

            ObjectProvider sm = om.findObjectProvider(obj);
            fieldNumbers = cmd.getNonPKMemberPositions();
            sm.replaceFields(fieldNumbers, new LocalFieldManager(jsonobj, sm, cmd, om));
            return obj;
        }
        catch (NucleusException ex)
        {
            // maybe the object was not found or had no identity values, so assume this is a new object
            sm1.disconnect();
            return obj;
        }
    }
 
    class LocalFieldManager implements FieldManager
    {
        JSONObject jsonobj;

        AbstractClassMetaData cmd;

        ObjectProvider sm;

        ExecutionContext pm;

        /**
         * @param jsonobj
         * @param sm may be null
         * @param cmd
         * @param pm
         */
        LocalFieldManager(JSONObject jsonobj, ObjectProvider sm, AbstractClassMetaData cmd, ExecutionContext pm)
        {
            this.jsonobj = jsonobj;
            this.cmd = cmd;
            this.sm = sm;
            this.pm = pm;
        }

        public String fetchStringField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return null;
            }
            try
            {
                String value = jsonobj.getString(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return null;
        }

        public short fetchShortField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                short value = (short) jsonobj.getInt(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public Object fetchObjectField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return null;
            }
            try
            {
                    if (jsonobj.isNull(fieldName))
                    {
                        return null;
                    }
                    Object value = jsonobj.get(fieldName);
                    if( value instanceof JSONObject)
                    {
                        value = getObjectFromJSONObject((JSONObject)value, ((JSONObject)value).getString("class"), pm);
                        if (sm != null)
                        {
                            sm.makeDirty(position);
                        }
                        return value;
                    }
                    if( value instanceof JSONArray)
                    {
                        value= fetchJSONArray((JSONArray)value,position);
                        if (sm != null)
                        {
                            sm.makeDirty(position);
                        }
                        return value;
                    }
                    if (sm != null)
                    {
                        sm.makeDirty(position);
                    }
                    return TypeConversionHelper.convertTo(value, cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getType());
            }
            catch(JSONException ex)
            {
                throw new RuntimeException(ex);
            }
        }

        private List fetchJSONArray(JSONArray array,  int position) throws JSONException
        {
            List elements = new ArrayList();
            for( int i=0; i<array.length(); i++)
            {
                if( array.isNull(i) )
                {
                    elements.add(null);
                }
                else
                {
                    Object value = array.get(i);
                    
                    if( value instanceof JSONObject)
                    {
                        elements.add(getObjectFromJSONObject((JSONObject)value, ((JSONObject)value).getString("class"), pm));
                    }
                    else if( value instanceof JSONArray)
                    {
                        elements.add(fetchJSONArray((JSONArray)value,position));
                    }
                    else
                    {
                        elements.add(TypeConversionHelper.convertTo(value, cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getType()));
                    }
                }
            }
            return elements;
        }
        
        public long fetchLongField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                long value = jsonobj.getLong(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public int fetchIntField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                int value = jsonobj.getInt(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public float fetchFloatField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                float value = (float) jsonobj.getDouble(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public double fetchDoubleField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                double value = jsonobj.getDouble(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public char fetchCharField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                String str = jsonobj.getString(fieldName);
                char value = 0;
                if (str != null && str.length() > 0)
                {
                    value = str.charAt(0);
                }
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public byte fetchByteField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return 0;
            }
            try
            {
                String str = jsonobj.getString(fieldName);
                byte value = 0;
                if (str != null && str.length() > 0)
                {
                    value = str.getBytes()[0];
                }
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return 0;
        }

        public boolean fetchBooleanField(int position)
        {
            String fieldName = cmd.getMetaDataForManagedMemberAtAbsolutePosition(position).getName();
            if (!jsonobj.has(fieldName))
            {
                return false;
            }
            try
            {
                boolean value = jsonobj.getBoolean(fieldName);
                if (sm != null)
                {
                    sm.makeDirty(position);
                }
                return value;
            }
            catch (JSONException e)
            {
                // should not happen
            }
            return false;
        }

        public void storeStringField(int arg0, String arg1)
        {
        }

        public void storeShortField(int arg0, short arg1)
        {
        }

        public void storeObjectField(int arg0, Object arg1)
        {
        }

        public void storeLongField(int arg0, long arg1)
        {
        }

        public void storeIntField(int arg0, int arg1)
        {
        }

        public void storeFloatField(int arg0, float arg1)
        {
        }

        public void storeDoubleField(int arg0, double arg1)
        {
        }

        public void storeCharField(int arg0, char arg1)
        {
        }

        public void storeByteField(int arg0, byte arg1)
        {
        }

        public void storeBooleanField(int arg0, boolean arg1)
        {
        }
    }

    // TODO Delete this. JDO specific code which has no place in a store plugin
    /**
     * This class is used to populate fields of PC objects
     */
    private class LocalSM implements javax.jdo.spi.StateManager
    {
        PersistenceCapable myPC;

        FieldManager fm;

        public LocalSM(Class cls)
        {
            myPC = JDOImplHelper.getInstance().newInstance(cls, this);
        }

        public boolean getBooleanField(PersistenceCapable arg0, int arg1, boolean arg2)
        {
            return false;
        }

        public byte getByteField(PersistenceCapable arg0, int arg1, byte arg2)
        {
            return 0;
        }

        public char getCharField(PersistenceCapable arg0, int arg1, char arg2)
        {
            return 0;
        }

        public double getDoubleField(PersistenceCapable arg0, int arg1, double arg2)
        {
            return 0;
        }

        public float getFloatField(PersistenceCapable arg0, int arg1, float arg2)
        {
            return 0;
        }

        public int getIntField(PersistenceCapable arg0, int arg1, int arg2)
        {
            return 0;
        }

        public long getLongField(PersistenceCapable arg0, int arg1, long arg2)
        {
            return 0;
        }

        public Object getObjectField(PersistenceCapable arg0, int arg1, Object arg2)
        {
            return null;
        }

        public Object getObjectId(PersistenceCapable arg0)
        {
            return null;
        }

        public PersistenceManager getPersistenceManager(PersistenceCapable arg0)
        {
            return null;
        }

        public short getShortField(PersistenceCapable arg0, int arg1, short arg2)
        {
            return 0;
        }

        public String getStringField(PersistenceCapable arg0, int arg1, String arg2)
        {
            return null;
        }

        public Object getTransactionalObjectId(PersistenceCapable arg0)
        {
            return null;
        }

        public Object getVersion(PersistenceCapable arg0)
        {
            return null;
        }

        public boolean isDeleted(PersistenceCapable arg0)
        {
            return false;
        }

        public boolean isDirty(PersistenceCapable arg0)
        {
            return false;
        }

        public boolean isLoaded(PersistenceCapable arg0, int arg1)
        {
            return false;
        }

        public boolean isNew(PersistenceCapable arg0)
        {
            return false;
        }

        public boolean isPersistent(PersistenceCapable arg0)
        {
            return false;
        }

        public boolean isTransactional(PersistenceCapable arg0)
        {
            return false;
        }

        public void makeDirty(PersistenceCapable arg0, String arg1)
        {
        }

        public void preSerialize(PersistenceCapable arg0)
        {
        }

        public void providedBooleanField(PersistenceCapable arg0, int arg1, boolean arg2)
        {
        }

        public void providedByteField(PersistenceCapable arg0, int arg1, byte arg2)
        {
        }

        public void providedCharField(PersistenceCapable arg0, int arg1, char arg2)
        {
        }

        public void providedDoubleField(PersistenceCapable arg0, int arg1, double arg2)
        {
        }

        public void providedFloatField(PersistenceCapable arg0, int arg1, float arg2)
        {
        }

        public void providedIntField(PersistenceCapable arg0, int arg1, int arg2)
        {
        }

        public void providedLongField(PersistenceCapable arg0, int arg1, long arg2)
        {
        }

        public void providedObjectField(PersistenceCapable arg0, int arg1, Object arg2)
        {
        }

        public void providedShortField(PersistenceCapable arg0, int arg1, short arg2)
        {
        }

        public void providedStringField(PersistenceCapable arg0, int arg1, String arg2)
        {
        }

        public boolean replacingBooleanField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchBooleanField(arg1);
        }

        public byte replacingByteField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchByteField(arg1);
        }

        public char replacingCharField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchCharField(arg1);
        }

        public Object[] replacingDetachedState(Detachable arg0, Object[] arg1)
        {
            return null;
        }

        public double replacingDoubleField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchDoubleField(arg1);
        }

        public byte replacingFlags(PersistenceCapable arg0)
        {
            return 0;
        }

        public float replacingFloatField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchFloatField(arg1);
        }

        public int replacingIntField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchIntField(arg1);
        }

        public long replacingLongField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchLongField(arg1);
        }

        public Object replacingObjectField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchObjectField(arg1);
        }

        public short replacingShortField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchShortField(arg1);
        }

        public javax.jdo.spi.StateManager replacingStateManager(PersistenceCapable arg0, javax.jdo.spi.StateManager arg1)
        {
            return null;
        }

        public String replacingStringField(PersistenceCapable arg0, int arg1)
        {
            return fm.fetchStringField(arg1);
        }

        public void setBooleanField(PersistenceCapable arg0, int arg1, boolean arg2, boolean arg3)
        {
        }

        public void setByteField(PersistenceCapable arg0, int arg1, byte arg2, byte arg3)
        {
        }

        public void setCharField(PersistenceCapable arg0, int arg1, char arg2, char arg3)
        {
        }

        public void setDoubleField(PersistenceCapable arg0, int arg1, double arg2, double arg3)
        {
        }

        public void setFloatField(PersistenceCapable arg0, int arg1, float arg2, float arg3)
        {
        }

        public void setIntField(PersistenceCapable arg0, int arg1, int arg2, int arg3)
        {
        }

        public void setLongField(PersistenceCapable arg0, int arg1, long arg2, long arg3)
        {
        }

        public void setObjectField(PersistenceCapable arg0, int arg1, Object arg2, Object arg3)
        {
        }

        public void setShortField(PersistenceCapable arg0, int arg1, short arg2, short arg3)
        {
        }

        public void setStringField(PersistenceCapable arg0, int arg1, String arg2, String arg3)
        {
        }

        void replaceFields(int[] fieldNumbers, FieldManager fm)
        {
            this.fm = fm;
            myPC.jdoReplaceFields(fieldNumbers);
        }

        public Object getObject()
        {
            return myPC;
        }

        public void disconnect()
        {
            try
            {
                // Calls to pc.jdoReplaceStateManager must be run privileged
                AccessController.doPrivileged(new PrivilegedAction()
                {
                    public Object run()
                    {
                        myPC.jdoReplaceStateManager(null);
                        return null;
                    }
                });
            }
            catch (SecurityException e)
            {
            }
        }
    }
 
    /**
     * Deserialise from JSON to an object. (Used for non PersistenceCapable classes)
     * @param jsonobj
     * @param cls
     * @return
     */
    private Object getObject(final ClassLoaderResolver clr, final JSONObject jsonobj, final Class cls)
    {
        if (cls.getName().equals("com.google.appengine.api.users.User"))
        {
            String email = null;
            String authDomain = null;
            try
            {
                email = jsonobj.getString("email");
            }
            catch (JSONException e)
            {
                // should not happen if the field exists
            }
            try
            {
                authDomain = jsonobj.getString("authDomain");
            }
            catch (JSONException e)
            {
                // should not happen if the field exists
            }
            return ClassUtils.newInstance(cls, new Class[]{String.class, String.class}, new String[]{email, authDomain});
        }
        else if (cls.getName().equals("com.google.appengine.api.datastore.Key"))
        {
            try
            {
                Object parent = null;
                if (jsonobj.has("parent") && !jsonobj.isNull("parent"))
                {
                    
                    //if it's a JSONObject
                    JSONObject parentobj = jsonobj.getJSONObject("parent");
                    parent = getObject(clr, parentobj,clr.classForName(jsonobj.getString("class")));
                }
                if (jsonobj.has("appId"))
                {
                    String appId = jsonobj.getString("appId");
                    String kind = jsonobj.getString("kind");
                    Class keyFactory = Class.forName("com.google.appengine.api.datastore.KeyFactory",false,cls.getClassLoader());
                    if( parent != null)
                    {
                        return ClassUtils.getMethodForClass(keyFactory, "createKey", new Class[]{cls,String.class,String.class}).invoke(null, new Object[]{parent,kind,appId});
                    }
                    else
                    {
                        return ClassUtils.getMethodForClass(keyFactory, "createKey", new Class[]{String.class,String.class}).invoke(null, new Object[]{kind,appId});
                    }
                }
                else
                {
                    long id = jsonobj.getLong("id");
                    String kind = jsonobj.getString("kind");
                    Class keyFactory = Class.forName("com.google.appengine.api.datastore.KeyFactory",false,cls.getClassLoader());
                    if( parent != null)
                    {
                        return ClassUtils.getMethodForClass(keyFactory, "createKey",
                            new Class[]{cls,String.class,long.class}).invoke(null, new Object[]{parent,kind,Long.valueOf(id)});
                    }
                    else
                    {
                        return ClassUtils.getMethodForClass(keyFactory, "createKey", 
                            new Class[]{String.class,long.class}).invoke(null, new Object[]{kind,Long.valueOf(id)});
                    }
                }
            }
            catch (Exception e)
            {
                throw new RuntimeException(e);
            }
        }
        
        else
        {
            try
            {
                return AccessController.doPrivileged(new PrivilegedAction()
                {
                    public Object run()
                    {
                        try
                        {
                            Constructor c = ClassUtils.getConstructorWithArguments(cls, new Class[]{});
                            c.setAccessible(true);
                            Object obj = c.newInstance(new Object[]{});
                            String[] fieldNames = JSONObject.getNames(jsonobj);
                            for (int i = 0; i < jsonobj.length(); i++)
                            {
                                //ignore class field
                                if(!fieldNames[i].equals("class"))
                                {
                                    Field field = cls.getField(fieldNames[i]);
                                    field.setAccessible(true);
                                    field.set(obj, jsonobj.get(fieldNames[i]));
                                }
                            }
                            return obj;
                        }
                        catch (InstantiationException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (IllegalAccessException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (IllegalArgumentException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (SecurityException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (NoSuchFieldException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (JSONException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        catch (InvocationTargetException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        return null;
                    }
                });
            }
            catch (SecurityException ex)
            {
                ex.printStackTrace();
            }

        }
        return null;
    }
    
}