/*
 * Copyright (c) 2008, 2009, 2011 Oracle, Inc. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.  The Eclipse Public License is available
 * at http://www.eclipse.org/legal/epl-v10.html and the Eclipse Distribution License
 * is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package javax.persistence;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies the primary table for the annotated entity. Additional tables may be specified using
 * {@link SecondaryTable} or {@link SecondaryTables} annotation.
 * <p>
 * If no <code>Table</code> annotation is specified for an entity class, the default values apply.
 *
 * <pre>
 *    Example:
 * 
 *    &#064;Entity
 *    &#064;Table(name="CUST", schema="RECORDS")
 *    public class Customer { ... }
 * </pre>
 * @since Java Persistence 1.0
 */
@Target(TYPE)
@Retention(RUNTIME)
public @interface Table
{
    /**
     * (Optional) The name of the table. Defaults to the entity name.
     * @return name
     */
    String name() default "";

    /**
     * (Optional) The catalog of the table. Defaults to the default catalog.
     * @return catalog
     */
    String catalog() default "";

    /**
     * (Optional) The schema of the table. Defaults to the default schema for user.
     * @return schema
     */
    String schema() default "";

    /**
     * (Optional) Unique constraints that are to be placed on the table. These are only used if table
     * generation is in effect. These constraints apply in addition to any constraints specified by the
     * <code>Column</code> and <code>JoinColumn</code> annotations and constraints entailed by primary key
     * mappings. Defaults to no additional constraints.
     * @return unique constrs
     */
    UniqueConstraint[] uniqueConstraints() default {};

    /**
     * (Optional) Indexes for the table. These are only used if table generation is in effect. Defaults to no
     * additional indexes.
     * @return The indexes
     */
    Index[] indexes() default {};
}
