/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.ditto.model.base.common;

import java.text.MessageFormat;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.function.LongFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import org.eclipse.ditto.model.base.common.ConditionChecker;

@Immutable
public final class DittoDuration
implements CharSequence {
    private final long amount;
    private final DittoTimeUnit dittoTimeUnit;

    private DittoDuration(long amount, DittoTimeUnit dittoTimeUnit) {
        this.amount = amount;
        this.dittoTimeUnit = dittoTimeUnit;
    }

    public static DittoDuration of(Duration duration) {
        ConditionChecker.checkNotNull(duration, "duration");
        ConditionChecker.checkArgument(duration, d -> !d.isNegative(), () -> MessageFormat.format("The duration must not be negative but was <{0}>!", duration));
        return new DittoDuration(duration.toMillis(), DittoTimeUnit.MILLISECONDS);
    }

    public static DittoDuration parseDuration(CharSequence duration) {
        return DittoDuration.parseDuration(ConditionChecker.checkNotNull(duration, "duration"), DittoTimeUnit.values());
    }

    private static DittoDuration parseDuration(CharSequence duration, DittoTimeUnit[] dittoTimeUnits) {
        DittoTimeUnit timeUnit = null;
        Long durationValue = null;
        for (int i = 0; null == durationValue && i < dittoTimeUnits.length; ++i) {
            timeUnit = dittoTimeUnits[i];
            durationValue = DittoDuration.parseDurationRegexBased(duration, timeUnit);
        }
        if (null == durationValue) {
            timeUnit = DittoTimeUnit.SECONDS_IMPLICIT;
            durationValue = DittoDuration.parseDurationPlain(duration, timeUnit.getSuffix());
        }
        return new DittoDuration(durationValue, timeUnit);
    }

    @Nullable
    private static Long parseDurationRegexBased(CharSequence duration, DittoTimeUnit dittoTimeUnit) {
        Long result = null;
        Matcher matcher = dittoTimeUnit.getRegexMatcher(duration);
        if (matcher.matches()) {
            result = DittoDuration.parseDurationPlain(matcher.group("amount"), dittoTimeUnit.getSuffix());
        }
        return result;
    }

    private static Long parseDurationPlain(CharSequence charSequence, CharSequence timeUnitSuffix) {
        long result = Long.parseLong(charSequence.toString());
        ConditionChecker.checkArgument(result, r -> 0L <= r, () -> {
            String msgPattern = "The duration must not be negative but was <{0}{1}>!";
            return MessageFormat.format("The duration must not be negative but was <{0}{1}>!", charSequence, timeUnitSuffix);
        });
        return result;
    }

    public boolean isZero() {
        return 0L == this.amount;
    }

    public Duration getDuration() {
        return this.dittoTimeUnit.getJavaDuration(this.amount);
    }

    public ChronoUnit getChronoUnit() {
        return this.dittoTimeUnit.getChronoUnit();
    }

    public DittoDuration setAmount(Duration duration) {
        long localAmount;
        Duration unit = this.dittoTimeUnit.getChronoUnit().getDuration();
        long seconds = duration.getSeconds();
        long nanoseconds = duration.getNano();
        long unitSeconds = unit.getSeconds();
        long unitNanoseconds = unit.getNano();
        if (unitSeconds != 0L) {
            localAmount = Math.max(1L, seconds / unitSeconds);
        } else {
            long withOverflow = seconds * (1000000000L / unitNanoseconds) + nanoseconds / unitNanoseconds;
            localAmount = Math.max(1L, withOverflow);
        }
        return new DittoDuration(localAmount, this.dittoTimeUnit);
    }

    @Override
    public int length() {
        return this.toString().length();
    }

    @Override
    public char charAt(int index) {
        return this.toString().charAt(index);
    }

    @Override
    public CharSequence subSequence(int start, int end) {
        return this.toString().subSequence(start, end);
    }

    @Override
    public String toString() {
        return this.amount + this.dittoTimeUnit.getSuffix();
    }

    public boolean equals(@Nullable Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DittoDuration that = (DittoDuration)o;
        return this.amount == that.amount && this.dittoTimeUnit == that.dittoTimeUnit;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.amount, this.dittoTimeUnit});
    }

    private static enum DittoTimeUnit {
        SECONDS("s", Duration::ofSeconds, ChronoUnit.SECONDS),
        SECONDS_IMPLICIT("", Duration::ofSeconds, ChronoUnit.SECONDS),
        MILLISECONDS("ms", Duration::ofMillis, ChronoUnit.MILLIS),
        MINUTES("m", Duration::ofMinutes, ChronoUnit.MINUTES),
        HOURS("h", Duration::ofHours, ChronoUnit.HOURS);

        private final String suffix;
        private final LongFunction<Duration> toJavaDuration;
        private final ChronoUnit chronoUnit;
        private final Pattern regexPattern;

        private DittoTimeUnit(String suffix, LongFunction<Duration> toJavaDuration, ChronoUnit chronoUnit) {
            this.suffix = suffix;
            this.toJavaDuration = toJavaDuration;
            this.chronoUnit = chronoUnit;
            this.regexPattern = Pattern.compile("(?<amount>[+-]?\\d++)(?<unit>" + suffix + ")");
        }

        private Matcher getRegexMatcher(CharSequence duration) {
            return this.regexPattern.matcher(duration);
        }

        private String getSuffix() {
            return this.suffix;
        }

        private Duration getJavaDuration(long amount) {
            return this.toJavaDuration.apply(amount);
        }

        private ChronoUnit getChronoUnit() {
            return this.chronoUnit;
        }
    }
}

