/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.edc.connector.core.policy;

import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalUnit;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.edc.policy.engine.spi.AtomicConstraintFunction;
import org.eclipse.edc.policy.engine.spi.PolicyContext;
import org.eclipse.edc.policy.model.Operator;
import org.eclipse.edc.policy.model.Permission;
import org.eclipse.edc.spi.EdcException;
import org.eclipse.edc.spi.types.domain.agreement.ContractAgreement;

public class ContractExpiryCheckFunction
implements AtomicConstraintFunction<Permission> {
    public static final String CONTRACT_EXPIRY_EVALUATION_KEY = "https://w3id.org/edc/v0.0.1/ns/inForceDate";
    private static final String EXPRESSION_REGEX = "(contract[A,a]greement)\\+(-?[0-9]+)(s|m|h|d)";
    private static final int REGEX_GROUP_NUMERIC = 2;
    private static final int REGEX_GROUP_UNIT = 3;

    public boolean evaluate(Operator operator, Object rightValue, Permission rule, PolicyContext context) {
        if (!(rightValue instanceof String)) {
            context.reportProblem("Right-value expected to be String but was " + rightValue.getClass());
            return false;
        }
        try {
            Instant now = this.getContextData(Instant.class, context);
            String rightValueStr = (String)rightValue;
            Instant bound = this.asInstant(rightValueStr);
            if (bound != null) {
                return this.checkFixedPeriod(now, operator, bound);
            }
            Duration duration = this.asDuration(rightValueStr);
            if (duration != null) {
                ContractAgreement agreement = this.getContextData(ContractAgreement.class, context);
                Instant signingDate = Instant.ofEpochSecond(agreement.getContractSigningDate());
                return this.checkFixedPeriod(now, operator, signingDate.plus(duration));
            }
            context.reportProblem(String.format("Unsupported right-value, expected either an ISO-8061 String or a expression matching '%s', but got '%s'", CONTRACT_EXPIRY_EVALUATION_KEY, rightValueStr));
        }
        catch (NullPointerException | DateTimeParseException ex) {
            context.reportProblem(ex.getMessage());
        }
        return false;
    }

    private Duration asDuration(String rightValueStr) {
        Matcher matcher = Pattern.compile(EXPRESSION_REGEX).matcher(rightValueStr);
        if (matcher.matches()) {
            int number = Integer.parseInt(matcher.group(2));
            String unit = matcher.group(3);
            return Duration.of(number, this.asChrono(unit));
        }
        return null;
    }

    private TemporalUnit asChrono(String unit) {
        return switch (unit) {
            case "s" -> ChronoUnit.SECONDS;
            case "m" -> ChronoUnit.MINUTES;
            case "h" -> ChronoUnit.HOURS;
            case "d" -> ChronoUnit.DAYS;
            default -> throw new EdcException(String.format("Cannot parse '%s' into a ChronoUnit", unit));
        };
    }

    private boolean checkFixedPeriod(Instant now, Operator operator, Instant bound) {
        int comparison = now.compareTo(bound);
        return switch (operator) {
            case Operator.EQ -> {
                if (comparison == 0) {
                    yield true;
                }
                yield false;
            }
            case Operator.NEQ -> {
                if (comparison != 0) {
                    yield true;
                }
                yield false;
            }
            case Operator.GT -> {
                if (comparison > 0) {
                    yield true;
                }
                yield false;
            }
            case Operator.GEQ -> {
                if (comparison >= 0) {
                    yield true;
                }
                yield false;
            }
            case Operator.LT -> {
                if (comparison < 0) {
                    yield true;
                }
                yield false;
            }
            case Operator.LEQ -> {
                if (comparison <= 0) {
                    yield true;
                }
                yield false;
            }
            default -> throw new IllegalStateException("Unexpected value: " + operator);
        };
    }

    private Instant asInstant(String isoString) {
        try {
            return Instant.parse(isoString);
        }
        catch (DateTimeParseException e) {
            return null;
        }
    }

    private <R> R getContextData(Class<R> clazz, PolicyContext context) {
        return (R)Objects.requireNonNull(context.getContextData(clazz), clazz.getSimpleName());
    }
}

