/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http2;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.http2.CloseState;
import org.eclipse.jetty.http2.ErrorCode;
import org.eclipse.jetty.http2.FlowControlStrategy;
import org.eclipse.jetty.http2.HTTP2Flusher;
import org.eclipse.jetty.http2.HTTP2Stream;
import org.eclipse.jetty.http2.ISession;
import org.eclipse.jetty.http2.IStream;
import org.eclipse.jetty.http2.api.Session;
import org.eclipse.jetty.http2.api.Stream;
import org.eclipse.jetty.http2.frames.DataFrame;
import org.eclipse.jetty.http2.frames.DisconnectFrame;
import org.eclipse.jetty.http2.frames.Frame;
import org.eclipse.jetty.http2.frames.FrameType;
import org.eclipse.jetty.http2.frames.GoAwayFrame;
import org.eclipse.jetty.http2.frames.HeadersFrame;
import org.eclipse.jetty.http2.frames.PingFrame;
import org.eclipse.jetty.http2.frames.PriorityFrame;
import org.eclipse.jetty.http2.frames.PushPromiseFrame;
import org.eclipse.jetty.http2.frames.ResetFrame;
import org.eclipse.jetty.http2.frames.SettingsFrame;
import org.eclipse.jetty.http2.frames.WindowUpdateFrame;
import org.eclipse.jetty.http2.generator.Generator;
import org.eclipse.jetty.http2.parser.Parser;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.util.Atomics;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.CountingCallback;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.Scheduler;

public abstract class HTTP2Session
implements ISession,
Parser.Listener {
    private static final Logger LOG = Log.getLogger(HTTP2Session.class);
    private final ConcurrentMap<Integer, IStream> streams = new ConcurrentHashMap<Integer, IStream>();
    private final AtomicInteger streamIds = new AtomicInteger();
    private final AtomicInteger lastStreamId = new AtomicInteger();
    private final AtomicInteger localStreamCount = new AtomicInteger();
    private final AtomicInteger remoteStreamCount = new AtomicInteger();
    private final AtomicInteger sendWindow = new AtomicInteger();
    private final AtomicInteger recvWindow = new AtomicInteger();
    private final AtomicReference<CloseState> closed = new AtomicReference<CloseState>(CloseState.NOT_CLOSED);
    private final Scheduler scheduler;
    private final EndPoint endPoint;
    private final Generator generator;
    private final Session.Listener listener;
    private final FlowControlStrategy flowControl;
    private final HTTP2Flusher flusher;
    private int maxLocalStreams;
    private int maxRemoteStreams;
    private long streamIdleTimeout;
    private boolean pushEnabled;

    public HTTP2Session(Scheduler scheduler, EndPoint endPoint, Generator generator, Session.Listener listener, FlowControlStrategy flowControl, int initialStreamId) {
        this.scheduler = scheduler;
        this.endPoint = endPoint;
        this.generator = generator;
        this.listener = listener;
        this.flowControl = flowControl;
        this.flusher = new HTTP2Flusher(this);
        this.maxLocalStreams = -1;
        this.maxRemoteStreams = -1;
        this.streamIds.set(initialStreamId);
        this.streamIdleTimeout = endPoint.getIdleTimeout();
        this.sendWindow.set(65535);
        this.recvWindow.set(65535);
        this.pushEnabled = true;
    }

    public FlowControlStrategy getFlowControlStrategy() {
        return this.flowControl;
    }

    public int getMaxLocalStreams() {
        return this.maxLocalStreams;
    }

    public void setMaxLocalStreams(int maxLocalStreams) {
        this.maxLocalStreams = maxLocalStreams;
    }

    public int getMaxRemoteStreams() {
        return this.maxRemoteStreams;
    }

    public void setMaxRemoteStreams(int maxRemoteStreams) {
        this.maxRemoteStreams = maxRemoteStreams;
    }

    public long getStreamIdleTimeout() {
        return this.streamIdleTimeout;
    }

    public void setStreamIdleTimeout(long streamIdleTimeout) {
        this.streamIdleTimeout = streamIdleTimeout;
    }

    public EndPoint getEndPoint() {
        return this.endPoint;
    }

    public Generator getGenerator() {
        return this.generator;
    }

    @Override
    public void onData(DataFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        int streamId = frame.getStreamId();
        final IStream stream = this.getStream(streamId);
        final int flowControlLength = frame.remaining() + frame.padding();
        this.flowControl.onDataReceived(this, stream, flowControlLength);
        if (stream != null) {
            if (this.getRecvWindow() < 0) {
                this.close(ErrorCode.FLOW_CONTROL_ERROR.code, "session_window_exceeded", Callback.NOOP);
            } else {
                stream.process(frame, new Callback(){

                    public void succeeded() {
                        HTTP2Session.this.flowControl.onDataConsumed(HTTP2Session.this, stream, flowControlLength);
                    }

                    public void failed(Throwable x) {
                        HTTP2Session.this.flowControl.onDataConsumed(HTTP2Session.this, stream, flowControlLength);
                    }
                });
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Ignoring {}, stream #{} not found", new Object[]{frame, streamId});
            }
            this.flowControl.onDataConsumed(this, null, flowControlLength);
        }
    }

    @Override
    public abstract void onHeaders(HeadersFrame var1);

    @Override
    public void onPriority(PriorityFrame frame) {
    }

    @Override
    public void onReset(ResetFrame frame) {
        IStream stream;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if ((stream = this.getStream(frame.getStreamId())) != null) {
            stream.process(frame, Callback.NOOP);
        } else {
            this.notifyReset(this, frame);
        }
    }

    @Override
    public void onSettings(SettingsFrame frame) {
        this.onSettings(frame, true);
    }

    public void onSettings(SettingsFrame frame, boolean reply) {
        if (frame.isReply()) {
            return;
        }
        block8: for (Map.Entry<Integer, Integer> entry : frame.getSettings().entrySet()) {
            int key = entry.getKey();
            int value = entry.getValue();
            switch (key) {
                case 1: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update HPACK header table size to {}", (long)value);
                    }
                    this.generator.setHeaderTableSize(value);
                    continue block8;
                }
                case 2: {
                    if (value != 0 && value != 1) {
                        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "invalid_settings_enable_push");
                        return;
                    }
                    this.pushEnabled = value == 1;
                    continue block8;
                }
                case 3: {
                    this.maxLocalStreams = value;
                    if (!LOG.isDebugEnabled()) continue block8;
                    LOG.debug("Update max local concurrent streams to {}", (long)this.maxLocalStreams);
                    continue block8;
                }
                case 4: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update initial window size to {}", (long)value);
                    }
                    this.flowControl.updateInitialStreamWindow(this, value, false);
                    continue block8;
                }
                case 5: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update max frame size to {}", (long)value);
                    }
                    if (value < 16384 || value > 0xFFFFFF) {
                        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "invalid_settings_max_frame_size");
                        return;
                    }
                    this.generator.setMaxFrameSize(value);
                    continue block8;
                }
                case 6: {
                    LOG.warn("NOT IMPLEMENTED max header list size to {}", new Object[]{value});
                    continue block8;
                }
            }
            LOG.debug("Unknown setting {}:{}", new Object[]{key, value});
        }
        this.notifySettings(this, frame);
        if (reply) {
            SettingsFrame replyFrame = new SettingsFrame(Collections.emptyMap(), true);
            this.settings(replyFrame, Callback.NOOP);
        }
    }

    @Override
    public void onPing(PingFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if (frame.isReply()) {
            this.notifyPing(this, frame);
        } else {
            PingFrame reply = new PingFrame(frame.getPayload(), true);
            this.control(null, Callback.NOOP, reply);
        }
    }

    @Override
    public void onGoAway(final GoAwayFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.REMOTELY_CLOSED)) continue block3;
                    this.control(null, new Callback(){

                        public void succeeded() {
                            HTTP2Session.this.notifyClose(HTTP2Session.this, frame);
                        }

                        public void failed(Throwable x) {
                            HTTP2Session.this.notifyClose(HTTP2Session.this, frame);
                        }
                    }, new DisconnectFrame());
                    return;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignored {}, already closed", new Object[]{frame});
        }
    }

    @Override
    public void onWindowUpdate(WindowUpdateFrame frame) {
        int streamId;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if ((streamId = frame.getStreamId()) > 0) {
            IStream stream = this.getStream(streamId);
            if (stream != null) {
                this.onWindowUpdate(stream, frame);
            }
        } else {
            this.onWindowUpdate(null, frame);
        }
    }

    @Override
    public void onConnectionFailure(int error, String reason) {
        this.close(error, reason, Callback.NOOP);
        this.notifyFailure(this, new IOException(String.format("%d/%s", error, reason)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void newStream(HeadersFrame frame, Promise<Stream> promise, Stream.Listener listener) {
        boolean queued;
        HTTP2Session hTTP2Session = this;
        synchronized (hTTP2Session) {
            int streamId = this.streamIds.getAndAdd(2);
            PriorityFrame priority = frame.getPriority();
            priority = priority == null ? null : new PriorityFrame(streamId, priority.getDependentStreamId(), priority.getWeight(), priority.isExclusive());
            frame = new HeadersFrame(streamId, frame.getMetaData(), priority, frame.isEndStream());
            IStream stream = this.createLocalStream(streamId, promise);
            if (stream == null) {
                return;
            }
            stream.setListener(listener);
            ControlEntry entry = new ControlEntry(frame, stream, new PromiseCallback(promise, stream));
            queued = this.flusher.append(entry);
        }
        if (queued) {
            this.flusher.iterate();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void push(IStream stream, Promise<Stream> promise, PushPromiseFrame frame, Stream.Listener listener) {
        boolean queued;
        HTTP2Session hTTP2Session = this;
        synchronized (hTTP2Session) {
            int streamId = this.streamIds.getAndAdd(2);
            frame = new PushPromiseFrame(frame.getStreamId(), streamId, frame.getMetaData());
            IStream pushStream = this.createLocalStream(streamId, promise);
            if (pushStream == null) {
                return;
            }
            pushStream.setListener(listener);
            ControlEntry entry = new ControlEntry(frame, pushStream, new PromiseCallback(promise, pushStream));
            queued = this.flusher.append(entry);
        }
        if (queued) {
            this.flusher.iterate();
        }
    }

    @Override
    public void settings(SettingsFrame frame, Callback callback) {
        this.control(null, callback, frame);
    }

    @Override
    public void ping(PingFrame frame, Callback callback) {
        if (frame.isReply()) {
            callback.failed((Throwable)new IllegalArgumentException());
        } else {
            this.control(null, callback, frame);
        }
    }

    protected void reset(ResetFrame frame, Callback callback) {
        this.control(this.getStream(frame.getStreamId()), callback, frame);
    }

    @Override
    public boolean close(int error, String reason, Callback callback) {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.LOCALLY_CLOSED)) continue block3;
                    byte[] payload = reason == null ? null : reason.getBytes(StandardCharsets.UTF_8);
                    GoAwayFrame frame = new GoAwayFrame(this.lastStreamId.get(), error, payload);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Sending {}", new Object[]{frame});
                    }
                    this.control(null, callback, frame);
                    return true;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignoring close {}/{}, already closed", new Object[]{error, reason});
        }
        callback.succeeded();
        return false;
    }

    @Override
    public boolean isClosed() {
        return this.closed.get() != CloseState.NOT_CLOSED;
    }

    private void control(IStream stream, Callback callback, Frame frame) {
        this.frames(stream, callback, frame, Frame.EMPTY_ARRAY);
    }

    @Override
    public void frames(IStream stream, Callback callback, Frame frame, Frame ... frames) {
        int length = frames.length;
        if (length == 0) {
            this.frame(new ControlEntry(frame, stream, callback), true);
        } else {
            callback = new CountingCallback(callback, 1 + length);
            this.frame(new ControlEntry(frame, stream, callback), false);
            for (int i = 1; i <= length; ++i) {
                this.frame(new ControlEntry(frames[i - 1], stream, callback), i == length);
            }
        }
    }

    @Override
    public void data(IStream stream, Callback callback, DataFrame frame) {
        this.frame(new DataEntry(frame, stream, callback), true);
    }

    private void frame(HTTP2Flusher.Entry entry, boolean flush) {
        boolean queued;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Sending {}", new Object[]{entry.frame});
        }
        boolean bl = queued = entry.frame.getType() == FrameType.PING ? this.flusher.prepend(entry) : this.flusher.append(entry);
        if (queued && flush) {
            this.flusher.iterate();
        }
    }

    protected IStream createLocalStream(int streamId, Promise<Stream> promise) {
        int localCount;
        do {
            localCount = this.localStreamCount.get();
            int maxCount = this.maxLocalStreams;
            if (maxCount < 0 || localCount < maxCount) continue;
            promise.failed((Throwable)new IllegalStateException("Max local stream count " + maxCount + " exceeded"));
            return null;
        } while (!this.localStreamCount.compareAndSet(localCount, localCount + 1));
        IStream stream = this.newStream(streamId);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream, true);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created local {}", new Object[]{stream});
            }
            return stream;
        }
        promise.failed((Throwable)new IllegalStateException("Duplicate stream " + streamId));
        return null;
    }

    protected IStream createRemoteStream(int streamId) {
        int remoteCount;
        do {
            remoteCount = this.remoteStreamCount.get();
            int maxCount = this.getMaxRemoteStreams();
            if (maxCount < 0 || remoteCount < maxCount) continue;
            this.reset(new ResetFrame(streamId, ErrorCode.REFUSED_STREAM_ERROR.code), Callback.NOOP);
            return null;
        } while (!this.remoteStreamCount.compareAndSet(remoteCount, remoteCount + 1));
        IStream stream = this.newStream(streamId);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            this.updateLastStreamId(streamId);
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream, false);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created remote {}", new Object[]{stream});
            }
            return stream;
        }
        this.close(ErrorCode.PROTOCOL_ERROR.code, "duplicate_stream", Callback.NOOP);
        return null;
    }

    public IStream createUpgradeStream() {
        this.remoteStreamCount.incrementAndGet();
        IStream stream = this.newStream(1);
        this.streams.put(1, stream);
        this.updateLastStreamId(1);
        stream.setIdleTimeout(this.getStreamIdleTimeout());
        this.flowControl.onStreamCreated(stream, false);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Created upgrade {}", new Object[]{stream});
        }
        return stream;
    }

    protected IStream newStream(int streamId) {
        return new HTTP2Stream(this.scheduler, this, streamId);
    }

    @Override
    public void removeStream(IStream stream, boolean local) {
        IStream removed = (IStream)this.streams.remove(stream.getId());
        if (removed != null) {
            assert (removed == stream);
            if (local) {
                this.localStreamCount.decrementAndGet();
            } else {
                this.remoteStreamCount.decrementAndGet();
            }
            this.flowControl.onStreamDestroyed(stream, local);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Removed {}", new Object[]{stream});
            }
        }
    }

    @Override
    public Collection<Stream> getStreams() {
        ArrayList<Stream> result = new ArrayList<Stream>();
        result.addAll(this.streams.values());
        return result;
    }

    @Override
    public IStream getStream(int streamId) {
        return (IStream)this.streams.get(streamId);
    }

    public int getSendWindow() {
        return this.sendWindow.get();
    }

    public int getRecvWindow() {
        return this.recvWindow.get();
    }

    @Override
    public int updateSendWindow(int delta) {
        return this.sendWindow.getAndAdd(delta);
    }

    @Override
    public int updateRecvWindow(int delta) {
        return this.recvWindow.getAndAdd(delta);
    }

    @Override
    public void onWindowUpdate(IStream stream, WindowUpdateFrame frame) {
        this.flusher.window(stream, frame);
    }

    @Override
    public boolean isPushEnabled() {
        return this.pushEnabled;
    }

    @Override
    public void onShutdown() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Shutting down {}", new Object[]{this});
        }
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Abrupt close for {}", new Object[]{this});
                }
                this.abort(new ClosedChannelException());
                break;
            }
            case LOCALLY_CLOSED: {
                this.control(null, Callback.NOOP, new DisconnectFrame());
                break;
            }
            case REMOTELY_CLOSED: {
                break;
            }
        }
    }

    @Override
    public void onIdleTimeout() {
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                this.close(ErrorCode.NO_ERROR.code, "idle_timeout", Callback.NOOP);
                break;
            }
            case LOCALLY_CLOSED: 
            case REMOTELY_CLOSED: {
                this.abort(new TimeoutException());
                break;
            }
        }
    }

    @Override
    public void onFrame(Frame frame) {
        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "upgrade");
    }

    public void disconnect() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Disconnecting {}", new Object[]{this});
        }
        this.endPoint.close();
    }

    private void terminate() {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: 
                case LOCALLY_CLOSED: 
                case REMOTELY_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.CLOSED)) continue block3;
                    this.flusher.close();
                    for (IStream stream : this.streams.values()) {
                        stream.close();
                    }
                    this.streams.clear();
                    this.disconnect();
                    return;
                }
            }
            break;
        }
    }

    protected void abort(Throwable failure) {
        this.terminate();
        this.notifyFailure(this, failure);
    }

    public boolean isDisconnected() {
        return !this.endPoint.isOpen();
    }

    private void updateLastStreamId(int streamId) {
        Atomics.updateMax((AtomicInteger)this.lastStreamId, (int)streamId);
    }

    protected Stream.Listener notifyNewStream(Stream stream, HeadersFrame frame) {
        try {
            return this.listener.onNewStream(stream, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
            return null;
        }
    }

    protected void notifySettings(Session session, SettingsFrame frame) {
        try {
            this.listener.onSettings(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyPing(Session session, PingFrame frame) {
        try {
            this.listener.onPing(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyReset(Session session, ResetFrame frame) {
        try {
            this.listener.onReset(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyClose(Session session, GoAwayFrame frame) {
        try {
            this.listener.onClose(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyFailure(Session session, Throwable failure) {
        try {
            this.listener.onFailure(session, failure);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    public String toString() {
        return String.format("%s@%x{queueSize=%d,sendWindow=%s,recvWindow=%s,streams=%d,%s}", this.getClass().getSimpleName(), this.hashCode(), this.flusher.getQueueSize(), this.sendWindow, this.recvWindow, this.streams.size(), this.closed);
    }

    private class PromiseCallback<C>
    implements Callback {
        private final Promise<C> promise;
        private final C value;

        private PromiseCallback(Promise<C> promise, C value) {
            this.promise = promise;
            this.value = value;
        }

        public void succeeded() {
            this.promise.succeeded(this.value);
        }

        public void failed(Throwable x) {
            this.promise.failed(x);
        }
    }

    private class DataEntry
    extends HTTP2Flusher.Entry {
        private int length;

        private DataEntry(DataFrame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
        }

        @Override
        public int dataRemaining() {
            return ((DataFrame)this.frame).remaining();
        }

        @Override
        public Throwable generate(ByteBufferPool.Lease lease) {
            try {
                int flowControlLength = this.dataRemaining();
                int sessionSendWindow = HTTP2Session.this.getSendWindow();
                if (sessionSendWindow < 0) {
                    throw new IllegalStateException();
                }
                int streamSendWindow = this.stream.updateSendWindow(0);
                if (streamSendWindow < 0) {
                    throw new IllegalStateException();
                }
                int window = Math.min(streamSendWindow, sessionSendWindow);
                int length = this.length = Math.min(flowControlLength, window);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Generated {}, length/window={}/{}", new Object[]{this.frame, length, window});
                }
                HTTP2Session.this.generator.data(lease, (DataFrame)this.frame, length);
                HTTP2Session.this.flowControl.onDataSending(this.stream, length);
                return null;
            }
            catch (Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failure generating frame " + this.frame, x);
                }
                return x;
            }
        }

        public void succeeded() {
            HTTP2Session.this.flowControl.onDataSent(this.stream, this.length);
            DataFrame dataFrame = (DataFrame)this.frame;
            if (dataFrame.remaining() > 0) {
                HTTP2Session.this.flusher.prepend(this);
            } else {
                if (this.stream.updateClose(dataFrame.isEndStream(), true)) {
                    HTTP2Session.this.removeStream(this.stream, true);
                }
                this.callback.succeeded();
            }
        }
    }

    private class ControlEntry
    extends HTTP2Flusher.Entry {
        private ControlEntry(Frame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
        }

        @Override
        public Throwable generate(ByteBufferPool.Lease lease) {
            try {
                HTTP2Session.this.generator.control(lease, this.frame);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Generated {}", new Object[]{this.frame});
                }
                this.prepare();
                return null;
            }
            catch (Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Failure generating frame " + this.frame, x);
                }
                return x;
            }
        }

        private void prepare() {
            switch (this.frame.getType()) {
                case SETTINGS: {
                    SettingsFrame settingsFrame = (SettingsFrame)this.frame;
                    Integer initialWindow = settingsFrame.getSettings().get(4);
                    if (initialWindow == null) break;
                    HTTP2Session.this.flowControl.updateInitialStreamWindow(HTTP2Session.this, initialWindow, true);
                    break;
                }
            }
        }

        public void succeeded() {
            switch (this.frame.getType()) {
                case HEADERS: {
                    HeadersFrame headersFrame = (HeadersFrame)this.frame;
                    if (!this.stream.updateClose(headersFrame.isEndStream(), true)) break;
                    HTTP2Session.this.removeStream(this.stream, true);
                    break;
                }
                case RST_STREAM: {
                    if (this.stream == null) break;
                    this.stream.close();
                    HTTP2Session.this.removeStream(this.stream, true);
                    break;
                }
                case PUSH_PROMISE: {
                    this.stream.updateClose(true, false);
                    break;
                }
                case GO_AWAY: {
                    HTTP2Session.this.getEndPoint().shutdownOutput();
                    break;
                }
                case WINDOW_UPDATE: {
                    HTTP2Session.this.flowControl.windowUpdate(HTTP2Session.this, this.stream, (WindowUpdateFrame)this.frame);
                    break;
                }
                case DISCONNECT: {
                    HTTP2Session.this.terminate();
                    break;
                }
            }
            this.callback.succeeded();
        }
    }
}

