/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http2.hpack;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashSet;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.HttpScheme;
import org.eclipse.jetty.http.HttpStatus;
import org.eclipse.jetty.http.HttpVersion;
import org.eclipse.jetty.http.MetaData;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.http2.hpack.HpackContext;
import org.eclipse.jetty.http2.hpack.HpackException;
import org.eclipse.jetty.http2.hpack.Huffman;
import org.eclipse.jetty.http2.hpack.NBitInteger;
import org.eclipse.jetty.util.StringUtil;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class HpackEncoder {
    private static final Logger LOG = Log.getLogger(HpackEncoder.class);
    private static final HttpField[] STATUSES = new HttpField[599];
    static final EnumSet<HttpHeader> DO_NOT_HUFFMAN = EnumSet.of(HttpHeader.AUTHORIZATION, HttpHeader.CONTENT_MD5, HttpHeader.PROXY_AUTHENTICATE, HttpHeader.PROXY_AUTHORIZATION);
    static final EnumSet<HttpHeader> DO_NOT_INDEX = EnumSet.of(HttpHeader.AUTHORIZATION, new HttpHeader[]{HttpHeader.CONTENT_MD5, HttpHeader.CONTENT_RANGE, HttpHeader.ETAG, HttpHeader.IF_MODIFIED_SINCE, HttpHeader.IF_UNMODIFIED_SINCE, HttpHeader.IF_NONE_MATCH, HttpHeader.IF_RANGE, HttpHeader.IF_MATCH, HttpHeader.LOCATION, HttpHeader.RANGE, HttpHeader.RETRY_AFTER, HttpHeader.LAST_MODIFIED, HttpHeader.SET_COOKIE, HttpHeader.SET_COOKIE2});
    static final EnumSet<HttpHeader> NEVER_INDEX = EnumSet.of(HttpHeader.AUTHORIZATION, HttpHeader.SET_COOKIE, HttpHeader.SET_COOKIE2);
    private static final EnumSet<HttpHeader> IGNORED_HEADERS = EnumSet.of(HttpHeader.CONNECTION, HttpHeader.KEEP_ALIVE, HttpHeader.PROXY_CONNECTION, HttpHeader.TRANSFER_ENCODING, HttpHeader.UPGRADE);
    private static final PreEncodedHttpField TE_TRAILERS = new PreEncodedHttpField(HttpHeader.TE, "trailers");
    private static final PreEncodedHttpField C_SCHEME_HTTP = new PreEncodedHttpField(HttpHeader.C_SCHEME, "http");
    private static final PreEncodedHttpField C_SCHEME_HTTPS = new PreEncodedHttpField(HttpHeader.C_SCHEME, "https");
    private static final EnumMap<HttpMethod, PreEncodedHttpField> C_METHODS = new EnumMap(HttpMethod.class);
    private final HpackContext _context;
    private final boolean _debug;
    private int _remoteMaxDynamicTableSize;
    private int _localMaxDynamicTableSize;
    private int _maxHeaderListSize;
    private int _headerListSize;
    private boolean _validateEncoding = true;

    public HpackEncoder() {
        this(4096, 4096, -1);
    }

    public HpackEncoder(int localMaxDynamicTableSize) {
        this(localMaxDynamicTableSize, 4096, -1);
    }

    public HpackEncoder(int localMaxDynamicTableSize, int remoteMaxDynamicTableSize) {
        this(localMaxDynamicTableSize, remoteMaxDynamicTableSize, -1);
    }

    public HpackEncoder(int localMaxDynamicTableSize, int remoteMaxDynamicTableSize, int maxHeaderListSize) {
        this._context = new HpackContext(remoteMaxDynamicTableSize);
        this._remoteMaxDynamicTableSize = remoteMaxDynamicTableSize;
        this._localMaxDynamicTableSize = localMaxDynamicTableSize;
        this._maxHeaderListSize = maxHeaderListSize;
        this._debug = LOG.isDebugEnabled();
    }

    public int getMaxHeaderListSize() {
        return this._maxHeaderListSize;
    }

    public void setMaxHeaderListSize(int maxHeaderListSize) {
        this._maxHeaderListSize = maxHeaderListSize;
    }

    public HpackContext getHpackContext() {
        return this._context;
    }

    public void setRemoteMaxDynamicTableSize(int remoteMaxDynamicTableSize) {
        this._remoteMaxDynamicTableSize = remoteMaxDynamicTableSize;
    }

    public void setLocalMaxDynamicTableSize(int localMaxDynamicTableSize) {
        this._localMaxDynamicTableSize = localMaxDynamicTableSize;
    }

    public boolean isValidateEncoding() {
        return this._validateEncoding;
    }

    public void setValidateEncoding(boolean validateEncoding) {
        this._validateEncoding = validateEncoding;
    }

    public void encode(ByteBuffer buffer, MetaData metadata) throws HpackException {
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("CtxTbl[%x] encoding", this._context.hashCode()), new Object[0]);
            }
            HttpFields fields = metadata.getFields();
            if (this.isValidateEncoding() && fields != null) {
                for (HttpField field : fields) {
                    String name = field.getName();
                    char firstChar = name.charAt(0);
                    if (firstChar > ' ' && firstChar != ':') continue;
                    throw new HpackException.StreamException("Invalid header name: '%s'", name);
                }
            }
            this._headerListSize = 0;
            int pos = buffer.position();
            int maxDynamicTableSize = Math.min(this._remoteMaxDynamicTableSize, this._localMaxDynamicTableSize);
            if (maxDynamicTableSize != this._context.getMaxDynamicTableSize()) {
                this.encodeMaxDynamicTableSize(buffer, maxDynamicTableSize);
            }
            if (metadata.isRequest()) {
                MetaData.Request request = (MetaData.Request)metadata;
                String scheme = request.getURI().getScheme();
                this.encode(buffer, (HttpField)(HttpScheme.HTTPS.is(scheme) ? C_SCHEME_HTTPS : C_SCHEME_HTTP));
                String method = request.getMethod();
                HttpMethod httpMethod = method == null ? null : HttpMethod.fromString((String)method);
                HttpField methodField = (HttpField)C_METHODS.get(httpMethod);
                this.encode(buffer, methodField == null ? new HttpField(HttpHeader.C_METHOD, method) : methodField);
                this.encode(buffer, new HttpField(HttpHeader.C_AUTHORITY, request.getURI().getAuthority()));
                this.encode(buffer, new HttpField(HttpHeader.C_PATH, request.getURI().getPathQuery()));
            } else if (metadata.isResponse()) {
                HttpField status;
                MetaData.Response response = (MetaData.Response)metadata;
                int code = response.getStatus();
                HttpField httpField = status = code < STATUSES.length ? STATUSES[code] : null;
                if (status == null) {
                    status = new HttpField.IntValueHttpField(HttpHeader.C_STATUS, code);
                }
                this.encode(buffer, status);
            }
            if (fields != null) {
                long contentLength;
                HashSet<String> hopHeaders = null;
                for (Object value : fields.getCSV(HttpHeader.CONNECTION, false)) {
                    if (hopHeaders == null) {
                        hopHeaders = new HashSet<String>();
                    }
                    hopHeaders.add(StringUtil.asciiToLowerCase((String)value));
                }
                boolean contentLengthEncoded = false;
                for (HttpField field : fields) {
                    HttpHeader header = field.getHeader();
                    if (header != null && IGNORED_HEADERS.contains(header)) continue;
                    if (header == HttpHeader.TE) {
                        if (!field.contains("trailers")) continue;
                        this.encode(buffer, (HttpField)TE_TRAILERS);
                        continue;
                    }
                    String name = field.getLowerCaseName();
                    if (hopHeaders != null && hopHeaders.contains(name)) continue;
                    if (header == HttpHeader.CONTENT_LENGTH) {
                        contentLengthEncoded = true;
                    }
                    this.encode(buffer, field);
                }
                if (!contentLengthEncoded && (contentLength = metadata.getContentLength()) >= 0L) {
                    this.encode(buffer, new HttpField(HttpHeader.CONTENT_LENGTH, String.valueOf(contentLength)));
                }
            }
            if (this._maxHeaderListSize > 0 && this._headerListSize > this._maxHeaderListSize) {
                LOG.warn("Header list size too large {} > {} for {}", new Object[]{this._headerListSize, this._maxHeaderListSize});
                if (LOG.isDebugEnabled()) {
                    LOG.debug("metadata={}", new Object[]{metadata});
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("CtxTbl[%x] encoded %d octets", this._context.hashCode(), buffer.position() - pos), new Object[0]);
            }
        }
        catch (HpackException x) {
            throw x;
        }
        catch (Throwable x) {
            HpackException.SessionException failure = new HpackException.SessionException("Could not hpack encode %s", metadata);
            failure.initCause(x);
            throw failure;
        }
    }

    public void encodeMaxDynamicTableSize(ByteBuffer buffer, int maxDynamicTableSize) {
        if (maxDynamicTableSize > this._remoteMaxDynamicTableSize) {
            throw new IllegalArgumentException();
        }
        buffer.put((byte)32);
        NBitInteger.encode(buffer, 5, maxDynamicTableSize);
        this._context.resize(maxDynamicTableSize);
    }

    public void encode(ByteBuffer buffer, HttpField field) {
        if (field.getValue() == null) {
            field = new HttpField(field.getHeader(), field.getName(), "");
        }
        int fieldSize = field.getName().length() + field.getValue().length();
        this._headerListSize += fieldSize + 32;
        int p = this._debug ? buffer.position() : -1;
        String encoding = null;
        HpackContext.Entry entry = this._context.get(field);
        if (entry != null) {
            if (entry.isStatic()) {
                buffer.put(((HpackContext.StaticEntry)entry).getEncodedField());
                if (this._debug) {
                    encoding = "IdxFieldS1";
                }
            } else {
                int index = this._context.index(entry);
                buffer.put((byte)-128);
                NBitInteger.encode(buffer, 7, index);
                if (this._debug) {
                    encoding = "IdxField" + (entry.isStatic() ? "S" : "") + (1 + NBitInteger.octectsNeeded(7, index));
                }
            }
        } else {
            boolean indexed;
            HttpHeader header = field.getHeader();
            if (header == null) {
                HpackContext.Entry name = this._context.get(field.getName());
                if (field instanceof PreEncodedHttpField) {
                    int i = buffer.position();
                    ((PreEncodedHttpField)field).putTo(buffer, HttpVersion.HTTP_2);
                    byte b = buffer.get(i);
                    boolean bl = indexed = b < 0 || b >= 64;
                    if (this._debug) {
                        encoding = indexed ? "PreEncodedIdx" : "PreEncoded";
                    }
                } else if (name == null) {
                    indexed = true;
                    this.encodeName(buffer, (byte)64, 6, field.getName(), null);
                    HpackEncoder.encodeValue(buffer, true, field.getValue());
                    if (this._debug) {
                        encoding = "LitHuffNHuffVIdx";
                    }
                } else {
                    indexed = false;
                    this.encodeName(buffer, (byte)0, 4, field.getName(), null);
                    HpackEncoder.encodeValue(buffer, true, field.getValue());
                    if (this._debug) {
                        encoding = "LitHuffNHuffV!Idx";
                    }
                }
            } else {
                HpackContext.Entry name = this._context.get(header);
                if (field instanceof PreEncodedHttpField) {
                    int i = buffer.position();
                    ((PreEncodedHttpField)field).putTo(buffer, HttpVersion.HTTP_2);
                    byte b = buffer.get(i);
                    boolean bl = indexed = b < 0 || b >= 64;
                    if (this._debug) {
                        encoding = indexed ? "PreEncodedIdx" : "PreEncoded";
                    }
                } else if (DO_NOT_INDEX.contains(header)) {
                    indexed = false;
                    boolean neverIndex = NEVER_INDEX.contains(header);
                    boolean huffman = !DO_NOT_HUFFMAN.contains(header);
                    this.encodeName(buffer, neverIndex ? (byte)16 : 0, 4, header.asString(), name);
                    HpackEncoder.encodeValue(buffer, huffman, field.getValue());
                    if (this._debug) {
                        encoding = "Lit" + (name == null ? "HuffN" : "IdxN" + (name.isStatic() ? "S" : "") + (1 + NBitInteger.octectsNeeded(4, this._context.index(name)))) + (huffman ? "HuffV" : "LitV") + (neverIndex ? "!!Idx" : "!Idx");
                    }
                } else if (fieldSize >= this._context.getMaxDynamicTableSize() || header == HttpHeader.CONTENT_LENGTH && field.getValue().length() > 2) {
                    indexed = false;
                    this.encodeName(buffer, (byte)0, 4, header.asString(), name);
                    HpackEncoder.encodeValue(buffer, true, field.getValue());
                    if (this._debug) {
                        encoding = "LitIdxNS" + (1 + NBitInteger.octectsNeeded(4, this._context.index(name))) + "HuffV!Idx";
                    }
                } else {
                    indexed = true;
                    boolean huffman = !DO_NOT_HUFFMAN.contains(header);
                    this.encodeName(buffer, (byte)64, 6, header.asString(), name);
                    HpackEncoder.encodeValue(buffer, huffman, field.getValue());
                    if (this._debug) {
                        encoding = (name == null ? "LitHuffN" : "LitIdxN" + (name.isStatic() ? "S" : "") + (1 + NBitInteger.octectsNeeded(6, this._context.index(name)))) + (huffman ? "HuffVIdx" : "LitVIdx");
                    }
                }
            }
            if (indexed) {
                this._context.add(field);
            }
        }
        if (this._debug) {
            int e = buffer.position();
            if (LOG.isDebugEnabled()) {
                LOG.debug("encode {}:'{}' to '{}'", new Object[]{encoding, field, TypeUtil.toHexString((byte[])buffer.array(), (int)(buffer.arrayOffset() + p), (int)(e - p))});
            }
        }
    }

    private void encodeName(ByteBuffer buffer, byte mask, int bits, String name, HpackContext.Entry entry) {
        buffer.put(mask);
        if (entry == null) {
            buffer.put((byte)-128);
            NBitInteger.encode(buffer, 7, Huffman.octetsNeededLC(name));
            Huffman.encodeLC(buffer, name);
        } else {
            NBitInteger.encode(buffer, bits, this._context.index(entry));
        }
    }

    static void encodeValue(ByteBuffer buffer, boolean huffman, String value) {
        if (huffman) {
            buffer.put((byte)-128);
            int needed = Huffman.octetsNeeded(value);
            if (needed >= 0) {
                NBitInteger.encode(buffer, 7, needed);
                Huffman.encode(buffer, value);
            } else {
                byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
                NBitInteger.encode(buffer, 7, Huffman.octetsNeeded(bytes));
                Huffman.encode(buffer, bytes);
            }
        } else {
            buffer.put((byte)0).mark();
            NBitInteger.encode(buffer, 7, value.length());
            for (int i = 0; i < value.length(); ++i) {
                char c = value.charAt(i);
                if (c < ' ' || c > '\u007f') {
                    buffer.reset();
                    byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
                    NBitInteger.encode(buffer, 7, bytes.length);
                    buffer.put(bytes, 0, bytes.length);
                    return;
                }
                buffer.put((byte)c);
            }
        }
    }

    static {
        for (HttpStatus.Code code : HttpStatus.Code.values()) {
            HpackEncoder.STATUSES[code.getCode()] = new PreEncodedHttpField(HttpHeader.C_STATUS, Integer.toString(code.getCode()));
        }
        for (HttpStatus.Code code : HttpMethod.values()) {
            C_METHODS.put((HttpMethod)code, new PreEncodedHttpField(HttpHeader.C_METHOD, code.asString()));
        }
    }
}

