/*
 * Copyright (c) 1998, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Oracle - initial API and implementation from Oracle TopLink
package org.eclipse.persistence.internal.oxm.record;

import java.util.List;

import org.eclipse.persistence.internal.core.sessions.CoreAbstractSession;
import org.eclipse.persistence.internal.oxm.Constants;
import org.eclipse.persistence.internal.oxm.ContainerValue;
import org.eclipse.persistence.internal.oxm.NamespaceResolver;
import org.eclipse.persistence.internal.oxm.NodeValue;
import org.eclipse.persistence.internal.oxm.XPathFragment;
import org.eclipse.persistence.internal.oxm.XPathNode;
import org.eclipse.persistence.internal.oxm.mappings.Mapping;
import org.eclipse.persistence.oxm.sequenced.Setting;

/**
 * An implementation of MarshalContext for handling sequenced objects that
 * are mapped to XML.
 */
public class SequencedMarshalContext implements MarshalContext {

    private List<Setting> settings;
    private XPathFragment indexFragment;
    private Object value;

    public SequencedMarshalContext(List<Setting> settings) {
        super();
        indexFragment = new XPathFragment();
        this.settings = settings;
    }

    public SequencedMarshalContext(Object value) {
        super();
        indexFragment = new XPathFragment();
        this.value = value;
    }


    @Override
    public MarshalContext getMarshalContext(int index) {
        Setting setting = settings.get(index);
        List<Setting> children = setting.getChildren();
        if(null == children) {
            return new SequencedMarshalContext(setting.getValue());
        } else {
            return new SequencedMarshalContext(children);
        }
    }

    @Override
    public int getNonAttributeChildrenSize(XPathNode xPathNode) {
        if(null == settings) {
            return 0;
        }
        return settings.size();
    }

    @Override
    public Object getNonAttributeChild(int index, XPathNode xPathNode) {
        Setting setting = settings.get(index);
        if(null == setting.getName()) {
            return xPathNode.getAnyNode();
        } else {
            if (setting.getName().equals(Constants.TEXT)) {
                return xPathNode.getTextNode();
            } else {
                indexFragment.setLocalName(null);
                indexFragment.setXPath(setting.getName());
                indexFragment.setNamespaceURI(setting.getNamespaceURI());
                return xPathNode.getNonAttributeChildrenMap().get(indexFragment);

            }
        }
    }

    @Override
    public Object getAttributeValue(Object object, Mapping mapping) {
        return value;
    }

    @Override
    public boolean marshal(NodeValue nodeValue, XPathFragment xPathFragment, MarshalRecord marshalRecord, Object object, CoreAbstractSession session, NamespaceResolver namespaceResolver) {
        if(nodeValue.isContainerValue()) {
            ((ContainerValue)nodeValue).marshalSingleValue(xPathFragment, marshalRecord, object, value, session, namespaceResolver, this);
            return true;
        } else {
            return nodeValue.marshal(xPathFragment, marshalRecord, object, session, namespaceResolver, this);
        }
    }

    @Override
    public boolean marshal(NodeValue nodeValue, XPathFragment xPathFragment, MarshalRecord marshalRecord, Object object, CoreAbstractSession session, NamespaceResolver namespaceResolver, XPathFragment rootFragment) {
        if(nodeValue.isContainerValue()) {
            ((ContainerValue)nodeValue).marshalSingleValue(xPathFragment, marshalRecord, object, value, session, namespaceResolver, this);
            return true;
        } else {
            return nodeValue.marshal(xPathFragment, marshalRecord, object, session, namespaceResolver, this, rootFragment);
        }
    }


}
