/*
 * Copyright (c) 1998, 2021 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     James Sutherland - initial API and implementation from Oracle TopLink
package org.eclipse.persistence.transaction.jotm;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedActionException;

import jakarta.transaction.TransactionManager;

import org.eclipse.persistence.internal.security.PrivilegedAccessHelper;
import org.eclipse.persistence.internal.security.PrivilegedClassForName;
import org.eclipse.persistence.internal.security.PrivilegedGetMethod;
import org.eclipse.persistence.internal.security.PrivilegedMethodInvoker;
import org.eclipse.persistence.transaction.JTATransactionController;

/**
 * <p>
 * <b>Purpose</b>: TransactionController implementation for JOTM
 * <p>
 * <b>Description</b>: Implements the required behavior for controlling JTA
 * transactions in JOTM.
 *
 * @see org.eclipse.persistence.transaction.JTATransactionController
 */
public class JotmTransactionController extends JTATransactionController {
    // Class and method to execute to obtain the TransactionManager
    protected final static String TX_CURRENT_FACTORY_CLASS = "org.objectweb.jotm.Current";
    protected final static String TX_CURRENT_FACTORY_METHOD = "getCurrent";
    protected final static String TX_MANAGER_FACTORY_METHOD = "getTransactionManager";

    public JotmTransactionController() {
        super();
    }

    /**
     * INTERNAL:
     * Obtain and return the JTA TransactionManager on this platform
     */
    @Override
    protected TransactionManager acquireTransactionManager() throws Exception {
        if (PrivilegedAccessHelper.shouldUsePrivilegedAccess()){
            try{
                Class<? extends TransactionManager> clazz = AccessController.doPrivileged(new PrivilegedClassForName<>(TX_CURRENT_FACTORY_CLASS));
                Method method = AccessController.doPrivileged(new PrivilegedGetMethod(clazz, TX_CURRENT_FACTORY_METHOD, null, false));
                Method txMethod = AccessController.doPrivileged(new PrivilegedGetMethod(clazz, TX_MANAGER_FACTORY_METHOD, null, false));
                TransactionManager current = AccessController.doPrivileged(new PrivilegedMethodInvoker<>(method, null, null));
                return AccessController.doPrivileged(new PrivilegedMethodInvoker<>(txMethod, current, null));
            }catch (PrivilegedActionException ex){
                if (ex.getCause() instanceof ClassNotFoundException){
                    throw (ClassNotFoundException)ex.getCause();
                }
                if (ex.getCause() instanceof NoSuchMethodException){
                    throw (NoSuchMethodException)ex.getCause();
                }
                if (ex.getCause() instanceof IllegalAccessException){
                    throw (IllegalAccessException)ex.getCause();
                }
                if (ex.getCause() instanceof InvocationTargetException){
                    throw (InvocationTargetException)ex.getCause();
                }
                throw (RuntimeException) ex.getCause();
            }
        }else{
            Class<? extends TransactionManager> clazz = PrivilegedAccessHelper.getClassForName(TX_CURRENT_FACTORY_CLASS);
            Method method = PrivilegedAccessHelper.getMethod(clazz, TX_CURRENT_FACTORY_METHOD, null, false);
            Method txMethod = PrivilegedAccessHelper.getMethod(clazz, TX_MANAGER_FACTORY_METHOD, null, false);
            TransactionManager current = PrivilegedAccessHelper.invokeMethod(method, null, null);
            return PrivilegedAccessHelper.invokeMethod(txMethod, current, null);
        }
    }
}
